% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grangersearch.R
\name{granger_search}
\alias{granger_search}
\title{Exhaustive Pairwise Granger Causality Search}
\usage{
granger_search(
  .data,
  ...,
  lag = 1,
  alpha = 0.05,
  test = "F",
  include_insignificant = FALSE
)
}
\arguments{
\item{.data}{A data frame or tibble containing the time series variables.}

\item{...}{<\code{\link[dplyr:dplyr_tidy_select]{tidy-select}}> Columns to include
in the analysis. If empty, all numeric columns are used.}

\item{lag}{Integer or integer vector. The lag order(s) for VAR models.
If a vector (e.g., \code{1:4}), tests are performed at each lag and the
best (lowest p-value) result is returned for each pair. Default is 1.}

\item{alpha}{Numeric. Significance level for hypothesis testing. Default is 0.05.}

\item{test}{Character. Test type, currently only "F" supported. Default is "F".}

\item{include_insignificant}{Logical. If FALSE (default), only return
significant causal relationships. If TRUE, return all pairwise results.}
}
\value{
A tibble with one row per directed pair tested, containing:
\describe{
\item{cause}{Character. The potential cause variable name.}
\item{effect}{Character. The potential effect variable name.}
\item{statistic}{Numeric. The F-test statistic.}
\item{p.value}{Numeric. The p-value of the test.}
\item{significant}{Logical. Whether the result is significant at alpha.}
\item{lag}{Integer. The lag order used (best lag if multiple were tested).}
}
}
\description{
Performs Granger causality tests on all pairwise combinations of variables
in a dataset. This is the core "search" functionality of the package,
enabling discovery of causal relationships among multiple time series.
}
\details{
This function tests all \eqn{n(n-1)} directed pairs for \eqn{n} variables.
For each pair (X, Y), it tests whether X Granger-causes Y.

When multiple lags are specified (e.g., \code{lag = 1:4}), the function tests
each pair at every lag and returns the result with the lowest p-value.
This is useful for discovering the optimal lag structure.

The function is useful for exploratory analysis when you have multiple
time series and want to discover which variables have predictive relationships.
}
\section{Multiple Testing}{

When testing many pairs (and especially many lags), consider adjusting for
multiple comparisons. The returned p-values are unadjusted. You can apply
corrections such as Bonferroni or Benjamini-Hochberg using \code{\link[stats:p.adjust]{stats::p.adjust()}}.
}

\examples{
# Create dataset with multiple time series
set.seed(123)
n <- 100
df <- data.frame(
  A = cumsum(rnorm(n)),
  B = cumsum(rnorm(n)),
  C = cumsum(rnorm(n))
)
# B is caused by lagged A
df$B <- c(0, 0.7 * df$A[1:(n-1)]) + rnorm(n, sd = 0.5)

# Search for all causal relationships
granger_search(df)

# Include all results, not just significant ones
granger_search(df, include_insignificant = TRUE)

# Select specific columns
granger_search(df, A, B)

# Search across multiple lags (returns best lag for each pair)
granger_search(df, lag = 1:4)

# Search with specific lag
granger_search(df, lag = 2)

}
\seealso{
\code{\link[=granger_causality_test]{granger_causality_test()}} for testing a single pair.
}
