/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Tpkg.h"     

static herr_t H5T__set_offset(const H5T_t *dt, size_t offset);

int
H5Tget_offset(hid_t type_id)
{
    H5T_t *dt;
    int    ret_value;

    FUNC_ENTER_API(-1)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an atomic data type");

    
    if ((ret_value = H5T_get_offset(dt)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "can't get offset for specified datatype");

done:
    FUNC_LEAVE_API(ret_value)
} 

int
H5T_get_offset(const H5T_t *dt)
{
    int ret_value = -1; 

    FUNC_ENTER_NOAPI(-1)

    
    while (dt->shared->parent)
        dt = dt->shared->parent;
    if (!H5T_IS_ATOMIC(dt->shared))
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "operation not defined for specified data type");

    
    ret_value = (int)dt->shared->u.atomic.offset;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5Tset_offset(hid_t type_id, size_t offset)
{
    H5T_t *dt;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an atomic data type");
    if (H5T_STATE_TRANSIENT != dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_CANTINIT, FAIL, "data type is read-only");
    if (H5T_STRING == dt->shared->type && offset != 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "offset must be zero for this type");
    if (H5T_ENUM == dt->shared->type && dt->shared->u.enumer.nmembs > 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "operation not allowed after members are defined");
    if (H5T_COMPOUND == dt->shared->type || H5T_REFERENCE == dt->shared->type ||
        H5T_OPAQUE == dt->shared->type)
        HGOTO_ERROR(H5E_DATATYPE, H5E_UNSUPPORTED, FAIL, "operation not defined for this datatype");

    
    if (H5T__set_offset(dt, offset) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to set offset");

done:
    FUNC_LEAVE_API(ret_value)
}

static herr_t
H5T__set_offset(const H5T_t *dt, size_t offset)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dt);
    assert(H5T_STRING != dt->shared->type || 0 == offset);
    assert(H5T_REFERENCE != dt->shared->type);
    assert(H5T_OPAQUE != dt->shared->type);
    assert(H5T_COMPOUND != dt->shared->type);
    assert(!(H5T_ENUM == dt->shared->type && 0 == dt->shared->u.enumer.nmembs));

    if (dt->shared->parent) {
        if (H5T__set_offset(dt->shared->parent, offset) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "unable to set offset for base type");

        
        if (dt->shared->type == H5T_ARRAY)
            dt->shared->size = dt->shared->parent->shared->size * dt->shared->u.array.nelem;
        else if (dt->shared->type == H5T_COMPLEX)
            dt->shared->size = 2 * dt->shared->parent->shared->size;
        else if (dt->shared->type != H5T_VLEN)
            dt->shared->size = dt->shared->parent->shared->size;
    }
    else {
        if (offset + dt->shared->u.atomic.prec > 8 * dt->shared->size)
            dt->shared->size = (offset + dt->shared->u.atomic.prec + 7) / 8;
        dt->shared->u.atomic.offset = offset;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
}
