/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5VLmodule.h" 

#include "H5private.h"   
#include "H5Aprivate.h"  
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5ESprivate.h" 
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5Gprivate.h"  
#include "H5Iprivate.h"  
#include "H5Mprivate.h"  
#include "H5MMprivate.h" 
#include "H5PLprivate.h" 
#include "H5Tprivate.h"  
#include "H5VLpkg.h"     

#include "H5VLnative_private.h"   
#include "H5VLpassthru_private.h" 

typedef struct H5VL_wrap_ctx_t {
    unsigned          rc;           
    H5VL_connector_t *connector;    
    void             *obj_wrap_ctx; 
} H5VL_wrap_ctx_t;

typedef struct {
    
    H5PL_vol_key_t key;

    
    hid_t found_id; 
} H5VL_get_connector_ud_t;

static herr_t            H5VL__free_cls(void *cls);
static void             *H5VL__wrap_obj(void *obj, H5I_type_t obj_type);
static H5VL_connector_t *H5VL__conn_create(H5VL_class_t *cls);
static herr_t            H5VL__conn_find(H5PL_vol_key_t *key, H5VL_connector_t **connector);
static herr_t            H5VL__conn_free(H5VL_connector_t *connector);
static herr_t            H5VL__conn_free_id(void *connector, void H5_ATTR_UNUSED **request);
static void             *H5VL__object(hid_t id, H5I_type_t obj_type);
static herr_t            H5VL__free_vol_wrapper(H5VL_wrap_ctx_t *vol_wrap_ctx);

bool H5_PKG_INIT_VAR = false;

static const H5I_class_t H5I_VOL_CLS[1] = {{
    H5I_VOL,                       
    0,                             
    0,                             
    (H5I_free_t)H5VL__conn_free_id 
}};

H5FL_DEFINE_STATIC(H5VL_class_t);

H5FL_DEFINE_STATIC(H5VL_connector_t);

H5FL_DEFINE(H5VL_object_t);

H5FL_DEFINE_STATIC(H5VL_wrap_ctx_t);

static H5VL_connector_t *H5VL_conn_list_head_g = NULL;

static H5VL_connector_prop_t H5VL_def_conn_s = {NULL, NULL};

herr_t
H5VL_init_phase1(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_init_phase2(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5T_init() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize datatype interface");
    if (H5D_init() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize dataset interface");
    if (H5F_init() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize file interface");
    if (H5G_init() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize group interface");
    if (H5A_init() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize attribute interface");
    if (H5M_init() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize map interface");

    
    if (H5VL__native_register() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to register native VOL connector");
    if (H5VL__passthru_register() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to register passthru VOL connector");

    
    assert(H5VL_def_conn_s.connector == NULL);
    assert(H5VL_def_conn_s.connector_info == NULL);

    
    if (H5VL__set_def_conn() < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "unable to set default VOL connector");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__init_package(void)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5I_register_type(H5I_VOL_CLS) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, FAIL, "unable to initialize H5VL interface");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5VL_term_package(void)
{
    int n = 0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        if (H5VL_def_conn_s.connector) {
            
            (void)H5VL_conn_prop_free(&H5VL_def_conn_s);
            H5VL_def_conn_s.connector      = NULL;
            H5VL_def_conn_s.connector_info = NULL;
            n++;
        } 
        else {
            if (H5I_nmembers(H5I_VOL) > 0) {
                
                (void)H5I_clear_type(H5I_VOL, true, false);

                
                (void)H5VL__native_unregister();
                (void)H5VL__passthru_unregister();

                n++;
            } 
            else {
                if (H5VL__num_opt_operation() > 0) {
                    
                    (void)H5VL__term_opt_operation();
                    n++;
                } 
                else {
                    
                    n += (H5I_dec_type_ref(H5I_VOL) > 0);

                    
                    if (0 == n)
                        H5_PKG_INIT_VAR = false;
                } 
            }     
        }         
    }             

    FUNC_LEAVE_NOAPI(n)
} 

static herr_t
H5VL__free_cls(void *_cls)
{
    H5VL_class_t *cls = (H5VL_class_t *)_cls;
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (cls->terminate) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                ret_value = cls->terminate();
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTCLOSEOBJ, FAIL, "VOL connector did not terminate cleanly");
    }

    
    H5MM_xfree_const(cls->name);
    H5FL_FREE(H5VL_class_t, cls);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__set_def_conn(void)
{
    H5P_genplist_t   *def_fapl;            
    H5P_genclass_t   *def_fapclass;        
    const char       *env_var;             
    char             *buf       = NULL;    
    H5VL_connector_t *connector = NULL;    
    void             *vol_info  = NULL;    
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    
    if (H5VL_def_conn_s.connector) {
        
        (void)H5VL_conn_prop_free(&H5VL_def_conn_s);
        H5VL_def_conn_s.connector      = NULL;
        H5VL_def_conn_s.connector_info = NULL;
    } 

    
    env_var = getenv(HDF5_VOL_CONNECTOR);

    
    if (env_var && *env_var) {
        char       *lasts = NULL;            
        const char *tok   = NULL;            
        htri_t      connector_is_registered; 

        
        if (NULL == (buf = H5MM_strdup(env_var)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, FAIL,
                        "can't allocate memory for environment variable string");

        
        if (NULL == (tok = HDstrtok_r(buf, " \t\n\r", &lasts)))
            HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "VOL connector environment variable set empty?");

        
        if ((connector_is_registered = H5VL__is_connector_registered_by_name(tok)) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't check if VOL connector already registered");
        else if (connector_is_registered) {
            
            if (NULL == (connector = H5VL__get_connector_by_name(tok)))
                HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get VOL connector ID");
        } 
        else {
            
            if (!strcmp(tok, "native")) {
                connector = H5VL_NATIVE_conn_g;

                
                H5VL_conn_inc_rc(connector);
            } 
            else if (!strcmp(tok, "pass_through")) {
                connector = H5VL_PASSTHRU_conn_g;

                
                H5VL_conn_inc_rc(connector);
            } 
            else {
                
                
                if (NULL == (connector = H5VL__register_connector_by_name(tok, H5P_VOL_INITIALIZE_DEFAULT)))
                    HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, FAIL, "can't register connector");
            } 
        }     

        
        if (NULL != (tok = HDstrtok_r(NULL, "\n\r", &lasts)))
            if (H5VL__connector_str_to_info(tok, connector, &vol_info) < 0)
                HGOTO_ERROR(H5E_VOL, H5E_CANTDECODE, FAIL, "can't deserialize connector info");

        
        H5VL_def_conn_s.connector      = connector;
        H5VL_def_conn_s.connector_info = vol_info;
    } 
    else {
        
        H5VL_def_conn_s.connector      = H5_DEFAULT_VOL;
        H5VL_def_conn_s.connector_info = NULL;

        
        H5VL_conn_inc_rc(H5VL_def_conn_s.connector);
    } 

    
    if (NULL == (def_fapclass = H5I_object(H5P_FILE_ACCESS)))
        HGOTO_ERROR(H5E_VOL, H5E_BADID, FAIL, "can't find object for default file access property class ID");

    
    if (H5P_reset_vol_class(def_fapclass, &H5VL_def_conn_s) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL,
                    "can't set default VOL connector for default file access property class");

    
    if (NULL == (def_fapl = H5I_object(H5P_FILE_ACCESS_DEFAULT)))
        HGOTO_ERROR(H5E_VOL, H5E_BADID, FAIL, "can't find object for default fapl ID");

    
    if (H5P_set_vol(def_fapl, H5VL_def_conn_s.connector, H5VL_def_conn_s.connector_info) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set default VOL connector for default FAPL");

done:
    
    if (ret_value < 0) {
        if (vol_info)
            if (H5VL_free_connector_info(connector, vol_info) < 0)
                HDONE_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "can't free VOL connector info");
        if (connector)
            
            if (H5VL_conn_dec_rc(connector) < 0)
                HDONE_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "unable to unregister VOL connector");
    } 

    
    H5MM_xfree(buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5VL__wrap_obj(void *obj, H5I_type_t obj_type)
{
    H5VL_wrap_ctx_t *vol_wrap_ctx = NULL; 
    void            *ret_value    = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(obj);

    
    if (H5CX_get_vol_wrap_ctx((void **)&vol_wrap_ctx) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, NULL, "can't get VOL object wrap context");

    
    if (vol_wrap_ctx) {
        
        if (NULL == (ret_value = H5VL_wrap_object(vol_wrap_ctx->connector->cls, vol_wrap_ctx->obj_wrap_ctx,
                                                  obj, obj_type)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, NULL, "can't wrap object");
    } 
    else
        ret_value = obj;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_object_t *
H5VL_new_vol_obj(H5I_type_t type, void *object, H5VL_connector_t *connector, bool wrap_obj)
{
    H5VL_object_t *new_vol_obj  = NULL;  
    bool           conn_rc_incr = false; 
    H5VL_object_t *ret_value    = NULL;  

    FUNC_ENTER_NOAPI(NULL)

    
    assert(object);
    assert(connector);

    
    if (type != H5I_ATTR && type != H5I_DATASET && type != H5I_DATATYPE && type != H5I_FILE &&
        type != H5I_GROUP && type != H5I_MAP)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, NULL, "invalid type number");

    
    if (NULL == (new_vol_obj = H5FL_CALLOC(H5VL_object_t)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, NULL, "can't allocate memory for VOL object");
    new_vol_obj->connector = connector;
    if (wrap_obj) {
        if (NULL == (new_vol_obj->data = H5VL__wrap_obj(object, type)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, NULL, "can't wrap library object");
    } 
    else
        new_vol_obj->data = object;
    new_vol_obj->rc = 1;

    
    H5VL_conn_inc_rc(connector);
    conn_rc_incr = true;

    
    if (H5I_DATATYPE == type) {
        if (NULL == (ret_value = (H5VL_object_t *)H5T_construct_datatype(new_vol_obj)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, NULL, "can't construct datatype object");
    } 
    else
        ret_value = (H5VL_object_t *)new_vol_obj;

done:
    
    if (NULL == ret_value) {
        if (conn_rc_incr && H5VL_conn_dec_rc(connector) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, NULL, "unable to decrement ref count on VOL connector");

        if (new_vol_obj) {
            if (wrap_obj && new_vol_obj->data)
                (void)H5VL_object_unwrap(new_vol_obj);
            (void)H5FL_FREE(H5VL_object_t, new_vol_obj);
        }
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_conn_prop_copy(H5VL_connector_prop_t *connector_prop)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (connector_prop) {
        
        if (connector_prop->connector) {
            
            H5VL_conn_inc_rc(connector_prop->connector);

            
            if (connector_prop->connector_info) {
                void *new_connector_info = NULL; 

                
                if (H5VL_copy_connector_info(connector_prop->connector, &new_connector_info,
                                             connector_prop->connector_info) < 0)
                    HGOTO_ERROR(H5E_PLIST, H5E_CANTCOPY, FAIL, "connector info copy failed");

                
                connector_prop->connector_info = new_connector_info;
            } 
        }     
    }         

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_conn_prop_cmp(int *cmp_value, const H5VL_connector_prop_t *prop1, const H5VL_connector_prop_t *prop2)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cmp_value);
    assert(prop1);
    assert(prop2);

    
    if (prop1 == prop2)
        
        *cmp_value = 0;
    else {
        H5VL_connector_t *conn1, *conn2;     
        int               tmp_cmp_value = 0; 

        
        conn1 = prop1->connector;
        conn2 = prop2->connector;
        if (H5VL_cmp_connector_cls(&tmp_cmp_value, conn1->cls, conn2->cls) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");
        if (tmp_cmp_value != 0)
            
            *cmp_value = tmp_cmp_value;
        else {
            

            
            assert(conn1->cls->info_cls.cmp == conn2->cls->info_cls.cmp);
            tmp_cmp_value = 0;
            if (H5VL_cmp_connector_info(conn1, &tmp_cmp_value, prop1->connector_info, prop2->connector_info) <
                0)
                HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector class info");

            
            *cmp_value = tmp_cmp_value;
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_conn_prop_free(const H5VL_connector_prop_t *connector_prop)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    if (connector_prop) {
        
        if (connector_prop->connector) {
            if (connector_prop->connector_info)
                
                if (H5VL_free_connector_info(connector_prop->connector, connector_prop->connector_info) < 0)
                    HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL,
                                "unable to release VOL connector info object");

            
            if (H5VL_conn_dec_rc(connector_prop->connector) < 0)
                HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "can't decrement reference count for connector");
        }
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5VL_register(H5I_type_t type, void *object, H5VL_connector_t *vol_connector, bool app_ref)
{
    H5VL_object_t *vol_obj   = NULL;            
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    assert(object);
    assert(vol_connector);

    
    
    if (NULL == (vol_obj = H5VL_new_vol_obj(type, object, vol_connector, false)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, H5I_INVALID_HID, "can't create VOL object");

    
    if ((ret_value = H5I_register(type, vol_obj, app_ref)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register handle");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_register_using_existing_id(H5I_type_t type, void *object, H5VL_connector_t *vol_connector, bool app_ref,
                                hid_t existing_id)
{
    H5VL_object_t *new_vol_obj = NULL;    
    herr_t         ret_value   = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(object);
    assert(vol_connector);

    
    
    if (NULL == (new_vol_obj = H5VL_new_vol_obj(type, object, vol_connector, true)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, FAIL, "can't create VOL object");

    
    if (H5I_register_using_existing_id(type, new_vol_obj, app_ref, existing_id) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, FAIL, "can't register object under existing ID");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_object_t *
H5VL_create_object(void *object, H5VL_connector_t *vol_connector)
{
    H5VL_object_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    assert(object);
    assert(vol_connector);

    
    
    if (NULL == (ret_value = H5FL_CALLOC(H5VL_object_t)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, NULL, "can't allocate memory for VOL object");
    ret_value->connector = vol_connector;
    ret_value->data      = object;
    ret_value->rc        = 1;

    
    H5VL_conn_inc_rc(vol_connector);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5VL_connector_t *
H5VL__conn_create(H5VL_class_t *cls)
{
    H5VL_connector_t *connector = NULL; 
    H5VL_connector_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (NULL == (connector = H5FL_CALLOC(H5VL_connector_t)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, NULL, "can't allocate VOL connector struct");
    connector->cls = cls;

    
    if (H5VL_conn_list_head_g) {
        connector->next             = H5VL_conn_list_head_g;
        H5VL_conn_list_head_g->prev = connector;
    }
    H5VL_conn_list_head_g = connector;

    
    ret_value = connector;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5VL_conn_register(H5VL_connector_t *connector)
{
    hid_t ret_value = H5I_INVALID_HID;

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    assert(connector);

    
    if ((ret_value = H5I_register(H5I_VOL, connector, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector ID");

    
    H5VL_conn_inc_rc(connector);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5VL__conn_find(H5PL_vol_key_t *key, H5VL_connector_t **connector)
{
    H5VL_connector_t *node; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(key);
    assert(connector);

    
    node = H5VL_conn_list_head_g;
    while (node) {
        if (H5VL_GET_CONNECTOR_BY_NAME == key->kind) {
            if (0 == strcmp(node->cls->name, key->u.name)) {
                *connector = node;
                break;
            } 
        }     
        else {
            assert(H5VL_GET_CONNECTOR_BY_VALUE == key->kind);
            if (node->cls->value == key->u.value) {
                *connector = node;
                break;
            } 
        }     

        
        node = node->next;
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int64_t
H5VL_conn_inc_rc(H5VL_connector_t *connector)
{
    int64_t ret_value = -1;

    FUNC_ENTER_NOAPI(-1)

    
    assert(connector);

    
    connector->nrefs++;

    
    ret_value = connector->nrefs;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int64_t
H5VL_conn_dec_rc(H5VL_connector_t *connector)
{
    int64_t ret_value = -1; 

    FUNC_ENTER_NOAPI(-1)

    
    assert(connector);

    
    connector->nrefs--;

    
    ret_value = connector->nrefs;

    
    if (0 == connector->nrefs)
        if (H5VL__conn_free(connector) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "unable to free VOL connector");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5VL_conn_same_class(const H5VL_connector_t *conn1, const H5VL_connector_t *conn2)
{
    htri_t ret_value = FAIL; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(conn1);
    assert(conn2);

    
    if (conn1 == conn2)
        HGOTO_DONE(true);
    else {
        int cmp_value = 0; 

        
        if (H5VL_cmp_connector_cls(&cmp_value, conn1->cls, conn2->cls) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");
        ret_value = (0 == cmp_value);
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5VL__conn_free(H5VL_connector_t *connector)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(connector);
    assert(0 == connector->nrefs);

    
    if (H5VL_conn_list_head_g == connector) {
        H5VL_conn_list_head_g = H5VL_conn_list_head_g->next;
        if (H5VL_conn_list_head_g)
            H5VL_conn_list_head_g->prev = NULL;
    }
    else {
        if (connector->prev)
            connector->prev->next = connector->next;
        if (connector->next)
            connector->next->prev = connector->prev;
    }

    if (H5VL__free_cls(connector->cls) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "can't free VOL class");

    H5FL_FREE(H5VL_connector_t, connector);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5VL__conn_free_id(void *_connector, void H5_ATTR_UNUSED **request)
{
    H5VL_connector_t *connector = (H5VL_connector_t *)_connector;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(connector);

    
    if (H5VL_conn_dec_rc(connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "unable to decrement ref count on VOL connector");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5VL_object_inc_rc(H5VL_object_t *vol_obj)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(vol_obj);

    
    FUNC_LEAVE_NOAPI(++vol_obj->rc)
} 

herr_t
H5VL_free_object(H5VL_object_t *vol_obj)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);

    if (--vol_obj->rc == 0) {
        
        if (H5VL_conn_dec_rc(vol_obj->connector) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "unable to decrement ref count on VOL connector");

        vol_obj = H5FL_FREE(H5VL_object_t, vol_obj);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_object_is_native(const H5VL_object_t *obj, bool *is_native)
{
    const H5VL_class_t *cls;                 
    H5VL_connector_t   *native;              
    int                 cmp_value = 0;       
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(obj);
    assert(is_native);

    
    cls = NULL;
    if (H5VL_introspect_get_conn_cls(obj, H5VL_GET_CONN_LVL_TERM, &cls) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get VOL connector class");

    
    native = H5VL_NATIVE_conn_g;

    
    if (H5VL_cmp_connector_cls(&cmp_value, cls, native->cls) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");

    
    *is_native = (cmp_value == 0);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_file_is_same(const H5VL_object_t *vol_obj1, const H5VL_object_t *vol_obj2, bool *same_file)
{
    const H5VL_class_t *cls1;                
    const H5VL_class_t *cls2;                
    int                 cmp_value = 0;       
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj1);
    assert(vol_obj2);
    assert(same_file);

    
    cls1 = NULL;
    if (H5VL_introspect_get_conn_cls(vol_obj1, H5VL_GET_CONN_LVL_TERM, &cls1) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get VOL connector class");
    cls2 = NULL;
    if (H5VL_introspect_get_conn_cls(vol_obj2, H5VL_GET_CONN_LVL_TERM, &cls2) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get VOL connector class");

    
    if (H5VL_cmp_connector_cls(&cmp_value, cls1, cls2) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");

    
    if (cmp_value)
        *same_file = false;
    else {
        void                     *obj2;        
        H5VL_file_specific_args_t vol_cb_args; 

        
        if (NULL == (obj2 = H5VL_object_data(vol_obj2)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get unwrapped object");

        
        vol_cb_args.op_type                 = H5VL_FILE_IS_EQUAL;
        vol_cb_args.args.is_equal.obj2      = obj2;
        vol_cb_args.args.is_equal.same_file = same_file;

        
        if (H5VL_file_specific(vol_obj1, &vol_cb_args, H5P_DATASET_XFER_DEFAULT, NULL) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTOPERATE, FAIL, "file specific failed");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_connector_t *
H5VL__register_connector(const H5VL_class_t *cls, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL;
    H5VL_class_t     *saved     = NULL;
    bool              init_done = false;
    H5VL_connector_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(cls);

    
    if (NULL == (saved = H5FL_MALLOC(H5VL_class_t)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, NULL, "memory allocation failed for VOL connector class struct");
    H5MM_memcpy(saved, cls, sizeof(H5VL_class_t));
    if (NULL == (saved->name = H5MM_strdup(cls->name)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, NULL, "memory allocation failed for VOL connector name");

    
    if (cls->initialize) {
        herr_t status;

        
        H5_BEFORE_USER_CB(NULL)
            {
                status = cls->initialize(vipl_id);
            }
        H5_AFTER_USER_CB(NULL)
        if (status < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, NULL, "unable to init VOL connector");
    }
    init_done = true;

    
    if (NULL == (connector = H5VL__conn_create(saved)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, NULL, "unable to create VOL connector");

    
    ret_value = connector;

done:
    if (NULL == ret_value) {
        if (connector) {
            if (H5VL__conn_free(connector) < 0)
                HDONE_ERROR(H5E_VOL, H5E_CANTRELEASE, NULL, "can't free VOL connector");
        }
        else if (init_done) {
            if (H5VL__free_cls(saved) < 0)
                HDONE_ERROR(H5E_VOL, H5E_CANTRELEASE, NULL, "can't free VOL class");
        }
        else if (saved) {
            if (saved->name)
                H5MM_xfree_const(saved->name);
            H5FL_FREE(H5VL_class_t, saved);
        }
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_connector_t *
H5VL__register_connector_by_class(const H5VL_class_t *cls, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL; 
    H5PL_vol_key_t    key;              
    H5VL_connector_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (!cls)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, NULL, "VOL connector class pointer cannot be NULL");
    if (H5VL_VERSION != cls->version)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL, "VOL connector has incompatible version");
    if (!cls->name)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL, "VOL connector class name cannot be the NULL pointer");
    if (0 == strlen(cls->name))
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL, "VOL connector class name cannot be the empty string");
    if (cls->info_cls.copy && !cls->info_cls.free)
        HGOTO_ERROR(
            H5E_VOL, H5E_CANTREGISTER, NULL,
            "VOL connector must provide free callback for VOL info objects when a copy callback is provided");
    if (cls->wrap_cls.get_wrap_ctx && !cls->wrap_cls.free_wrap_ctx)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL,
                    "VOL connector must provide free callback for object wrapping contexts when a get "
                    "callback is provided");

    
    key.kind   = H5VL_GET_CONNECTOR_BY_NAME;
    key.u.name = cls->name;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTFIND, NULL, "can't search VOL connectors");

    
    if (NULL == connector)
        if (NULL == (connector = H5VL__register_connector(cls, vipl_id)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL, "unable to register VOL connector");

    
    H5VL_conn_inc_rc(connector);

    
    ret_value = connector;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_connector_t *
H5VL__register_connector_by_name(const char *name, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL; 
    H5PL_vol_key_t    key;              
    H5VL_connector_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    key.kind   = H5VL_GET_CONNECTOR_BY_NAME;
    key.u.name = name;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTFIND, NULL, "can't search VOL connectors");

    
    if (NULL == connector) {
        H5PL_key_t          plugin_key;
        const H5VL_class_t *cls;

        
        plugin_key.vol.kind   = H5VL_GET_CONNECTOR_BY_NAME;
        plugin_key.vol.u.name = name;
        if (NULL == (cls = H5PL_load(H5PL_TYPE_VOL, &plugin_key)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, NULL, "unable to load VOL connector");

        
        if (NULL == (connector = H5VL__register_connector(cls, vipl_id)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL, "unable to register VOL connector");
    } 

    
    H5VL_conn_inc_rc(connector);

    
    ret_value = connector;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_connector_t *
H5VL__register_connector_by_value(H5VL_class_value_t value, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL; 
    H5PL_vol_key_t    key;              
    H5VL_connector_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    key.kind    = H5VL_GET_CONNECTOR_BY_VALUE;
    key.u.value = value;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTFIND, NULL, "can't search VOL connectors");

    
    if (NULL == connector) {
        H5PL_key_t          plugin_key;
        const H5VL_class_t *cls;

        
        plugin_key.vol.kind    = H5VL_GET_CONNECTOR_BY_VALUE;
        plugin_key.vol.u.value = value;
        if (NULL == (cls = H5PL_load(H5PL_TYPE_VOL, &plugin_key)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, NULL, "unable to load VOL connector");

        
        if (NULL == (connector = H5VL__register_connector(cls, vipl_id)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, NULL, "unable to register VOL connector ID");
    } 

    
    H5VL_conn_inc_rc(connector);

    
    ret_value = connector;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5_ATTR_PURE htri_t
H5VL__is_connector_registered_by_name(const char *name)
{
    H5VL_connector_t *connector = NULL;  
    H5PL_vol_key_t    key;               
    htri_t            ret_value = false; 

    FUNC_ENTER_PACKAGE

    
    key.kind   = H5VL_GET_CONNECTOR_BY_NAME;
    key.u.name = name;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTFIND, FAIL, "can't search VOL connectors");

    
    if (connector)
        ret_value = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5_ATTR_PURE htri_t
H5VL__is_connector_registered_by_value(H5VL_class_value_t value)
{
    H5VL_connector_t *connector = NULL;  
    H5PL_vol_key_t    key;               
    htri_t            ret_value = false; 

    FUNC_ENTER_PACKAGE

    
    key.kind    = H5VL_GET_CONNECTOR_BY_VALUE;
    key.u.value = value;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTFIND, FAIL, "can't search VOL connectors");

    
    if (connector)
        ret_value = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_connector_t *
H5VL__get_connector_by_name(const char *name)
{
    H5VL_connector_t *connector = NULL; 
    H5PL_vol_key_t    key;              
    H5VL_connector_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    key.kind   = H5VL_GET_CONNECTOR_BY_NAME;
    key.u.name = name;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_BADITER, NULL, "can't find VOL connector");

    if (connector)
        
        H5VL_conn_inc_rc(connector);

    
    ret_value = connector;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_connector_t *
H5VL__get_connector_by_value(H5VL_class_value_t value)
{
    H5VL_connector_t *connector = NULL; 
    H5PL_vol_key_t    key;              
    H5VL_connector_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    key.kind    = H5VL_GET_CONNECTOR_BY_VALUE;
    key.u.value = value;

    
    if (H5VL__conn_find(&key, &connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_BADITER, NULL, "can't find VOL connector");

    if (connector)
        
        H5VL_conn_inc_rc(connector);

    
    ret_value = connector;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

size_t
H5VL__get_connector_name(const H5VL_connector_t *connector, char *name , size_t size)
{
    size_t len = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(connector);

    len = strlen(connector->cls->name);
    if (name) {
        strncpy(name, connector->cls->name, size);
        if (len >= size)
            name[size - 1] = '\0';
    } 

    FUNC_LEAVE_NOAPI(len)
} 

H5VL_object_t *
H5VL_vol_object(hid_t id)
{
    H5VL_object_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    if (NULL == (ret_value = H5VL_vol_object_verify(id, H5I_get_type(id))))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, NULL, "can't retrieve object for ID");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5VL_object_t *
H5VL_vol_object_verify(hid_t id, H5I_type_t obj_type)
{
    void          *obj       = NULL;
    H5VL_object_t *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    if (H5I_FILE == obj_type || H5I_GROUP == obj_type || H5I_ATTR == obj_type || H5I_DATASET == obj_type ||
        H5I_DATATYPE == obj_type || H5I_MAP == obj_type) {
        
        if (NULL == (obj = H5I_object_verify(id, obj_type)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "identifier is not of specified type");

        
        if (H5I_DATATYPE == obj_type)
            if (NULL == (obj = H5T_get_named_type((H5T_t *)obj)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a named datatype");
    } 
    else
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "invalid identifier type to function");

    ret_value = (H5VL_object_t *)obj;

done:
    FUNC_LEAVE_NOAPI(ret_value)
}

void *
H5VL_object_data(const H5VL_object_t *vol_obj)
{
    void *ret_value = NULL;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    if (vol_obj->connector->cls->wrap_cls.get_object) {
        
        H5_BEFORE_USER_CB_NOERR(NULL)
            {
                ret_value = (vol_obj->connector->cls->wrap_cls.get_object)(vol_obj->data);
            }
        H5_AFTER_USER_CB_NOERR(NULL)
    }
    else
        ret_value = vol_obj->data;

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL_object_unwrap(const H5VL_object_t *vol_obj)
{
    void *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    if (NULL == (ret_value = H5VL_unwrap_object(vol_obj->connector->cls, vol_obj->data)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, NULL, "can't unwrap object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static void *
H5VL__object(hid_t id, H5I_type_t obj_type)
{
    H5VL_object_t *vol_obj   = NULL;
    void          *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    switch (obj_type) {
        case H5I_GROUP:
        case H5I_DATASET:
        case H5I_FILE:
        case H5I_ATTR:
        case H5I_MAP:
            
            if (NULL == (vol_obj = H5I_object(id)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "invalid identifier");
            break;

        case H5I_DATATYPE: {
            H5T_t *dt = NULL;

            
            if (NULL == (dt = H5I_object(id)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "invalid identifier");

            
            if (NULL == (vol_obj = H5T_get_named_type(dt)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a named datatype");
            break;
        }

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_DATASPACE:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unknown data object type");
    } 

    
    ret_value = H5VL_object_data(vol_obj);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL_object(hid_t id)
{
    void *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    if (NULL == (ret_value = H5VL__object(id, H5I_get_type(id))))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, NULL, "can't retrieve object for ID");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL_object_verify(hid_t id, H5I_type_t obj_type)
{
    void *ret_value = NULL;

    FUNC_ENTER_NOAPI(NULL)

    
    if (obj_type != H5I_get_type(id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "invalid identifier");

    
    if (NULL == (ret_value = H5VL__object(id, obj_type)))
        HGOTO_ERROR(H5E_ARGS, H5E_CANTGET, NULL, "can't retrieve object for ID");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_cmp_connector_cls(int *cmp_value, const H5VL_class_t *cls1, const H5VL_class_t *cls2)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cls1);
    assert(cls2);

    
    if (cls1 == cls2) {
        *cmp_value = 0;
        HGOTO_DONE(SUCCEED);
    } 

    
    if (cls1->value < cls2->value) {
        *cmp_value = -1;
        HGOTO_DONE(SUCCEED);
    } 
    if (cls1->value > cls2->value) {
        *cmp_value = 1;
        HGOTO_DONE(SUCCEED);
    } 
    assert(cls1->value == cls2->value);

    
    if (cls1->name == NULL && cls2->name != NULL) {
        *cmp_value = -1;
        HGOTO_DONE(SUCCEED);
    } 
    if (cls1->name != NULL && cls2->name == NULL) {
        *cmp_value = 1;
        HGOTO_DONE(SUCCEED);
    } 
    if (0 != (*cmp_value = strcmp(cls1->name, cls2->name)))
        HGOTO_DONE(SUCCEED);

    
    if (cls1->version < cls2->version) {
        *cmp_value = -1;
        HGOTO_DONE(SUCCEED);
    } 
    if (cls1->version > cls2->version) {
        *cmp_value = 1;
        HGOTO_DONE(SUCCEED);
    } 
    assert(cls1->version == cls2->version);

    
    if (cls1->info_cls.size < cls2->info_cls.size) {
        *cmp_value = -1;
        HGOTO_DONE(SUCCEED);
    } 
    if (cls1->info_cls.size > cls2->info_cls.size) {
        *cmp_value = 1;
        HGOTO_DONE(SUCCEED);
    } 
    assert(cls1->info_cls.size == cls2->info_cls.size);

    
    *cmp_value = 0;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_retrieve_lib_state(void **state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(state);

    
    if (H5CX_retrieve_state((H5CX_state_t **)state) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get API context state");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_start_lib_state(void **context)
{
    H5CX_node_t *cnode     = NULL;    
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(context);

    
    if (NULL == (cnode = H5MM_calloc(sizeof(H5CX_node_t))))
        HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, FAIL, "can't allocate library context");

    
    if (H5CX_push(cnode) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't push API context");

    
    *context = cnode;

done:
    if (ret_value < 0)
        if (cnode)
            H5MM_xfree(cnode);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_restore_lib_state(const void *state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(state);

    
    if (H5CX_restore_state((const H5CX_state_t *)state) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set API context state");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_finish_lib_state(void *context)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(context);

    
    if (H5CX_pop(false) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTRESET, FAIL, "can't pop API context");

    
    H5MM_xfree(context);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_free_lib_state(void *state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(state);

    
    if (H5CX_free_state((H5CX_state_t *)state) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "can't free API context state");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5VL__free_vol_wrapper(H5VL_wrap_ctx_t *vol_wrap_ctx)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(vol_wrap_ctx);
    assert(0 == vol_wrap_ctx->rc);
    assert(vol_wrap_ctx->connector);
    assert(vol_wrap_ctx->connector->cls);

    
    if (vol_wrap_ctx->obj_wrap_ctx) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                
                ret_value =
                    (*vol_wrap_ctx->connector->cls->wrap_cls.free_wrap_ctx)(vol_wrap_ctx->obj_wrap_ctx);
            }
        H5_AFTER_USER_CB(FAIL)
        if (ret_value < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL,
                        "unable to release connector's object wrapping context");
    }

    
    if (H5VL_conn_dec_rc(vol_wrap_ctx->connector) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "unable to decrement ref count on VOL connector");

    
    H5FL_FREE(H5VL_wrap_ctx_t, vol_wrap_ctx);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_set_vol_wrapper(const H5VL_object_t *vol_obj)
{
    H5VL_wrap_ctx_t *vol_wrap_ctx = NULL;    
    herr_t           ret_value    = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);

    
    if (H5CX_get_vol_wrap_ctx((void **)&vol_wrap_ctx) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get VOL object wrap context");

    
    if (NULL == vol_wrap_ctx) {
        void *obj_wrap_ctx = NULL; 

        
        assert(vol_obj->data);
        assert(vol_obj->connector);

        
        if (vol_obj->connector->cls->wrap_cls.get_wrap_ctx) {
            
            assert(vol_obj->connector->cls->wrap_cls.free_wrap_ctx);

            
            H5_BEFORE_USER_CB(FAIL)
                {
                    
                    ret_value =
                        (vol_obj->connector->cls->wrap_cls.get_wrap_ctx)(vol_obj->data, &obj_wrap_ctx);
                }
            H5_AFTER_USER_CB(FAIL)
            if (ret_value < 0)
                HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't retrieve VOL connector's object wrap context");
        } 

        
        if (NULL == (vol_wrap_ctx = H5FL_MALLOC(H5VL_wrap_ctx_t)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTALLOC, FAIL, "can't allocate VOL wrap context");

        
        H5VL_conn_inc_rc(vol_obj->connector);

        
        vol_wrap_ctx->rc           = 1;
        vol_wrap_ctx->connector    = vol_obj->connector;
        vol_wrap_ctx->obj_wrap_ctx = obj_wrap_ctx;
    } 
    else
        
        vol_wrap_ctx->rc++;

    
    if (H5CX_set_vol_wrap_ctx(vol_wrap_ctx) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set VOL object wrap context");

done:
    if (ret_value < 0 && vol_wrap_ctx)
        
        H5FL_FREE(H5VL_wrap_ctx_t, vol_wrap_ctx);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_inc_vol_wrapper(void *_vol_wrap_ctx)
{
    H5VL_wrap_ctx_t *vol_wrap_ctx = (H5VL_wrap_ctx_t *)_vol_wrap_ctx; 
    herr_t           ret_value    = SUCCEED;                          

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == vol_wrap_ctx)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "no VOL object wrap context?");
    if (0 == vol_wrap_ctx->rc)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "bad VOL object wrap context refcount?");

    
    vol_wrap_ctx->rc++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_dec_vol_wrapper(void *_vol_wrap_ctx)
{
    H5VL_wrap_ctx_t *vol_wrap_ctx = (H5VL_wrap_ctx_t *)_vol_wrap_ctx; 
    herr_t           ret_value    = SUCCEED;                          

    FUNC_ENTER_NOAPI(FAIL)

    
    if (NULL == vol_wrap_ctx)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "no VOL object wrap context?");
    if (0 == vol_wrap_ctx->rc)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "bad VOL object wrap context refcount?");

    
    vol_wrap_ctx->rc--;

    
    if (0 == vol_wrap_ctx->rc)
        if (H5VL__free_vol_wrapper(vol_wrap_ctx) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "unable to release VOL object wrapping context");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_reset_vol_wrapper(void)
{
    H5VL_wrap_ctx_t *vol_wrap_ctx = NULL;    
    herr_t           ret_value    = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    if (H5CX_get_vol_wrap_ctx((void **)&vol_wrap_ctx) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get VOL object wrap context");

    
    if (NULL == vol_wrap_ctx)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "no VOL object wrap context?");

    
    vol_wrap_ctx->rc--;

    
    if (0 == vol_wrap_ctx->rc) {
        
        if (H5VL__free_vol_wrapper(vol_wrap_ctx) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "unable to release VOL object wrapping context");

        
        if (H5CX_set_vol_wrap_ctx(NULL) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set VOL object wrap context");
    } 
    else
        
        if (H5CX_set_vol_wrap_ctx(vol_wrap_ctx) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set VOL object wrap context");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5VL_wrap_register(H5I_type_t type, void *obj, bool app_ref)
{
    H5VL_wrap_ctx_t *vol_wrap_ctx = NULL;         
    void            *new_obj;                     
    hid_t            ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI(H5I_INVALID_HID)

    
    assert(obj);

    
    if (H5CX_get_vol_wrap_ctx((void **)&vol_wrap_ctx) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, H5I_INVALID_HID, "can't get VOL object wrap context");
    if (NULL == vol_wrap_ctx || NULL == vol_wrap_ctx->connector)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, H5I_INVALID_HID,
                    "VOL object wrap context or its connector is NULL???");

    
    if (type == H5I_DATATYPE)
        if (vol_wrap_ctx->connector == H5VL_NATIVE_conn_g)
            if (true == H5T_already_vol_managed((const H5T_t *)obj))
                HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, H5I_INVALID_HID, "can't wrap an uncommitted datatype");

    
    if (NULL == (new_obj = H5VL__wrap_obj(obj, type)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, H5I_INVALID_HID, "can't wrap library object");

    
    if ((ret_value = H5VL_register(type, new_obj, vol_wrap_ctx->connector, app_ref)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to get an ID for the object");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_check_plugin_load(const H5VL_class_t *cls, const H5PL_key_t *key, bool *success)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cls);
    assert(key);
    assert(success);

    
    if (key->vol.kind == H5VL_GET_CONNECTOR_BY_NAME) {
        
        if (cls->name && !strcmp(cls->name, key->vol.u.name))
            *success = true;
    } 
    else {
        
        assert(key->vol.kind == H5VL_GET_CONNECTOR_BY_VALUE);

        
        if (cls->value == key->vol.u.value)
            *success = true;
    } 

    
    if (*success && cls->version != H5VL_VERSION)
        *success = false;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5VL__is_default_conn(hid_t fapl_id, const H5VL_connector_t *connector, bool *is_default)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(is_default);

    
    *is_default = (H5VL_def_conn_s.connector == H5_DEFAULT_VOL) &&
                  (H5P_FILE_ACCESS_DEFAULT == fapl_id || connector == H5_DEFAULT_VOL);

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5VL_setup_args(hid_t loc_id, H5I_type_t id_type, H5VL_object_t **vol_obj)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);

    
    if (NULL == (*vol_obj = H5I_object_verify(loc_id, id_type)))
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "not the correct type of ID");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set collective metadata read");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_setup_loc_args(hid_t loc_id, H5VL_object_t **vol_obj, H5VL_loc_params_t *loc_params)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);
    assert(loc_params);

    
    if (NULL == (*vol_obj = (H5VL_object_t *)H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "not the correct type of ID");

    
    if (H5CX_set_loc(loc_id) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set collective metadata read");

    
    loc_params->type     = H5VL_OBJECT_BY_SELF;
    loc_params->obj_type = H5I_get_type(loc_id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_setup_acc_args(hid_t loc_id, const H5P_libclass_t *libclass, bool is_collective, hid_t *acspl_id,
                    H5VL_object_t **vol_obj, H5VL_loc_params_t *loc_params)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(libclass);
    assert(acspl_id);
    assert(vol_obj);
    assert(loc_params);

    
    if (H5CX_set_apl(acspl_id, libclass, loc_id, is_collective) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (*vol_obj = (H5VL_object_t *)H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params->type     = H5VL_OBJECT_BY_SELF;
    loc_params->obj_type = H5I_get_type(loc_id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_setup_self_args(hid_t loc_id, H5VL_object_t **vol_obj, H5VL_loc_params_t *loc_params)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);
    assert(loc_params);

    
    if (NULL == (*vol_obj = (H5VL_object_t *)H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params->type     = H5VL_OBJECT_BY_SELF;
    loc_params->obj_type = H5I_get_type(loc_id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_setup_name_args(hid_t loc_id, const char *name, bool is_collective, hid_t lapl_id,
                     H5VL_object_t **vol_obj, H5VL_loc_params_t *loc_params)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);
    assert(loc_params);

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, is_collective) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (*vol_obj = (H5VL_object_t *)H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params->type                         = H5VL_OBJECT_BY_NAME;
    loc_params->loc_data.loc_by_name.name    = name;
    loc_params->loc_data.loc_by_name.lapl_id = lapl_id;
    loc_params->obj_type                     = H5I_get_type(loc_id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_setup_idx_args(hid_t loc_id, const char *name, H5_index_t idx_type, H5_iter_order_t order, hsize_t n,
                    bool is_collective, hid_t lapl_id, H5VL_object_t **vol_obj, H5VL_loc_params_t *loc_params)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);
    assert(loc_params);

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be NULL");
    if (!*name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "name parameter cannot be an empty string");
    if (idx_type <= H5_INDEX_UNKNOWN || idx_type >= H5_INDEX_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index type specified");
    if (order <= H5_ITER_UNKNOWN || order >= H5_ITER_N)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid iteration order specified");

    
    if (H5CX_set_apl(&lapl_id, H5P_CLS_LACC, loc_id, is_collective) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't set access property list info");

    
    if (NULL == (*vol_obj = (H5VL_object_t *)H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params->type                         = H5VL_OBJECT_BY_IDX;
    loc_params->loc_data.loc_by_idx.name     = name;
    loc_params->loc_data.loc_by_idx.idx_type = idx_type;
    loc_params->loc_data.loc_by_idx.order    = order;
    loc_params->loc_data.loc_by_idx.n        = n;
    loc_params->loc_data.loc_by_idx.lapl_id  = lapl_id;
    loc_params->obj_type                     = H5I_get_type(loc_id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_setup_token_args(hid_t loc_id, H5O_token_t *obj_token, H5VL_object_t **vol_obj,
                      H5VL_loc_params_t *loc_params)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_obj);
    assert(loc_params);

    
    if (NULL == (*vol_obj = (H5VL_object_t *)H5VL_vol_object(loc_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid location identifier");

    
    loc_params->type                        = H5VL_OBJECT_BY_TOKEN;
    loc_params->loc_data.loc_by_token.token = obj_token;
    loc_params->obj_type                    = H5I_get_type(loc_id);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL_conn_prop_get_cap_flags(const H5VL_connector_prop_t *connector_prop, uint64_t *cap_flags)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(connector_prop);

    
    if (connector_prop->connector) {
        
        if (H5VL_introspect_get_cap_flags(connector_prop->connector_info, connector_prop->connector->cls,
                                          cap_flags) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't query connector's capability flags");
    } 
    else
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "connector ID not set?");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
