#' Generic plot function for dpmeta object.
#'
#' This function is built with `ggplot2`. Parameters such as `binwidth`, `adjust`, `density.color`, `density.lwd`, `density.lty`, and `density.alpha` correspond directly to arguments in the `ggplot2` geoms.
#'
#' @param x The object generated by the dpmeta function.
#' @param n.samples The number of posterior samples to plot.
#' @param x.lim Numeric vector of length 2 specifying the x-axis limits.
#' @param y.lim Numeric vector of length 2 specifying the y-axis limits.
#' @param x.lab Text with the label of the x-axis. Defaults to "Treatment Effect".
#' @param y.lab Text with the label of the y-axis. Defaults to "Posterior Predictive Distribution".
#' @param title.plot Text for setting a title in the plot. Defaults to "Dirichlet Process ".
#' @param show.histogram A logical value indicating whether to show the histogram.
#' @param show.rug A logical value indicating whether to show the rug plot.
#' @param binwidth A numeric value specifying the bin width for the histogram. Defaults to 0.08.
#' @param adjust A numeric value to adjust the bandwidth of the density curves. Defaults to 1.
#' @param density.color A character value for the density curve color. Defaults to "blue".
#' @param density.lty A character value for the density curve line type. Defaults to "solid".
#' @param density.lwd A numeric value for the density curve line width. Defaults to 1.
#' @param density.alpha A numeric value between 0 and 1 for the density curve transparency. Defaults to 0.2.
#' @param hist.color A character value for the histogram border color. Defaults to "black".
#' @param hist.fill A character value for the histogram fill color. Defaults to "grey95".
#' @param rug.color A character value for the rug plot color. Defaults to "green".
#' @param rug.alpha A numeric value between 0 and 1 for the rug plot transparency. Defaults to 0.5.
#'
#' @param ... \dots
#'
#' @import ggplot2
#'
#' @export
plot.dpmeta = function(x,
                       n.samples = NULL,
                       x.lim = NULL,
                       y.lim = NULL,
                       x.lab = expression(theta^{new}),
                       y.lab = "Posterior Predictive Distribution",
                       title.plot = "Dirichlet Process ",
                       show.histogram = TRUE,
                       show.rug = TRUE,
                       binwidth = 0.08,
                       adjust = 1,
                       density.color = "blue",
                       density.lty = "solid",
                       density.lwd = 1,
                       density.alpha = 0.2,
                       hist.color = "black",
                       hist.fill = "grey95",
                       rug.color = "green",
                       rug.alpha = 0.5,
                       ...) {

  N = length(x$data$TE) + 1

  # Use all samples unless a smaller number is specified
  total_sims = dim(x$BUGSoutput$sims.list$theta)[1]
  if (is.null(n.samples)) {
    n.samples = total_sims
  }

  # Extract posterior predictive samples
  data_plot = data.frame(
    effect = x$BUGSoutput$sims.list$theta[1:n.samples, N]
  )

  # Calculate default x.lim if not specified
  if (is.null(x.lim)) {
    q1 = quantile(data_plot$effect, 0.25, na.rm = TRUE)
    q3 = quantile(data_plot$effect, 0.75, na.rm = TRUE)
    iqr = q3 - q1
    lower_lim = q1 - 3 * iqr
    upper_lim = q3 + 3 * iqr
    x.lim = c(lower_lim, upper_lim)
  }

  # Create the plot
  plot1 = ggplot(data = data_plot, aes(x = effect))

  if (show.histogram) {
    plot1 = plot1 + geom_histogram(aes(y = after_stat(density)), binwidth = binwidth, colour = hist.color, fill = hist.fill)
  }

  plot1 = plot1 + geom_density(color = density.color, lwd = density.lwd, lty = density.lty, alpha = density.alpha, adjust = adjust)

  if (show.rug) {
    plot1 = plot1 + geom_rug(data = data.frame(TE = x$data$TE[1:(N-1)]), aes(x = TE), sides = "b", lwd = 1, col = rug.color, alpha = rug.alpha, show.legend = FALSE)
  }

  plot1 = plot1 +
    coord_cartesian(xlim = x.lim, ylim = y.lim) +
    xlab(x.lab) +
    ylab(y.lab) +
    geom_vline(aes(xintercept = 0), lty = 2, color = "black") +
    theme_bw() +
    ggtitle(title.plot) +
    theme(legend.position = "none")


  return(suppressWarnings(plot1))
}
