#' Agent: A General-Purpose LLM Agent
#'
#' @description
#' The `Agent` class defines a modular LLM-based agent capable of responding to prompts using a defined role/instruction.
#' It wraps an OpenAI-compatible chat model via the [`ellmer`](https://github.com/llrs/ellmer) package.
#'
#' Each agent maintains its own message history and unique identity.
#'
#' @importFrom R6 R6Class
#' @importFrom uuid UUIDgenerate
#' @importFrom checkmate assert_string assert_flag assert_character assert_integerish
#' @importFrom cli cli_abort cli_alert_success cli_alert_warning cli_alert_info cli_rule cli_text cli_ul
#' @export
Agent <- R6::R6Class(
  classname = "Agent",

  public = list(
    #' @description
    #' Initializes a new Agent with a specific role/instruction.
    #'
    #' @param name A short identifier for the agent (e.g. `"translator"`).
    #' @param instruction The system prompt that defines the agent's role.
    #' @param llm_object The LLM object generate by ellmer (eg. output of ellmer::chat_openai)
    #' @param budget Numerical value denoting the amount to set for the budget in US$ to a specific agent,
    #' if the budget is reached, an error will be thrown.
    #' @examples
    #'   # An API KEY is required in order to invoke the Agent
    #'   openai_4_1_mini <- ellmer::chat(
    #'     name = "openai/gpt-4.1-mini",
    #'     api_key = Sys.getenv("OPENAI_API_KEY"),
    #'     echo = "none"
    #'   )
    #'
    #'   polar_bear_researcher <- Agent$new(
    #'     name = "POLAR BEAR RESEARCHER",
    #'     instruction = paste0(
    #'     "You are an expert in polar bears, ",
    #'     "you task is to collect information about polar bears. Answer in 1 sentence max."
    #'     ),
    #'     llm_object = openai_4_1_mini
    #'   )
    #'
    initialize = function(name, instruction, llm_object, budget = NA) {
      checkmate::assert_string(name)
      checkmate::assert_string(instruction)

      if (!"Chat" %in% class(llm_object)) {
        cli::cli_abort("The llm_object must be generated by ellmer")
      }

      self$name <- name
      self$instruction <- instruction
      self$llm_object <- llm_object$clone(deep = TRUE)

      meta_data <- self$llm_object$get_provider()
      self$model_provider <- meta_data@name
      self$model_name <- meta_data@model

      self$llm_object$set_system_prompt(value = instruction)

      private$._messages <- list(
        list(role = "system", content = instruction)
      )

      self$agent_id <- uuid::UUIDgenerate()

      checkmate::assert_numeric(budget)

      self$budget <- budget

      self$budget_policy <- list(
        on_exceed = "abort",
        warn_at = 0.8
      )

    },

    #' @description
    #' Sends a user prompt to the agent and returns the assistant's response.
    #'
    #' @param prompt A character string prompt for the agent to respond to.
    #' @return The LLM-generated response as a character string.
    #' @examples \dontrun{
    #' # An API KEY is required in order to invoke the Agent
    #' openai_4_1_mini <- ellmer::chat(
    #'     name = "openai/gpt-4.1-mini",
    #'     api_key = Sys.getenv("OPENAI_API_KEY"),
    #'     echo = "none"
    #' )
    #' agent <- Agent$new(
    #'  name = "translator",
    #'  instruction = "You are an Algerian citizen",
    #'  llm_object = openai_4_1_mini
    #' )
    #' agent$invoke("Continue this sentence: 1 2 3 viva")
    #' }
    invoke = function(prompt) {

      checkmate::assert_string(prompt)

      if (!is.na(self$budget)) {
        private$.check_budget()
      }

      private$.add_user_message(prompt)
      response <- self$llm_object$chat(prompt)
      response <- as.character(response)
      private$.add_assistant_message(response)
      return(response)
    },

    #' @description
    #' Generate R code from natural language descriptions and optionally validate/execute it
    #'
    #' @param code_description Character string describing the R code to generate
    #' @param validate Logical indicating whether to validate the generated code syntax
    #' @param execute Logical indicating whether to execute the generated code (use with caution)
    #' @param interactive Logical; if TRUE, ask for user confirmation before executing generated code
    #' @param env Environment in which to execute the code if execute = TRUE. Default to \code{globalenv}
    #' @return A list containing the generated code and validation/execution results
    #' @examples
    #' \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' r_assistant <- Agent$new(
    #'   name = "R Code Assistant",
    #'   instruction = paste("You are an expert R programmer",
    #'   llm_object = openai_4_1_mini
    #' )
    #' # Generate code for data manipulation
    #' result <- r_assistant$generate_execute_r_code(
    #'   code_description = "Calculate the summary of the mtcars dataframe",
    #'   validate = TRUE,
    #'   execute = TRUE,
    #'   interactive = TRUE
    #' )
    #' print(result)
    #' }
    generate_execute_r_code = function(
    code_description,
    validate = FALSE,
    execute = FALSE,
    interactive = TRUE,
    env = globalenv()
    ) {

      checkmate::assert_string(code_description)
      checkmate::assert_flag(validate)
      checkmate::assert_flag(execute)
      checkmate::assert_environment(env)
      checkmate::assert_flag(interactive)

      code_prompt <- paste0(
        "Generate R code for the following task. Return ONLY the R code without any explanations, ",
        "markdown formatting, or additional text:\n\n",
        "if you run several commands, use the ';' character to separate them. \n\n",
        "Do not add additional spaces that would be interpreted later with '\n'. \n\n",
        code_description
      )

      generated_code <- self$invoke(code_prompt)

      clean_code <- gsub("```\\{?r\\}?|```", "", generated_code)
      clean_code <- gsub("```\\{?r\\}?|```", "", generated_code)
      clean_code <- trimws(clean_code)

      result <- list(
        description = code_description,
        code = clean_code,
        validated = FALSE,
        validation_message = NA_character_,
        executed = FALSE,
        execution_result = NULL,
        execution_error = NULL
      )

      if (validate) {
        result <- private$.validate_r_code(r_code = clean_code, result = result)
      }

      if (execute) {
        if (!validate || !result$validated) {
          cli::cli_alert_warning("Code execution skipped: code must be validated first")
          return(result)
        }

        if (interactive) {
          cli::cli_h1("Generated code preview:")
          cat(paste0("\n", clean_code, "\n\n"))
          user_input <- readline(prompt = "Do you want to execute this code? [y/N]: ")
          if (tolower(user_input) != "y") {
            cli::cli_alert_info("Execution cancelled by user.")
            return(result)
          }
        }

        cli::cli_alert_info("Executing generated R code...")

        execution_result <- tryCatch({
          output <- capture.output({
            eval_result <- eval(parse(text = clean_code), envir = env)
          })

          result$executed <- TRUE
          result$execution_result <- list(
            value = eval_result,
            output = output
          )
          result$execution_error <- NULL

          cli::cli_alert_success("Code executed successfully")

        }, error = function(e) {
          result$executed <- FALSE
          result$execution_error <- e$message
          cli::cli_alert_danger(paste("Execution error:", e$message))
        })
      }

      return(result)
    },

    #' @description
    #' Set a budget to a specific agent, if the budget is reached, an error will be thrown
    #'
    #' @param amount_in_usd Numerical value denoting the amount to set for the budget,
    #' @examples \dontrun{
    #' # An API KEY is required in order to invoke the Agent
    #' openai_4_1_mini <- ellmer::chat(
    #'     name = "openai/gpt-4.1-mini",
    #'     api_key = Sys.getenv("OPENAI_API_KEY"),
    #'     echo = "none"
    #' )
    #' agent <- Agent$new(
    #'  name = "translator",
    #'  instruction = "You are an Algerian citizen",
    #'  llm_object = openai_4_1_mini
    #' )
    #' agent$set_budget(amount_in_usd = 10.5) # this is equivalent to 10.5$
    #' }
    set_budget = function(amount_in_usd) {

      checkmate::assert_number(amount_in_usd, lower = 0)

      self$budget <- amount_in_usd

      cli::cli_alert_success(glue::glue("Budget successfully set to {amount_in_usd}$"))
      cli::cli_alert_info(glue::glue("Budget policy: on_exceed='{self$budget_policy$on_exceed}', warn_at={self$budget_policy$warn_at}"))
      cli::cli_alert_info("Use the set_budget_policy() method to configure the budget policy.")
      invisible(self)
    },

    #' @description
    #' Configure how the agent behaves as it approaches or exceeds its budget.
    #' Use `warn_at` (0-1) to emit a one-time warning when spending reaches the
    #' specified fraction of the budget. When the budget is exceeded, `on_exceed`
    #' controls behavior: abort, warn and proceed, or ask interactively.
    #' @param on_exceed One of "abort", "warn", or "ask".
    #' @param warn_at Numeric in (0,1); fraction of budget to warn at. Default 0.8.
    #' @examples \dontrun{
    #' agent$set_budget(5)
    #' agent$set_budget_policy(on_exceed = "ask", warn_at = 0.9)
    #' }
    set_budget_policy = function(on_exceed = "abort", warn_at = 0.8) {

      checkmate::assert_choice(on_exceed, c("abort", "warn", "ask"))
      checkmate::assert_number(warn_at, lower = 0, upper = 1)

      self$budget_policy <- list(
        on_exceed = on_exceed,
        warn_at = warn_at
      )

      cli::cli_alert_success(glue::glue(
        "Budget policy set: on_exceed='{on_exceed}', warn_at={warn_at}"
      ))

      invisible(self)
    },

    #' @description
    #' Keep only the most recent `n` messages, discarding older ones while keeping
    #' the system prompt.
    #' @param n Number of most recent messages to keep.
    #' @examples \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "capital finder",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #' )
    #' agent$invoke("What is the capital of Algeria")
    #' agent$invoke("What is the capital of Germany")
    #' agent$invoke("What is the capital of Italy")
    #' agent$keep_last_n_messages(n = 2)
    #' }
    keep_last_n_messages = function(n = 2) {

      checkmate::assert_integerish(n, lower = 1)

      ln_messags <- length(self$messages)

      messages_to_keep <- self$messages[(ln_messags - n + 1):ln_messags]

      system_prompt <- self$llm_object$get_system_prompt()

      tmp_sp <- list(
        list(role = "system", content = system_prompt)
      )

      private$._messages <- append(tmp_sp, messages_to_keep)

      private$.set_turns_from_messages()

      cli::cli_alert_success("Conversation truncated to last {n} messages.")

      invisible(self)

    },

    #' @description
    #' Summarises the agent's conversation history into a concise form and appends it
    #' to the system prompt. Unlike `update_instruction()`, this method does not override
    #' the existing instruction but augments it with a summary for future context.
    #'
    #' After creating the summary, the method clears the conversation history and
    #' retains only the updated system prompt. This ensures that subsequent interactions
    #' start fresh but with the summary preserved as context.
    #'
    #' @examples \dontrun{
    #'   # Requires an OpenAI-compatible LLM from `ellmer`
    #'   openai_4_1_mini <- ellmer::chat(
    #'     name = "openai/gpt-4.1-mini",
    #'     api_key = Sys.getenv("OPENAI_API_KEY"),
    #'     echo = "none"
    #'   )
    #'
    #'   agent <- Agent$new(
    #'     name = "summariser",
    #'     instruction = "You are a summarising assistant",
    #'     llm_object = openai_4_1_mini
    #'   )
    #'
    #'   agent$invoke("The quick brown fox jumps over the lazy dog.")
    #'   agent$invoke("This is another example sentence.")
    #'
    #'   # Summarises and resets history
    #'   agent$summarise_messages()
    #'
    #'   # Now only the system prompt (with summary) remains
    #'   agent$messages
    #' }
    #'

    clear_and_summarise_messages = function() {

      if (length(self$messages) <= 1) {
        cli::cli_alert_info("No conversation history to summarise.")
        return(invisible(NULL))
      }

      summary_prompt <- paste0(
        "Summarise the following conversation history in a concise paragraph:\n\n",
        paste(
          vapply(self$messages, function(m) {
            paste0(m$role, ": ", m$content)
          }, character(1)),
          collapse = " \n "
        )
      )

      summary <- self$llm_object$chat(summary_prompt)
      summary <- as.character(summary)

      new_system_prompt <- paste(
        self$instruction,
        "\n\n--- Conversation Summary ---\n",
        summary
      )

      self$llm_object$set_system_prompt(value = new_system_prompt)

      private$._messages <- list(
        list(role = "system", content = new_system_prompt)
      )

      private$.set_turns_from_messages()

      cli::cli_alert_success("Conversation history summarised and appended to system prompt.")
      cli::cli_alert_info("Summary: {substr(summary, 1, 100)}...")

      invisible(self)
    },

    #' @description
    #' Update the system prompt/instruction
    #' @param new_instruction New instruction to use. Not that the new instruction
    #' will override the old one
    #' @examples \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "assistant",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #' )
    #' agent$update_instruction("You are a concise assistant.")
    #' }
    update_instruction = function(new_instruction) {

      checkmate::assert_string(new_instruction)

      old_instruction <- self$instruction
      self$instruction <- new_instruction
      self$llm_object$set_system_prompt(value = new_instruction)

      private$._messages[[1]]$content <- new_instruction

      cli::cli_alert_success("Instruction successfully updated")
      cli::cli_alert_info("Old: {substr(old_instruction, 1, 50)}...")
      cli::cli_alert_info("New: {substr(new_instruction, 1, 50)}...")

      invisible(self)
    },

    #' @description
    #' Get the current token count and estimated cost of the conversation
    #'
    #' @return A list with token counts and cost information
    #' @examples
    #' \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "assistant",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #' )
    #' agent$set_budget(1)
    #' agent$invoke("What is the capital of Algeria?")
    #' stats <- agent$get_usage_stats()
    #' stats
    #' }
    get_usage_stats = function() {

      current_cost <- self$llm_object$get_cost()

      budget_remaining <- NA

      if (!is.na(self$budget)) {
        budget_remaining <- self$budget - as.numeric(current_cost)
      }

      total_tokens_df <- self$llm_object$get_tokens()

      total_tokens <- sum(total_tokens_df$tokens_total)

      llm_costs <- list(
        total_tokens = total_tokens,
        estimated_cost = round(as.numeric(current_cost), 4),
        budget = round(self$budget, 4),
        budget_remaining = round(budget_remaining, 4)
      )

      llm_costs

    },

    #' @description
    #' Add a pre-formatted message to the conversation history
    #'
    #' @param role The role of the message ("user", "assistant", or "system")
    #' @param content The content of the message
    #' @examples
    #' \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "AI assistant",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #')
    #' agent$add_message("user", "Hello, how are you?")
    #' agent$add_message("assistant", "I'm doing well, thank you!")
    #' }
    add_message = function(role, content) {
      checkmate::assert_string(role)
      checkmate::assert_string(content)
      checkmate::assert_choice(role, c("user", "assistant", "system"))

      private$.add_message(content, role)
      private$.set_turns_from_messages()

      cli::cli_alert_success("Added {role} message: {substr(content, 1, 50)}...")
      invisible(self)
    },

    #' @description
    #' Reset the agent's conversation history while keeping the system instruction
    #'
    #' @examples
    #' \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "AI assistant",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #')
    #' agent$invoke("Hello, how are you?")
    #' agent$invoke("Tell me about machine learning")
    #' agent$reset_conversation_history()  # Clears all messages except system prompt
    #' }
    reset_conversation_history = function() {
      system_prompt <- self$llm_object$get_system_prompt()

      private$._messages <- list(
        list(role = "system", content = system_prompt)
      )

      private$.set_turns_from_messages()

      cli::cli_alert_success("Conversation history reset. System prompt preserved.")
      invisible(self)
    },

    #' @description
    #' Saves the agent's current conversation history as a JSON file on disk.
    #' @param file_path Character string specifying the file path where the JSON
    #' file should be saved. Defaults to a file named
    #' `"<agent_name>_messages.json"` in the current working directory.
    #'
    #' @examples
    #' \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "capital_finder",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #')
    #' agent$invoke("What is the capital of Algeria")
    #' agent$invoke("What is the capital of Italy")
    #' agent$export_messages_history()
    #' }
    #'
    #' @seealso [load_messages_history()] for reloading a saved message history.
    #'
    export_messages_history = function(
    file_path = paste0(getwd(), "/", paste0(self$name, "_messages.json"))
    ) {

      checkmate::assert_string(file_path)

      jsonlite::write_json(
        self$messages,
        path = file_path,
        auto_unbox = TRUE,
        pretty = TRUE
      )

      cli::cli_alert_success(glue::glue("Conversation saved to {file_path}"))

    },

    #' @description
    #' Saves the agent's current conversation history as a JSON file on disk.
    #' @param file_path Character string specifying the file path where the JSON
    #' file is stored. Defaults to a file named
    #' `"<agent_name>_messages.json"` in the current working directory.
    #'
    #' @examples
    #' \dontrun{
    #' openai_4_1_mini <- ellmer::chat(
    #'   name = "openai/gpt-4.1-mini",
    #'   api_key = Sys.getenv("OPENAI_API_KEY"),
    #'   echo = "none"
    #' )
    #' agent <- Agent$new(
    #'   name = "capital_finder",
    #'   instruction = "You are an assistant.",
    #'   llm_object = openai_4_1_mini
    #')
    #' agent$load_messages_history("path/to/messages.json")
    #' agent$messages
    #' agent$llm_object
    #' }
    #'
    #' @seealso [export_messages_history()] for exporting the messages object to json.
    #'
    load_messages_history = function(
    file_path = paste0(getwd(), "/", paste0(self$name, "_messages.json"))
    ) {

      checkmate::assert_string(file_path)

      if (!file.exists(file_path)) {
        cli::cli_abort("File does not exist.")
      }

      messages <- jsonlite::read_json(file_path, simplifyVector = FALSE)

      self$messages <- messages

      cli::cli_alert_success(glue::glue("Conversation history loaded from {file_path}"))

    },

    #' @field name The agent's name.
    name = NULL,
    #' @field instruction The agent's role/system prompt.
    instruction = NULL,
    #' @field llm_object The underlying `ellmer::chat_openai` object.
    llm_object = NULL,
    #' @field agent_id A UUID uniquely identifying the agent.
    agent_id = NULL,
    #'@field model_provider The name of the entity providing the model (eg. OpenAI)
    model_provider = NULL,
    #'@field model_name The name of the model to be used (eg. gpt-4.1-mini)
    model_name = NULL,
    #'@field broadcast_history A list of all past broadcast interactions.
    broadcast_history = list(),
    #'@field budget A budget in $ that the agent should not exceed.
    budget = NULL,
    #'@field budget_policy A list controlling budget behavior: on_exceed and warn_at.
    budget_policy = NULL,
    #'@field budget_warned Internal flag indicating whether warn_at notice was emitted.
    budget_warned = NULL
  ),

  active = list(
    #' @field messages Public active binding for the conversation history.
    #' Assignment is validated automatically.
    messages = function(value) {
      if (missing(value)) {
        return(private$._messages)
      }

      if (!is.list(value)) {
        cli::cli_abort("messages must be a list of message objects")
      }

      for (msg in value) {
        if (!is.list(msg) || !all(c("role", "content") %in% names(msg))) {
          cli::cli_abort("Each message must be a list with 'role' and 'content'")
        }
        if (!msg$role %in% c("system", "user", "assistant")) {
          cli::cli_abort(paste0("Invalid role: ", msg$role))
        }
      }

      private$._messages <- value

      private$.set_turns_from_messages()

    }
  ),

  private = list(
    ._messages = NULL,
    .add_message = function(message, type) {
      private$._messages[[length(private$._messages) + 1]] <- list(
        role = type,
        content = message
      )
    },

    .add_assistant_message = function(message, type = "assistant") {
      private$.add_message(message, type)
    },

    .add_user_message = function(message, type = "user") {
      private$.add_message(message, type)
    },

    .set_turns_from_messages = function() {

      messages <- self$messages
      turns <- list()

      for (msg in messages) {
        turn <- ellmer::Turn(
          role = msg$role,
          contents = list(ellmer::ContentText(msg$content))
        )
        turns <- append(turns, list(turn))
      }

      self$llm_object$set_turns(turns)

    },

    .validate_r_code = function(r_code, result) {

      validation <- tryCatch({
        parsed <- parse(text = r_code)
        result$validated <- TRUE
        result$validation_message <- "Syntax is valid"
        return(result)
      }, error = function(e) {
        result$validated <- FALSE
        result$validation_message <- paste("Syntax error:", e$message)
        return(result)
      })

    },

    .check_budget = function() {

      current_cost <- as.numeric(self$llm_object$get_cost())

      warn_at <- self$budget_policy$warn_at
      ratio <- current_cost / as.numeric(self$budget)

      budget_exceeded <- current_cost > self$budget

      if (ratio >= warn_at && !budget_exceeded) {
        cli::cli_alert_warning(
          glue::glue(
            "{self$name} budget nearing limit: Cost {round(current_cost, 4)} / . ",
            "Budget {round(self$budget, 4)} ({round(ratio * 100, 1)}%)"
          ))
      }

      if (!budget_exceeded) {
        return(invisible(NULL))
      }

      policy <- self$budget_policy$on_exceed

      if (policy == "warn") {
        cli::cli_alert_warning(glue::glue(
          "{self$name} exceeded budget: Cost {round(current_cost,4)} > ",
          "Budget {round(self$budget,4)}. Proceeding per policy 'warn'."
        ))
      }

      if (policy == "ask") {
        user_input <- readline(prompt = glue::glue(
          "Budget exceeded (Cost {round(current_cost,4)} > Budget {round(self$budget,4)}). ",
          "Continue? [y/N]: "
        ))
        if (tolower(user_input) != "y") {
          cli::cli_abort(glue::glue(
            "{self$name} agent cancelled due to budget exceedance. ",
            "Cost: {round(current_cost,4)}, Budget: {round(self$budget,4)}"
          ))
          return(invisible(NULL))
        }
      }

      if (policy == "abort") {
        cli::cli_abort(glue::glue(
          "{self$name} agent has exceeded its budget. ",
          "Cost: {round(current_cost, 4)}, Budget: {round(self$budget, 4)}"
        ))
      }
    }
  )
)
