% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesopt_parego.R
\name{mlr_loop_functions_parego}
\alias{mlr_loop_functions_parego}
\alias{bayesopt_parego}
\title{Multi-Objective Bayesian Optimization via ParEGO}
\usage{
bayesopt_parego(
  instance,
  surrogate,
  acq_function,
  acq_optimizer,
  init_design_size = NULL,
  q = 1L,
  s = 100L,
  rho = 0.05,
  random_interleave_iter = 0L
)
}
\arguments{
\item{instance}{(\link[bbotk:OptimInstanceBatchMultiCrit]{bbotk::OptimInstanceBatchMultiCrit})\cr
The \link[bbotk:OptimInstanceBatchMultiCrit]{bbotk::OptimInstanceBatchMultiCrit} to be optimized.}

\item{surrogate}{(\link{SurrogateLearner})\cr
\link{SurrogateLearner} to be used as a surrogate.}

\item{acq_function}{(\link{AcqFunction})\cr
\link{AcqFunction} to be used as acquisition function.}

\item{acq_optimizer}{(\link{AcqOptimizer})\cr
\link{AcqOptimizer} to be used as acquisition function optimizer.}

\item{init_design_size}{(\code{NULL} | \code{integer(1)})\cr
Size of the initial design.
If \code{NULL} and the \link[bbotk:ArchiveBatch]{bbotk::ArchiveBatch} contains no evaluations, \code{4 * d} is used with \code{d} being the
dimensionality of the search space.
Points are generated via a Sobol sequence.}

\item{q}{(\code{integer(1)})\cr
Batch size, i.e., the number of candidates to be obtained for a single batch.
Default is \code{1}.}

\item{s}{(\code{integer(1)})\cr
\eqn{s} in Equation 1 in Knowles (2006).
Determines the total number of possible random weight vectors.
Default is \code{100}.}

\item{rho}{(\code{numeric(1)})\cr
\eqn{\rho} in Equation 2 in Knowles (2006) scaling the linear part of the augmented Tchebycheff function.
Default is \code{0.05}}

\item{random_interleave_iter}{(\code{integer(1)})\cr
Every \code{random_interleave_iter} iteration (starting after the initial design), a point is
sampled uniformly at random and evaluated (instead of a model based proposal).
For example, if \code{random_interleave_iter = 2}, random interleaving is performed in the second,
fourth, sixth, ... iteration.
Default is \code{0}, i.e., no random interleaving is performed at all.}
}
\value{
invisible(instance)\cr
The original instance is modified in-place and returned invisible.
}
\description{
Loop function for multi-objective Bayesian Optimization via ParEGO.
Normally used inside an \link{OptimizerMbo}.

In each iteration after the initial design, the observed objective function values are normalized and \code{q} candidates are
obtained by scalarizing these values via the augmented Tchebycheff function, updating the surrogate with respect to
these scalarized values and optimizing the acquisition function.
}
\note{
\itemize{
\item The \code{acq_function$surrogate}, even if already populated, will always be overwritten by the \code{surrogate}.
\item The \code{acq_optimizer$acq_function}, even if already populated, will always be overwritten by \code{acq_function}.
\item The \code{surrogate$archive}, even if already populated, will always be overwritten by the \link[bbotk:ArchiveBatch]{bbotk::ArchiveBatch} of the \link[bbotk:OptimInstanceBatchMultiCrit]{bbotk::OptimInstanceBatchMultiCrit}.
\item The scalarizations of the objective function values are stored as the \code{y_scal} column in the
\link[bbotk:ArchiveBatch]{bbotk::ArchiveBatch} of the \link[bbotk:OptimInstanceBatchMultiCrit]{bbotk::OptimInstanceBatchMultiCrit}.
\item To make use of parallel evaluations in the case of `q > 1, the objective
function of the \link[bbotk:OptimInstanceBatchMultiCrit]{bbotk::OptimInstanceBatchMultiCrit} must be implemented accordingly.
}
}
\examples{
\donttest{
if (requireNamespace("mlr3learners") &
    requireNamespace("DiceKriging") &
    requireNamespace("rgenoud")) {

  library(bbotk)
  library(paradox)
  library(mlr3learners)

  fun = function(xs) {
    list(y1 = xs$x^2, y2 = (xs$x - 2) ^ 2)
  }
  domain = ps(x = p_dbl(lower = -10, upper = 10))
  codomain = ps(y1 = p_dbl(tags = "minimize"), y2 = p_dbl(tags = "minimize"))
  objective = ObjectiveRFun$new(fun = fun, domain = domain, codomain = codomain)

  instance = OptimInstanceBatchMultiCrit$new(
    objective = objective,
    terminator = trm("evals", n_evals = 5))

  surrogate = default_surrogate(instance, n_learner = 1)

  acq_function = acqf("ei")

  acq_optimizer = acqo(
    optimizer = opt("random_search", batch_size = 100),
    terminator = trm("evals", n_evals = 100))

  optimizer = opt("mbo",
    loop_function = bayesopt_parego,
    surrogate = surrogate,
    acq_function = acq_function,
    acq_optimizer = acq_optimizer)

  optimizer$optimize(instance)
}
}
}
\references{
\itemize{
\item Knowles, Joshua (2006).
\dQuote{ParEGO: A Hybrid Algorithm With On-Line Landscape Approximation for Expensive Multiobjective Optimization Problems.}
\emph{IEEE Transactions on Evolutionary Computation}, \bold{10}(1), 50--66.
}
}
\seealso{
Other Loop Function: 
\code{\link{loop_function}},
\code{\link{mlr_loop_functions}},
\code{\link{mlr_loop_functions_ego}},
\code{\link{mlr_loop_functions_emo}},
\code{\link{mlr_loop_functions_mpcl}},
\code{\link{mlr_loop_functions_smsego}}
}
\concept{Loop Function}
