% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/all_class.R, R/neurovec.R
\docType{class}
\name{NeuroVec-class}
\alias{NeuroVec-class}
\alias{NeuroVec}
\title{NeuroVec Class}
\usage{
NeuroVec(data, space = NULL, mask = NULL, label = "")
}
\arguments{
\item{data}{The image data. This can be:
\itemize{
  \item A matrix (voxels x time points)
  \item A 4D array
  \item A list of \code{\linkS4class{NeuroVol}} objects
}
If a list of NeuroVol objects is provided, the geometric space (\code{NeuroSpace})
will be inferred from the constituent volumes, which must all be identical.}

\item{space}{An optional \code{\linkS4class{NeuroSpace}} object defining the spatial
properties of the image. Not required if \code{data} is a list of NeuroVol objects.}

\item{mask}{An optional logical array specifying which voxels to include. If provided,
a SparseNeuroVec object will be created.}

\item{label}{A character string providing a label for the NeuroVec object. Default is an empty string.}
}
\value{
A concrete instance of the \code{\linkS4class{NeuroVec}} class:
  \itemize{
    \item If \code{mask} is provided: a \code{\linkS4class{SparseNeuroVec}} object
    \item Otherwise: a \code{\linkS4class{DenseNeuroVec}} object
  }
}
\description{
This S4 class represents a four-dimensional brain image, which is used to store
and process time series neuroimaging data such as fMRI or 4D functional
connectivity maps. The class extends the basic functionality of \code{\linkS4class{NeuroObj}}.

The \code{NeuroVec} class represents a vectorized form of neuroimaging data, supporting both in-memory
and file-backed data modes. It provides efficient data storage and access methods and integrates with
the spatial reference system provided by \code{\linkS4class{NeuroSpace}}.
}
\details{
NeuroVec objects are designed to handle 4D neuroimaging data, where the first
three dimensions represent spatial coordinates, and the fourth dimension typically
represents time or another series dimension. This structure is particularly
useful for storing and analyzing functional MRI data, time series of brain
states, or multiple 3D volumes in a single object.

The function performs several operations:
\itemize{
  \item If \code{data} is a list of NeuroVol objects, it combines them into a single 4D array.
  \item It checks that the dimensions of \code{data} match the provided \code{space}.
  \item Depending on whether a \code{mask} is provided, it creates either a DenseNeuroVec
    or a SparseNeuroVec object.
}
}
\section{Slots}{

\describe{
  \item{space}{A \code{\linkS4class{NeuroSpace}} object defining the spatial properties of the image.}
  \item{label}{A character string providing a label for the NeuroVec object.}
}
}

\section{Methods}{

Methods specific to NeuroVec objects may include operations for time series
analysis, 4D data manipulation, and extraction of 3D volumes or time courses.
}

\section{Usage}{

To create a NeuroVec object, use the constructor function \code{NeuroVec()}.
This function should handle the appropriate initialization of the 4D data
structure and associated spatial information.
}

\examples{

# Load an example 4D brain image
example_4d_image <- read_vec(system.file("extdata", "global_mask_v4.nii", package = "neuroim2"))

# Create a NeuroVec object
neuro_vec <- NeuroVec(data = array(rnorm(64*64*32*10), dim = c(64, 64, 32, 10)),
                      space = NeuroSpace(dim = c(64, 64, 32,10),
                      origin = c(0, 0, 0),
                      spacing = c(3, 3, 4)))


dim(neuro_vec)

# Extract a single 3D volume (e.g., the first time point)
first_volume <- neuro_vec[[1]]


# Load an example 4D brain image
example_file <- system.file("extdata", "global_mask_v4.nii", package = "neuroim2")
example_4d_image <- read_vec(example_file)

# Create a DenseNeuroVec object
dense_vec <- NeuroVec(data = example_4d_image@.Data,
                      space = space(example_4d_image))
print(dense_vec)

# Create a SparseNeuroVec object with a mask
mask <- array(runif(prod(dim(example_4d_image)[1:3])) > 0.5,
              dim = dim(example_4d_image)[1:3])
sparse_vec <- NeuroVec(data = example_4d_image@.Data,
                       space = space(example_4d_image),
                       mask = mask)
print(sparse_vec)

}
\seealso{
\code{\link{NeuroObj-class}} for the parent class.
\code{\link{DenseNeuroVec-class}} and \code{\link{SparseNeuroVec-class}} for specific implementations.

\code{\linkS4class{NeuroSpace}} for spatial information,
\code{\link{sub_vector}} for subsetting routines, and
\code{\link{index_to_coord}} for coordinate conversion.
\code{\link{DenseNeuroVec-class}}, \code{\link{SparseNeuroVec-class}} for the
specific NeuroVec types.
\code{\link{NeuroVol-class}} for 3D volumetric data.
}
