rand.test.aov1 <-
  function(x, g,
           var.equal = FALSE, median.test = FALSE,
           R = 9999, parallel = FALSE, cl = NULL,
           perm.dist = TRUE){
    # One-Way ANOVA Randomization Tests (Mean/Median)
    # Nathaniel E. Helwig (helwig@umn.edu)
    # last updated: 2026-01-14
    
    
    #########   INITIAL CHECKS   #########
    
    ### check x and g
    x <- as.matrix(x)
    g <- as.factor(g)
    N <- nrow(x)
    nvar <- ncol(x)
    if(length(g) != N) stop("Inputs 'x' and 'g' must satisfy:  nrow(x) == length(g)")
    gsize <- summary(g)
    y <- x
    
    ### check var.equal
    var.equal <- as.logical(var.equal[1])
    
    ### check median.test
    median.test <- as.logical(median.test[1])
    
    ### check R
    R <- as.integer(R)
    if(R < 1) stop("Input 'R' must be a positive integer.")
    
    ### check parallel
    parallel <- as.logical(parallel[1])
    
    ### check 'cl'
    make.cl <- FALSE
    if(parallel){
      if(is.null(cl)){
        make.cl <- TRUE
        cl <- parallel::makeCluster(2L)
      } else {
        if(!any(class(cl) == "cluster")) stop("Input 'cl' must be an object of class 'cluster'.")
      }
    }
    
    ### build design
    if(median.test){
      x <- xtx <- xtxi <- xinv <- NULL
    } else {
      y <- scale(y, scale = FALSE)
      x <- scale(model.matrix(~ g, contrasts.arg = list(g = "contr.sum"))[,-1], scale = FALSE)
      xtx <- crossprod(x)
      xtxi <- psdinv(xtx)
      xinv <- tcrossprod(xtxi, x)
    } # if(median.test)
    
    
    #########   ANOVA TEST   #########
    
    ### univariate or multivariate
    if(nvar == 1L){
      
      ## UNIVARIATE TEST
      
      ## vectorize y
      y <- as.numeric(y)
      
      ## observed test statistic
      Tstat <- Tstat.aov1(y = y, g = g, gsize = gsize, 
                          var.equal = var.equal, median.test = median.test, 
                          x = x, xtx = xtx, xtxi = xtxi, xinv = xinv)
      
      ## approximate permutation test (given input R)
      nperm <- R + 1L
      permdist <- rep(0, nperm)
      permdist[1] <- Tstat
      
      ## parallel or sequential computation?
      if(parallel){
        permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                 FUN = Tperm.aov1, 
                                                 y = y, g = g, gsize = gsize, 
                                                 var.equal = var.equal, 
                                                 median.test = median.test, 
                                                 x = x, xtx = xtx, 
                                                 xtxi = xtxi, xinv = xinv)
      } else {
        permdist[2:nperm] <- sapply(X = integer(R),
                                    FUN = Tperm.aov1, 
                                    y = y, g = g, gsize = gsize, 
                                    var.equal = var.equal, 
                                    median.test = median.test, 
                                    x = x, xtx = xtx, 
                                    xtxi = xtxi, xinv = xinv)
      } # end if(parallel)
      
      ## permutation p-value (greater than alternative = only option)
      p.value <- mean(permdist >= Tstat)
      
      
    } else {
      
      ## MULTIVARIATE TEST
      
      ## observed test statistic
      Tuni <- Tstat.aov1.mv(y = y, g = g, gsize = gsize, 
                            var.equal = var.equal, median.test = median.test,
                            x = x, xtx = xtx, xtxi = xtxi, xinv = xinv, 
                            combine = FALSE)
      Tstat <- max(Tuni)
      
      ## approximate permutation test (given input R)
      nperm <- R + 1L
      permdist <- rep(0, nperm)
      permdist[1] <- Tstat
      
      ## parallel or sequential computation?
      if(parallel){
        permdist[2:nperm] <- parallel::parSapply(cl = cl, X = integer(R), 
                                                 FUN = Tperm.aov1.mv, 
                                                 y = y, g = g, gsize = gsize, 
                                                 var.equal = var.equal, 
                                                 median.test = median.test,
                                                 x = x, xtx = xtx, 
                                                 xtxi = xtxi, xinv = xinv)
      } else {
        permdist[2:nperm] <- sapply(X = integer(R),
                                    FUN = Tperm.aov1.mv, 
                                    y = y, g = g, gsize = gsize, 
                                    var.equal = var.equal, 
                                    median.test = median.test,
                                    x = x, xtx = xtx, 
                                    xtxi = xtxi, xinv = xinv)
      } # end if(parallel)
      
      ## permutation p-value
      uni.p.value <- rep(NA, nvar)
      p.value <- mean(permdist >= Tstat)
      for(v in 1:nvar) uni.p.value[v] <- mean(permdist >= Tuni[v])
      
    } # end if(nvar == 1L)
    
    
    #########   RESULTS   #########
    
    ### return results
    if(make.cl) parallel::stopCluster(cl)
    if(!perm.dist) permdist <- NULL
    res <- list(statistic = Tstat, p.value = p.value,
                perm.dist = permdist, repeated = FALSE,
                var.equal = var.equal, median.test = median.test, R = R)
    if(nvar > 1L) {
      res$univariate <- Tuni
      res$adj.p.values <- uni.p.value
    }
    class(res) <- "rand.test.aov1"
    return(res)
    
  } # end rand.test.aov1


### permutation replication (univariate)
Tperm.aov1 <-
  function(i, y, g, gsize = summary(g), var.equal = FALSE, median.test = FALSE, 
           x = NULL, xtx = NULL, xtxi = NULL, xinv = NULL){
    Tstat <- Tstat.aov1(y = sample(y), g = g, gsize = gsize, 
                        var.equal = var.equal, median.test = median.test,
                        x = x, xtx = xtx, xtxi = xtxi, xinv = xinv)
    return(Tstat)
  } # end Tperm.aov1.R

### permutation replication (multivariate)
Tperm.aov1.mv <-
  function(i, y, g, gsize = summary(g), var.equal = FALSE, median.test = FALSE, 
           x = NULL, xtx = NULL, xtxi = NULL, xinv = NULL, combine = TRUE){
    n <- nrow(y)
    Tstat <- Tstat.aov1.mv(y = y[sample.int(n),], g = g, gsize = gsize, 
                           var.equal = var.equal, median.test = median.test,
                           x = x, xtx = xtx, xtxi = xtxi, xinv = xinv, 
                           combine = combine)
    return(Tstat)
  } # end Tperm.aov1.mv.R

### test statistic (univariate)
Tstat.aov1 <-
  function(y, g, gsize = summary(g), var.equal = FALSE, median.test = FALSE, 
           x = NULL, xtx = NULL, xtxi = NULL, xinv = NULL){
    
    if(median.test){
      if(var.equal){
        # Kruskal Wallace ANOVA
        N <- length(y)
        ry <- rank(y)
        ryg <- tapply(ry, g, mean)
        top <- sum(gsize * (ryg - (N + 1)/2)^2)
        bot <- N * (N + 1) / 12
        Tstat <- top / bot
      } else {
        # studentized Kruskal-Wallace ANOVA
        N <- length(y)
        ry <- rank(y)
        ryg <- tapply(ry, g, mean)
        top <- sum(gsize * (ryg - (N + 1)/2)^2)
        #bot^2 <- var(top) under H0: muk = mu0
        #bot^2 <- sum( gsize^2 * var( (ryg - (N + 1)/2)^2 ) )
        #bot^2 <- sum( gsize^2 * E( (ryg - (N + 1)/2)^2 ) )
        rygv <- tapply(ry, g, function(x) var(x) / length(x))
        bot <- sqrt(sum(gsize^2 * rygv))
        Tstat <- top / bot
      }
    } else {
      Tstat <- Tstat.lm1(x = x, y = y, homosced = var.equal,
                         xtx = xtx, xtxi = xtxi, xinv = xinv)
      
    } # end if(median.test)
    Tstat
  } # end Tstat.aov1.R

### test statistic (multivariate)
Tstat.aov1.mv <-
  function(y, g, gsize = summary(g), var.equal = FALSE, median.test = FALSE, 
           x = NULL, xtx = NULL, xtxi = NULL, xinv = NULL, combine = TRUE){
    nvar <- ncol(y)
    Tstat <- rep(0.0, nvar)
    for(j in 1:nvar) {
      Tstat[j] <- Tstat.aov1(y = y[,j], g = g, gsize = gsize, 
                             var.equal = var.equal, median.test = median.test,
                             x = x, xtx = xtx, xtxi = xtxi, xinv = xinv)
    }
    if(combine) Tstat <- max(Tstat)
    Tstat
  } # end Tstat.aov1.mv.R
