% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anthropic_live.R
\name{submit_anthropic_pairs_live}
\alias{submit_anthropic_pairs_live}
\title{Live Anthropic (Claude) comparisons for a tibble of pairs}
\usage{
submit_anthropic_pairs_live(
  pairs,
  model,
  trait_name,
  trait_description,
  prompt_template = set_prompt_template(),
  api_key = NULL,
  anthropic_version = "2023-06-01",
  reasoning = c("none", "enabled"),
  verbose = TRUE,
  status_every = 1,
  progress = TRUE,
  include_raw = FALSE,
  include_thoughts = NULL,
  ...
)
}
\arguments{
\item{pairs}{Tibble or data frame with at least columns \code{ID1},
\code{text1}, \code{ID2}, \code{text2}. Typically created by
\code{\link{make_pairs}}, \code{\link{sample_pairs}}, and
\code{\link{randomize_pair_order}}.}

\item{model}{Anthropic model name (for example \code{"claude-sonnet-4-5"},
\code{"claude-haiku-4-5"}, or \code{"claude-opus-4-5"}).}

\item{trait_name}{Trait name to pass to \code{anthropic_compare_pair_live}.}

\item{trait_description}{Trait description to pass to
\code{anthropic_compare_pair_live}.}

\item{prompt_template}{Prompt template string, typically from
\code{\link{set_prompt_template}}.}

\item{api_key}{Optional Anthropic API key. Defaults to
\code{Sys.getenv("ANTHROPIC_API_KEY")}.}

\item{anthropic_version}{Anthropic API version string passed as the
\code{anthropic-version} HTTP header. Defaults to \code{"2023-06-01"}.}

\item{reasoning}{Character scalar passed to
\code{\link{anthropic_compare_pair_live}} (one of \code{"none"} or
\code{"enabled"}).}

\item{verbose}{Logical; if \code{TRUE}, prints status, timing, and result
summaries.}

\item{status_every}{Integer; print status / timing for every
\code{status_every}-th pair. Defaults to 1 (every pair). Errors are always
printed.}

\item{progress}{Logical; if \code{TRUE}, shows a textual progress bar.}

\item{include_raw}{Logical; if \code{TRUE}, each row of the returned tibble
will include a \code{raw_response} list-column with the parsed JSON body
from Anthropic.}

\item{include_thoughts}{Logical or \code{NULL}; forwarded to
\code{\link{anthropic_compare_pair_live}}. When \code{TRUE} and
\code{reasoning = "none"}, the underlying calls upgrade to extended
thinking mode (\code{reasoning = "enabled"}), which implies
\code{temperature = 1} and adds a \code{thinking} block. When
\code{FALSE} or \code{NULL}, \code{reasoning} is used as-is.}

\item{...}{Additional Anthropic parameters (for example \code{temperature},
\code{top_p}, \code{max_tokens}) passed on to
\code{\link{anthropic_compare_pair_live}}.}
}
\value{
A tibble with one row per pair and the same columns as
\code{\link{anthropic_compare_pair_live}}.
}
\description{
This is a thin row-wise wrapper around
\code{\link{anthropic_compare_pair_live}}. It takes a tibble of pairs
(ID1 / text1 / ID2 / text2), submits each pair to the Anthropic
Messages API, and binds the results into a single tibble.
}
\details{
The output has the same columns as \code{\link{anthropic_compare_pair_live}},
with one row per pair, making it easy to pass into
\code{\link{build_bt_data}} and \code{\link{fit_bt_model}}.

\strong{Temperature and reasoning behaviour}

Temperature and extended-thinking behaviour are controlled by
\code{\link{anthropic_compare_pair_live}}:
\itemize{
\item When \code{reasoning = "none"} (no extended thinking), the default
\code{temperature} is \code{0} (deterministic) unless you explicitly
supply a different \code{temperature} via \code{...}.
\item When \code{reasoning = "enabled"} (extended thinking), Anthropic
requires \code{temperature = 1}. If you supply a different value, an
error is raised by \code{\link{anthropic_compare_pair_live}}.
}

If you set \code{include_thoughts = TRUE} while \code{reasoning = "none"},
the underlying calls upgrade to \code{reasoning = "enabled"}, which in turn
implies \code{temperature = 1} and adds a \code{thinking} block to the API
request. When \code{include_thoughts = FALSE} (the default), and you leave
\code{reasoning = "none"}, the effective default temperature is \code{0}.
}
\examples{
\dontrun{
# Requires ANTHROPIC_API_KEY and network access.
library(pairwiseLLM)

data("example_writing_samples", package = "pairwiseLLM")

pairs <- example_writing_samples |>
  make_pairs() |>
  sample_pairs(n_pairs = 5, seed = 123) |>
  randomize_pair_order(seed = 456)

td <- trait_description("overall_quality")
tmpl <- set_prompt_template()

# Deterministic comparisons with no extended thinking and temperature = 0
res_claude <- submit_anthropic_pairs_live(
  pairs             = pairs,
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "none",
  verbose           = TRUE,
  status_every      = 2,
  progress          = TRUE,
  include_raw       = FALSE
)

res_claude$better_id

# Comparisons with extended thinking and temperature = 1
res_claude_reason <- submit_anthropic_pairs_live(
  pairs             = pairs,
  model             = "claude-sonnet-4-5",
  trait_name        = td$name,
  trait_description = td$description,
  prompt_template   = tmpl,
  reasoning         = "enabled",
  include_thoughts  = TRUE,
  verbose           = TRUE,
  status_every      = 2,
  progress          = TRUE,
  include_raw       = TRUE
)

res_claude_reason$better_id
}

}
