% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lifetab.R
\name{lifetab}
\alias{lifetab}
\alias{lifeexp}
\title{Calculate Life Tables or Life Expectancies}
\usage{
lifetab(
  data,
  mx = NULL,
  qx = NULL,
  age = age,
  sex = NULL,
  ax = NULL,
  by = NULL,
  infant = c("constant", "linear", "CD", "AK"),
  child = c("constant", "linear", "CD"),
  closed = c("constant", "linear"),
  open = "constant",
  radix = 1e+05,
  suffix = NULL,
  n_core = 1
)

lifeexp(
  data,
  mx = NULL,
  qx = NULL,
  at = 0,
  age = age,
  sex = NULL,
  ax = NULL,
  by = NULL,
  infant = c("constant", "linear", "CD", "AK"),
  child = c("constant", "linear", "CD"),
  closed = c("constant", "linear"),
  open = "constant",
  suffix = NULL,
  n_core = 1
)
}
\arguments{
\item{data}{Data frame with mortality data.}

\item{mx}{<\code{\link[tidyselect:language]{tidyselect}}>
Mortality rates, expressed as deaths per
person-year lived. Possibly an \link[rvec:rvec]{rvec}.}

\item{qx}{<\code{\link[tidyselect:language]{tidyselect}}>
Probability of dying within age interval.
An alternative to \code{mx}. Possibly an \link[rvec:rvec]{rvec}.}

\item{age}{<\code{\link[tidyselect:language]{tidyselect}}>
Age group labels. The labels must be
interpretable by functions
such as \code{\link[=reformat_age]{reformat_age()}} and \code{\link[=age_group_type]{age_group_type()}}.
The first age group must start at age 0, and the
last age group must be "open", with no upper
limit.}

\item{sex}{<\code{\link[tidyselect:language]{tidyselect}}>
Biological sex, with labels that can be
interpreted by \code{\link[=reformat_sex]{reformat_sex()}}. Needed only when
\code{infant} is \code{"CD"} or \code{"AK"}, or \code{child} is
\code{"CD"}.}

\item{ax}{<\code{\link[tidyselect:language]{tidyselect}}>
Average age at death within age group.
Optional. See Details.}

\item{by}{<\code{\link[tidyselect:language]{tidyselect}}>
Separate life tables, or life expectancies,
calculated for each combination the \code{by} variables.
If a \code{sex} variable was specified, then that
variable is automatically included among the \code{by}
variables. If \code{data} is a
\href{https://dplyr.tidyverse.org/reference/group_data.html}{grouped}
data frame, then the grouping variables
take precedence over \code{by}.}

\item{infant}{Method used to calculate
life table values in age group \code{"0"}.
Ignored if \code{age} does not include age group \code{"0"}.
Default is \code{"constant"}.}

\item{child}{Method used to calculate
life table values in age group \code{"1-4"}.
Ignored if \code{age} does not include age group \code{"0"}.
Default is \code{"constant"}.}

\item{closed}{Method used to calculate
life table values in closed age intervals
other than \code{"0"} and \code{"1-4"} (ie intervals
such as "10-14" or "12"). Default is \code{"constant"}.}

\item{open}{Method used to calculate
life table values in the final, open age group
(eg \code{"80+"} or \code{"110+"}).
Currently the only option is `"constant".}

\item{radix}{Initial population for the
\code{lx} column. Default is \code{100000}.}

\item{suffix}{Optional suffix added to new
columns in result.}

\item{n_core}{Number of cores to use for parallel
processing. If \code{n_core} is \code{1} (the default),
no parallel processing is done.}

\item{at}{Age at which life expectancy is calculated
(\verb{lifeexp() only). Default is }0`.
Can be a vector with length > 1.}
}
\value{
A \link[tibble:tibble]{tibble}.
}
\description{
Calculate life table quantities. Function
\code{lifetab()} returns an entire life table.
Function \code{lifeexp()} returns life expectancy at birth.
The inputs can be mortality rates (\code{mx}) or
probabilities of dying (\code{qx}), though not both.
}
\section{Definitions of life table quantities}{

\itemize{
\item \code{mx} Deaths per person-year lived.
\item \code{qx} Probability of surviving from the start
of age group 'x' to the end.
\item \code{lx} Number of people alive at
the start of age group \code{x}.
\item \code{dx} Number of deaths in age group \code{x}
\item \code{Lx} Expected number of person years lived in
age group \code{x}.
\item \code{ex} Life expectancy, calculated at the start
of age group \code{x}.
}

Mortality rates \code{mx} are sometimes expressed
as deaths per 1000 person-years lived, or per 100,000
person-years lived. \code{lifetab()} and \code{lifeexp()}
assumed that they are expressed as deaths per
person-year lived.
}

\section{Calculation methods}{


\code{lifetab()} and \code{lifeexp()} implement several
methods for calculating life table quantities
from mortality rates. Each method makes
different assumptions about
the way that mortality rates vary within
age intervals:
\itemize{
\item \code{"constant"} Mortality rates are constant
within each interval.
\item \code{"linear"}. Life table quantity \code{lx}
is a straight line within each interval.
Equivalently, deaths are distributed uniformly
within each interval.
\item \code{"CD"}. Used only with age groups "0"
and "1-4". Mortality rates decline
over the  age interval,
with the slope depending on the absolute
level of infant mortality. The formulas were
developed by Coale and Demeny (1983),
and used in Preston et al (2001).
\item \code{"AK"}. Used only with age group "0".
Mortality rates decline over the age interval,
with the slope depending on the absolute
level of infant mortality. The formulas
were formulas developed by Andreev and Kingkade (2015),
and are used in the Human Mortality Database
\href{https://www.mortality.org/File/GetDocument/Public/Docs/MethodsProtocolV6.pdf}{methods protocol}.
}

For a detailed description of the methods,
see the vignette for \strong{poputils}.
}

\section{ax}{


\code{ax} is the average number of years
lived in an age interval by people who
die in that interval. Demographers sometimes
refer to it as the 'separation factor'. If a non-\code{NA}
value of \code{ax} is supplied for an age group,
then the results for that age group are based
on the formula

\deqn{m_x = d_x / (n_x l_x + a_x d_x)},

(where \code{n_x} is the width of the age interval),
over-riding any methods specified via the \code{infant}, \code{child},
\code{closed} and \code{open} arguments.
}

\section{Open age group when inputs are qx}{


The probability of dying, \code{qx}, is always 1 in the
final (open) age group. \code{qx} therefore provides
no direct information on mortality conditions
within the final age group. \code{lifetab()} and
\code{lifeexp()} use conditions in the second-to-final
age group as a proxy for conditions in the final
age group. When \code{open} is \code{"constant"} (which
is currently the only option), and no value
for \code{ax} in the final age group is provided,
\code{lifetab()} and \code{lifeexp()} assume
that \eqn{m_A = m_{A-1}}, and set
\eqn{L_{A} = l_A / m_A}.

In practice, mortality is likely to be higher
in the final age group than in the second-to-final
age group, so the default procedure is likely to
lead to inaccuracies. When the size of the final
age group is very small, these inaccuracies will
be inconsequential. But in other cases, it may
be necessary to supply an explicit value for
\code{ax} for the final age group, or to use \code{mx}
rather than \code{qx} as inputs.
}

\section{Using rvecs to represent uncertainty}{


An \link[rvec:rvec]{rvec} is a 'random vector',
holding multiple draws from a distribution.
Using an rvec for the \code{mx} argument to
\code{lifetab()} or \code{lifeexp()} is a way of representing
uncertainty. This uncertainty is propagated
through to the life table values, which will
also be rvecs.
}

\section{Parallel processing}{


Calculations can be slow when working
with rvecs and many combinations of 'by' variables.
In these cases, setting \code{n_core} to a number
greater than 1, which triggers parallel processing,
may help.
}

\examples{
library(dplyr)

## life table for females based on 'level 1'
## mortality rates "West" model life table
west_lifetab |>
    filter(sex == "Female",
           level == 1) |>
    lifetab(mx = mx)

## change method for infant and children from
## default ("constant") to "CD"
west_lifetab |>
    filter(sex == "Female",
           level == 1) |>
    lifetab(mx = mx,
            sex = sex,
            infant = "CD",
            child = "CD")

## calculate life expectancies
## for all levels, using the 'by'
## argument to distinguish levels
west_lifetab |>
    lifeexp(mx = mx,
            sex = sex,
            infant = "CD",
            child = "CD",
            by = level)

## obtain the same result using
## 'group_by'
west_lifetab |>
  group_by(level) |>
  lifeexp(mx = mx,
          sex = sex,
          infant = "CD",
          child = "CD")

## calculations based on 'qx'
west_lifetab |>
  lifeexp(qx = qx,
          sex = sex,
          by = level)

## life expectancy at age 60
west_lifetab |>
  filter(level == 10) |>
  lifeexp(mx = mx,
          at = 60,
          sex = sex)

## life expectancy at ages 0 and 60
west_lifetab |>
  filter(level == 10) |>
  lifeexp(mx = mx,
          at = c(0, 60),
          sex = sex)
}
\references{
\itemize{
\item Preston SH, Heuveline P, and Guillot M. 2001.
\emph{Demography: Measuring and Modeling Population Processes}
Oxford: Blackwell.
\item Coale AJ, Demeny P,  and Vaughn B. 1983.
\emph{Regional model life tables and stable populations}
New York: Academic Press.
\item Andreev, E.M. and Kingkade, W.W., 2015.
Average age at death in infancy and infant mortality level:
Reconsidering the Coale-Demeny formulas at current
levels of low mortality. \emph{Demographic Research},
33, pp.363-390.
\item Human Mortality Database \href{https://www.mortality.org/File/GetDocument/Public/Docs/MethodsProtocolV6.pdf}{Methods Protocol}.
\item \href{https://demographicestimation.iussp.org}{Tools for Demographic Estimation}.
}
}
\seealso{
\itemize{
\item \code{\link[=ex_to_lifetab_brass]{ex_to_lifetab_brass()}} Calculate life table from minimal inputs
\item \code{\link[=q0_to_m0]{q0_to_m0()}} Convert between infant mortality measures
\item \code{\link[=tfr]{tfr()}} Calculate total fertility rate
}
}
