% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weights.R
\name{wt_ate}
\alias{wt_ate}
\alias{wt_ate.data.frame}
\alias{wt_att}
\alias{wt_att.data.frame}
\alias{wt_atu}
\alias{wt_atu.data.frame}
\alias{wt_atm}
\alias{wt_atm.data.frame}
\alias{wt_ato}
\alias{wt_ato.data.frame}
\alias{wt_entropy}
\alias{wt_entropy.data.frame}
\alias{wt_atc}
\alias{wt_cens}
\alias{wt_cens.data.frame}
\title{Calculate propensity score weights}
\usage{
wt_ate(
  .propensity,
  .exposure,
  .sigma = NULL,
  exposure_type = c("auto", "binary", "categorical", "continuous"),
  .focal_level = NULL,
  .reference_level = NULL,
  stabilize = FALSE,
  stabilization_score = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_ate}{data.frame}(
  .propensity,
  .exposure,
  .sigma = NULL,
  exposure_type = c("auto", "binary", "categorical", "continuous"),
  .focal_level = NULL,
  .reference_level = NULL,
  stabilize = FALSE,
  stabilization_score = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)

wt_att(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_att}{data.frame}(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)

wt_atu(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_atu}{data.frame}(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)

wt_atm(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_atm}{data.frame}(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)

wt_ato(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_ato}{data.frame}(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)

wt_entropy(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_entropy}{data.frame}(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)

wt_atc(
  .propensity,
  .exposure,
  exposure_type = c("auto", "binary", "categorical"),
  .focal_level = NULL,
  .reference_level = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

wt_cens(
  .propensity,
  .exposure,
  .sigma = NULL,
  exposure_type = c("auto", "binary", "categorical", "continuous"),
  .focal_level = NULL,
  .reference_level = NULL,
  stabilize = FALSE,
  stabilization_score = NULL,
  ...,
  .treated = NULL,
  .untreated = NULL
)

\method{wt_cens}{data.frame}(
  .propensity,
  .exposure,
  .sigma = NULL,
  exposure_type = c("auto", "binary", "categorical", "continuous"),
  .focal_level = NULL,
  .reference_level = NULL,
  stabilize = FALSE,
  stabilization_score = NULL,
  ...,
  .propensity_col = NULL,
  .treated = NULL,
  .untreated = NULL
)
}
\arguments{
\item{.propensity}{Propensity scores in one of several forms:
\itemize{
\item A \strong{numeric vector} of predicted probabilities (binary/continuous).
\item A \strong{data frame} or matrix with one column per exposure level
(categorical), or two columns for binary (see \code{.propensity_col}).
\item A fitted \strong{\code{glm}} object -- fitted values are extracted automatically.
\item A modified propensity score created by \code{\link[=ps_trim]{ps_trim()}}, \code{\link[=ps_trunc]{ps_trunc()}},
\code{\link[=ps_refit]{ps_refit()}}, or \code{\link[=ps_calibrate]{ps_calibrate()}}.
}}

\item{.exposure}{The exposure (treatment) variable. For binary exposures, a
numeric 0/1 vector, logical, or two-level factor. For categorical
exposures, a factor or character vector. For continuous exposures, a
numeric vector. Optional when \code{.propensity} is a \code{glm} object (extracted
from the model).}

\item{.sigma}{Numeric vector of observation-level standard deviations for
continuous exposures (e.g., \code{influence(model)$sigma}). Extracted
automatically when \code{.propensity} is a \code{glm} object.}

\item{exposure_type}{Type of exposure: \code{"auto"} (default), \code{"binary"},
\code{"categorical"}, or \code{"continuous"}. \code{"auto"} detects the type from
\code{.exposure}.}

\item{.focal_level}{The value of \code{.exposure} representing the focal
(treated) group. For binary exposures, defaults to the higher value.
Required for \code{wt_att()} and \code{wt_atu()} with categorical exposures.}

\item{.reference_level}{The value of \code{.exposure} representing the reference
(control) group. Automatically detected if not supplied.}

\item{stabilize}{If \code{TRUE}, multiply weights by an estimate of the marginal
treatment probability (binary) or density (continuous). Only supported by
\code{wt_ate()} and \code{wt_cens()}. See \strong{Stabilization} in Details.}

\item{stabilization_score}{Optional numeric value to use as the
stabilization multiplier instead of the default (the marginal mean of
\code{.exposure}). Ignored when \code{stabilize = FALSE}.}

\item{...}{These dots are for future extensions and must be empty.}

\item{.treated}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{.focal_level} instead.}

\item{.untreated}{\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} Use \code{.reference_level} instead.}

\item{.propensity_col}{Column to use when \code{.propensity} is a data frame
with a binary exposure. Accepts a column name (quoted or unquoted) or
numeric index. Defaults to the second column. Ignored for categorical
exposures, where all columns are used.}
}
\value{
A \code{\link{psw}} vector (a double vector with class \code{psw}) carrying
these attributes:
\itemize{
\item \code{estimand}: character, e.g. \code{"ate"}, \code{"att"}, \code{"uncensored"}.
\item \code{stabilized}: logical, whether stabilization was applied.
\item \code{trimmed}: logical, whether the propensity scores were trimmed.
\item \code{truncated}: logical, whether the propensity scores were truncated.
\item \code{calibrated}: logical, whether the propensity scores were calibrated.
}
}
\description{
Compute inverse probability weights for causal inference under different
estimands. Each function targets a different population:
\itemize{
\item \code{wt_ate()}: \strong{Average Treatment Effect} -- the full population.
\item \code{wt_att()}: \strong{Average Treatment Effect on the Treated} -- the treated
(focal) group.
\item \code{wt_atu()}: \strong{Average Treatment Effect on the Untreated} -- the
untreated (reference) group. \code{wt_atc()} is an alias.
\item \code{wt_atm()}: \strong{Average Treatment Effect for the Evenly Matchable} --
units with the most overlap.
\item \code{wt_ato()}: \strong{Average Treatment Effect for the Overlap Population} --
weights proportional to overlap.
\item \code{wt_entropy()}: \strong{Entropy-weighted Average Treatment Effect} --
an entropy-balanced population.
\item \code{wt_cens()}: \strong{Inverse probability of censoring weights} -- uses the
same formula as \code{wt_ate()} but labels the estimand \code{"uncensored"}. Use
these to adjust for censoring in survival analysis, not for treatment
weighting.
}

\code{.propensity} accepts a numeric vector of predicted probabilities, a
\code{data.frame} of per-level probabilities, a fitted \code{glm} object, or a
modified propensity score created by \code{\link[=ps_trim]{ps_trim()}}, \code{\link[=ps_trunc]{ps_trunc()}},
\code{\link[=ps_refit]{ps_refit()}}, or \code{\link[=ps_calibrate]{ps_calibrate()}}.

All functions return a \code{\link{psw}} object -- a numeric vector that tracks the
estimand, stabilization status, and any trimming or truncation applied.
}
\details{
\subsection{Exposure types}{

All weight functions support binary exposures. \code{wt_ate()} and \code{wt_cens()}
also support continuous exposures. All except \code{wt_cens()} support
categorical exposures.
\itemize{
\item \strong{Binary}: \code{.exposure} is a two-level vector (e.g., 0/1, logical, or a
two-level factor). \code{.propensity} is a numeric vector of P(treatment | X).
\item \strong{Categorical}: \code{.exposure} is a factor or character vector with 3+
levels. \code{.propensity} must be a matrix or data frame with one column per
level, where rows sum to 1.
\item \strong{Continuous}: \code{.exposure} is a numeric vector. \code{.propensity} is a
vector of conditional means (fitted values). Weights use a normal density
ratio; stabilization is strongly recommended.
\item \strong{Auto} (default): Detects the exposure type from \code{.exposure}.
}
}

\subsection{Stabilization}{

Setting \code{stabilize = TRUE} multiplies the base weight by an estimate of
P(A) (binary) or f_A(A) (continuous), reducing variance. When no
\code{stabilization_score} is supplied, the marginal mean of \code{.exposure} is
used. Stabilization is supported for ATE and censoring weights
(\code{wt_ate()} and \code{wt_cens()}) and is strongly recommended for continuous
exposures.
}

\subsection{Handling extreme weights}{

Extreme weights signal positivity violations, poor model fit, or limited
overlap. You can address them by:
\itemize{
\item Choosing an overlap-focused estimand (\code{wt_ato()}, \code{wt_atm()},
\code{wt_entropy()}), which down-weight units in regions of poor overlap.
\item Trimming (\code{\link[=ps_trim]{ps_trim()}}) or truncating (\code{\link[=ps_trunc]{ps_trunc()}}) propensity scores
before computing weights.
\item Calibrating weights with \code{\link[=ps_calibrate]{ps_calibrate()}}.
\item Stabilizing ATE weights (\code{stabilize = TRUE}).
}

See the \href{https://CRAN.R-project.org/package=halfmoon}{halfmoon} package
for weight diagnostics and visualization.
}

\subsection{Weight formulas}{
\subsection{Binary exposures}{

For binary treatments (\eqn{A \in \{0, 1\}}), with propensity score
\eqn{e(X) = P(A=1 \mid X)}:
\itemize{
\item \strong{ATE}: \eqn{w = \frac{A}{e(X)} + \frac{1-A}{1-e(X)}}
\item \strong{ATT}: \eqn{w = A + \frac{(1-A) \cdot e(X)}{1-e(X)}}
\item \strong{ATU}: \eqn{w = \frac{A \cdot (1-e(X))}{e(X)} + (1-A)}
\item \strong{ATM}: \eqn{w = \frac{\min(e(X), 1-e(X))}{A \cdot e(X) + (1-A) \cdot (1-e(X))}}
\item \strong{ATO}: \eqn{w = A \cdot (1-e(X)) + (1-A) \cdot e(X)}
\item \strong{Entropy}: \eqn{w = \frac{h(e(X))}{A \cdot e(X) + (1-A) \cdot (1-e(X))}}, where \eqn{h(e) = -[e \log(e) + (1-e) \log(1-e)]}
}
}

\subsection{Continuous exposures}{

Weights use the density ratio
\eqn{w = f_A(A) / f_{A|X}(A \mid X)}, where \eqn{f_A} is the marginal
density and \eqn{f_{A|X}} is the conditional density (both assumed
normal). Only \code{wt_ate()} and \code{wt_cens()} support continuous exposures.
}

\subsection{Categorical exposures}{

For \eqn{K}-level treatments, weights take the tilting-function form
\eqn{w_i = h(\mathbf{e}_i) / e_{i,Z_i}}, where \eqn{e_{i,Z_i}} is the
propensity for unit \eqn{i}'s observed level and \eqn{h(\cdot)} depends
on the estimand:
\itemize{
\item \strong{ATE}: \eqn{h(\mathbf{e}) = 1}
\item \strong{ATT}: \eqn{h(\mathbf{e}) = e_{\text{focal}}}
\item \strong{ATU}: \eqn{h(\mathbf{e}) = 1 - e_{\text{focal}}}
\item \strong{ATM}: \eqn{h(\mathbf{e}) = \min(e_1, \ldots, e_K)}
\item \strong{ATO}: \eqn{h(\mathbf{e}) = \bigl(\sum_k 1/e_k\bigr)^{-1}}
\item \strong{Entropy}: \eqn{h(\mathbf{e}) = -\sum_k e_k \log(e_k)}
}
}

}
}
\examples{
# -- Binary exposure, numeric propensity scores ----------------------
set.seed(123)
ps <- runif(100, 0.1, 0.9)
trt <- rbinom(100, 1, ps)

wt_ate(ps, trt)
wt_att(ps, trt)
wt_atu(ps, trt)
wt_atm(ps, trt)
wt_ato(ps, trt)
wt_entropy(ps, trt)

# Stabilized ATE weights (reduces variance)
wt_ate(ps, trt, stabilize = TRUE)

# Inspect the result
w <- wt_ate(ps, trt)
estimand(w)
summary(w)

# -- Overlap-focused estimands handle extreme PS better --------------
ps_extreme <- c(0.01, 0.02, 0.98, 0.99, rep(0.5, 4))
trt_extreme <- c(0, 0, 1, 1, 0, 1, 0, 1)

max(wt_ate(ps_extreme, trt_extreme))
max(wt_ato(ps_extreme, trt_extreme))

# -- From a fitted GLM -----------------------------------------------
x1 <- rnorm(100)
x2 <- rnorm(100)
trt2 <- rbinom(100, 1, plogis(0.5 * x1 + 0.3 * x2))
ps_model <- glm(trt2 ~ x1 + x2, family = binomial)

# Exposure is extracted from the model automatically
wt_ate(ps_model)

# -- Data frame input ------------------------------------------------
ps_df <- data.frame(
  control = c(0.9, 0.7, 0.3, 0.1),
  treated = c(0.1, 0.3, 0.7, 0.9)
)
exposure <- c(0, 0, 1, 1)
wt_ate(ps_df, exposure)
wt_ate(ps_df, exposure, .propensity_col = "treated")

# -- Censoring weights -----------------------------------------------
cens_ps <- runif(50, 0.6, 0.95)
cens_ind <- rbinom(50, 1, cens_ps)
wt_cens(cens_ps, cens_ind)
estimand(wt_cens(cens_ps, cens_ind))  # "uncensored"

}
\references{
Barrett, M., D'Agostino McGowan, L., & Gerke, T. \emph{Causal Inference in R}.
\url{https://www.r-causal.org/}

Rosenbaum, P. R., & Rubin, D. B. (1983). The central role of the propensity
score in observational studies for causal effects. \emph{Biometrika}, 70(1),
41--55.

Li, L., & Greene, T. (2013). A weighting analogue to pair matching in
propensity score analysis. \emph{The International Journal of Biostatistics},
9(2), 215--234. (ATM weights)

Li, F., Morgan, K. L., & Zaslavsky, A. M. (2018). Balancing covariates via
propensity score weighting. \emph{Journal of the American Statistical
Association}, 113(521), 390--400. (ATO weights)

Zhou, Y., Matsouaka, R. A., & Thomas, L. (2020). Propensity score weighting
under limited overlap and model misspecification. \emph{Statistical Methods in
Medical Research}, 29(12), 3721--3756. (Entropy weights)

Hirano, K., & Imbens, G. W. (2004). The propensity score with continuous
treatments. In \emph{Applied Bayesian Modeling and Causal Inference from
Incomplete-Data Perspectives} (pp. 73--84).

Austin, P. C., & Stuart, E. A. (2015). Moving towards best practice when
using inverse probability of treatment weighting (IPTW). \emph{Statistics in
Medicine}, 34(28), 3661--3679.
}
\seealso{
\itemize{
\item \code{\link[=psw]{psw()}} for the returned weight vector class.
\item \code{\link[=ps_trim]{ps_trim()}}, \code{\link[=ps_trunc]{ps_trunc()}}, \code{\link[=ps_refit]{ps_refit()}}, and \code{\link[=ps_calibrate]{ps_calibrate()}} for
modifying propensity scores before weighting.
\item \code{\link[=ipw]{ipw()}} for inverse-probability-weighted estimation of causal effects.
}
}
