% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smartrank.R
\name{smartrank}
\alias{smartrank}
\title{Rank a vector based on either alphabetical or frequency order}
\usage{
smartrank(
  x,
  sort_by = c("alphabetical", "frequency"),
  desc = FALSE,
  ties.method = "average",
  na.last = TRUE,
  freq_tiebreak = c("match_desc", "asc", "desc"),
  verbose = TRUE
)
}
\arguments{
\item{x}{A numeric, character, or factor vector}

\item{sort_by}{Sort ranking either by "alphabetical" or "frequency" . Default is "alphabetical"}

\item{desc}{A logical indicating whether the ranking should be in descending ( TRUE ) or ascending ( FALSE ) order.
When input is numeric, ranking is always based on numeric order.}

\item{ties.method}{a character string specifying how ties are treated,
    see \sQuote{Details}; can be abbreviated.}

\item{na.last}{a logical or character string controlling the treatment
    of \code{\link[base]{NA}}s. If \code{TRUE}, missing values in the data are
    put last; if \code{FALSE}, they are put first; if \code{NA}, they
    are removed; if \code{"keep"} they are kept with rank \code{NA}.}

\item{freq_tiebreak}{Controls how alphabetical tie-breaking works when
\code{sort_by = "frequency"} and \code{x} is character/factor/logical. Must be
one of:
\itemize{
\item \code{"match_desc"} (default): alphabetical tie-breaking direction
follows \code{desc} (ascending when \code{desc = FALSE}, descending when
\code{desc = TRUE}).
\item \code{"asc"}: ties are always broken by \strong{ascending} alphabetical
order, regardless of \code{desc}.
\item \code{"desc"}: ties are always broken by \strong{descending} alphabetical
order, regardless of \code{desc}.
}}

\item{verbose}{verbose (flag)}
}
\value{
The ranked vector
}
\description{
This function acts as a drop-in replacement for the base \code{rank()} function with the added option to:
\enumerate{
\item Rank categorical factors based on frequency instead of alphabetically
\item Rank in descending or ascending order
}
}
\details{
If \code{x} includes ‘ties’ (equal values), the \code{ties.method} argument determines how the rank value is decided. Must be one of:
\itemize{
\item \strong{average}: replaces integer ranks of tied values with their average  (default)
\item \strong{first}: first-occurring value is assumed to be the lower rank (closer to one)
\item \strong{last}: last-occurring value is assumed to be the lower rank (closer to one)
\item \strong{max} or \strong{min}: integer ranks of tied values are replaced with their maximum and minimum respectively (latter is typical in sports-ranking)
\item \strong{random} which of the tied values are higher / lower rank is randomly decided.
}

NA values are never considered to be equal:
for na.last = TRUE and na.last = FALSE
they are given distinct ranks in the order in which they occur in x.
}
\note{
When \code{sort_by = "frequency"}, ties based on frequency are broken by
alphabetical order of the terms. Use \code{freq_tiebreak} to control whether
that alphabetical tie-breaking is ascending, descending, or follows
\code{desc}.

When \code{sort_by = "frequency"} and input is character, ties.method is ignored. Each distinct element level gets its own rank, and each rank is 1 unit away from the next element, irrespective of how many duplicates
}
\examples{

# ------------------
## CATEGORICAL INPUT
# ------------------

fruits <- c("Apple", "Orange", "Apple", "Pear", "Orange")

# rank alphabetically
smartrank(fruits)
#> [1] 1.5 3.5 1.5 5.0 3.5

# rank based on frequency
smartrank(fruits, sort_by = "frequency")
#> [1] 2.5 4.5 2.5 1.0 4.5

# rank based on descending order of frequency
smartrank(fruits, sort_by = "frequency", desc = TRUE)
#> [1] 1.5 3.5 1.5 5.0 3.5

# sort fruits vector based on rank
ranks <- smartrank(fruits,sort_by = "frequency", desc = TRUE)
fruits[order(ranks)]
#> [1] "Apple"  "Apple"  "Orange" "Orange" "Pear"


# ------------------
## NUMERICAL INPUT
# ------------------

# rank numerically
smartrank(c(1, 3, 2))
#> [1] 1 3 2

# rank numerically based on descending order
smartrank(c(1, 3, 2), desc = TRUE)
#> [1] 3 1 2

# always rank numeric vectors based on values, irrespective of sort_by
smartrank(c(1, 3, 2), sort_by = "frequency")
#> smartrank: Sorting a non-categorical variable. Ignoring `sort_by` and sorting numerically
#> [1] 1 3 2
}
