\name{recluster.cons}
\alias{recluster.cons}
\title{
Consensus tree from row-order permutations of a dissimilarity matrix
}
\description{
This function builds a set of hierarchical clustering trees by repeatedly permuting the row order of the input dissimilarity matrix (or of the community matrix used to compute it). A majority-rule consensus tree is then computed from these trees. The aim is to obtain a topology that is robust to artefacts caused by ties and zero values, and therefore insensitive to the original row order.
}
\usage{
recluster.cons(mat, phylo = NULL, tr = 100, p = 0.5,
dist = "simpson", method = "average",
blenghts = TRUE, select = FALSE)
}
\arguments{
  \item{mat}{
A community-by-site matrix (rows = sites/areas, columns = species) or a \code{\link[stats]{dist}} object.
Row names are required.
}

  \item{phylo}{
An ultrametric rooted phylogeny with tip labels matching \code{colnames(mat)}.
Only required to compute phylogenetic beta-diversity in \code{\link{recluster.dist}}.
}

  \item{tr}{
Number of permutations/trees used to compute the consensus.
}

  \item{p}{
A numeric value between 0.5 and 1 giving the minimum proportion of trees in which a clade must occur to be represented in the consensus tree (as in \code{\link[ape]{consensus}}).
}

  \item{dist}{
A beta-diversity index supported by \code{\link{recluster.dist}} (default: Simpson turnover) or a custom binary dissimilarity specified using \code{\link[vegan]{designdist}} syntax. Ignored when \code{mat} is a \code{dist} object.
}

  \item{method}{
Any clustering method allowed by \code{\link[stats]{hclust}} (e.g. \code{"average"}, \code{"complete"}, \code{"ward.D2"}).
}

  \item{blenghts}{
Logical. If \code{TRUE}, branch lengths are estimated by non-negative least squares using \code{\link[phangorn]{nnls.tree}} (and the consensus is first assigned Grafen branch lengths).
}

  \item{select}{
Logical. If \code{TRUE}, residual sum of squares (RSS) from \code{\link[phangorn]{nnls.tree}} is computed for each permuted tree and only trees with RSS below the median are retained for the consensus.
}
}
\details{
If \code{mat} is not a \code{dist} object, dissimilarities are computed by \code{\link{recluster.dist}} (optionally using \code{phylo} for phylogenetic indices).
For each permutation, a hierarchical clustering is computed from the permuted dissimilarity matrix and converted to a \code{phylo} object with \code{\link[ape]{as.phylo}}.

When \code{select = TRUE}, trees are optionally filtered based on their least-squares fit to the distance matrix (via \code{\link[phangorn]{nnls.tree}}), which may reduce polytomies in the consensus by discarding poorly fitting topologies.
}
\value{
A list with components:
\describe{
  \item{\code{cons}}{
The consensus tree (an object of class \code{phylo}; forced to be fully dichotomous via \code{\link[ape]{multi2di}}).
  }
  \item{\code{trees}}{
The list of permuted trees used to compute the consensus.
  }
  \item{\code{RSS}}{
If \code{select = TRUE}, a list (or vector) of RSS values for each tree; otherwise \code{NULL}.
  }
}
}
\references{
Dapporto L., Ramazzotti M., Fattorini S., Talavera G., Vila R., Dennis R. (2013).
recluster: an unbiased clustering procedure for beta-diversity turnover.
\emph{Ecography}, 36, 1070--1075.
}
\author{
Leonardo Dapporto and Matteo Ramazzotti
}
\examples{
data(datamod, treemod)

tree <- recluster.cons(datamod, tr = 10)
plot(tree$cons, direction = "downwards")

# Calculate any dissimilarity matrix with another package
#An example with betapart

data(dataisl)
if (requireNamespace("betapart", quietly = TRUE)) {

  dsim <- betapart::beta.pair(dataisl)$beta.sim
  tree <- recluster.cons(dsim, tr = 10, method = "average")
  plot(tree$cons, direction = "downwards")

}


}
\keyword{cluster}
