test_that("viz_auc_svyglm runs on synthetic data and returns plot + stats", {

  set.seed(123)
  n <- 800

  # ---------------------------
  # synthetic data
  # ---------------------------
  synthetic_svy <- data.frame(
    psu      = sample(1:80, n, replace = TRUE),
    strata   = sample(1:40, n, replace = TRUE),
    weight   = runif(n, 0.5, 3),
    exposure = rbinom(n, 1, 0.45),
    age      = round(rnorm(n, 50, 12)),
    sex      = factor(sample(c("Male", "Female"), n, replace = TRUE)),
    bmi      = round(rnorm(n, 27, 4), 1)
  )

  linpred <- -2 + 0.8 * synthetic_svy$exposure + 0.03 * synthetic_svy$age
  synthetic_svy$outcome <- rbinom(n, 1, plogis(linpred))

  # ---------------------------
  # fit main model
  # ---------------------------
  fit_main <- final_svyglm(
    data = synthetic_svy,
    dep_var = "outcome",
    covariates = c("age", "sex", "bmi"),
    id_var = "psu",
    strata_var = "strata",
    weight_var = "weight",
    family = "binomial",
    level = 0.95,
    interaction_terms = NULL
  )

  # ---------------------------
  # generate ROC output
  # ---------------------------
  res <- viz_auc_svyglm(fit_main)

  # ---------------------------
  # structure checks
  # ---------------------------
  expect_type(res, "list")
  expect_named(res, c("plot", "stats"))

  # ---------------------------
  # plot checks
  # ---------------------------
  expect_s3_class(res$plot, "ggplot")
  expect_gt(length(res$plot$layers), 0)

  # ---------------------------
  # stats checks
  # ---------------------------
  expect_named(res$stats, c("AUC", "SE", "LCI", "UCI"))
  expect_true(is.numeric(res$stats))
  expect_length(res$stats, 4)

  # sanity bounds
  expect_gte(res$stats["AUC"], 0)
  expect_lte(res$stats["AUC"], 1)
  expect_lte(res$stats["LCI"], res$stats["AUC"])
  expect_gte(res$stats["UCI"], res$stats["AUC"])
})
