################################################################################
# Performs two one-sided tests for 0th-order stochastic equivalence in unpaired 
# data
# Author: Alexis Dinno <alexis.dinno@pdx.edu>
# version 3.1.9 
# Date: Feb 06, 2026

equivalence.types <- c("delta", "epsilon")

tost.rank.sum <- function(
  x,
  by,
  eqv.type    = equivalence.types,
  eqv.level   = 1,
  upper       = NA,
  conf.level  = 0.95, 
  x.name      = "",
  by.name     = "",
  by.values   = NULL,
  ccontinuity = FALSE,
  relevance   = TRUE) {

  # Make eqv.type lower case
  # Validate and sanitize eqv.type
  if (length(eqv.type) > 1) {
    eqv.type <- "epsilon"
    }
  eqv.type <- tolower(eqv.type)
  if ( eqv.type != "delta" & eqv.type != "epsilon" ) {
    rlang::abort(message="option eqv.type() must be either delta or epsilon")
    }
  if (eqv.type == "delta") {
    eqv.type.display <- "\U0394"
    }
  if (eqv.type == "epsilon") {
    eqv.type.display <- "\U03B5"
    }

  # Validate eqv.level and upper
  if ( eqv.level <= 0 && is.na(upper) ) {
    rlang::abort(message="eqv.level incorrectly specified\n the tolerance must be a positive real value")
    }

  if ( eqv.level <= 0 && !is.na(upper) && upper>0 && eqv.type=="epsilon" ) {
    lower <- abs(eqv.level)
    }

  # Validate upper
  if (!is.na(upper) && upper<0) {
    rlang::abort(message="upper incorrectly specified\n the tolerance must be a positive real value for asymmatric equivalence intervals")
    }
 
  if (is.na(upper) || upper == abs(eqv.level)) {
    upper <- abs(eqv.level)
    lower <- abs(eqv.level)
    if (eqv.type == "delta") {
      eqv.level.display <- trimws(sprintf("%-7.0f",eqv.level))
      }
    if (eqv.type == "epsilon") {
      eqv.level.display <- trimws(sprintf("%-8.4f",eqv.level))
      }
    }

  if (!is.na(upper) && upper>0) {
    upper <- abs(upper)
    lower <- abs(eqv.level)
    if (eqv.type == "delta") {
      eqv.type.lower.display <- "\U0394l"
      eqv.type.upper.display <- "\U0394u"
      }
    if (eqv.type == "epsilon") {
      eqv.type.lower.display <- "\U03B5l"
      eqv.type.upper.display <- "\U03B5u"
      }
    upper.display <- trimws(sprintf("%#-8.5g",upper))
    lower.display <- trimws(sprintf("%#-8.5g",-1*lower))
    }

  # Validate conf.level
  if (conf.level > 1 | conf.level < 0) {
    rlang::abort(message="conf.level must be between 0 and 1 inclusive")
    }

  # Validate conf.level and create alpha
  alpha <- 1 - conf.level
  if (alpha <= 0 | alpha >= 1) {
    rlang::abort(message="conf.level must be >0 and <1")
    }
  # Set levels for groups
  by.levels <- levels(as.factor(by))

  # Create conclusion strings
  pos.decision <- "Reject"
  neg.decision <- "Reject"

## VALIDATE THE BY OPTION HERE
  ## Come back and deal with NA values
  if ( length(unique(by)) != 2 ) {
    rlang::abort(message="by must be a grouping variable with exactly two values")
    }
## Possibly create weights, if, and by options like in Stata
  # Set display names for variables
  # Check if the by variable is labeled, and if so assign those labels
  if (!is.null(attr(by,"labels"))) {
    by.names <- names(attributes(sort(by))$labels)
    }
  # Otherwise use the values of the by variable
   else {
    by.names <- levels(as.factor(sort(by)))
    }
  if (is.null(by.values)) {
    by.values <- by.names
    }
  group.1.name <- pad.left(by.values[1], 12)
  group.2.name <- pad.left(by.values[2], 12)
  if (x.name=="") {
    x.name <- pad.left(gsub(".*\\$","",deparse(substitute(x))),8)
    }
  if (by.name=="") {
    by.name <- pad.left(gsub(".*\\$","",deparse(substitute(by))),12)
    }
  # Will will display x & y this way in the output if by is specified
  name.stem <- trimws(x.name)
  # Create table.title, mid.bar, and bottom.bar
  table.title <- paste0(
                   "\n",
                   pad.spaces(8),
                   "Sign \U2502",
                   pad.spaces(6),
                   "Obs",
                   pad.spaces(4),
                   "Sum ranks",
                   pad.spaces(4),
                   "Expected", collapse="")
  mid.bar     <- paste0(pad.horizontal(13),"\U253C",pad.horizontal(34), collapse="")
  bottom.bar  <- paste0(pad.spaces(21),pad.horizontal(10), collapse="")
  # Identify groups 1 and 2, and the group1 and 2 data
  g1         <- as.numeric(by.levels[1])
  x1         <- x[by==g1]
  n1         <- length(x1)
  n1.display <- sprintf("%8.0f",n1)
  g2         <- as.numeric(by.levels[2])
  x2         <- x[by==g2]
  n2         <- length(x2)
  n2.display <- sprintf("%8.0f",n2)
  n          <- n1 + n2
  n.display  <- sprintf("%8.0f",n)

  # Rank the data
  ranks <- rank(x,ties.method="average",na.last=NA)
  rank.sum.display   <- sprintf("%10.0f",sum(c(1:n)))
  rank.sum.2         <- sum(ranks[by==g2])
  rank.sum.2.display <- sprintf("%10.0f",rank.sum.2)
  
  # Create group 1 statistics
  W             <- sum(ranks[by==g1])
  W.display     <- sprintf("%10.0f",W)
  mu.W          <- (n1*(n+1))/2
  mu.W.display  <- sprintf("%10.0f",mu.W)
  sigma.sq.W    <- (n1*n2*(n+1))/12
  if (sigma.sq.W < 1e7) {
    sigma.sq.W.display <- sprintf("%10.2f",sigma.sq.W) 
    }
   else {
    sigma.sq.W.display <- sprintf("%12.0f",sigma.sq.W) 
    }
  v             <- n1*n2*var(ranks)/n
  ties.adjust   <- v - sigma.sq.W
  if (ties.adjust < 1e7) {
    ties.adjust.display <- sprintf("%10.2f",ties.adjust) 
    }
   else {
    ties.adjust.display <- sprintf("%12.0f",ties.adjust) 
    }
  sigma.sq.W.adjust <- sigma.sq.W + ties.adjust
  if (sigma.sq.W.adjust < 1e7) {
    sigma.sq.W.adjust.display <- sprintf("%10.2f",sigma.sq.W.adjust) 
    }
   else {
    sigma.sq.W.adjust.display <- sprintf("%12.0f",sigma.sq.W.adjust) 
    }
  sigma.W       <- sqrt(sigma.sq.W)
  # Set continuity correction if requested
  cont.cor <- 0
    if (ccontinuity) {
    cont.cor <- 0.5
    }
  # Positivist test statistic
  z.pos         <- (sign(W-mu.W)*(abs(W-mu.W) - cont.cor))/sqrt(sigma.sq.W.adjust)
  z.pos.display <- sprintf("%-7.3f",z.pos)
  p.pos         <- 2*pnorm(abs(z.pos),lower.tail=FALSE)
  p.pos.display <- format.extreme.p.vals(p.pos)
  # Rejection decision for two-sided test
  if ( p.pos > alpha ) {
    pos.decision <- "Fail to reject"
    }
  # Negativist test statistics
  if ( eqv.type == "delta" ) {
    z1 <- (upper - (sign(W-mu.W)*(abs(W-mu.W) - cont.cor)))/sqrt(sigma.sq.W.adjust)
    z2 <- ((sign(W-mu.W)*(abs(W-mu.W) - cont.cor)) + lower)/sqrt(sigma.sq.W.adjust)
    }
  if ( eqv.type == "epsilon" ) {
    z1 <- upper - ( (sign(W-mu.W)*(abs(W-mu.W) - cont.cor))/sqrt(sigma.sq.W.adjust) )
    z2 <- ( (sign(W-mu.W)*(abs(W-mu.W) - cont.cor))/sqrt(sigma.sq.W.adjust) ) + lower
    }        
  z1.display <- sprintf("%-8.4g",z1)
  z2.display <- sprintf("%-8.4g",z2)
  neg.z.stats.out <-paste0(pad.spaces(8),"z1 = ",z1.display,pad.spaces(17),"z2 = ",z2.display,"\n",collapse="")
  # p values and rejection decision for two one-sided tests
  p1 <- pnorm(z1, lower.tail = FALSE)
  p2 <- pnorm(z2, lower.tail = FALSE)    
  p1.display <- format.extreme.p.vals(p1)
  p2.display <- format.extreme.p.vals(p2)
  neg.p.vals.out <- paste0(pad.spaces(3),"Pr(Z \U2265 z) ",p1.display,pad.spaces(12),"Pr(Z \U2265 z) ",p2.display,collapse="")
  if (p1 > alpha | p2 > alpha) {
    neg.decision <- "Fail to reject"
    }
  if (relevance==TRUE) {
    rlang::inform(message="\nTwo-sample rank-sum relevance test\n")
    rlang::inform(message="Two-sample test for 0th-order stochastic dominance")
    rlang::inform(message=table.title)
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(group.1.name," \u2502 ",n1.display,pad.spaces(3),W.display,pad.spaces(2),mu.W.display,sep=""))
    rlang::inform(message=paste0(group.2.name," \U2502 ",n2.display,pad.spaces(3),rank.sum.2.display,pad.spaces(2),mu.W.display,sep=""))
    rlang::inform(message=mid.bar)
    rlang::inform(message=paste0(pad.spaces(4),"combined \U2502 ",n.display,pad.spaces(3),rank.sum.display,pad.spaces(2),rank.sum.display,"\n",sep=""))
    rlang::inform(message=paste0("unadjusted variance",pad.spaces(2),sigma.sq.W.display,sep=""))
    rlang::inform(message=paste0("adjustment for ties",pad.spaces(2),ties.adjust.display,sep=""))
    rlang::inform(message=bottom.bar)
    rlang::inform(message=paste0("adjusted variance",pad.spaces(4),sigma.sq.W.adjust.display,"\n",sep=""))
    rlang::inform(message=paste0("Ho: Pr(",name.stem,"|",trimws(by.name),"=",by.values[1]," > ",name.stem,"|",trimws(by.name),"=",by.values[2],") = 0.5",sep=""))
    rlang::inform(message=paste0("Ha: Pr(",name.stem,"|",trimws(by.name),"=",by.values[1]," > ",name.stem,"|",trimws(by.name),"=",by.values[2],") \U2260 0.5\n",sep=""))
    if (ccontinuity) {
      rlang::inform(message="Using continuity correction\n")
      }
    rlang::inform(message=paste0(pad.spaces(14),"z = ",z.pos.display,sep=""))
    rlang::inform(message=paste0(pad.spaces(4),"Pr(Z > |z|) ",p.pos.display,"\n",sep=""))
    }
  rlang::inform(message="\nTwo-sample test for 0th-order stochastic equivalence")
  rlang::inform(message=table.title)
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(group.1.name," \u2502 ",n1.display,pad.spaces(3),W.display,pad.spaces(2),mu.W.display,sep=""))
  rlang::inform(message=paste0(group.2.name," \U2502 ",n2.display,pad.spaces(3),rank.sum.2.display,pad.spaces(2),mu.W.display,sep=""))
  rlang::inform(message=mid.bar)
  rlang::inform(message=paste0(pad.spaces(4),"combined \U2502 ",n.display,pad.spaces(3),rank.sum.display,pad.spaces(2),rank.sum.display,"\n",sep=""))
  rlang::inform(message=paste0("unadjusted variance  ",sigma.sq.W.display,sep=""))
  rlang::inform(message=paste0("adjustment for ties  ",ties.adjust.display,sep=""))
  rlang::inform(message=bottom.bar)
  rlang::inform(message=paste0("adjusted variance    ",sigma.sq.W.adjust.display,"\n",sep=""))
  if (eqv.type == "delta") {
    if (upper == lower) {
      rlang::inform(message=paste0(pad.spaces(9),eqv.type.display," = ",pad.right(upper.display,8),eqv.type.display," expressed in units of signed ranks (T)",sep=""))
      }
    if (upper != lower) {
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.lower.display," = ",pad.right(lower.display,8),eqv.type.lower.display," expressed in units of signed ranks (T)",sep=""))
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.upper.display," =  ",pad.right(upper.display,7),eqv.type.upper.display," expressed in units of signed ranks (T)",sep=""))
      }
    critical.value <- sigma.W*qnorm(alpha,lower.tail=FALSE)
    critical.value.display <- sprintf("%-6.4g",critical.value)
    if (upper == lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U0394 \U2264 z-crit*s.e. (",trimws(critical.value.display),"). See help(tost.rank.sum).",sep=""))
      }
    if (upper != lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if |\U0394l| \U2264 z-crit*s.e. (",trimws(critical.value.display),"). See help(tost.rank.sum).",sep=""))
      }
    if (upper != lower & upper <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U0394u \U2264 z-crit*s.e. (",trimws(critical.value.display),"). See help(tost.rank.sum).\n",sep=""))
      }
    if (upper == lower) {
      rlang::inform(message="\nHo: |W-E(W)| \U2265 \U0394:")
      if (ccontinuity) {
        rlang::inform(message="Using continuity correction\n")
        }
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394-[W-E(W)] \U2264 0",pad.spaces(18),"Ho2: [W-E(W)]+\U0394 \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394-[W-E(W)] > 0",pad.spaces(18),"Ha2: [W-E(W)]+\U0394 > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    if (upper != lower) {
      rlang::inform(message="\nHo: [W-E(W)] \U2264 \U0394l, or [W-E(W)] \U2265 \U0394u:")
      if (ccontinuity) {
        rlang::inform(message="Using continuity correction\n")
        }
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U0394u-[W-E(W)] \U2264 0",pad.spaces(17),"Ho2: [W-E(W)]-\U0394l \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U0394u-[W-E(W)] > 0",pad.spaces(17),"Ha2: [W-E(W)]-\U0394l > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    }
  if (eqv.type == "epsilon") {
    if (upper == lower) {
      rlang::inform(message=paste0(pad.spaces(9),"\U03B5 = ",pad.right(upper.display,8),eqv.type.display," expressed in units of the z distribution",sep=""))
      }
    if (upper != lower) {
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.lower.display," = ",pad.right(lower.display,8),eqv.type.lower.display," expressed in units of the z distribution",sep=""))
      rlang::inform(message=paste0(pad.spaces(8),eqv.type.upper.display," =  ",pad.right(upper.display,7),eqv.type.upper.display," expressed in units of the z distribution",sep=""))
      }
    critical.value <- qnorm(alpha,lower.tail=FALSE)
    critical.value.display <- sprintf("%-6.4g",critical.value)
    if (upper == lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U03B5 \U2264 z-crit (",trimws(critical.value.display),"). See help(tost.rank.sum).",sep=""))
      }
    if (upper != lower & lower <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if |\U03B5l| \U2264 z-crit (",trimws(critical.value.display),"). See help(tost.rank.sum).",sep=""))
      }
    if (upper != lower & upper <= critical.value) {
      rlang::inform(message=paste0("\nImpossible to reject any Ho if \U03B5u \U2264 z-crit (",trimws(critical.value.display),"). See help(tost.rank.sum).",sep=""))
      }
    if (upper == lower) {
      rlang::inform(message="\nHo: |Z| \U2265 \U03B5:")
      if (ccontinuity) {
        rlang::inform(message="Using continuity correction\n")
        }
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5-Z \U2264 0",pad.spaces(18),"Ho2: Z+\U03B5 \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5-Z > 0",pad.spaces(18),"Ha2: Z+\U03B5 > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      } 
    if (upper != lower) {
      rlang::inform(message="\nHo: Z \U2264 \U03B5l, or Z \U2265 \U03B5u:")
      if (ccontinuity) {
        rlang::inform(message="Using continuity correction\n")
        }
      rlang::inform(message=neg.z.stats.out)
      rlang::inform(message=paste0(pad.spaces(3),"Ho1: \U03B5u-Z \U2264 0",pad.spaces(17),"Ho2: Z-\U03B5l \U2264 0",sep=""))
      rlang::inform(message=paste0(pad.spaces(3),"Ha1: \U03B5u-Z > 0",pad.spaces(17),"Ha2: Z-\U03B5l > 0",sep=""))
      rlang::inform(message=neg.p.vals.out)
      }
    }
  # Output combined tests results if relevance test is requested
  if (relevance) {
    # Format alpha to remove trailing zeros
    alpha.display <- sub("0+$", "", as.character(alpha))
    # Format Delta or epsilon to remove trailing zeros
    rlang::inform(message="")
    if (upper == lower) {
      if (eqv.type == "delta") {
        rlang::inform(message=paste0("\nRelevance test conclusion for \U03B1 = ",alpha," and \U0394 = ",trimws(upper.display),":",sep=""))
        }
      if (eqv.type == "epsilon") {
        rlang::inform(message=paste0("\nRelevance test conclusion for \U03B1 = ",alpha," and \U03B5 = ",trimws(upper.display),":",sep=""))
        }
      }
    if (upper != lower) {
      if (eqv.type == "delta") {
        rlang::inform(message=paste0("\nRelevance test conclusion for \U03B1 = ",alpha," \U0394l = ",lower.display,", and \U0394u = ",upper.display,sep=""))
        }
      if (eqv.type == "epsilon") {
        rlang::inform(message=paste0("\nRelevance test conclusion for \U03B1 = ",alpha," \U03B5l = ",lower.display,", and \U03B5u = ",upper.display,sep=""))
        }
      }
    rlang::inform(message=paste0("  Ho test for 0th-order stochastic dominance:   ",pos.decision,sep=""))
    rlang::inform(message=paste0("  Ho test for 0th-order stochastic equivalence: ",neg.decision,sep=""))
    if ((pos.decision == "Reject") & (neg.decision == "Reject")) {
      rel.conclusion <- "Trivial 0th-order stochastic dominance (overpowered test)"
      }
    if ((pos.decision == "Reject") & (neg.decision == "Fail to reject")) {
      rel.conclusion <- "Relevant 0th-order stochastic dominance"
      }
    if ((pos.decision == "Fail to reject") & (neg.decision == "Reject")) {
      rel.conclusion <- "Stochastic equivalence"
      }
    if ((pos.decision == "Fail to reject") & (neg.decision == "Fail to reject")) {
      rel.conclusion <- "Indeterminate (underpowered test)"
      }
    rlang::inform(message=paste0("\nConclusion from combined tests: ",rel.conclusion,sep=""))
    }
###############################################################################
# Program end. Close up shop and return things.                               #
###############################################################################

  out <- list() 
  # Prepare return stuff for two-sample test
  if (!relevance) {
    out$statistics <- c(z1,z2)
    names(out$statistics) <- c("z1","z2")
    out$p.values <- c(p1,p2)
    names(out$p.values) <- c("p1","p2")
    }
   else {
    out$statistics <- c(z1,z2,z.pos)
    names(out$statistics) <- c("z1","z2","z")
    out$p.values <- c(p1,p2,p.pos)
    names(out$p.values) <- c("p1","p2","p")
    }
  out$rank_sums <- c(W, rank.sum.2, mu.W)
  names(out$rank_sums) <- c(paste0(trimws(group.1.name)," rank sum",collapse=""),paste0(trimws(group.2.name)," rank sum",collapse=""),"expected under Ho+")
  out$sample_sizes <- c(n1,n2,n)
  names(out$sample_sizes) <- c("n1","n2","n1+n2")
  out$var_adj <- sigma.sq.W.adjust
  names(out$var_adj) <- "adjusted \U03C3\U00B2"
  if (eqv.type=="delta") {
    if (upper==abs(eqv.level)) {
      out$threshold <- lower
      names(out$threshold) <- "\U394"
      }
     else {
       out$threshold <- c(upper, lower)
       names(out$threshold) <- c("\U0394u", "\U0394l")
      }
    }
   else {
    if (upper==abs(eqv.level)) {
      out$threshold <- lower
      names(out$threshold) <- "\U3B5"
      }
     else {
       out$threshold <- c(upper, lower)
       names(out$threshold) <- c("\U03B5u", "\U03B5l")
      }
    }
  if(relevance) {
    out$conclusion <- rel.conclusion
    names(out$conclusion) <- "relevance conclusion"
    }

  invisible(out)
  }
