#' gg_color
#'
#' @keywords internal
#' @importFrom grDevices hcl
#'
gg_color <- function (n) {
  hues <- seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}


#' prediction.variable.balance
#'
#' @description Function that graphs the balance of the different categories of a column of a data frame.
#'
#' @param data A data frame.
#' @param predict.variable Character type. The name of the variable to predict. This name must be part of the columns of the data frame.
#' @param ylab A character string that describes the y-axis on the graph.
#' @param xlab A character string that describes the x-axis on the graph.
#' @param main Character type. The main title of the chart.
#' @param col A vector that specifies the colors of the categories represented by bars within the chart.
#'
#' @seealso \code{\link[ggplot2]{ggplot}}
#'
#' @return A ggplot object.
#'
#' @note With this function we can identify if the data is balanced or not, according to the variable to be predicted.
#'
#' @import ggplot2
#' @importFrom rlang .data
#' @export
#'
#' @examples
#'
#' prediction.variable.balance(iris,"Species")
#'
prediction.variable.balance <- function(
    data, predict.variable,
    ylab = "Number of individuals",
    xlab = "",
    main = paste("Variable Distribution", predict.variable),
    col = NA
) {

  if (missing(predict.variable) || !(predict.variable %in% colnames(data))) {
    stop("predict.variable must be entered and be part of the table columns", call. = FALSE)
  }

  if (is.character(data[[predict.variable]]) || is.factor(data[[predict.variable]])) {

    if (length(col) == 0 || all(is.na(col))) {
      col <- gg_color(length(unique(data[[predict.variable]])))
    } else {
      col <- rep(col, length(unique(data[[predict.variable]])))
    }

    ggplot2::ggplot(
      data = data,
      mapping = ggplot2::aes(
        x = .data[[predict.variable]],
        fill = .data[[predict.variable]]
      )
    ) +
      ggplot2::geom_bar() +
      ggplot2::scale_fill_manual(values = col, name = predict.variable) +
      ggplot2::labs(x = xlab, y = ylab, title = main) +
      ggplot2::theme_minimal() +
      ggplot2::theme(legend.position = "bottom")

  } else {
    stop("The variable to predict must be of type factor or character", call. = FALSE)
  }
}

#' numerical.predictive.power
#'
#' @description Function that graphs the density of individuals and shows their category according to a numerical variable.
#'
#' @param data A data frame.
#' @param predict.variable Character type. The name of the variable to predict. This name must be part of the columns of the data frame.
#' @param variable.to.compare Character type. The name of the numeric variable to compare. This name must be part of the columns of the data frame.
#' @param ylab A character string that describes the y-axis on the graph.
#' @param xlab A character string that describes the x-axis on the graph.
#' @param main Character type. The main title of the chart.
#' @param col A vector that specifies the colors of the categories of the variable to predict.
#'
#' @seealso \code{\link[ggplot2]{ggplot}}
#'
#' @return A ggplot object.
#'
#' @note With this function we can analyze the predictive power of a numerical variable.
#'
#' @import ggplot2
#' @importFrom rlang .data
#' @export
#'
#' @examples
#'
#' numerical.predictive.power(iris,"Species","Sepal.Length")
#'
numerical.predictive.power <- function(
    data, predict.variable, variable.to.compare,
    ylab = "",
    xlab = "",
    main = paste("Variable Density", variable.to.compare, "according to", predict.variable),
    col = NA
) {

  if (missing(predict.variable) || !(predict.variable %in% colnames(data))) {
    stop("predict.variable must be entered and be part of the table columns", call. = FALSE)
  }

  if (missing(variable.to.compare) || !(variable.to.compare %in% colnames(data)) ||
      !is.numeric(data[[variable.to.compare]])) {
    stop("variable.to.compare must be entered and be part of the numeric columns of the table", call. = FALSE)
  }

  if (is.character(data[[predict.variable]]) || is.factor(data[[predict.variable]])) {

    if (length(col) == 0 || all(is.na(col))) {
      col <- gg_color(length(unique(data[[predict.variable]])))
    } else {
      col <- rep(col, length(unique(data[[predict.variable]])))
    }

    ggplot2::ggplot(
      data = data,
      mapping = ggplot2::aes(
        x = .data[[variable.to.compare]],
        fill = .data[[predict.variable]]
      )
    ) +
      ggplot2::geom_density(alpha = 0.7, color = NA) +
      ggplot2::scale_fill_manual(values = col) +
      ggplot2::labs(title = main, y = ylab, x = xlab, fill = predict.variable) +
      ggplot2::theme_minimal() +
      ggplot2::theme(
        legend.position = "bottom",
        legend.title = ggplot2::element_blank(),
        text = ggplot2::element_text(size = 15)
      )

  } else {
    stop("The variable to predict must be of type factor or character", call. = FALSE)
  }
}

#' categorical.predictive.power
#'
#' @description Function that graphs the distribution of individuals and shows their category according to a categorical variable.
#'
#' @param data A data frame.
#' @param predict.variable Character type. The name of the variable to predict. This name must be part of the columns of the data frame.
#' @param variable.to.compare Character type. The name of the categorical variable to compare. This name must be part of the columns of the data frame.
#' @param ylab A character string that describes the y-axis on the graph.
#' @param xlab A character string that describes the x-axis on the graph.
#' @param main Character type. The main title of the chart.
#' @param col A vector that specifies the colors of the categories of the variable to predict.
#'
#' @seealso \code{\link[ggplot2]{ggplot}}
#'
#' @return A ggplot object.
#'
#' @note With this function we can analyze the predictive power of a categorical variable.
#'
#' @import ggplot2
#' @import dplyr
#' @importFrom rlang .data
#' @export
#'
#' @examples
#'
#' cars <- datasets::mtcars
#' cars$cyl <- as.factor(cars$cyl)
#' cars$vs <- as.factor(cars$vs)
#' categorical.predictive.power(cars,"vs","cyl")
#'
categorical.predictive.power <- function(
    data, predict.variable, variable.to.compare,
    ylab = "",
    xlab = "",
    main = paste("Variable Distribution", variable.to.compare, "according to", predict.variable),
    col = NA
) {

  if (missing(predict.variable) || !(predict.variable %in% colnames(data))) {
    stop("predict.variable must be entered and be part of the table columns", call. = FALSE)
  }

  if (missing(variable.to.compare) || !(variable.to.compare %in% colnames(data)) ||
      !(is.factor(data[[variable.to.compare]]) || is.character(data[[variable.to.compare]]))) {
    stop("variable.to.compare must be entered and be part of the categorical columns of the table", call. = FALSE)
  }

  if (is.character(data[[predict.variable]]) || is.factor(data[[predict.variable]])) {

    if (length(col) == 0 || all(is.na(col))) {
      col <- gg_color(length(unique(data[[predict.variable]])))
    } else {
      col <- rep(col, length(unique(data[[predict.variable]])))
    }

    suppressWarnings(
      datos2 <- data |>
        dplyr::group_by(dplyr::across(dplyr::all_of(c(variable.to.compare, predict.variable)))) |>
        dplyr::summarise(count = dplyr::n(), .groups = "drop")
    )

    if (variable.to.compare != predict.variable) {
      suppressWarnings(
        datos2 <- datos2 |>
          dplyr::group_by(dplyr::across(dplyr::all_of(variable.to.compare)))
      )
    }

    datos2 <- datos2 |>
      dplyr::mutate(prop = round(count / sum(count), 4)) |>
      dplyr::ungroup()

    ggplot2::ggplot(
      data = datos2,
      mapping = ggplot2::aes(
        x = .data[[variable.to.compare]],
        y = prop,
        fill = .data[[predict.variable]]
      )
    ) +
      ggplot2::geom_col(position = "fill") +
      ggplot2::geom_text(
        ggplot2::aes(label = paste0(round(prop * 100, 2), "% (", count, ")")),
        position = ggplot2::position_stack(vjust = 0.5),
        color = "white"
      ) +
      ggplot2::scale_y_continuous(labels = function(x) paste0(x * 100, "%")) +
      ggplot2::labs(y = xlab, x = ylab, title = main) +
      ggplot2::scale_fill_manual(values = col, name = predict.variable) +
      ggplot2::theme(legend.position = "bottom") +
      ggplot2::coord_flip()

  } else {
    stop("The variable to predict must be of type factor or character", call. = FALSE)
  }
}


#' importance.plot
#'
#' @description Function that graphs the importance of the variables.
#'
#' @param model fitted model object.
#' @param col the color of the chart bars.
#'
#' @seealso \code{\link[ggplot2]{ggplot}}, \code{\link[traineR]{train.adabag}}, \code{\link[adabag]{boosting}}
#'
#' @return A ggplot object.
#'
#' @note With this function we can identify how important the variables are for the generation of a predictive model.
#'
#' @import ggplot2
#' @importFrom rlang .data
#' @importFrom stats reorder
#' @importFrom gbm summary.gbm
#' @importFrom xgboost xgb.importance
#'
#' @export
#'
#' @examples
#'
#'data <- iris
#'n <- nrow(data)
#'
#'sam <- sample(1:n,n*0.75)
#'training <- data[sam,]
#'testing <- data[-sam,]
#'
#'model <- train.adabag(formula = Species~.,data = training,minsplit = 2,
#'  maxdepth = 20, mfinal = 10)
#'importance.plot(model)
#'

importance.plot <- function(model, col = "steelblue") {

  # 1) Si viene envuelto en prmdt (como tu 'modelo'), extrae el booster real
  if (is.list(model) && "model" %in% names(model) && inherits(model$model, "xgb.Booster")) {
    model <- model$model
  }

  # 2) Manejo por tipo/clase
  if ("adabag.prmdt" %in% class(model)) {

    df <- data.frame(x = names(model$importance), y = model$importance)

  } else if ("randomForest.prmdt" %in% class(model)) {

    aux <- data.frame(model$importance)
    df <- data.frame(x = row.names(aux), y = aux$MeanDecreaseGini)

  } else if (inherits(model, "xgb.Booster") || ("xgb.Booster.prmdt" %in% class(model))) {

    # 3) Asegura que el dispatch no se rompa por clases raras
    class(model) <- "xgb.Booster"

    # 4) Llama explícitamente al namespace correcto
    aux <- data.frame(xgboost::xgb.importance(model = model))
    df <- data.frame(x = aux$Feature, y = aux$Frequency)

  } else if ("gbm.prmdt" %in% class(model)) {

    aux <- gbm::summary.gbm(model, plotit = FALSE)
    df <- data.frame(x = aux$var, y = aux$rel.inf)

  } else {
    stop("The model does not have this functionality.", call. = FALSE)
  }

  df$y <- round(df$y, digits = 2)

  ggplot2::ggplot(
    data = df,
    ggplot2::aes(x = stats::reorder(x, .data$y), y = y)
  ) +
    ggplot2::geom_bar(stat = "identity", fill = col, width = 0.6) +
    ggplot2::theme_minimal() +
    ggplot2::labs(
      title = "Variable Importance",
      y = "Percentage of Importance",
      x = "Variable Names"
    ) +
    ggplot2::coord_flip()
}

