#' Ruled surface along a 3D parametric curve
#'
#' @description
#' Constructs a ruled surface generated by a three-dimensional parametric
#' curve and a chosen direction field. At each sampled point on the curve,
#' a straight segment is extended in a specified direction, producing a
#' surface composed of line elements. Optionally, the resulting surface can
#' be visualized using \pkg{plotly}.
#'
#' @details
#' The function samples the base curve at \code{n_t} parameter values and,
#' for each sampled point, generates a set of points along a line segment
#' determined by the ruling parameter. These segments are interpolated over
#' the interval specified in \code{s_range}.
#'
#' In this implementation, the ruling direction is given by a fixed
#' three-dimensional vector \code{dir}. This vector is normalized
#' internally before constructing the surface.
#'
#' If \code{plot = TRUE}, a 3D visualization is produced using \pkg{plotly}.
#' The plot may include the ruled surface, grid lines on the surface,
#' boundary edges corresponding to the extremes of the ruling parameter,
#' and a selection of rulings. The generating curve can also be displayed.
#'
#' @param X,Y,Z Functions of one variable \code{t} returning the coordinate
#' components of the base curve.
#' @param a,b Numeric values giving the endpoints of the parameter interval.
#' @param s_range Numeric vector of length two giving the lower and upper
#' bounds for the ruling parameter.
#' @param dir Numeric vector of length three \code{c(ux, uy, uz)} giving
#' the ruling direction. It will be normalized internally.
#' @param n_t,n_s Integers giving the sampling resolution along the \code{t}
#' and \code{s} directions.
#' @param plot Logical; if \code{TRUE}, displays the ruled surface using
#' \pkg{plotly}.
#' @param surface_colorscale Character string or vector specifying the
#' colorscale used for the surface.
#' @param surface_opacity Numeric value between 0 and 1 controlling the
#' opacity of the surface.
#' @param show_surface_grid Logical; if \code{TRUE}, draws grid lines on the
#' surface.
#' @param surface_grid_color Color for the surface grid lines.
#' @param surface_grid_width Numeric width for the grid lines.
#' @param show_curve Logical; if \code{TRUE}, overlays the generating curve.
#' @param curve_line List of \pkg{plotly} style options for the curve.
#' @param show_edge_a,show_edge_b Logical; if \code{TRUE}, draws the boundary
#' edges at the extremes of the ruling parameter.
#' @param edge_line List of \pkg{plotly} style options for boundary edges.
#' @param show_rulings Logical; if \code{TRUE}, draws a subset of rulings on
#' the surface.
#' @param rulings_count Integer giving the number of rulings to draw when
#' \code{rulings_at} is not provided.
#' @param rulings_at Optional numeric vector giving the parameter values at
#' which rulings should be displayed.
#' @param rulings_line List of \pkg{plotly} style options for displayed
#' rulings.
#' @param show_axis_grid Logical; if \code{TRUE}, shows axis gridlines in the
#' 3D scene.
#' @param scene Optional list with 3D scene settings for \pkg{plotly}.
#' @param bg Optional list with background color settings for the figure.
#' @param lighting Optional list with lighting parameters for surface
#' shading in \pkg{plotly}.
#'
#' @return
#' A list with:
#' \itemize{
#'   \item \code{t_seq}, \code{s_seq}: parameter grids used to build the mesh,
#'   \item \code{Xmat}, \code{Ymat}, \code{Zmat}: matrices of coordinates
#'         for the ruled surface,
#'   \item \code{curve}: data frame with the sampled generating curve,
#'   \item \code{edge_a}, \code{edge_b}: data frames for boundary edges
#'         (possibly \code{NULL} if not requested),
#'   \item \code{u_hat}: normalized ruling direction vector.
#' }
#'
#' @examples
#' X <- function(t) cos(t)
#' Y <- function(t) sin(t)
#' Z <- function(t) 0.3 * t
#' dir_vec <- c(0, 0, 1)
#' rs <- cylindrical_surface3d(
#'   X, Y, Z,
#'   a = 0, b = 2 * pi,
#'   s_range = c(-0.2, 0.2),
#'   dir = dir_vec,
#'   n_t = 100, n_s = 40,
#'   plot = FALSE
#' )
#'
#' @importFrom tibble tibble
#' @export
cylindrical_surface3d <- function(
    X, Y, Z,
    a, b,
    s_range,
    dir,
    n_t = 200,
    n_s = 60,
    plot = FALSE,
    surface_colorscale = "Blues",
    surface_opacity    = 0.35,
    show_surface_grid  = TRUE,
    surface_grid_color = "rgba(60,80,200,0.25)",
    surface_grid_width = 1,
    show_curve = TRUE,
    curve_line = list(color = "red", width = 2, dash = "solid"),
    show_edge_a = TRUE,
    show_edge_b = FALSE,
    edge_line = list(color = "blue", width = 2, dash = "solid"),
    show_rulings = TRUE,
    rulings_count = 12,
    rulings_at = NULL,
    rulings_line = list(color = "black", width = 1, dash = "solid"),
    show_axis_grid = FALSE,
    scene = list(
      aspectmode = "data",
      xaxis = list(title = "x"),
      yaxis = list(title = "y"),
      zaxis = list(title = "z")
    ),
    bg = list(paper = "white", plot = "white"),
    lighting = list(
      ambient  = 1,
      diffuse  = 0.15,
      specular = 0,
      roughness = 1,
      fresnel   = 0
    )
) {
  # --- Validations ----------------------------------------------------
  if (!is.function(X) || !is.function(Y) || !is.function(Z)) {
    stop("X, Y, Z must be functions of one variable t.", call. = FALSE)
  }
  if (!is.numeric(a) || !is.numeric(b) ||
      length(a) != 1L || length(b) != 1L ||
      !is.finite(a) || !is.finite(b)) {
    stop("'a' and 'b' must be finite numeric scalars.", call. = FALSE)
  }
  if (b < a) {
    stop("'b' must be >= 'a'.", call. = FALSE)
  }
  if (!is.numeric(s_range) || length(s_range) != 2L ||
      any(!is.finite(s_range))) {
    stop("'s_range' must be a finite numeric vector c(c, d).", call. = FALSE)
  }
  if (!is.numeric(dir) || length(dir) != 3L || any(!is.finite(dir))) {
    stop("'dir' must be a finite numeric vector c(ux, uy, uz).", call. = FALSE)
  }

  # --- Normalized direction ------------------------------------------
  u <- as.numeric(dir)
  nu <- sqrt(sum(u * u))
  if (!is.finite(nu) || nu == 0) {
    stop("'dir' cannot be the zero vector.", call. = FALSE)
  }
  u <- u / nu  # \u00FB

  # --- Mesh -----------------------------------------------------------
  t_seq <- seq(a, b, length.out = n_t)
  s_seq <- seq(s_range[1], s_range[2], length.out = n_s)

  # r(t) sampled
  Rt <- cbind(X(t_seq), Y(t_seq), Z(t_seq))  # n_t x 3

  # Surface G(t,s) = r(t) + s u  -> matrices n_s x n_t
  Xmat <- matrix(NA_real_, n_s, n_t)
  Ymat <- matrix(NA_real_, n_s, n_t)
  Zmat <- matrix(NA_real_, n_s, n_t)

  for (j in seq_len(n_t)) {
    Xmat[, j] <- Rt[j, 1] + s_seq * u[1]
    Ymat[, j] <- Rt[j, 2] + s_seq * u[2]
    Zmat[, j] <- Rt[j, 3] + s_seq * u[3]
  }

  # Generating curve
  curve_df <- tibble::tibble(
    t = t_seq,
    x = Rt[, 1],
    y = Rt[, 2],
    z = Rt[, 3]
  )

  # Boundary curves G(a,s) and G(b,s)
  edge_a <- edge_b <- NULL
  if (isTRUE(show_edge_a)) {
    ra <- Rt[1, ]
    edge_a <- tibble::tibble(
      s = s_seq,
      x = ra[1] + s_seq * u[1],
      y = ra[2] + s_seq * u[2],
      z = ra[3] + s_seq * u[3]
    )
  }
  if (isTRUE(show_edge_b)) {
    rb <- Rt[n_t, ]
    edge_b <- tibble::tibble(
      s = s_seq,
      x = rb[1] + s_seq * u[1],
      y = rb[2] + s_seq * u[2],
      z = rb[3] + s_seq * u[3]
    )
  }

  # --- Plot -----------------------------------------------------------
  if (isTRUE(plot)) {
    if (!requireNamespace("plotly", quietly = TRUE)) {
      warning("To plot you need 'plotly' installed.", call. = FALSE)
    } else {
      contours_arg <- if (isTRUE(show_surface_grid)) {
        list(
          x = list(show = TRUE, color = surface_grid_color,
                   width = surface_grid_width),
          y = list(show = TRUE, color = surface_grid_color,
                   width = surface_grid_width),
          z = list(show = FALSE)
        )
      } else {
        NULL
      }

      plt <- plotly::plot_ly() |>
        plotly::add_surface(
          x = Xmat, y = Ymat, z = Zmat,
          colorscale = surface_colorscale,
          showscale  = FALSE,
          opacity    = surface_opacity,
          lighting   = lighting,
          contours   = contours_arg
        )

      if (isTRUE(show_curve)) {
        plt <- plt |>
          plotly::add_trace(
            data = curve_df, x = ~x, y = ~y, z = ~z,
            type = "scatter3d", mode = "lines",
            line = curve_line, hoverinfo = "none", showlegend = FALSE
          )
      }

      if (!is.null(edge_a)) {
        plt <- plt |>
          plotly::add_trace(
            data = edge_a, x = ~x, y = ~y, z = ~z,
            type = "scatter3d", mode = "lines",
            line = edge_line, hoverinfo = "none", showlegend = FALSE
          )
      }
      if (!is.null(edge_b)) {
        plt <- plt |>
          plotly::add_trace(
            data = edge_b, x = ~x, y = ~y, z = ~z,
            type = "scatter3d", mode = "lines",
            line = edge_line, hoverinfo = "none", showlegend = FALSE
          )
      }

      # Rulings
      if (isTRUE(show_rulings)) {
        t_pos <- if (is.null(rulings_at)) {
          seq(a, b, length.out = rulings_count)
        } else {
          as.numeric(rulings_at)
        }
        t_pos <- t_pos[is.finite(t_pos) & t_pos >= a & t_pos <= b]
        if (length(t_pos)) {
          for (ti in t_pos) {
            ri <- c(X(ti), Y(ti), Z(ti))
            xr <- ri[1] + s_seq * u[1]
            yr <- ri[2] + s_seq * u[2]
            zr <- ri[3] + s_seq * u[3]
            plt <- plt |>
              plotly::add_trace(
                x = xr, y = yr, z = zr,
                type = "scatter3d", mode = "lines",
                line = rulings_line, hoverinfo = "none", showlegend = FALSE
              )
          }
        }
      }

      # Axis grids if requested
      scene_final <- scene
      for (ax in c("xaxis", "yaxis", "zaxis")) {
        if (is.null(scene_final[[ax]])) scene_final[[ax]] <- list()
        scene_final[[ax]]$showgrid <- isTRUE(show_axis_grid)
      }

      plt <- plt |>
        plotly::layout(
          title = "Cylindrical surface G(t,s) = r(t) + s \u00FB",
          scene = scene_final,
          paper_bgcolor = bg$paper,
          plot_bgcolor  = bg$plot
        )
      print(plt)
    }
  }

  list(
    t_seq  = t_seq,
    s_seq  = s_seq,
    Xmat   = Xmat,
    Ymat   = Ymat,
    Zmat   = Zmat,
    curve  = curve_df,
    edge_a = edge_a,
    edge_b = edge_b,
    u_hat  = u
  )
}
