% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/curvature_torsion3d.R
\name{curvature_torsion3d}
\alias{curvature_torsion3d}
\title{Curvature and torsion of a 3D parametric curve}
\usage{
curvature_torsion3d(
  X,
  Y,
  Z,
  t0,
  h = 1e-04,
  plot = FALSE,
  window = 1,
  n_samples = 200,
  line = list(color = "red", width = 2, dash = "solid"),
  point = list(color = "black", size = 5, symbol = "circle"),
  scene = list(aspectmode = "data", xaxis = list(title = "x(t)"), yaxis = list(title =
    "y(t)"), zaxis = list(title = "z(t)")),
  bg = list(paper = "white", plot = "white"),
  tol = 1e-10
)
}
\arguments{
\item{X, Y, Z}{Functions of \code{t} returning the three coordinate
components of the parametric curve.}

\item{t0}{Value of the parameter at which curvature and torsion are
evaluated.}

\item{h}{Step size for the finite-difference approximations.
Smaller values give more accuracy but may amplify numerical noise.}

\item{plot}{Logical; if \code{TRUE}, displays a 3D plot of a short
segment of the curve around the evaluation point.}

\item{window}{Length of the parameter interval shown when
\code{plot = TRUE}. The interval is centered at \code{t0}.}

\item{n_samples}{Number of points used to draw the curve segment in
the 3D plot.}

\item{line}{A list defining the visual style of the curve in the
3D plot.}

\item{point}{A list defining the visual style of the marker placed
at the evaluation point.}

\item{scene}{A list with 3D axis settings for \pkg{plotly}.}

\item{bg}{Background colors for the \pkg{plotly} figure.}

\item{tol}{Numeric tolerance used to detect degenerate situations in
which curvature or torsion cannot be reliably computed.}
}
\value{
A list with:
\itemize{
\item \code{kappa}: numerical curvature at the evaluation point.
\item \code{tau}: numerical torsion at the evaluation point, or
\code{NA} if the computation is unstable.
\item \code{t0}: the parameter value where the evaluation was made.
\item \code{point}: a numeric vector containing the coordinates of
the curve at \code{t0}.
\item \code{r1}, \code{r2}, \code{r3}: numeric approximations to the
first, second and third derivative vectors at \code{t0}.
}
}
\description{
Computes numerical curvature and torsion of a three-dimensional
parametric curve at a specific value of the parameter. The curve is
described by three functions that give the coordinate components.
All derivatives are approximated using centered finite differences
of first, second and third order.
}
\details{
The curvature at the evaluation point measures how sharply the curve
bends at that location. It is computed from the first and second
derivative vectors. The torsion measures how the curve deviates from
being planar and is computed from the first, second and third
derivative vectors. If the first derivative vector is nearly zero, or
if the first and second derivative vectors are nearly parallel, the
torsion becomes undefined; in such cases the function returns
\code{NA} and provides a diagnostic message.

Optionally, the function can display a small segment of the curve
around the evaluation point using \pkg{plotly}. The point where the
curvature and torsion are computed is highlighted in the 3D plot.
}
\examples{
# Example curve
X <- function(t) t^2 * cos(t)
Y <- function(t) t^3 * sin(3 * t)
Z <- function(t) t
res <- curvature_torsion3d(X, Y, Z, t0 = pi)
res$kappa
res$tau

# \donttest{ if (requireNamespace("plotly", quietly = TRUE)) {
#   curvature_torsion3d(
#     X, Y, Z, t0 = pi, plot = TRUE,
#     window = 1.0, n_samples = 200,
#     line  = list(color = "red",  width = 2),
#     point = list(color = "black", size = 5, symbol = "circle")
#   )
# } }

}
