//===-- Half-precision atanf16(x) function --------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception.
//
//===----------------------------------------------------------------------===//

#include "src/math/atanf16.h"
#include "hdr/errno_macros.h"
#include "hdr/fenv_macros.h"
#include "src/__support/FPUtil/FEnvImpl.h"
#include "src/__support/FPUtil/FPBits.h"
#include "src/__support/FPUtil/PolyEval.h"
#include "src/__support/FPUtil/cast.h"
#include "src/__support/FPUtil/except_value_utils.h"
#include "src/__support/FPUtil/multiply_add.h"
#include "src/__support/FPUtil/sqrt.h"
#include "src/__support/macros/optimization.h"

namespace LIBC_NAMESPACE_DECL {

// Generated by Solly using the following command:
// > round(pi/2, SG, RN);
static constexpr float PI_2 = 0x1.921fb6p0;

#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
static constexpr size_t N_EXCEPTS = 6;

static constexpr fputil::ExceptValues<float16, N_EXCEPTS> ATANF16_EXCEPTS{{
    // (input, RZ output, RU offset, RD offset, RN offset)
    {0x2745, 0x2744, 1, 0, 1},
    {0x3099, 0x3090, 1, 0, 1},
    {0x3c6c, 0x3aae, 1, 0, 1},
    {0x466e, 0x3daa, 1, 0, 1},
    {0x48ae, 0x3ddb, 1, 0, 0},
    {0x5619, 0x3e3d, 1, 0, 1},
}};
#endif // !LIBC_MATH_HAS_SKIP_ACCURATE_PASS

LLVM_LIBC_FUNCTION(float16, atanf16, (float16 x)) {
  using FPBits = fputil::FPBits<float16>;
  FPBits xbits(x);

  uint16_t x_u = xbits.uintval();
  uint16_t x_abs = x_u & 0x7fff;
  bool x_sign = x_u >> 15;
  float sign = (x_sign ? -1.0 : 1.0);

  // |x| >= +/-inf
  if (LIBC_UNLIKELY(x_abs >= 0x7c00)) {
    if (xbits.is_nan()) {
      if (xbits.is_signaling_nan()) {
        fputil::raise_except_if_required(FE_INVALID);
        return FPBits::quiet_nan().get_val();
      }
      return x;
    }

    // atanf16(+/-inf) = +/-pi/2
    return fputil::cast<float16>(sign * PI_2);
  }

  float xf = x;
  float xsq = xf * xf;
#ifndef LIBC_MATH_HAS_SKIP_ACCURATE_PASS
  // Handle exceptional values
  if (auto r = ATANF16_EXCEPTS.lookup_odd(x_abs, x_sign);
      LIBC_UNLIKELY(r.has_value()))
    return r.value();
#endif

  // |x| <= 0x1p0, |x| <= 1
  if (x_abs <= 0x3c00) {
    // atanf16(+/-0) = +/-0
    if (LIBC_UNLIKELY(x_abs == 0))
      return x;

    // Degree-14 minimax odd polynomial of atan(x) generated by Sollya with:
    // > P = fpminimax(atan(x)/x, [|0, 2, 4, 6, 8, 10, 12, 14|], [|SG...|],
    // [0, 1]);
    float result = fputil::polyeval(
        xsq, 0x1.fffffcp-1f, -0x1.55519ep-2f, 0x1.98f6a8p-3f, -0x1.1f0a92p-3f,
        0x1.95b654p-4f, -0x1.e65492p-5f, 0x1.8c0c36p-6f, -0x1.32316ep-8f);
    return fputil::cast<float16>(xf * result);
  }

  // If |x| > 1
  // y = atan(x) = sign(x) * atan(|x|)
  // atan(|x|) = pi/2 - atan(1/|x|)
  // Recall, 1/|x| < 1
  float x_inv_sq = 1.0f / xsq;
  float x_inv = fputil::sqrt<float>(x_inv_sq);

  // Degree-14 minimax odd polynomial of atan(x) generated by Sollya with:
  // > P = fpminimax(atan(x)/x, [|0, 2, 4, 6, 8, 10, 12, 14|], [|SG...|],
  // [0, 1]);
  float interm =
      fputil::polyeval(x_inv_sq, 0x1.fffffcp-1f, -0x1.55519ep-2f,
                       0x1.98f6a8p-3f, -0x1.1f0a92p-3f, 0x1.95b654p-4f,
                       -0x1.e65492p-5f, 0x1.8c0c36p-6f, -0x1.32316ep-8f);

  return fputil::cast<float16>(sign *
                               fputil::multiply_add(x_inv, -interm, PI_2));
}

} // namespace LIBC_NAMESPACE_DECL
