/* OpenCP Module Player
 * copyright (c) 2021-'26 Stian Skjelstad <stian.skjelstad@gmail.com>
 *
 * unit test for filesystem-gzip.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define INPUTBUFFERSIZE 128
#define OUTPUTBUFFERSIZE 64
#define FILEHANDLE_CACHE_DISABLE

#include "filesystem-gzip.c"
#include "filesystem-dir-mem.h"
#include "filesystem-file-mem.h"

#define ANSI_COLOR_RED     "\x1b[31m"
#define ANSI_COLOR_GREEN   "\x1b[32m"
#define ANSI_COLOR_YELLOW  "\x1b[33m"
#define ANSI_COLOR_BLUE    "\x1b[34m"
#define ANSI_COLOR_MAGENTA "\x1b[35m"
#define ANSI_COLOR_CYAN    "\x1b[36m"
#define ANSI_COLOR_RESET   "\x1b[0m"

void preemptive_framelock (void)
{
}

uint32_t dirdbRef (uint32_t ref, enum dirdb_use use)
{
	return ref;
}

void dirdbUnref (uint32_t ref, enum dirdb_use use)
{
}

void dirdbGetName_internalstr (uint32_t ref, const char **retval)
{
	*retval = 0;
	if (ref == 1) *retval = "test1.txt.gz";
	if (ref == 2) *retval = "test1.txt";
	if (ref == 3) *retval = "test2.txt.gz";
	if (ref == 4) *retval = "test2.txt";
	if (ref == 5) *retval = "test3.txt.gz";
	if (ref == 6) *retval = "test3.txt";
	if (ref == 7) *retval = "test4.txt.gz";
	if (ref == 8) *retval = "test4.txt";
	if (ref == 9) *retval = "test5.txt.gz.gz";
	if (ref == 10) *retval = "test5.txt.gz";
	if (ref == 11) *retval = "test5.txt";

}

void dirdbGetName_malloc (uint32_t ref, char **retval)
{
	const char *temp = 0;
	dirdbGetName_internalstr (ref, &temp);
	if (!temp)
	{
		*retval = 0;
		return;
	}
	*retval = strdup (temp);
}

uint32_t dirdbFindAndRef (uint32_t parent, const char *name, enum dirdb_use use)
{
	if ((parent == 1) && (!strcmp (name, "test1.txt"))) return 2;
	return 0;
}

struct ocpdir_t *ocpdir_t_fill_default_readdir_dir  (struct ocpdir_t *_self, uint32_t dirdb_ref)
{
	fprintf (stderr, "Dummy symbol ocpdir_t_fill_default_readdir_dir called?\n");
	_exit(1);
}

struct ocpfile_t *ocpdir_t_fill_default_readdir_file (struct ocpdir_t *_self, uint32_t dirdb_ref)
{
	fprintf (stderr, "Dummy symbol ocpdir_t_fill_default_readdir_file called?\n");
	_exit(1);
}

const char *ocpfile_t_fill_default_filename_override (struct ocpfile_t *file)
{
	return 0;
}

int ocpfilehandle_t_fill_default_ioctl (struct ocpfilehandle_t *s, const char *cmd, void *ptr)
{
	return -1;
}

const char *ocpfilehandle_t_fill_default_filename_override (struct ocpfilehandle_t *fh)
{
	return 0;
}

void register_dirdecompressor (const struct ocpdirdecompressor_t *ofd)
{
}

int adbMetaAdd (const char *filename, const uint64_t filesize, const char *SIG, const unsigned char  *data, const uint32_t  datasize)
{
	return 0;
}

int adbMetaGet (const char *filename, const uint64_t filesize, const char *SIG,       unsigned char **data,       uint32_t *datasize)
{
	return -1;
}

int gzip_test1 (void)
{
	const uint8_t src[] =
	{
		0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x02, 0x03, 0xf3, 0x48, 0xcd, 0xc9, 0xc9, 0x57,
		0x28, 0xcf, 0x2f, 0xca, 0x49, 0xd1, 0x2b, 0xa9,
		0x28, 0xe1, 0xf2, 0x18, 0xe6, 0x7c, 0x00, 0x1d,
		0x6c, 0xb3, 0x05, 0xf0, 0x00, 0x00, 0x00
	};
	char dst[1024];
	const uint8_t verify[] = {
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
		'H', 'e', 'l', 'l', 'o', ' ', 'w', 'o', 'r', 'l', 'd', '.', 't', 'x', 't', '\n',
	};
	char *dsrc = malloc (sizeof (src));
	int fill;
	int retval = 0;
	struct ocpdir_t *test_dir; // the top-level container-directory
	struct ocpfile_t *osrc;   // the gzip source file
	struct ocpdir_t *oddst;
	struct ocpfile_t *odst; // the gzip'd decompressed virtual file
	struct ocpfilehandle_t *hdst; // the handle for gzip'd decompressed file

	printf ("Testing simple small linear compression:  ");

	memcpy (dsrc, src, sizeof (src));
	test_dir = ocpdir_mem_getdir_t(ocpdir_mem_alloc (0, "test:"));
	osrc = mem_file_open (test_dir, /* dirdb_ref */ 1, dsrc, sizeof (src));
	test_dir->unref (test_dir); test_dir = 0;

	oddst = gzip_check_steal (osrc, 2);
	odst = oddst->readdir_file(oddst, 2);
	hdst = odst->open (odst);
	oddst->unref (oddst); oddst = 0;
	odst->unref (odst); odst = 0;
	osrc->unref (osrc); osrc = 0;

	fill = hdst->read (hdst, dst, sizeof (dst));
	hdst->unref (hdst); hdst = 0;

	if (fill != sizeof (verify))
	{
		printf (ANSI_COLOR_RED " size missmatch" ANSI_COLOR_RESET "\n");
		retval |= 1;
	} else if (memcmp (verify, dst, fill))
	{
		printf (ANSI_COLOR_RED " data missmatch" ANSI_COLOR_RESET "\n");
		retval |= 2;
	} else {
		printf (ANSI_COLOR_GREEN " OK" ANSI_COLOR_RESET "\n");
	}

	return retval;
}

int gzip_test2 (void)
{
	const uint8_t src[] =
	{
		0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x02, 0x03, 0xf3, 0x48, 0xcd, 0xc9, 0xc9, 0x57,
		0x28, 0xcf, 0x2f, 0xca, 0x49, 0xd1, 0x2b, 0xa9,
		0x28, 0xe1, 0xf2, 0x18, 0xe6, 0x00,/*0x7c,*/ 0x00, 0x1d,
		0x6c, 0xb3, 0x05, 0xf0, 0x00, 0x00, 0x00
	};
	char *dsrc = malloc (sizeof (src));
	int retval = 0;
	struct ocpdir_t *test_dir; // the top-level container-directory
	struct ocpfile_t *osrc;   // the gzip source file
	struct ocpdir_t *oddst;
	struct ocpfile_t *odst; // the gzip'd decompressed virtual file

	printf ("Testing file size prediction:  ");

	memcpy (dsrc, src, sizeof (src));
	test_dir = ocpdir_mem_getdir_t(ocpdir_mem_alloc (0, "test:"));
	osrc = mem_file_open (test_dir, 3, dsrc, sizeof (src));
	test_dir->unref (test_dir); test_dir = 0;
	oddst = gzip_check_steal (osrc, 4);
	odst = oddst->readdir_file(oddst, 4);
	oddst->unref (oddst); oddst = 0;
	osrc->unref (osrc); osrc = 0;

	if (odst->filesize (odst) != 240)
	{
		printf (ANSI_COLOR_RED " size missmatch" ANSI_COLOR_RESET "\n");
		retval |= 1;
	} else {
		printf (ANSI_COLOR_GREEN " OK" ANSI_COLOR_RESET "\n");
	}

	odst->unref (odst); odst = 0;

	return retval;
}

int gzip_test3 (void)
{
	const uint8_t src[] =
	{
		0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x02, 0x03, 0xf3, 0x48, 0xcd, 0xc9, 0xc9, 0x57,
		0x28, 0xcf, 0x2f, 0xca, 0x49, 0xd1, 0x2b, 0xa9,
		0x28, 0xe1, 0xf2, 0x18, 0xe6, 0x7c, 0x00, 0x1d,
		0x6c, 0xb3, 0x05, 0xf0, 0x00, 0x00, 0x00
	};
	char *dsrc = malloc (sizeof (src));
	int retval = 0;
	struct ocpdir_t *test_dir; // the top-level container-directory
	struct ocpfile_t *osrc;   // the gzip source file
	struct ocpdir_t *oddst;
	struct ocpfile_t *odst; // the gzip'd decompressed virtual file
	struct ocpfilehandle_t *hdst; // the handle for gzip'd decompressed file


	printf ("Testing filehandle size (does not do prediction):  ");

	memcpy (dsrc, src, sizeof (src));
	test_dir = ocpdir_mem_getdir_t(ocpdir_mem_alloc (0, "test:"));
	osrc = mem_file_open (test_dir, 5, dsrc, sizeof (src));
	test_dir->unref (test_dir); test_dir = 0;

	oddst = gzip_check_steal (osrc, 6);
	odst = oddst->readdir_file(oddst, 6);
	hdst = odst->open (odst);
	oddst->unref (oddst); oddst = 0;
	odst->unref (odst); odst = 0;
	osrc->unref (osrc); osrc = 0;

	if (hdst->filesize (hdst) != 240)
	{
		printf (ANSI_COLOR_RED " size missmatch" ANSI_COLOR_RESET "\n");
		retval |= 1;
	} else {
		printf (ANSI_COLOR_GREEN " OK" ANSI_COLOR_RESET "\n");
	}

	hdst->unref (hdst); hdst = 0;

	return retval;
}

int gzip_test4 (void)
{
	const uint8_t src[] =
	{
		0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x03, 0x0d, 0xd3, 0xb7, 0x61, 0xc4, 0x40,
		0x00, 0xc0, 0xb0, 0xde, 0xd3, 0x3c, 0x95, 0x35, 0x8e, 0xd2, 0xed, 0x3f, 0x82, 0x51, 0xb0, 0x64,
		0x89, 0xdf, 0xef, 0xf7, 0xf7, 0xfb, 0xa5, 0x49, 0xb3, 0x16, 0xad, 0xda, 0xb4, 0xeb, 0xd0, 0xa9,
		0x4b, 0xb7, 0x1e, 0xbd, 0xfa, 0x34, 0xfe, 0x7e, 0xf9, 0xf3, 0xe7, 0xcf, 0x9f, 0x3f, 0x7f, 0xfe,
		0xfc, 0xf9, 0xf3, 0xe7, 0xcf, 0x9f, 0x3f, 0x7f, 0xfe, 0xfc, 0x93, 0x7f, 0xf2, 0x4f, 0xfe, 0xc9,
		0x3f, 0xf9, 0x27, 0xff, 0xe4, 0x9f, 0xfc, 0x93, 0x7f, 0xf2, 0x4f, 0xfe, 0xc9, 0x3f, 0xf9, 0x27,
		0xff, 0xe4, 0x9f, 0xfc, 0xb3, 0x7f, 0xf6, 0xcf, 0xfe, 0xd9, 0x3f, 0xfb, 0x67, 0xff, 0xec, 0x9f,
		0xfd, 0xb3, 0x7f, 0xf6, 0xcf, 0xfe, 0xd9, 0x3f, 0xfb, 0x67, 0xff, 0xec, 0x9f, 0xfd, 0x8b, 0x7f,
		0xf1, 0x2f, 0xfe, 0xc5, 0xbf, 0xf8, 0x17, 0xff, 0xe2, 0x5f, 0xfc, 0x8b, 0x7f, 0xf1, 0x2f, 0xfe,
		0xc5, 0xbf, 0xf8, 0x17, 0xff, 0xe2, 0x5f, 0xfc, 0xab, 0x7f, 0xf5, 0xaf, 0xfe, 0xd5, 0xbf, 0xfa,
		0x57, 0xff, 0xea, 0x5f, 0xfd, 0xab, 0x7f, 0xf5, 0xaf, 0xfe, 0xd5, 0xbf, 0xfa, 0x57, 0xff, 0xea,
		0x5f, 0xfd, 0x9b, 0x7f, 0xf3, 0x6f, 0xfe, 0xcd, 0xbf, 0xf9, 0x37, 0xff, 0xe6, 0xdf, 0xfc, 0x9b,
		0x7f, 0xf3, 0x6f, 0xfe, 0xcd, 0xbf, 0xf9, 0x37, 0xff, 0xe6, 0xdf, 0xfc, 0xbb, 0x7f, 0xf7, 0xef,
		0xfe, 0xdd, 0xbf, 0xfb, 0x77, 0xff, 0xee, 0xdf, 0xfd, 0xbb, 0x7f, 0xf7, 0xef, 0xfe, 0xdd, 0xbf,
		0xfb, 0x77, 0xff, 0xee, 0xdf, 0xfd, 0x87, 0xff, 0xf0, 0x1f, 0xfe, 0xc3, 0x7f, 0xf8, 0x0f, 0xff,
		0xe1, 0x3f, 0xfc, 0x87, 0xff, 0xf0, 0x1f, 0xfe, 0xc3, 0x7f, 0xf8, 0x0f, 0xff, 0xe1, 0x3f, 0xfc,
		0xa7, 0xff, 0xf4, 0x9f, 0xfe, 0xd3, 0x7f, 0xfa, 0x4f, 0xff, 0xe9, 0x3f, 0xfd, 0xa7, 0xff, 0xf4,
		0x9f, 0xfe, 0xd3, 0x7f, 0xfa, 0x4f, 0xff, 0xe9, 0x3f, 0xfd, 0x97, 0xff, 0xf2, 0x5f, 0xfe, 0xcb,
		0x7f, 0xf9, 0x2f, 0xff, 0xe5, 0xbf, 0xfc, 0x97, 0xff, 0xf2, 0x5f, 0xfe, 0xcb, 0x7f, 0xf9, 0x2f,
		0xff, 0xe5, 0xbf, 0xfc, 0xb7, 0xff, 0xf6, 0xdf, 0xfe, 0xdb, 0x7f, 0xfb, 0x6f, 0xff, 0xed, 0xbf,
		0xfd, 0xb7, 0xff, 0xf6, 0xdf, 0xfe, 0xdb, 0x7f, 0xfb, 0x6f, 0xff, 0xed, 0xbf, 0xfd, 0x8f, 0xff,
		0xf1, 0x3f, 0xfe, 0xc7, 0xff, 0xf8, 0x1f, 0xff, 0xe3, 0x7f, 0xfc, 0x8f, 0xff, 0xf1, 0x3f, 0xfe,
		0xc7, 0xff, 0xf8, 0x1f, 0xff, 0xe3, 0x7f, 0xfc, 0xaf, 0xff, 0xf5, 0xbf, 0xfe, 0xd7, 0xff, 0xfa,
		0x5f, 0xff, 0xeb, 0x7f, 0xfd, 0xaf, 0xff, 0xf5, 0xbf, 0xfe, 0xd7, 0xff, 0xfa, 0x5f, 0xff, 0xeb,
		0x7f, 0xfd, 0x9f, 0xff, 0xf3, 0x7f, 0xfe, 0xcf, 0xff, 0xf9, 0x3f, 0xff, 0xe7, 0xff, 0xfc, 0x9f,
		0xff, 0xf3, 0x7f, 0xfe, 0xcf, 0xff, 0xf9, 0x3f, 0xff, 0xe7, 0xff, 0xfc, 0xc3, 0x3f, 0xfc, 0xc3,
		0x3f, 0xfc, 0xc3, 0x3f, 0xfc, 0xc3, 0x3f, 0xfc, 0xc3, 0x3f, 0xfc, 0xc3, 0x3f, 0xfc, 0xc3, 0x3f,
		0xfc, 0xc3, 0x3f, 0xc6, 0x5f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7,
		0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f,
		0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc, 0xc7, 0x7f, 0xfc,
		0xc7, 0x7f, 0xfc, 0xc7, 0xff, 0x3f, 0x16, 0xf8, 0xc6, 0xd3, 0x00, 0x08, 0x00, 0x00
	};
	const struct test4_subtest
	{
		uint32_t offset, count;
		const char *data;
	} subtests[] =
	{
		{0x000*4, 4, "000\n"},
		{0x143*4, 4, "143\n"},
		{0x0f0*4, 4, "0f0\n"},
		{0x1ff*4, 4, "1ff\n"},
		{0x001*4, 4, "001\n"},
		{0x1fd*4, 4, "1fd\n"},
	};
	char *dsrc = malloc (sizeof (src));
	int retval = 0;
	struct ocpdir_t *test_dir; // the top-level container-directory
	struct ocpfile_t *osrc;   // the bzip2 source file
	struct ocpdir_t *oddst;
	struct ocpfile_t *odst; // the bzip2'd decompressed virtual file
	struct ocpfilehandle_t *hdst; // the handle for bzip2'd decompressed file
	int i;
	char *dst = malloc (4);

	printf ("Testing seek and reads (no cache):  ");

	memcpy (dsrc, src, sizeof (src));
	test_dir = ocpdir_mem_getdir_t(ocpdir_mem_alloc (0, "test:"));
	osrc = mem_file_open (test_dir, /* dirdb_ref */ 7, dsrc, sizeof (src));
	test_dir->unref (test_dir); test_dir = 0;

	oddst = gzip_check_steal (osrc, 8);
	odst = oddst->readdir_file(oddst, 8);
	hdst = odst->open (odst);
	oddst->unref (oddst); oddst = 0;
	odst->unref (odst); odst = 0;
	osrc->unref (osrc); osrc = 0;

	for (i=0; i < 5; i++)
	{
		if (hdst->seek_set (hdst, subtests[i].offset))
		{
			printf ("s");
			retval |= 1;
		} else if (hdst->read (hdst, dst, subtests[i].count) != 4)
		{
			printf ("r");
			retval |= 2;
		} else if (memcmp (dst, subtests[i].data, subtests[i].count))
		{
			printf ("d");
			retval |= 4;
		} else {
			printf ("%d", i+1);
		}
	}
	if (retval)
	{
		printf (ANSI_COLOR_RED " Failed" ANSI_COLOR_RESET "\n");
	} else {
		printf (ANSI_COLOR_GREEN " OK" ANSI_COLOR_RESET "\n");
	}

	hdst->unref (hdst); hdst = 0;
	free (dst);

	return retval;
}

int gzip_test5 (void)
{
	const uint8_t src[] =
	{
		0x1f, 0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x03, 0xed, 0xd3, 0xad, 0x4b, 0x43, 0x61,
		0x14, 0xc7, 0xf1, 0xa1, 0xc5, 0x22, 0xb6, 0xd9, 0xbc, 0xd1, 0xb8, 0xa2, 0x58, 0x84, 0xc7, 0x2a,
		0x86, 0x61, 0x32, 0x5e, 0x04, 0xc1, 0xb8, 0x68, 0xf3, 0x80, 0x45, 0x64, 0xa0, 0x41, 0xc4, 0x60,
		0xb8, 0x49, 0x44, 0x44, 0x36, 0x98, 0xb0, 0x20, 0xf2, 0x08, 0xd3, 0x0d, 0x91, 0x31, 0xf1, 0x22,
		0xa2, 0xa2, 0x17, 0x5f, 0x87, 0xce, 0x39, 0xb7, 0xb9, 0xdd, 0xb7, 0xe7, 0x39, 0x47, 0x2d, 0xc2,
		0x01, 0xbb, 0xc5, 0xc3, 0x37, 0x9c, 0xcf, 0x1f, 0xf0, 0x33, 0x92, 0x5d, 0x91, 0xef, 0xeb, 0xe8,
		0xec, 0xb6, 0xb3, 0x93, 0x07, 0x23, 0x91, 0xfd, 0xf4, 0x8d, 0x3d, 0xbc, 0x32, 0xb8, 0x78, 0x5a,
		0x15, 0x73, 0xe3, 0xe9, 0xa9, 0x05, 0xa7, 0xe6, 0xba, 0xe1, 0xfa, 0x68, 0x26, 0xba, 0x7d, 0xb9,
		0x53, 0x39, 0xd9, 0x1c, 0xcb, 0xf6, 0xed, 0x05, 0x03, 0x38, 0xeb, 0x37, 0xcb, 0x25, 0x4b, 0x00,
		0xaa, 0x9f, 0x67, 0x19, 0x1a, 0x71, 0x3c, 0x12, 0x7e, 0x3f, 0x3d, 0x58, 0x1c, 0x19, 0x68, 0x97,
		0xf0, 0x42, 0x84, 0xd3, 0xf4, 0x6a, 0x69, 0x86, 0x24, 0xd4, 0x63, 0x78, 0x28, 0xbd, 0x5e, 0xba,
		0x33, 0x15, 0xc3, 0x16, 0xb4, 0x52, 0x78, 0x26, 0x83, 0x09, 0x7a, 0x31, 0x35, 0xc3, 0x1a, 0x34,
		0x13, 0x58, 0x94, 0xfe, 0x10, 0x3d, 0x39, 0x8a, 0x61, 0x17, 0xdc, 0x1a, 0x5e, 0xcb, 0x70, 0x86,
		0xde, 0x1c, 0xcd, 0x30, 0x4f, 0xef, 0x06, 0xe6, 0xc0, 0xeb, 0xa1, 0x5b, 0xa1, 0x18, 0x36, 0xe8,
		0xc3, 0x42, 0x1b, 0x82, 0x38, 0x3d, 0x0b, 0xcd, 0xb0, 0x4a, 0x0d, 0x13, 0x8f, 0xc1, 0x8f, 0xd1,
		0xa3, 0x54, 0x0c, 0x59, 0x6a, 0x3b, 0x78, 0x05, 0x61, 0x82, 0xaa, 0x52, 0x33, 0x2c, 0x51, 0x5d,
		0x60, 0x81, 0x3c, 0x83, 0xee, 0x41, 0x31, 0xa4, 0xa8, 0x25, 0xf1, 0x9c, 0x02, 0x93, 0x2a, 0xa0,
		0x19, 0x2c, 0x6a, 0x02, 0x96, 0xc8, 0x17, 0x54, 0x26, 0xc5, 0x90, 0x13, 0xbf, 0x96, 0x37, 0x55,
		0x01, 0xfe, 0xfb, 0xbb, 0x48, 0x44, 0xbd, 0xbc, 0x1d, 0xf9, 0x1a, 0xce, 0x27, 0x34, 0xa6, 0x08,
		0xda, 0x3e, 0x03, 0x00, 0x00,
	};
	const struct test5_subtest
	{
		uint32_t offset, count;
		const char *data;
	} subtests[] =
	{
		{0x000*4, 4, "000\n"},
		{0x143*4, 4, "143\n"},
		{0x0f0*4, 4, "0f0\n"},
		{0x1ff*4, 4, "1ff\n"},
		{0x001*4, 4, "001\n"},
		{0x1fd*4, 4, "1fd\n"},
	};
	char *dsrc = malloc (sizeof (src));
	int retval = 0;
	struct ocpdir_t *test_dir;
	struct ocpfile_t *osrc;
	struct ocpdir_t *oddst;
	struct ocpfile_t *odst;
	struct ocpdir_t *oddst2;
	struct ocpfile_t *odst2;
	struct ocpfilehandle_t *hdst2;
	int i;
	char *dst = malloc (4);

	printf ("Testing seek and reads (no cache, recursive compression):  ");

	memcpy (dsrc, src, sizeof (src));
	test_dir = ocpdir_mem_getdir_t(ocpdir_mem_alloc (0, "test:"));
	osrc = mem_file_open (test_dir, 9, dsrc, sizeof (src));
	test_dir->unref (test_dir); test_dir = 0;

	oddst  = gzip_check_steal (osrc, 10);
	odst = oddst->readdir_file(oddst, 10);
	oddst2 = gzip_check_steal (odst, 11);
	odst2 = oddst2->readdir_file(oddst2, 11);
	hdst2 = odst->open (odst2);
	oddst2->unref (oddst2); oddst2 = 0;
	odst2->unref (odst2); odst2 = 0;
	oddst->unref (oddst); oddst = 0;
	odst->unref (odst); odst  = 0;
	osrc->unref (osrc); osrc  = 0;

	for (i=0; i < 5; i++)
	{
		if (hdst2->seek_set (hdst2, subtests[i].offset))
		{
			printf ("s");
			retval |= 1;
		} else if (hdst2->read (hdst2, dst, subtests[i].count) != 4)
		{
			printf ("r");
			retval |= 2;
		} else if (memcmp (dst, subtests[i].data, subtests[i].count))
		{
			printf ("d");
			retval |= 4;
		} else {
			printf ("%d", i+1);
		}
	}
	if (retval)
	{
		printf (ANSI_COLOR_RED " Failed" ANSI_COLOR_RESET "\n");
	} else {
		printf (ANSI_COLOR_GREEN " OK" ANSI_COLOR_RESET "\n");
	}

	hdst2->unref (hdst2); hdst2 = 0;
	free (dst);

	return retval;
}


int main(int argc, char *argv[])
{
	int retval = 0;

	printf ( ANSI_COLOR_CYAN "Testing GZIP" ANSI_COLOR_RESET "\n");
	retval |= gzip_test1 ();
	retval |= gzip_test2 ();
	retval |= gzip_test3 ();
	retval |= gzip_test4 ();
	retval |= gzip_test5 ();
	printf ("\n");

	return retval;
}
