//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.cif2cif;

import org.eclipse.escet.common.java.output.WarnOutput;

/**
 * Linearization transformation that creates the Cartesian product of all edges for non-tau events, combining them in
 * all possible combinations. This results in self loops for all non-tau events, where the combination of all self loops
 * maintains all non-deterministic choices of the original specification. Worst case, the model size of the resulting
 * specification compared to the model size of the original specification could blow up exponentially.
 *
 * <p>
 * Where possible, exponential blow-up is prevented by merging multiple alternatives, via which an automaton may
 * participate in synchronization, into a single alternative. Essentially, if there is no guard overlap, it does
 * something similar to {@link LinearizeMerge}.
 * </p>
 *
 * <p>
 * This transformation produces linearized edges in the order that adheres to the transition execution order as defined
 * by the controller properties checker. However, since it may generate multiple linearized edges for the same event, it
 * is better to use the {@link LinearizeMerge} transformation instead, if compliance to the execution scheme of the
 * controller properties checker is desired.
 * </p>
 *
 * <p>
 * This transformation may produce conditional updates ('if' updates) when merging multiple synchronization alternatives
 * of a single automaton into a single alternative.
 * </p>
 *
 * <p>
 * See the {@link LinearizeBase} class for further details.
 * </p>
 */
public class LinearizeProductWithMerge extends LinearizeProduct {
    /**
     * Constructor for the {@link LinearizeProductWithMerge} class.
     *
     * <p>
     * Does not allow optimization of initialization of newly introduced location pointers, by analyzing declarations
     * (used for instance in initialization predicates) to see whether they have constant values.
     * </p>
     *
     * @param warnOutput Callback to send warnings to the user.
     */
    public LinearizeProductWithMerge(WarnOutput warnOutput) {
        super(true, false, warnOutput);
    }

    /**
     * Constructor for the {@link LinearizeProductWithMerge} class.
     *
     * @param optInits Whether to allow optimization of initialization of newly introduced location pointers, by
     *     analyzing declarations (used for instance in initialization predicates) to see whether they have constant
     *     values.
     * @param warnOutput Callback to send warnings to the user.
     */
    public LinearizeProductWithMerge(boolean optInits, WarnOutput warnOutput) {
        super(true, optInits, warnOutput);
    }
}
