/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */

package org.eclipse.lsat.external.api.util;

import org.eclipse.lsat.external.api.model.ActionBase;
import org.eclipse.lsat.external.api.model.ArrayValues;
import org.eclipse.lsat.external.api.model.Claim;
import org.eclipse.lsat.external.api.model.LsatData;
import org.eclipse.lsat.external.api.model.Move;
import org.eclipse.lsat.external.api.model.PeripheralAction;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventBase;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventEntry;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventExit;
import org.eclipse.lsat.external.api.model.ProductLifeCycleEventTransfer;
import org.eclipse.lsat.external.api.model.ProductPropertyBase;
import org.eclipse.lsat.external.api.model.ProductPropertyBoolean;
import org.eclipse.lsat.external.api.model.ProductPropertyInteger;
import org.eclipse.lsat.external.api.model.ProductPropertyString;
import org.eclipse.lsat.external.api.model.RaiseEvent;
import org.eclipse.lsat.external.api.model.Release;
import org.eclipse.lsat.external.api.model.RequireEvent;
import org.eclipse.lsat.external.api.model.ScalarValue;
import org.eclipse.lsat.external.api.model.SyncBar;
import org.eclipse.lsat.external.api.model.TimingSetting;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.typeadapters.RuntimeTypeAdapterFactory;

public class LsatDataUtil {

    private LsatDataUtil() {
    }

    /** Factory for ProductPropertyBase hierarchy (discriminator: propertyType). */
    private static RuntimeTypeAdapterFactory<ActionBase> actionFactory() {
        return RuntimeTypeAdapterFactory
                .of(ActionBase.class, "actionType")
                .registerSubtype(Move.class, "Move")
                .registerSubtype(Claim.class, "Claim")
                .registerSubtype(Release.class, "Release")
                .registerSubtype(PeripheralAction.class, "PeripheralAction")
                .registerSubtype(RaiseEvent.class, "RaiseEvent")
                .registerSubtype(RequireEvent.class, "RequireEvent")
                .registerSubtype(SyncBar.class, "SyncBar");
    }

    /** Factory for ProductLifeCycleEventBase hierarchy (discriminator: eventType). */
    private static RuntimeTypeAdapterFactory<ProductLifeCycleEventBase> productLifeCycleEventFactory() {
        return RuntimeTypeAdapterFactory
                .of(ProductLifeCycleEventBase.class, "eventType")
                .registerSubtype(ProductLifeCycleEventEntry.class, "Entry")
                .registerSubtype(ProductLifeCycleEventExit.class, "Exit")
                .registerSubtype(ProductLifeCycleEventTransfer.class, "Transfer");
    }

    /** Factory for ProductPropertyBase hierarchy (discriminator: propertyType). */
    private static RuntimeTypeAdapterFactory<ProductPropertyBase> productPropertyFactory() {
        return RuntimeTypeAdapterFactory
                .of(ProductPropertyBase.class, "propertyType")
                .registerSubtype(ProductPropertyBoolean.class, "Boolean")
                .registerSubtype(ProductPropertyInteger.class, "Integer")
                .registerSubtype(ProductPropertyString.class, "String");
    }

    /** Factory for TimingSetting hierarchy (discriminator: type). */
    private static RuntimeTypeAdapterFactory<TimingSetting> timingSettingFactory() {
        return RuntimeTypeAdapterFactory
                .of(TimingSetting.class, "type")
                .registerSubtype(ScalarValue.class, "Scalar")
                .registerSubtype(ArrayValues.class, "Array");
    }

    /** Builder: a Gson pre-configured with all LSAT polymorphic factories. */
    private static Gson newGsonWithLsatFactories() {
        return new GsonBuilder()
                .registerTypeAdapterFactory(actionFactory())
                .registerTypeAdapterFactory(productLifeCycleEventFactory())
                .registerTypeAdapterFactory(productPropertyFactory())
                .registerTypeAdapterFactory(timingSettingFactory())
                .create();
    }

    /** Utility function to parse a JSON string to an LsatData object. */
    public static LsatData fromJson(String jsonText) {
        var gson = newGsonWithLsatFactories();
        var lsatData = gson.fromJson(jsonText, LsatData.class);
        lsatData.resolveReferences(lsatData);
        return lsatData;
    }
}
