/*******************************************************************************
 * Copyright (c) 2011, 2024 Google, Inc.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * https://www.eclipse.org/legal/epl-2.0.
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Google, Inc. - initial API and implementation
 *******************************************************************************/
package org.eclipse.wb.core.gef.policy.helpers;

import org.eclipse.wb.gef.core.policies.IEditPartDecorationListener;
import org.eclipse.wb.gef.graphical.policies.LayoutEditPolicy;
import org.eclipse.wb.gef.graphical.policies.SelectionEditPolicy;

import org.eclipse.gef.EditPart;
import org.eclipse.gef.EditPolicy;

import java.util.Map;
import java.util.WeakHashMap;

/**
 * Helper for replacing {@link EditPolicy#SELECTION_FEEDBACK_ROLE} during child {@link EditPart} decoration
 * and restoring original {@link SelectionEditPolicy} during undecoration.
 *
 * @author scheglov_ke
 * @coverage core.gef.policy
 */
public abstract class SelectionPolicyDecorationHelper {
	private final LayoutEditPolicy m_layoutPolicy;
	private final Map<EditPart, EditPolicy> m_policies = new WeakHashMap<>();

	////////////////////////////////////////////////////////////////////////////
	//
	// Constructor
	//
	////////////////////////////////////////////////////////////////////////////
	public SelectionPolicyDecorationHelper(LayoutEditPolicy layoutPolicy) {
		m_layoutPolicy = layoutPolicy;
		m_layoutPolicy.addEditPartListener(new IEditPartDecorationListener() {
			@Override
			public void decorate(EditPart child) {
				if (shouldChangePolicy(child)) {
					rememberOldPolicy(child);
					setNewPolicy(child);
				}
			}

			@Override
			public void undecorate(EditPart child) {
				if (shouldChangePolicy(child)) {
					restoreOldPolicy(child);
				}
			}
		});
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Implementation
	//
	////////////////////////////////////////////////////////////////////////////
	private void rememberOldPolicy(EditPart child) {
		EditPolicy selectionPolicy = child.getEditPolicy(EditPolicy.SELECTION_FEEDBACK_ROLE);
		if (selectionPolicy != null) {
			m_policies.put(child, selectionPolicy);
		}
	}

	private void setNewPolicy(EditPart child) {
		EditPolicy newPolicy = getNewPolicy(child);
		child.installEditPolicy(EditPolicy.SELECTION_FEEDBACK_ROLE, newPolicy);
	}

	private void restoreOldPolicy(EditPart child) {
		EditPolicy selectionPolicy = m_policies.get(child);
		child.installEditPolicy(EditPolicy.SELECTION_FEEDBACK_ROLE, selectionPolicy);
	}

	////////////////////////////////////////////////////////////////////////////
	//
	// Implementation specific
	//
	////////////////////////////////////////////////////////////////////////////
	/**
	 * @return <code>true</code> if {@link EditPolicy#SELECTION_FEEDBACK_ROLE} of given {@link EditPart} should
	 *         be changed.
	 */
	protected boolean shouldChangePolicy(EditPart child) {
		return true;
	}

	/**
	 * @return new {@link EditPolicy} to use for {@link EditPolicy#SELECTION_FEEDBACK_ROLE}, may be
	 *         <code>null</code>.
	 */
	protected EditPolicy getNewPolicy(EditPart child) {
		return null;
	}
}
