/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.core.slm;

import java.io.IOException;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.snapshots.SnapshotId;
import org.elasticsearch.snapshots.SnapshotState;
import org.elasticsearch.xcontent.ConstructingObjectParser;
import org.elasticsearch.xcontent.ParseField;
import org.elasticsearch.xcontent.ToXContent;
import org.elasticsearch.xcontent.ToXContentObject;
import org.elasticsearch.xcontent.XContentBuilder;
import org.elasticsearch.xcontent.XContentParser;

public class SnapshotRetentionConfiguration
implements ToXContentObject,
Writeable {
    public static final SnapshotRetentionConfiguration EMPTY = new SnapshotRetentionConfiguration(null, null, null);
    private static final ParseField EXPIRE_AFTER = new ParseField("expire_after", new String[0]);
    private static final ParseField MINIMUM_SNAPSHOT_COUNT = new ParseField("min_count", new String[0]);
    private static final ParseField MAXIMUM_SNAPSHOT_COUNT = new ParseField("max_count", new String[0]);
    private static final Logger logger = LogManager.getLogger(SnapshotRetentionConfiguration.class);
    private static final Set<SnapshotState> UNSUCCESSFUL_STATES = EnumSet.of(SnapshotState.FAILED, SnapshotState.PARTIAL);
    private static final ConstructingObjectParser<SnapshotRetentionConfiguration, Void> PARSER = new ConstructingObjectParser("snapshot_retention", true, a -> {
        TimeValue expireAfter = a[0] == null ? null : TimeValue.parseTimeValue((String)((String)a[0]), (String)EXPIRE_AFTER.getPreferredName());
        Integer minCount = (Integer)a[1];
        Integer maxCount = (Integer)a[2];
        return new SnapshotRetentionConfiguration(expireAfter, minCount, maxCount);
    });
    private final LongSupplier nowSupplier;
    private final TimeValue expireAfter;
    private final Integer minimumSnapshotCount;
    private final Integer maximumSnapshotCount;

    SnapshotRetentionConfiguration(StreamInput in) throws IOException {
        this.nowSupplier = System::currentTimeMillis;
        this.expireAfter = in.readOptionalTimeValue();
        this.minimumSnapshotCount = in.readOptionalVInt();
        this.maximumSnapshotCount = in.readOptionalVInt();
    }

    public SnapshotRetentionConfiguration(@Nullable TimeValue expireAfter, @Nullable Integer minimumSnapshotCount, @Nullable Integer maximumSnapshotCount) {
        this(System::currentTimeMillis, expireAfter, minimumSnapshotCount, maximumSnapshotCount);
    }

    public SnapshotRetentionConfiguration(LongSupplier nowSupplier, @Nullable TimeValue expireAfter, @Nullable Integer minimumSnapshotCount, @Nullable Integer maximumSnapshotCount) {
        this.nowSupplier = nowSupplier;
        this.expireAfter = expireAfter;
        this.minimumSnapshotCount = minimumSnapshotCount;
        this.maximumSnapshotCount = maximumSnapshotCount;
        if (this.minimumSnapshotCount != null && this.minimumSnapshotCount < 1) {
            throw new IllegalArgumentException("minimum snapshot count must be at least 1, but was: " + this.minimumSnapshotCount);
        }
        if (this.maximumSnapshotCount != null && this.maximumSnapshotCount < 1) {
            throw new IllegalArgumentException("maximum snapshot count must be at least 1, but was: " + this.maximumSnapshotCount);
        }
        if (maximumSnapshotCount != null && minimumSnapshotCount != null && this.minimumSnapshotCount > this.maximumSnapshotCount) {
            throw new IllegalArgumentException("minimum snapshot count " + this.minimumSnapshotCount + " cannot be larger than maximum snapshot count " + this.maximumSnapshotCount);
        }
    }

    public static SnapshotRetentionConfiguration parse(XContentParser parser, String name) {
        return (SnapshotRetentionConfiguration)PARSER.apply(parser, null);
    }

    public boolean isSnapshotEligibleForDeletion(SnapshotId snapshotId, RepositoryData.SnapshotDetails snapshotDetails, Map<SnapshotId, RepositoryData.SnapshotDetails> allSnapshots) {
        assert (org.elasticsearch.common.Strings.hasText((String)snapshotDetails.getSlmPolicy()));
        SnapshotState snapshotState = snapshotDetails.getSnapshotState();
        long startTimeMillis = snapshotDetails.getStartTimeMillis();
        String snapshotName = snapshotId.getName();
        int totalSnapshotCount = allSnapshots.size();
        List<Map.Entry> sortedSnapshots = allSnapshots.entrySet().stream().sorted(Comparator.comparingLong(e -> ((RepositoryData.SnapshotDetails)e.getValue()).getStartTimeMillis())).toList();
        int successCount = 0;
        long latestSuccessfulTimestamp = Long.MIN_VALUE;
        for (RepositoryData.SnapshotDetails snapshot : allSnapshots.values()) {
            assert (Objects.equals(snapshot.getSlmPolicy(), snapshotDetails.getSlmPolicy()));
            if (snapshot.getSnapshotState() != SnapshotState.SUCCESS) continue;
            ++successCount;
            latestSuccessfulTimestamp = Math.max(latestSuccessfulTimestamp, snapshot.getStartTimeMillis());
        }
        long newestSuccessfulTimestamp = latestSuccessfulTimestamp;
        int successfulSnapshotCount = successCount;
        if (this.expireAfter == null && UNSUCCESSFUL_STATES.contains(snapshotState) && newestSuccessfulTimestamp > startTimeMillis) {
            logger.trace("[{}]: ELIGIBLE as it is {} and there is a more recent successful snapshot", (Object)snapshotName, (Object)snapshotState);
            return true;
        }
        if (this.maximumSnapshotCount != null && successfulSnapshotCount > this.maximumSnapshotCount) {
            long successfulSnapsToDelete = successfulSnapshotCount - this.maximumSnapshotCount;
            boolean found = false;
            int successfulSeen = 0;
            for (Map.Entry s2 : sortedSnapshots) {
                if (((RepositoryData.SnapshotDetails)s2.getValue()).getSnapshotState() == SnapshotState.SUCCESS) {
                    ++successfulSeen;
                }
                if ((long)successfulSeen > successfulSnapsToDelete) break;
                if (!((SnapshotId)s2.getKey()).equals((Object)snapshotId)) continue;
                found = true;
                break;
            }
            if (found) {
                logger.trace("[{}]: ELIGIBLE as it is one of the {} oldest snapshots with {} non-failed snapshots ({} total), over the limit of {} maximum snapshots", (Object)snapshotName, (Object)successfulSnapsToDelete, (Object)successfulSnapshotCount, (Object)totalSnapshotCount, (Object)this.maximumSnapshotCount);
                return true;
            }
            logger.trace("[{}]: SKIPPING as it is not one of the {} oldest snapshots with {} non-failed snapshots ({} total), over the limit of {} maximum snapshots", (Object)snapshotName, (Object)successfulSnapsToDelete, (Object)successfulSnapshotCount, (Object)totalSnapshotCount, (Object)this.maximumSnapshotCount);
        }
        if (this.minimumSnapshotCount != null && successfulSnapshotCount <= this.minimumSnapshotCount) {
            if (!UNSUCCESSFUL_STATES.contains(snapshotState)) {
                logger.trace("[{}]: INELIGIBLE as there are {} non-failed snapshots ({} total) and {} minimum snapshots needed", (Object)snapshotName, (Object)successfulSnapshotCount, (Object)totalSnapshotCount, (Object)this.minimumSnapshotCount);
                return false;
            }
            logger.trace("[{}]: SKIPPING minimum snapshot count check as this snapshot is {} and not counted towards the minimum snapshot count.", (Object)snapshotName, (Object)snapshotState);
        }
        if (this.expireAfter != null) {
            long snapshotAge;
            if (this.minimumSnapshotCount != null) {
                boolean maybeEligible;
                if (snapshotState == SnapshotState.SUCCESS) {
                    maybeEligible = sortedSnapshots.stream().filter(snap -> SnapshotState.SUCCESS.equals((Object)((RepositoryData.SnapshotDetails)snap.getValue()).getSnapshotState())).limit(Math.max(0, successfulSnapshotCount - this.minimumSnapshotCount)).anyMatch(s -> ((SnapshotId)s.getKey()).equals((Object)snapshotId));
                } else if (UNSUCCESSFUL_STATES.contains(snapshotState)) {
                    maybeEligible = allSnapshots.containsKey(snapshotId);
                } else {
                    logger.trace("[{}] INELIGIBLE because snapshot is in state [{}]", (Object)snapshotName, (Object)snapshotState);
                    return false;
                }
                if (!maybeEligible) {
                    logger.trace("[{}]: INELIGIBLE as snapshot expiration would pass the minimum number of configured snapshots ({}) to keep, regardless of age", (Object)snapshotName, (Object)this.minimumSnapshotCount);
                    return false;
                }
            }
            if ((snapshotAge = this.nowSupplier.getAsLong() - startTimeMillis) > this.expireAfter.getMillis()) {
                logger.trace(() -> Strings.format((String)"[%s]: ELIGIBLE as snapshot age of %s is older than %s", (Object[])new Object[]{snapshotName, new TimeValue(snapshotAge).toHumanReadableString(3), this.expireAfter.toHumanReadableString(3)}));
                return true;
            }
            logger.trace(() -> Strings.format((String)"[%s]: INELIGIBLE as snapshot age of [%sms] is newer than %s", (Object[])new Object[]{snapshotName, new TimeValue(snapshotAge).toHumanReadableString(3), this.expireAfter.toHumanReadableString(3)}));
            return false;
        }
        logger.trace("[{}]: INELIGIBLE as no retention predicates matched", (Object)snapshotName);
        return false;
    }

    public void writeTo(StreamOutput out) throws IOException {
        out.writeOptionalTimeValue(this.expireAfter);
        out.writeOptionalVInt(this.minimumSnapshotCount);
        out.writeOptionalVInt(this.maximumSnapshotCount);
    }

    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        builder.startObject();
        if (this.expireAfter != null) {
            builder.field(EXPIRE_AFTER.getPreferredName(), this.expireAfter.getStringRep());
        }
        if (this.minimumSnapshotCount != null) {
            builder.field(MINIMUM_SNAPSHOT_COUNT.getPreferredName(), this.minimumSnapshotCount);
        }
        if (this.maximumSnapshotCount != null) {
            builder.field(MAXIMUM_SNAPSHOT_COUNT.getPreferredName(), this.maximumSnapshotCount);
        }
        builder.endObject();
        return builder;
    }

    public int hashCode() {
        return Objects.hash(this.expireAfter, this.minimumSnapshotCount, this.maximumSnapshotCount);
    }

    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (obj.getClass() != this.getClass()) {
            return false;
        }
        SnapshotRetentionConfiguration other = (SnapshotRetentionConfiguration)obj;
        return Objects.equals(this.expireAfter, other.expireAfter) && Objects.equals(this.minimumSnapshotCount, other.minimumSnapshotCount) && Objects.equals(this.maximumSnapshotCount, other.maximumSnapshotCount);
    }

    public String toString() {
        return org.elasticsearch.common.Strings.toString((ToXContent)this);
    }

    static {
        PARSER.declareString(ConstructingObjectParser.optionalConstructorArg(), EXPIRE_AFTER);
        PARSER.declareInt(ConstructingObjectParser.optionalConstructorArg(), MINIMUM_SNAPSHOT_COUNT);
        PARSER.declareInt(ConstructingObjectParser.optionalConstructorArg(), MAXIMUM_SNAPSHOT_COUNT);
    }
}

