/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 *  epix.h -- Automatically created on Mon Nov 10 02:29:20 PM UTC 2025  *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <iostream>
#include <vector>
#include <list>
#include <set>
#include <algorithm>
#include <string>
#include <sstream>
#include <cstdlib>
#include <cmath>

#ifndef EPIX_H
#define EPIX_H

/* 
 * enums.h -- ePiX's user-visible enumeration types
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.0.23
 * Last Change: January 10, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_ENUMS
#define EPIX_ENUMS

namespace ePiX {

  enum epix_mark_type {PATH, CIRC, SPOT, RING, DOT, DDOT, PLUS, OPLUS, 
		       TIMES, OTIMES, DIAMOND, UP, DOWN, BOX, BBOX, 
		       HTICK, VTICK, TEXT};

  enum epix_label_posn {none, c, r, tr, rt, t, tl, lt, l, bl, lb, b, br, rb};

  enum epix_integral_type {LEFT, RIGHT, UPPER, LOWER, TRAP, MIDPT};

  enum epix_field_type {SLOPE, DART, VECTOR};

} // end of namespace

#endif /* EPIX_ENUMS */
/*
 * length.h -- ePiX true and LaTeX length manipulation
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.2.0
 * Last Change: September 22, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * This file provides the length class from ePiX2:
 *
 *   - length(std::string) assumes argument is "double [space(s)] unit"
 *     Recognized units are pt (points), in, cm, mm, pc (picas = 1/12 in), 
 *     bp (big points = 1/72 in)
 *
 *     Malformed strings are handled as follows:
 *     Missing doubles are assumed to be 0, units are assumed to be pt
 *
 *   - length(double) sets length to specified number of pt
 *
 *   - operator += Increments a length, returning answer in LHS units, e.g.
 *     1cm+=2in is 6.08cm
 *     2in+=1cm is 2.3937in
 *
 *   - operator *= scales a length by a double
 *   - to(string) converts a length to specified units
 *   - operator< compares true lengths
 *   - get_lengths(string, length, length) parses a string into ht and width
 *
 * A length can be negative.
 */

#ifndef EPIX_LENGTH
#define EPIX_LENGTH


namespace ePiX {

  class length {
  public:
    length(std::string); // not const std::string&
    explicit length(double = 0);

    length& operator+= (const length&); // increment by another length
    length& operator*= (double);  // scale
    length& to(std::string);            // convert to specified units

    // compare, with units conversion
    bool operator== (const length&) const;
    bool operator!= (const length& len) const { return !((*this) == len); }

    // for output
    double magnitude() const;
    std::string units() const;
    std::string name() const;

  private:
    double m_mag;
    double pts_per_unit; // constants defined in length.cc
    std::string m_units;
  }; // end of class length

  // value-returning operators
  length operator+ (length, const length&);
  length operator* (double, length len);

  // Compare true size; must pass by value...
  bool operator< (length arg1, length arg2);

  /*
   * Parse a string into two lengths (width and height).
   *
   * The "sz" argument is expected to look like "4in x 10cm" or "4 x 6 in",
   * specifically a double, an optional two-letter unitlength, an "x", a
   * double, and a two-letter unitlength (defaults to pt). Spaces are
   * unimportant, as is initial or trailing garbage. Badly malformed input
   * (e.g., "4,6 in") may confuse the parsing code on some platforms.
   */
  void get_lengths(std::string sz, length& length1, length& length2);

} // end of namespace

#endif /* EPIX_LENGTH */
/* 
 * interval.h -- ePiX::interval class and operations
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.0.25
 * Last Change: May 20, 2007
 *
 * 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 *
 *
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * 
 * This file provides the interval class:
 *
 *     - Minkowski sum, intersection
 */

#ifndef EPIX_INTERVAL
#define EPIX_INTERVAL


namespace ePiX {

  class interval {
  public:
    interval(std::string); // allow implicit conversion, must mangle argument
    interval(double, double); // closed interval

    static interval emptyset;

    double min() const;
    double max() const;
    double avg() const;

    bool contains(double) const;

    // Minkowski sum
    interval& operator +=(const interval&);

    // intersection
    interval& operator *=(const interval&);

    bool operator== (const interval&) const;
    bool operator!= (const interval&) const;
    bool is_empty() const;

  private:
    double m_rmin, m_rmax;

    bool m_closed_l, m_closed_r;

    static interval literal(double a, double b); // can force b<a
  }; // end of interval class

  interval operator+ (interval I1, const interval& I2);
  interval operator* (interval I1, const interval& I2);

} // end of namespace

#endif /* EPIX_INTERVAL */
/* 
 *  triples.h -- ePiX::P class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.2.17
 * Last Change: July 20, 2017
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2017
 * Andrew D. Hwang <rot 13 nujnat at ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 *   This file provides:
 *
 *   The P class (ordered triple of <double>s) and operators
 *
 *    - P(), P() (default to origin), E_1, E_2, E_3 (standard basis)
 *    - Vector space operators
 *    - Other vector operators, operands (a,b,c), (x,y,z)
 *       | dot product                      ax + by + cz
 *       norm(p) = sqrt(p|p)
 *       & componentwise product            (ax, by, cz)
 *       * cross product                    (bz-cy, cx-az, ay-bx)
 *       J rotate (x1,x2,0)-plane 1/4 turn  (-b,a,0)
 *       % orthogonalization                u%v = u - ((u|v)/(v|v))*v
 */
#ifndef EPIX_TRIPLES
#define EPIX_TRIPLES

namespace ePiX {

  class Complex;
  class P {
  public:
    P(double arg1=0, double arg2=0, double arg3=0);
    P(const Complex&);
    
    double x1() const;
    double x2() const;
    double x3() const;

    bool is_valid() const; // coords not inf or nan

    // increment operators
    P& operator += (const P&);
    P& operator -= (const P&);
    P& operator *= (double); // scalar multipication

    // cross product
    P& operator *= (const P&);

    // componentwise product
    P& operator &= (const P&);

    // orthogonalization: u %= v is the vector of the form u-k*v perp to v
    P& operator %= (const P&);

  private:
    double m_x1, m_x2, m_x3;
  }; // end of class P


  // standard basis in global namespace
  const P E_1(1,0,0);
  const P E_2(0,1,0);
  const P E_3(0,0,1);

  // vector space operations
  P operator- (P u);  // unary negation
  P operator+ (P u, const P& v);
  P operator- (P u, const P& v);
  // scalar multiplication
  P operator* (double c, P v);

  // cross product
  P operator* (P, const P&);
  P J(P); // quarter turn about E_3-axis

  // dot product
  double operator | (const P&, const P&);
  double norm(const P&);

  // componentwise product (a,b,c)&(x,y,z)=(ax,by,cz)
  P operator& (P, const P&);
  // orthogonalization
  P operator% (P, const P&);

  // (in)equality
  bool operator == (const P&, const P&);
  bool operator != (const P&, const P&);

} // end of namespace

#endif /* EPIX_TRIPLES */
/* 
 * Complex.h -- ePiX::Complex number class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.2.18
 * Last Change: September 17, 2017
 *
 * 
 * Copyright (C) 2007, 2008, 2017
 * Andrew D. Hwang <rot 13 nujnat at ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 *
 *
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef EPIX_COMPLEX
#define EPIX_COMPLEX

namespace ePiX {

  class Complex {
  public:
    Complex(double real=0, double imag=0);

    Complex& operator += (const Complex& arg);
    Complex& operator -= (const Complex& arg);
    Complex& operator *= (const Complex& arg);
    Complex& operator /= (const Complex& arg);
    Complex& conj();

    const bool operator == (const Complex& arg) const;
    const bool operator != (const Complex& arg) const;

    double re() const;
    double im() const;
    double Arg(int branch = 0) const; // Arg(0, 0) = 0;
    double norm() const;

  private:
    double m_real;
    double m_imag;
  }; // end of class Complex

  const Complex operator+ (Complex arg1, const Complex& arg2);
  const Complex operator- (Complex arg1, const Complex& arg2);
  const Complex operator- (Complex arg);

  const Complex operator* (Complex arg1, const Complex& arg2);
  const Complex operator/ (Complex arg1, const Complex& arg2);

  double Arg(const Complex& arg, int branch = 0);
  double norm(const Complex& arg);

  Complex expC(const Complex&);
  Complex logC(const Complex&, int branch=0);

  Complex sqrtC(const Complex&, int branch=0);
  Complex rootC(const Complex&, int order, int branch=0);
  Complex powC(const Complex&, int);

  Complex SinC(const Complex&);
  Complex CosC(const Complex&);

  Complex TanC(const Complex&);
  Complex CotC(const Complex&);

  Complex SecC(const Complex&);
  Complex CscC(const Complex&);

  Complex SinhC(const Complex&);
  Complex CoshC(const Complex&);

  Complex TanhC(const Complex&);
  Complex CothC(const Complex&);

  Complex SechC(const Complex&);
  Complex CschC(const Complex&);
} // end of namespace

#endif /* EPIX_COMPLEX */
/*
 * functions.h  -- ePiX auxiliary functions; Deriv, Integral classes
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.22
 * Last Change: September 24, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_FUNCTIONS
#define EPIX_FUNCTIONS

namespace ePiX {

  class P;

  // sensitive to angle units
  double Cos(double t);
  double Sin(double t);
  double Tan(double t);

  double Sec(double t);
  double Csc(double t);
  double Cot(double t);

  double Acos(double t);
  double Asin(double t);
  double Atan(double t);

  double Atan2(double y, double x);

  // additional hyperbolic functions and inverses
  double sech(double);
  double csch(double);
  double coth(double);

  double asech(double);
  double acsch(double);
  double acoth(double);

  // zero functions
  double zero(double);
  double zero(double, double);
  double zero(double, double, double);
  double zero(const P&);

  // P constructors
  P xyz(double x, double y, double z=0);
  P cyl(double r, double t, double z);
  P sph(double r, double t, double phi);

  // for log data plotting
  P log_log(double x, double y, double z=0);
  P log_lin(double x, double y, double z=0);
  P lin_log(double x, double y, double z=0);

  P cylindrical(P); // not const P&
  P spherical(P);

  P polar(double r, double t);
  P cis(double t);

  // utility functions
  double recip (double);
  double sinx (double); // discontinuity removed
  double sgn (double);

  // period-2 extension of absolute value on [-1,1]: \/\/\/
  double cb (double);

  int gcd (int, int);

  double min(double, double);
  double max(double, double);

  double snip_to(double var, double arg1, double arg2);

  double inf (double f(double), double, double);
  double sup (double f(double), double, double);


  // derivative class
  class Deriv {
  private:
    double (*f)(double);
    double dt;

  public:
    Deriv(double func(double));
    Deriv(double func(double), double eps);

    P operator() (const P&) const; // for plotting

    // numerical values
    double eval(double t) const;

    // one-sided derivatives
    double right(double t) const;
    double left(double t) const;
  }; // end of class Deriv


  // definite integral class
  class Integral {
  private:
    double (*f)(double);
    double x0; // lower limit

  public:
    Integral(double func(double), double a=0);

    P operator() (const P&) const;

    double eval(double) const;
  }; // end of class Integral

  double newton (double f(double), double g(double), double start);
  double newton (double f(double), double start);

} // end of namespace

#endif /* EPIX_FUNCTIONS */
/* 
 * pairs.h -- ePiX pair:: class and mathematical operators
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.10
 * Last Change: August 04, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_PAIRS
#define EPIX_PAIRS

namespace ePiX {

  class P;

  class pair {
  public:
    pair(double arg1=0, double arg2=0)
      : m_x1(arg1), m_x2(arg2) { }

    pair(const P& arg); // take first two coordinates

    double x1() const { return m_x1; }
    double x2() const { return m_x2; }

    // unary increment operators
    pair& operator += (const pair& arg);
    pair& operator -= (const pair& arg);
    pair& operator *= (double c);

    // complex multiplication and division
    pair& operator *= (const pair& arg);
    pair& operator /= (const pair& arg);
    bool operator== (const pair& u) const;
    bool operator!= (const pair& u) const;

  private:
    double m_x1;
    double m_x2;

  }; // end of class pair
  
  pair operator- (pair u);
  pair operator+ (pair u, const pair& v);
  pair operator- (pair u, const pair& v);
  pair operator* (double c, pair u);

  // complex arithmetic
  pair J(pair p);
  pair operator* (pair u, const pair& v);
  pair operator/ (pair u, const pair& v);

  // dot product
  double operator| (const pair& u, const pair& v);
  double norm (const pair& u);

  // componentwise product (a,b)&(x,y)=(ax,by)
  pair operator& (const pair& u, const pair& v);

} // end of namespace

#endif /* EPIX_PAIRS */
/* 
 * affine.h -- ePiX::affine class (affine screen maps)
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.21
 * Last Change: September 23, 2007
 *
 * 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 *
 *
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * An affine map is constructed from the images of the "standard
 * triple" (0,0), (1,0), and (0,1).
 *
 * An affine map may be (post)-composed with translation, rotation,
 * reflection, scaling along either or both coordinate axes, shearing,
 * and may be inverted. An optional pair argument, defaulting to
 * (0,0), gives the center of the operation (if appropriate).
 *
 * An affine map may be post-composed via member function, and
 * pre-composed with operator syntax. The evaluation operator
 * returns the image of the argument.
 */

#ifndef EPIX_AFFINE
#define EPIX_AFFINE


namespace ePiX {

  class affine {
  public:
    // the identity map
    affine();

    // images of (1,0), (0,1), (0,0)
    affine(const pair&, const pair&, const pair& loc=pair(0,0));
    affine(const P&, const P&, const P& loc=P(0,0));

    // post-operations
    // translate by arg
    affine&   shift(const pair& arg);
    affine&   shift(const P& arg);

    // rotate by theta about ctr
    affine&  rotate(double theta, const pair& ctr = pair(0,0));
    affine&  rotate(double theta, const P& ctr);

    // reflect in angle-theta line through ctr
    affine& reflect(double theta, const pair& ctr = pair(0,0));
    affine& reflect(double theta, const P& ctr);

    // scale coord direction(s) fixing ctr
    affine& h_scale(double, const pair& ctr=pair(0,0));
    affine& v_scale(double, const pair& ctr=pair(0,0));
    affine&   scale(double, const pair& ctr=pair(0,0));

    affine& h_scale(double, const P& ctr);
    affine& v_scale(double, const P& ctr);
    affine&   scale(double, const P& ctr);

    // shear, fixing ctr
    affine& h_shear(double, const pair& ctr=pair(0,0));
    affine& v_shear(double, const pair& ctr=pair(0,0));

    affine& h_shear(double, const P& ctr);
    affine& v_shear(double, const P& ctr);

    // post-compose
    affine& postcomp(const affine&);

    affine& invert();

    // evaluation
    pair operator() (const pair&) const;
    pair operator() (const P&) const;

    // pre-compose
    affine operator() (const affine&) const;

    bool reverses_orientation() const;

  private:
    pair m_00;
    pair m_10;
    pair m_01;
  };
} // end of ePiX namespace

#endif /* EPIX_AFFINE */
/*
 * Color.h -- ePiX::Color class and helpers
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.2.2
 * Last Change: November 12, 2007
 *
 * 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 *
 *
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * This file defines the ePiX Color user interface:
 *       operator*=           Modify intensity
 *       blend(Color, double) Color mixing
 *       filter(Color)        Apply us as a filter
 *       superpose(Color)     Add colors, clip intensities
 *       invert()             Color negative
 *
 *   - Named primary constructors
 *
 * Scaling and filtering are implemented by converting to the filter's
 * color model. Color model conversions come from the LaTeX xcolor
 * package documentation, V 1.11 (2044/05/09) by Uwe Kern.
 */

#ifndef EPIX_COLOR
#define EPIX_COLOR


namespace ePiX {

  class Color_Base;

  class Color {
  public:
    Color();
    Color(const Color_Base*);
    Color(const Color_Base&);
    Color(const Color&);

    Color& operator= (const Color&);

    ~Color();

    const Color_Base* operator->() const;
    double alpha() const;

    bool is_unset() const;

    Color filter(const Color&) const;

    Color& operator*= (double c);     // scale the intensity
    Color& blend(const Color&, double);
    Color& superpose(const Color&);
    Color& invert();
    Color& alpha(double);

    std::string model() const;
    std::string name() const;
    std::vector<double> densities() const;

  private:
    Color_Base* m_color;
    double m_alpha;

  }; // end of class Color

  Color operator* (double, const Color&);
  const bool operator== (const Color&, const Color&);
  const bool operator!= (const Color&, const Color&);

  // for palette ordering
  const bool operator<  (const Color&, const Color&);


  /*
   * * *  Primary color functions in the global namespace * * *
   *
   * A primary color constructor accepts a numerical argument, which is
   * reduced mod 4, then interpreted as a color density as follows:
   * d = -2 or 2 : white
   * d = -1: full anti-saturation
   * d =  0: black
   * d =  1: full saturation (default)
   *
   * The primary changes continuously, and linearly on each interval
   * [-2,-1], [-1,0], [0,1], and [1,2].
   *
   * Example:
   * Red(0.3) = rgb(0.3, 0, 0) = Red(-3.7)
   * Red(1.3) = rgb(1,0.3,0.3) = Red(-2.7)
   * Red(2.3) = rgb(0.7, 1, 1) = Red(-1.7)
   * Red(3.3) = rgb(0,0.7,0.7) = Red(-0.7)
   */
  Color RGB(double r=0, double g=0, double b=0);
  Color CMY(double c=1, double m=1, double y=1);
  Color CMYK(double c=1, double m=1, double y=1,
	     double k=0); // void same as CMYK(0,0,0,1);
  Color Gray(double d=0);

  Color Red(double d=1);
  Color Green(double d=1);
  Color Blue(double d=1);
  Color White(double d=1);
  Color Black(double d=1);

  Color Cyan(double d=1);
  Color Magenta(double d=1);
  Color Yellow(double d=1);
  Color CMY_White(double d=1);
  Color CMY_Black(double d=1);

  Color CyanK(double d=1);
  Color MagentaK(double d=1);
  Color YellowK(double d=1);
  Color CMYK_White(double d=1);
  Color CMYK_Black(double d=1);

  // color separation
  Color C_Process(double d=1);
  Color M_Process(double d=1);
  Color Y_Process(double d=1);
  Color K_Process(double d=1);

  // "conversion filters"
  Color Neutral();
  Color RGB_Neutral();
  Color CMY_Neutral();
  Color CMYK_Neutral();
  Color Gray_Neutral();

} // end of namespace

#endif /* EPIX_COLOR */
/* 
 * state.h -- ePiX global functions to control current drawing state
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.22
 * Last Change: September 24, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * This file contains the (old) user interface governing:
 *
 *  [] Angular mode
 *  [] Clipping and cropping
 *  [] Font size and face, label angle, mask color
 *  [] Gray depth, path filling, fill color
 *  [] Whether to use PSTricks macros
 *  [] Arrowhead parameters
 *  [] Line style, dash length, dot size
 *  [] Line width
 *  [] Line/text color
 */

#ifndef EPIX_STATE
#define EPIX_STATE


namespace ePiX {

  class Color;
  class P;

  // set camera position
  void viewpoint(const P&);
  void viewpoint(double, double, double);

  // Set angular mode
  void radians();
  void degrees();
  void revolutions();

  // number of current angle units in one full turn
  double full_turn();

  // set clip box
  void clip(bool arg=true); // no effect
  void clip_box(); // default (very large)
  void clip_box(const P& arg1, const P& arg2);
  void clip_to(const P& arg);
  void clip_box(const P& arg);

  // add a clip face; perp points inward
  void clip_face(const P& loc, const P& perp);

  // add parallel clip planes
  void clip_slice(const P& loc, const P& perp);
  void clip_slice(const P& loc, const P& perp, double thickness);

  // loc = origin
  void clip_slice(const P& perp);
  void clip_slice(const P& perp, double thickness);

  // remove user-specified clip planes, preserve clip box settings
  void clip_restore();

  // set flag on active screen; contents unchanged
  void set_crop(bool arg=true);

  // crop contents of active screen
  void crop();

  //// New cropping functions ////
  // set crop mask of active screen
  void crop_rectangle(const P&, const P&);
  void crop_ellipse(const P&, const P&);
  void crop_diamond(const P&, const P&);

  void crop_rectangle();
  void crop_ellipse();
  void crop_diamond();

  void crop_box(const P&, const P&);
  void crop_box(); // (re)set crop box to bounding box

  // Set label attributes
  void font_size(const std::string& arg="nsz");
  void font_face(const std::string& arg="rm");

  void label_color(const Color&);

  void label_angle(double t=0);
  void label_mask(const Color& col);
  void label_mask();

  void label_pad(std::string);
  void label_border(const Color&, double);
  void label_border(const Color&, std::string);
  void label_border(const Color&);

  void label_border(double);
  void label_border(std::string);
  void no_label_border();

  // Gray depth, path filling, [fill color]
  void gray(double depth=0.3);
  void fill(const Color&); // set fill color
  void fill(bool arg=true); // set fill flag
  void nofill();
  // void fill_color(const std::string& arg);

  // Arrowhead parameters
  void arrow_width(double w=3);
  void arrow_ratio(double r=5.5);
  void arrow_inset(double arg=0);
  void arrow_fill(bool arg=true);

  // Dash pattern length
  void dash_size(double len=0);
  void dot_sep(double len=0);

  // Dot and box marker size
  void dot_size(double diam=0);

  // Tick length
  void tick_size(double len=0);

  // Line style
  void line_style(std::string style="-"); // solid by default

  void solid();
  void dashed();
  void dotted();

  // Line width
  void pen(const std::string&);
  void pen(double);

  void bbold();
  void  bold();
  void plain();

  // Line width and color
  void bbold(const Color&);
  void  bold(const Color&);
  void plain(const Color&);

  void pen(const Color&);
  void pen(const Color&, double);
  void pen(const Color&, std::string);

  void base(const Color&);
  void base(const Color&, double);
  void base(const Color&, std::string);

  // Line and text color
  // ePiX provides rgb/cmyk colors via xcolor. The following are equivalent:
  //    red();
  //    rgb(1,0,0);
  //
  // cmyk(0,1,1,0); is visually equivalent but not the same.
  //
  // Densities outside the unit interval [0,1] are truncated, e.g., 
  //    rgb(2, -1, 0.3) = rgb(1, 0, 0.3).

  // red-green-blue
  void rgb(double r, double g, double b); // must pass args by value
  // cyan-magenta-yellow-black
  void cmyk(double c, double m, double y, double k);

  // P arguments, for easy function control
  void rgb(const P&);
  void cmyk(const P&);

  // primary colors
  void red(double d=1);
  void green(double d=1);
  void blue(double d=1);
  void white(double d=1);
  void black(double d=1);

  void cyan(double d=1);
  void magenta(double d=1);
  void yellow(double d=1);

} // end of namespace

#endif /* EPIX_STATE */
/* 
 * frame.h -- Orthonormal basis
 *
 * This file is part of ePiX, a preprocessor for creating high-quality 
 * line figures in LaTeX 
 *
 * Version 1.0.15
 * Last Change: October 10, 2006
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * In geometry, a "frame" is a right-handed orthonormal basis, that is,
 * an ordered set of three mututally perpendicular unit vectors, oriented
 * according to the right-hand rule. A frame has nothing to do with
 * picture frames.
 *
 *
 * This file provides:
 *   - the frame class, and routines for rotating a frame about the axes
 *     determined by its elements. (Note that there are no methods for
 *     rotating about fixed coordinate axes, and that rotations in R^3 do
 *     not generally commute.
 */

#ifndef EPIX_FRAME
#define EPIX_FRAME


namespace ePiX {

  class frame {
  public:
    // standard basis
    frame();

    // Gram-Schmidt
    frame(P arg1, P arg2, P arg3); // need args by value

    // frame elements
    P sea() const;
    P sky() const;
    P eye() const;

    // rotations about frame elements
    frame& rot1(double angle);
    frame& rot2(double angle);
    frame& rot3(double angle);

  private:
    P m_e1; // sea
    P m_e2; // sky
    P m_e3; // eye
  }; // end of class frame

} // end of namespace

#endif /* EPIX_FRAME */
/* 
 * domain.h -- ePiX mesh and domain classes
 *
 * This file is part of ePiX, a preprocessor for creating high-quality 
 * line figures in LaTeX 
 *
 * Version 1.1.6
 * Last Change: July 04, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_DOMAIN
#define EPIX_DOMAIN



namespace ePiX {

  class mesh {
  public:
    mesh(int n1, int n2, int n3=1);
    mesh(int n); // mesh(n, n, n);
    mesh();      // mesh(1, 1, 1);

    int n1() const { return n_1; }
    int n2() const { return n_2; }
    int n3() const { return n_3; }

  private:
    int n_1;
    int n_2;
    int n_3;
  }; // end of class mesh

  // A domain is a gridded coordinate box with specified corners
  // and coarse/fine meshes, used for plotting. The "coarse" step
  // sizes are step1, etc., and the "fine" step sizes are dx1, etc.
  // Steps may be negative.
  //
  // A domain may be "sliced" by setting one of its coordinates to
  // a specified constant.
  //
  // Data members are public for convenience; each member ensures
  // its own well-constructedness.
  //
  class domain {
  public:
    domain(const P& arg1, const P& arg2, const mesh& c, const mesh& f);
    domain(const P& arg1, const P& arg2, const mesh& c);

    // 1-dim'l domain
    domain(double t_min, double t_max, int num_pts);

    int dim() const; // dimension

    // big and small steps
    double step1() const;
    double step2() const;
    double step3() const;

    double dx1() const;
    double dx2() const;
    double dx3() const;

    // resizing attempts to preserve real resolution
    domain resize1(double a1, double b1) const;
    domain resize2(double a2, double b2) const;
    domain resize3(double a3, double b3) const;

    // coordinate slices
    domain slice1(double a1) const;
    domain slice2(double a2) const;
    domain slice3(double a3) const;

    // "n=0": use number from coarse mesh
    std::list<domain> slices1(const unsigned int n=0) const;
    std::list<domain> slices2(const unsigned int n=0) const;
    std::list<domain> slices3(const unsigned int n=0) const;

    // state  functions for internal use
    P corner1() const;
    P corner2() const;

    double corner1_x1() const;
    double corner1_x2() const;
    double corner1_x3() const;

    double corner2_x1() const;
    double corner2_x2() const;
    double corner2_x3() const;

    int coarse_n1() const;
    int coarse_n2() const;
    int coarse_n3() const;

    int fine_n1() const;
    int fine_n2() const;
    int fine_n3() const;

  private:
    P m_corner1;
    P m_corner2;
    mesh m_coarse;
    mesh m_fine;
  }; // end of class domain

  class domain_list {
  public:
    domain_list(std::list<domain>); // implicitly converts "slices" output
    domain_list& add(const domain&);
    domain_list& add(const domain_list&);

    std::list<domain> m_list; // public: we're a simple wrapper
  }; // end of class domain_list
} // end of namespace

#endif /* EPIX_DOMAIN */
/* 
 * camera.h -- ePiX::Camera
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.21
 * Last Change: September 22, 2007
 *
 * 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 *
 *
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * A Camera has a location, orientation, lens, and color filter.  The
 * viewer sits at given <distance> from the <target>, in the direction
 * of the <eye>. The vectors <sea> and <sky> point horizontally to the
 * right and vertically upward on the screen.  The viewing angle is
 * controlled by changing the viewer, target, and/or the distance
 * between them.
 *
 * The implementation data comprise:
 *   - a frame ({sea, sky, eye}, with the intuitive meanings:)
 *   - a viewpoint
 *   - a target (at the center of the field of view)
 *   - the distance from the viewpoint to the target
 *   - a "filter" color, Neutral by default, for color separation, etc.
 *   - a Lens
 *
 * A Lens is a mapping P -> pair. In the implementation, the Camera
 * ("body") passes the orientation, target, and distance to the Lens
 * operator.
 */

#ifndef EPIX_CAMERA
#define EPIX_CAMERA


namespace ePiX {

  class halfspace;
  class Lens;
  class pair;

  class Camera {
  public:
    Camera();
    Camera(const P& vpt);

    Camera(const Camera&);
    Camera& operator= (const Camera&);
    ~Camera();

    // rotate
    Camera& tilt(double angle); // up/down (pitch)
    Camera& pan(double angle);  // left/right (yaw)
    Camera& roll(double angle); // about viewing axis

    // fix target, move viewpt radially along eye()
    Camera& range(double);
    // fix viewpt, move target radially along eye()
    Camera& focus(double);

    // set clip distance
    Camera& clip_range(double);

    Camera& at(const P& arg);      // fix target, set viewpt
    Camera& look_at(const P& arg); // fix viewpt, set target

    // alternative syntax
    Camera& at(double, double, double);
    Camera& look_at(double, double, double);

    Camera& filter(const Color&);

    // set lens
    Camera& perspective(); // default
    Camera& orthog(); // faster
    Camera& fisheye();
    Camera& bubble();

    // Functions for internal use
    // Camera maps
    Color operator() (const Color&) const; // filter
    pair  operator() (const P&) const; // lens map
    bool  is_linear() const; // lens preserves lines?

    bool needs_clip() const; // lens needs scene pre-clipping
    halfspace clip_plane() const;

    // situation
    P eye() const;
    P viewpt() const;

  private:
    P the_viewpt; // center of projection
    P the_target; // mapped to Screen origin
    frame the_orient; // our {sea, sky, eye}
    double the_distance;
    double the_clip_range;

    Color the_filter;
    void adjust(); // re-orient according to target, viewpt
    Lens* the_lens;
  }; // end of class Camera

  // global Camera
  Camera& cam();
  extern Camera& camera;
} // end of namespace

#endif /* EPIX_CAMERA */
/* 
 * screen.h -- ePiX::screen class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.18
 * Last Change: September 15, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * This file defines the screen class interface, a Cartesian rectangle
 * into which objects are drawn. Object-creating commands add to the
 * the "global" screen the_picture(). Users may create and activate
 * additional screens, then import their content to the_picture(). 
 *
 * Constructor:
 *   screen(P(a,c), P(b,d));  // corners of bounding box
 *
 * Markup functions:
 *   crop();         // remove elements outside bounding box
 *   crop_to(mask);  // or specified mask
 *   paste(screen);  // overlay contents of screen
 *   paste(screen, P(a,c), P(b,d));  // or inset into specified rectangle
 *
 * Decorations:
 *   border(Color, "1pt"); // set border color/width
 *   backing(Color);       // and backing color
 *
 * This file wraps the implementation, which is hidden from the user.
 */
#ifndef EPIX_SCREEN
#define EPIX_SCREEN



namespace ePiX {

  class Color;
  class affine;
  class screen_data;
  class P;

  // global function
  void write(const std::string&);

  class screen {
    // There's no user-visible form for screen contents, so
    // we give these classes access to our implementation.
    friend class arrow_data;
    friend class label_data;
    friend class legend;
    friend class path_data;
    friend class picture_data;
    friend void write(const std::string&);

  public:
    screen();
    screen(const P&, const P&);

    screen(const screen&);
    screen& operator= (const screen&);
    ~screen();

    // Set mask; one named function for each type of screen_mask
    screen& crop_mask_rectangle(const P&, const P&);
    screen& crop_mask_ellipse(const P&, const P&);
    screen& crop_mask_diamond(const P&, const P&);

    screen& crop_mask_rectangle();
    screen& crop_mask_ellipse();
    screen& crop_mask_diamond();

    // return selected region
    screen extract(const P&, const P&) const;
    screen extract_diamond(const P&, const P&) const;
    screen extract_ellipse(const P&, const P&) const;

    // set flag only
    screen& set_crop(bool arg=true);

    // crop contents to current mask
    screen& crop();

    // inset and import
    screen& paste(const screen& child, const P& inset_sw, const P& inset_ne);
    screen& paste(const screen& child);

    screen& clear();

    // border(Color, length)?
    screen& border(const Color&, double);
    screen& border(const Color&, const std::string&);
    screen& backing(const Color&);

    screen& border();
    screen& backing();

    //// pass-through functions ////
    // affine operations on contents (not border, backing)
    screen&  apply(const affine& f);

    // convenience operators for single affine ops
    screen&   shift(const P& arg);

    // rotate by theta about ctr
    screen&  rotate(double theta, const P& ctr = P(0,0));

    // reflect in angle-theta line through ctr
    screen& reflect(double theta, const P& ctr = P(0,0));

    // scale coord direction(s) fixing ctr
    screen& h_scale(double, const P& ctr=P(0,0));
    screen& v_scale(double, const P& ctr=P(0,0));
    screen&   scale(double, const P& ctr=P(0,0));

    // shear, fixing ctr, e.g. h_shear(sc) : (x,y) -> (x+sc*y, y)
    screen&  h_shear(double sc, const P& ctr=P(0,0));
    screen&  v_shear(double sc, const P& ctr=P(0,0));

    P tr() const;
    P tl() const;
    P bl() const;
    P br() const;

    P  t() const;
    P  b() const;
    P  l() const;
    P  r() const;
    P  c() const;

    double h_min()  const;
    double h_max()  const;
    double h_size() const;
    double h_avg()  const;

    double v_min()  const;
    double v_max()  const;
    double v_size() const;
    double v_avg()  const;

  private:
    screen_data* m_screen;

  }; // end of class screen

} // end of namespace

#endif /* EPIX_SCREEN */
/* 
 * picture.h -- ePiX globals functions for picture interface
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.16
 * Last Change: September 11, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_PICTURE
#define EPIX_PICTURE


namespace ePiX {

  class Color;
  class P;
  class picture_data;
  class screen;

  // "global" references
  picture_data& the_picture();

  double& xmin();
  double& xmax();
  double& xsize();

  double& ymin();
  double& ymax();
  double& ysize();

  double& tix();

  // preferred initialization functions
  // corners and dimensions, e.g. "4in x 15cm" or "4 x 6in"
  // (width units take precedence, dimension string must be passed by value)
  void picture(const P& arg1, const P& arg2, std::string);

  // same syntax as for picture dimensions
  void offset(std::string);

  //// Old global functions for manipulating the page markup region ////
  void bounding_box(const P&, const P&);

  void picture(double, double);
  void picture(const P&);

  void offset(double hoff, double voff);
  void offset(const P& arg);

  void unitlength(const std::string& units);

  // activate default screen
  void begin();
  // write the output file
  void end_picture();

  //// New global functions ////
  // the_picture().the_canvas
  screen& canvas();

  // specify active screen; pass non-const ref only to assign/compare pointer
  void activate(screen&);
  void deactivate(screen&);

  // add active or specified screen to the_picture
  void import();
  void inset(const P& sw, const P& ne);

  void import(const screen& child);
  void inset(const screen& child, const P& sw, const P& ne);

  void inset(const screen& child); // use child's corners

  // interpret child's coordinates as true pt on the page,
  // place origin at P
  void inlay(const screen& child, const P& loc);

  // interpret stated coord as true pt, the other as Cartesian
  void inlay_vertical(const screen& child, const P& loc);
  void inlay_horizontal(const screen& child, const P& loc);

  // set decorations on active screen
  void backing(const Color&);
  void border(const Color&, double); // width in pt
  void border(const Color&, const std::string&);

  void backing(); // use current fill style
  void border();  // use current line pen

  // write verbatim string to output file
  void write(const std::string&);

  void pre_write(const std::string&); // before picture header
  void post_write(const std::string&); // after picture footer

  // set output format
  void eepic_format();
  //  void mp_format();
  void tikz_format();
  void pst_format();

  // write in specified format to named output file
  void print_eepic(const std::string& filename);
  // void print_mp(const std::string& filename);
  void print_tikz(const std::string& filename);
  void print_pst(const std::string& filename);

  // allow user to override in-file format request on the command line
  inline void end()
    {
#ifdef EPIX_FMT_PSTRICKS
      pst_format();
#else
#ifdef EPIX_FMT_TIKZ
      tikz_format();
#else
#ifdef EPIX_FMT_EEPIC
      eepic_format();
#endif /* EPIX_FMT_EEPIC */
#endif /* EPIX_FMT_TIKZ */
#endif /* EPIX_FMT_PSTRICKS */
      end_picture();
    }

  // for implementors of true-size elements
  double pic_units_per_pt();   // exact
  double units_per_pic_unit(); // approximate, exact at true aspect ratio
  double pt_to_screen(double); // true length to screen length

  ////////////////////////////////////////////////////////////////
  ////                                                        ////
  ////    DEPRECATED global variables required for backward   ////
  ////    compatibility. Will be removed in a future release. ////
  ////    Use reference functions xmin() et. al.              ////
  ////                                                        ////
  ////////////////////////////////////////////////////////////////

  // bounding_box corners and dimensions
  extern double x_min, x_max, x_size, y_min, y_max, y_size;

} // end of namespace

#endif /* EPIX_PICTURE */
/* 
 * markers.h -- ePiX markers and labels
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.16
 * Last Change: September 09, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * label -- put string constant <label_text> at Cartesian position <base>
 * translated by (offset.x1, offset.x2) true points (i.e., 3rd component
 * of <offset> is discarded).
 *
 * Accepts an optional LaTeX-style positioning argument.
 * If no offset is specified, the label is centered at the given Cartesian
 * location.
 * masklabel requires the "color" package, and places the text in 
 * a white box that masks whatever is underneath and earlier in the file.
 */
#ifndef EPIX_MARKERS
#define EPIX_MARKERS



namespace ePiX {

  class P;

  // position, [offset], label text, [alignment]
  void label(const P&, const P&, const std::string&);
  void label(const P&, const std::string&);
  void label(const P&, const P&, const std::string&, epix_label_posn);

  void masklabel(const P&, const P&, const std::string&);
  void masklabel(const P&, const std::string&);
  void masklabel(const P&, const P&, const std::string&, epix_label_posn);


  // Empty and filled LaTeX circles of diameter get_dotsize() true pt
  void marker (const P&, epix_mark_type);

  void circ(const P& posn, const P& offset=P(0,0),
	    const std::string& label_text="",
	    epix_label_posn align=none); // filled white circ

  void ring(const P& posn, const P& offset=P(0,0),
	    const std::string& label_text="",
	    epix_label_posn align=none); // unfilled circ 

  void spot(const P& posn, const P& offset=P(0,0),
	    const std::string& label_text="",
	    epix_label_posn align=none);

  void dot(const P& posn, const P& offset=P(0,0),
	   const std::string& label_text="",
	   epix_label_posn align=none);

  void ddot(const P& posn, const P& offset=P(0,0),
	    const std::string& label_text="",
	    epix_label_posn align=none);

  void box(const P& posn, const P& offset=P(0,0),
	   const std::string& label_text="",
	   epix_label_posn align=none);

  void bbox(const P& posn, const P& offset=P(0,0),
	    const std::string& label_text="",
	    epix_label_posn align=none);

  void h_axis_tick(const P& location, epix_label_posn align=c);
  void v_axis_tick(const P& location, epix_label_posn align=c);

  // Arrow with aligned label at tail
  void arrow(const P& tail, const P& head, const P& offset,
	     const std::string& label_text,
	     epix_label_posn align, double scale=1);

  //// Path-like glyphs ////
  // angle indicators; default args are true sizes in pt
  void right_angle(const P& loc, P leg1, P leg2, double scale=8);
  void arc_measure(const P& loc, P leg1, P leg2,
		   const P& offset, const std::string&, epix_label_posn,
		   double scale=8);
  void arc_measure(const P& loc, P leg1, P leg2, double scale=8);

  void axis_break(const P& tail, const P& head, double scale=12);
  void h_error_bar(const P& loc, double err, epix_mark_type mk, double ht=6);
  void v_error_bar(const P& loc, double err, epix_mark_type mk, double wd=6);
} // end of namespace

#endif /* EPIX_MARKERS */
/* 
 * axis.h -- Decorable coordinate axis class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.13
 * Last Change: August 23, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_AXIS
#define EPIX_AXIS



namespace ePiX {

  class axis {
  public:
    axis(const P&, const P&, unsigned int n,
	 const P& offset = P(0,0), epix_label_posn POSN=none);

    // attribute-setting
    axis& dec();  // default
    axis& frac(); // fractional labels
    axis& trig(); // fractions times pi
    axis& sci();  // scientific notation

    // f(x, precision, base) returns a string representing x
    axis& label_rep(std::string f(double, unsigned int, unsigned int));

    // unmark selected location (non-log labels only)
    axis& unmark(double arg=0);

    // add or remove logarithmic tags ("sublabels")
    axis& log(unsigned int base=10); // log ticks and labels
    axis& tag(double);               // put label at arg in log mode
    axis& tag235();                  // tags at 2, 3, 5 in log10 mode
    axis& tags();                    // tags at 2, ..., b-1 in logb mode

    axis& untag();                   // remove all log tags
    axis& untag(double);             // remove selected tag

    axis& align_labels(epix_label_posn);
    axis& align(epix_label_posn); // align tick marks

    // number of minor ticks per segment
    axis& subdivide(unsigned int);

    // set length of minor ticks
    axis& tick_ratio(double);

    // set precision for axis labels
    axis& precision(unsigned int digits=0); // 0: reset to default

    void draw_ticks() const;
    void draw_labels() const;
    void draw() const;

  private:
    P m_tail;
    P m_head;

    unsigned int m_major_segs;
    unsigned int m_minor_segs;

    double m_tick_ratio;

    epix_label_posn m_align_ticks;

    P m_offset;
    epix_label_posn m_align_labels;

    unsigned int m_log_base;
    bool m_log;
    unsigned int m_precision;

    double (*m_coord)(const P&);  // value to place at arg

    // represent arg as string
    std::string (*m_rep)(double arg, unsigned int prec, unsigned int base);

    std::set<double> m_omitted;
    std::set<double> m_log_tags;
  }; // end of axis class

  // axes along sides of bounding box, ticks automatically aligned
  axis top_axis(unsigned int n, const P& offset=P(0,0),
		epix_label_posn POSN=t);
  axis bottom_axis(unsigned int n, const P& offset=P(0,0),
		   epix_label_posn POSN=b);
  axis left_axis(unsigned int n, const P& offset=P(0,0),
		 epix_label_posn POSN=l);
  axis right_axis(unsigned int n, const P& offset=P(0,0),
		  epix_label_posn POSN=r);


  // Global functions from Version 1.0
  void h_axis_labels(const P& tail, const P& head, unsigned int n, 
		     const P& offset, epix_label_posn POSN=none);

  void v_axis_labels(const P& tail, const P& head, unsigned int n, 
		     const P& offset, epix_label_posn POSN=none);

  void h_axis_masklabels(const P& tail, const P& head, unsigned int n, 
			 const P& offset, epix_label_posn POSN=none);

  void v_axis_masklabels(const P& tail, const P& head, unsigned int n, 
			 const P& offset, epix_label_posn POSN=none);

  // Axis labels with default endpoints
  void h_axis_labels(unsigned int n, const P& offset, 
		     epix_label_posn POSN=none);
  void h_axis_masklabels(unsigned int n, const P& offset,
			 epix_label_posn POSN=none);

  void v_axis_labels(unsigned int n, const P& offset,
		     epix_label_posn POSN=none);
  void v_axis_masklabels(unsigned int n, const P& offset,
			 epix_label_posn POSN=none);

  //// logarithmic labels ////
  // labels written $k x base^i$ at i + log_b(k)
  void h_axis_log_labels(const P& tail, const P& head, unsigned int n,
			 const P& offset, epix_label_posn POSN=none,
			 unsigned int base=10);

  void v_axis_log_labels(const P& tail, const P& head, unsigned int n,
			 const P& offset, epix_label_posn POSN=none,
			 unsigned int base=10);

  // labels written $base^i$ at coord i
  void h_axis_log_labels(const P& tail, const P& head,
			 const P& offset, epix_label_posn POSN=none,
			 unsigned int base=10);

  void v_axis_log_labels(const P& tail, const P& head,
			 const P& offset, epix_label_posn POSN=none,
			 unsigned int base=10);


  void h_axis_log_masklabels(const P& tail, const P& head, unsigned int n,
			     const P& offset, epix_label_posn POSN=none,
			     unsigned int base=10);

  void v_axis_log_masklabels(const P& tail, const P& head, unsigned int n,
			     const P& offset, epix_label_posn POSN=none,
			     unsigned int base=10);

  void h_axis_log_masklabels(const P& tail, const P& head,
			     const P& offset, epix_label_posn POSN=none,
			     unsigned int base=10);

  void v_axis_log_masklabels(const P& tail, const P& head,
			     const P& offset, epix_label_posn POSN=none,
			     unsigned int base=10);


  // Coordinate axes, specified by initial and final points, number of
  // tick marks. h/v_axis are identical except for style of tick marks.

  // n subintervals
  void h_axis(const P& tail, const P& head, unsigned int n,
	      epix_label_posn align=c);
  void v_axis(const P& tail, const P& head, unsigned int n,
	      epix_label_posn align=c);

  // Default endpoints (xmin(), 0), xmax(), 0), etc.
  void h_axis(unsigned int n = xsize(), epix_label_posn align=c);
  void v_axis(unsigned int n = ysize(), epix_label_posn align=c);

  void h_log_axis(const P& tail, const P& head, unsigned int segs,
		  epix_label_posn align=c, unsigned int base=10);

  void v_log_axis(const P& tail, const P& head, unsigned int segs,
		  epix_label_posn align=c, unsigned int base=10);

} // end of namespace

#endif /* EPIX_AXIS */
/* 
 * legend.h -- Plot/figure legend class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.11
 * Last Change: August 16, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_LEGEND
#define EPIX_LEGEND




namespace ePiX {

  class legend_item;
  class P;

  class legend {
  public:
    legend();

    legend(const legend&);

    legend& operator=(const legend&);

    ~legend();

    // add items; get key style from drawing state
    legend& fill_item(const std::string&); // solid color
    legend& path_item(const std::string&); // line
    legend& mark_item(epix_mark_type, const std::string&); // marker

    // decorations
    legend& backing(const Color&);
    legend& border(const Color&, double);
    legend& border(double); // line width in pt

    // item attributes
    legend& item_border(const Color&, double);
    legend& item_border(double);

    legend& label_skip(double); // 
    legend& key_size(double);   // size of legend keys in pt

    void draw(const P& loc, const P& offset=P(0,0),
	      epix_label_posn align=tr) const;

  private:
    Color m_backing;
    Color m_bord;
    double m_bord_width;

    Color m_item_bord;
    double m_item_bord_width;

    double m_key_size;
    double m_label_skip;

    std::list<legend_item*> m_items;
  }; // end of legend class

} // end of namespace

#endif /* EPIX_LEGEND */
/* 
 * path.h -- ePiX user class for polygons and paths
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.2.5
 * Last Change: May 04, 2008
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_PATHS
#define EPIX_PATHS


namespace ePiX {

  class Camera;
  class Color;
  class P;
  class path_data;
  class pen_data;
  class screen;
  class Sphere;

  class path {
    friend class facet;
  public:

    // path was visible to users, but is now implemented as a p_impl.
    path(const path&);
    path& operator= (const path&);
    ~path();

    // Interface dictated by backward compatibility
    path(const std::vector<P>& data, bool closed, bool filled);
    path();

    // line constructors
    path(const P& tail, const P& head, double expand=0);
    path(const P&, const P&, double expand, unsigned int num_pts);


    // Constructors that specify the number of points have two prototypes
    // rather than accepting a default argument. This hides the global
    // (but user-inaccessible) default number of points (see constants.h).

    // ellipse
    path(const P& center, const P& axis1, const P& axis2, 
	 double t_min, double t_max,
	 unsigned int num_pts);

    path(const P& center, const P& axis1, const P& axis2, 
	 double t_min, double t_max);


    // spline
    path(const P& p1, const P& p2, const P& p3, unsigned int n);
    path(const P& p1, const P& p2, const P& p3);

    path(const P& p1, const P& p2, const P& p3, const P& p4, unsigned int n);
    path(const P& p1, const P& p2, const P& p3, const P& p4);


    // parametric path
    path(P f(double), double t_min, double t_max, unsigned int num_pts);
    path(P f(double), double t_min, double t_max);

    // function graph
    path(double f(double), double t_min, double t_max, unsigned int num_pts);
    path(double f(double), double t_min, double t_max);

    // append a point
    path& pt(double, double, double=0);
    path& pt(const P&);

    // concatenate path segments
    path& operator+= (const path&);
    // concatenate, reversing second sequence
    path& operator-= (const path&);

    bool is_closed() const;
    bool is_filled() const;

    // set attributes
    path& close();
    path&  fill(const bool arg=true);


    void clip();

    // assumes path lies on sphere
    void clip_to(const Sphere&, const P& viewpt, bool back=false);

    std::vector<P> data() const;

    void draw() const;
    void draw(const Color&, const pen_data&) const;

  private:
    path_data*   m_segments;

  }; // end of class path

  /*
    Slated for removal
    // polygon/polyline with variable number of vertices
    path polygon(unsigned int num_pts ...);
    path polyline(unsigned int num_pts ...);
  */

} // end of namespace

#endif /* EPIX_PATHS */
/*
 * curves.h -- Ellipses, arcs, splines, coordinate grids
 *
 * This file is part of ePiX, a C++ library for creating high-quality
 * figures in LaTeX
 *
 * Version 1.1.10
 * Last Change: August 08, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_CURVES
#define EPIX_CURVES



namespace ePiX {

  class P;
  class mesh;

  // lines can be "stretched" by double parameter
  void line(const P& tail, const P& head, double expand=0); 
  void line(const P& tail, const P& head, double expand,
	    unsigned int num_pts);

  // "Visible" portion of the line through p1, p2
  void Line(const P& tail, const P& head);

  // point-slope form
  void Line(const P&, double);

  void triangle(const P&, const P&, const P&);
  void quad(const P&, const P&, const P&, const P&);

  void rect(const P&, const P&);
  void rect(const P&, const P&, bool solid);

  // arrows
  void arrow(const P& tail, const P& head, double scale=1);
  void dart(const P& tail, const P& head);

  // double-tipped
  void aarrow(const P& tail, const P& head, double scale=1);

  // arbitrary elliptical arc
  void arrow(const P& center, const P& axis1, const P& axis2,
	     double t_min, double t_max, double scale=1);

  // Algebraic curves (elliptical and circular arcs, splines)
  void ellipse(const P& center, const P& axis1, const P& axis2); // full turn

  void ellipse(const P& center, const P& axis1, const P& axis2,  
	       double t_min, double t_max); // angle range

  void ellipse(const P& center, const P& axis1, const P& axis2,  
	       double t_min, double t_max, unsigned int num_pts);

  // for backward compatibility
  void ellipse_arc(const P& center, const P& axis1, const P& axis2,
		   double t_min, double t_max);

  // old style "center and polyradius"
  void ellipse (const P& center, const P& radius);

  // Standard half-ellipse functions
  void ellipse_left (const P&, const P&);
  void ellipse_right (const P&, const P&);
  void ellipse_top (const P&, const P&);
  void ellipse_bottom (const P&, const P&);

  // Circular arcs parallel to (x,y)-plane
  void arc(const P& center, double r, 
	   double start,  double finish);

  void arc_arrow (const P& center, double r, 
		  double start, double finish, 
		  double scale=1);


  // Quadratic and cubic splines/spline arrows
  void spline(const P& p1, const P& p2, const P& p3, unsigned int num_pts);
  void spline(const P& p1, const P& p2, const P& p3);

  void arrow(const P& p1, const P& p2, const P& p3, double scale=1);

  void spline (const P& p1, const P& p2, const P& p3, const P& p4,
	       unsigned int num_pts);
  void spline (const P& p1, const P& p2, const P& p3, const P& p4);

  void arrow(const P&, const P&, const P&, const P&, double scale=1);

  // natural spline
  void spline(const std::vector<P>&, unsigned int num_pts);


  // A "mesh" is an ordered pair of positive integers, and is used to
  // specify the "fineness" of a grid. Grids, like parametric surface
  // meshes, have a "coarse mesh" -- the numbers of grid intervals in
  // each direction, and a "fine mesh" -- the numbers of points used
  // to render the grid lines. Since an ePiX camera does not always
  // map lines in object space to lines on the screen, grid lines cannot
  // generally be drawn using only two points.
  // A grid may look strange unless each component of fine is a multiple
  // of the corresponding entry of coarse, of course. :)

  // Cartesian grid of specified size, mesh, and resolution
  void grid(const P& arg1, const P& arg2, mesh coarse, mesh fine);

  // coarse = fine = (n1,n2)
  void grid(const P& arg1, const P& arg2,
	    unsigned int n1=1, unsigned int n2=1);
  void grid(unsigned int n1=1, unsigned int n2=1);

  // polar grid of specified radius, mesh (rings and sectors), and resolution
  void polar_grid(double r, mesh coarse, mesh fine);

  // polar grid with n1 rings and n2 sectors
  void polar_grid(double r, unsigned int n1, unsigned int n2);


  // (semi-)logarithmic grids specified by corners and numbers of orders of
  // magnitude or grid divisions in each direction. Optional arguments
  // specify the log base (10 by default). If corners are omitted, the grid
  // fills the bounding box.
  void log_grid(const P& arg1, const P& arg2,
		unsigned int segs1, unsigned int segs2,
		unsigned int base1=10, unsigned int base2=10);

  void log1_grid(const P& arg1, const P& arg2,
		 unsigned int segs1, unsigned int segs2,
		 unsigned int base1=10);

  void log2_grid(const P& arg1, const P& arg2,
		 unsigned int segs1, unsigned int segs2,
		 unsigned int base2=10);

  void log_grid(unsigned int segs1, unsigned int segs2,
		unsigned int base1=10, unsigned int base2=10);

  void log1_grid(unsigned int segs1, unsigned int segs2,
		 unsigned int base1=10);

  void log2_grid(unsigned int segs1, unsigned int segs2,
		 unsigned int base2=10);

  // fractal generation
  //
  // The basic "level-1" recursion unit is a piecewise-linear path whose 
  // segments are parallel to spokes on a wheel, labelled modulo <spokes>.
  // Recursively up to <depth>, each segment is replaced by a copy of the
  // recursion unit.
  //
  // Sample data for _/\_ standard Koch snowflake:
  // const int pre_seed[] = {6, 4, 0, 1, -1, 0};
  // pre_seed[0] = spokes, pre_seed[1] = seed_length;

  void fractal (const P& p, const P& q, const int depth, const int *pre_seed);

} // end of namespace

#endif /* EPIX_CURVES */
/* 
 * circle.h -- ePiX::Circle class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.9
 * Last Change: July 30, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 *   This file provides:
 *
 *   The Circle class (center, radius, unit normal).
 *
 *    - Circle() (defaults to origin, unit radius, in (x1,x2,0)-plane
 *    - Circle(ctr, pt) (parallel to (x1,x2,0)-plane, center ctr, through pt)
 *    - Circle(pt1, pt2, pt3) (arbitrary non-collinear points)
 *    - center(), radius(), perp()
 *    - affine operations: shift(P), move_to(P), scale(double)
 *    - draw() (ePiX ellipse)
 */
#ifndef EPIX_CIRCLE
#define EPIX_CIRCLE


namespace ePiX {

  class Circle {
  public:
    // defaults to unit circle in (x1,x2,0) plane
    Circle(const P& ctr=P(0,0,0), double rad=1, const P& perp=E_3);

    Circle(const P& ctr, const P& pt); // center, and point, normal = E_3
    Circle(const P& pt1, const P& pt2, const P& pt3); // three points

    Circle(bool); // malformed Circle for intersection operators

    P center() const;
    double radius() const;
    P perp() const;

    bool malformed() const;

    // translate
    Circle& shift(const P&);
    Circle& move_to(const P&);
    // scale radius by c
    Circle& scale(double c);

    void draw() const;

  private:
    P m_center;
    double m_radius;
    P m_perp; // unit normal, even if rad = 0

    bool m_malformed;
  }; // end of Circle class

  // global object-drawing commands
  void circle(const P& ctr=P(0,0,0), double rad=1, const P& perp=E_3);

  void circle(const P& ctr, const P& pt); // center, and point, normal = E_3
  void circle(const P& pt1, const P& pt2, const P& pt3); // three points

} // end of namespace

#endif /* EPIX_CIRCLE */
/* 
 *  plane.h -- ePiX::Plane class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.17
 * Last Change: September 13, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_PLANE
#define EPIX_PLANE


namespace ePiX {

  class Circle;
  class Plane;
  class Segment;
  class Sphere;

  class Plane {
  public:
    Plane(const P& pt=P(0,0,0), const P& perp=E_3);
    Plane(const P&, const P&, const P&);

    P pt() const;
    P perp() const;
    bool malformed() const;

    // affine operations
    Plane& shift(const P&);
    Plane& move_to(const P&);

    void draw() const;

  private:
    P m_pt;
    P m_perp; // unit normal

    bool m_malformed;
  }; // end of Plane class

  void plane(const P& pt=P(0,0,0), const P& perp=E_3);
  void plane(const P&, const P&, const P&);

} // end of namespace

#endif /* EPIX_PLANE */
/* 
 * segment.h -- ePiX::Segment class
 * 
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.2.5
 * Last Change: May 04, 2008
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 *   This file provides:
 *
 *   The Segment class (unordered pair of points), and operators:
 *    - end1(), end2() (no guaranteed order)
 *    - Translation operators shift(P), move_to(P)
 *    - Stretch about midpoint: scale(double)
 *    - draw(double) (ePiX line)
 *    - midpoint();
 */

#ifndef EPIX_SEGMENT
#define EPIX_SEGMENT


namespace ePiX {

  class Segment {
  public:
    Segment(const P&, const P&); // endpoints

    Segment(bool); // malformed Segment for intersection operators

    // Segment ends are not distinguished geometrically,
    // so these functions should normally be used in tandem.
    P end1() const;
    P end2() const;

    bool malformed() const;

    // translate
    Segment& shift(const P&);
    Segment& move_to(const P&);

    Segment& scale(double);

    P midpoint(double t=0.5) const;

    void draw(double stretch=0) const;

  private:
    P m_endpt1;
    P m_endpt2;

    bool m_malformed;
  }; // end of Segment class

} // end of namespace
#endif /* EPIX_SEGMENT */
/* 
 * sphere.h -- ePiX::Sphere class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.9
 * Last Change: July 30, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 *   This file provides:
 *
 *   The Sphere class (center, radius), and operators:
 *    - Sphere += P (translate by <P>)
 *    - Sphere *= double (scale about center)
 *    - draw() (ePiX line)
 *    - center();
 *    - radius();
 *
 *   Hiding test, latitude and longitudes on a sphere.
 */
#ifndef EPIX_SPHERE
#define EPIX_SPHERE


namespace ePiX {

  class Sphere {
  public:
    Sphere(const P& ctr=P(0,0,0), double rad=1);
    Sphere(const P& ctr, const P& pt);

    P center() const;
    double radius() const;

    bool malformed() const; // always returns false, currently

    // translation
    Sphere& shift(const P&);
    Sphere& move_to(const P&);

    // scale radius
    Sphere& scale(double&);

    void draw() const;

  private:
    P m_ctr;
    double m_rad;

    bool m_malformed;
  }; // end of Sphere class


  // global horizon-drawing functions
  void sphere(const P& ctr=P(0,0,0), double rad=1);
  void sphere(const P& ctr, const P& pt);

} // end of namespace

#endif /* EPIX_SPHERE */
/* 
 * intersections.h -- Circle, Plane, Segment, Sphere intersection operators 
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.9
 * Last Change: July 30, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_INTERSECTIONS
#define EPIX_INTERSECTIONS

namespace ePiX {

  class Circle;
  class Plane;
  class Segment;
  class Sphere;

  Segment operator* (const Circle&,  const Circle&);
  Segment operator* (const Circle&,  const Plane&);
  Segment operator* (const Circle&,  const Segment&);
  Segment operator* (const Circle&,  const Sphere&);

  Segment operator* (const Plane&,   const Plane&);
  P       operator* (const Plane&,   const Segment&);
  Circle  operator* (const Plane&,   const Sphere&);

  P       operator* (const Segment&, const Segment&);
  Segment operator* (const Segment&, const Sphere&);

  Circle  operator* (const Sphere&,  const Sphere&);

  // derived operators with reversed argument order
  Segment operator* (const Plane&,   const Circle&);
  Segment operator* (const Segment&, const Circle&);
  Segment operator* (const Sphere&,  const Circle&);

  P       operator* (const Segment&, const Plane&);
  Circle  operator* (const Sphere&,  const Plane&);

  Segment operator* (const Sphere&,  const Segment&);

} // end of namespace

#endif /* EPIX_INTERSECTIONS */
/*
 * plots.h -- parametrized plotting algorithms
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Andrew D. Hwang   <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 *
 * Version 1.0.23
 * Last Change: January 13, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_PLOTS
#define EPIX_PLOTS


namespace ePiX {

  class domain;
  class domain_list;
  class Deriv;
  class Integral;
  class P;

  //// paths ////
  // f:R -> R^3
  void plot (P f(double), double t_min, double t_max, unsigned int n);

  // f:R -> R x R x R
  void plot (double f1(double), double f2(double), double f3(double),
	     double t_min, double t_max, unsigned int n);

  // f:R -> R
  void plot (double f(double), double t_min, double t_max, unsigned int n);
  void plot (const Deriv&, double t_min, double t_max, unsigned int n);
  void plot (const Integral&, double t_min, double t_max, unsigned int n);

  // f: R -> R x R
  void plot (double f1(double), double f2(double), 
	     double t_min, double t_max, unsigned int n);

  void polarplot (double f(double), double t_min, double t_max, unsigned int);

  //// surfaces ////
  // f:R^2 -> R x R x R
  void plot(double f1(double u1, double u2),
	    double f2(double u1, double u2),
	    double f3(double u1, double u2), 
	    const P& min, const P& max, const mesh& coarse, const mesh& fine);

  void plot(double f1(double u1, double u2),
	    double f2(double u1, double u2),
	    double f3(double u1, double u2), 
	    const domain& R);

  // f:R^2 -> R
  void plot(double f(double u1, double u2),
	    const P& p1, const P& p2, const mesh& coarse, const mesh& fine);

  // f:R^2 -> R^3
  void plot(P f(double, double), const domain&);
  void plot(P f(double, double), const domain_list&);

  // f:R^3 -> R^3
  void plot(P f(double, double, double), const domain&);
  void plot(P f(double, double, double), const domain_list&);


  // Derivatives and integrals
  void plot_deriv(double f(double), double a, double b, unsigned int n);

  void plot_int(double f(double), double a, double b, unsigned int n);

  void plot_int(double f(double), double x0, double a, double b, unsigned int);

  void tan_line(double f1(double), double f2(double), double t0);
  void tan_line(double f(double), double t0);
  void tan_line(P f(double), double t0);

  void envelope(double f1(double), double f2(double),
		double, double, unsigned int);
  void envelope(double f(double), double, double, unsigned int);
  void envelope(P f(double), double, double, unsigned int);

  void tan_field(double f1(double), double f2(double), double, double,
		 unsigned int);
  void tan_field(P f(double), double, double, unsigned int);


  // Slope, dart, and vector fields
  // For slope and dart fields, the optional scale argument affects the
  // (constant) drawn length of field elements. For vector fields, elements
  // are drawn at true length, and "scale" affects the arrowhead size.

  // planar fields
  void  slope_field(P F(double, double), const P& p, const P& q,
		    unsigned int, unsigned int, double scale=1.0);
  void   dart_field(P F(double, double), const P& p, const P& q,
		    unsigned int, unsigned int, double scale=1.0);
  void vector_field(P F(double, double), const P& p, const P& q,
		    unsigned int, unsigned int, double scale=1.0);

  void  slope_field(P F(double, double), const domain& R,
		    double scale=1.0);
  void   dart_field(P F(double, double), const domain& R,
		    double scale=1.0);
  void vector_field(P F(double, double), const domain& R,
		    double scale=1.0);

  // spatial fields
  void  slope_field(P F(double, double, double), const P&, const P&,
		    unsigned int, unsigned int, double scale=1.0);
  void   dart_field(P F(double, double, double), const P&, const P&,
		    unsigned int, unsigned int, double scale=1.0);
  void vector_field(P F(double, double, double), const P&, const P&,
		    unsigned int, unsigned int, double scale=1.0);

  // spatial fields over a (3-D) domain
  void  slope_field(P F(double, double, double), const domain&,
		    double scale=1.0);
  void   dart_field(P F(double, double, double), const domain&,
		    double scale=1.0);
  void vector_field(P F(double, double, double), const domain&,
		    double scale=1.0);


  // Solutions of ODE systems

  // start at time 0
  void ode_plot (P F(double, double),
		 const P& start, double t_max, unsigned int);

  void ode_plot (P F(double, double, double),
		 const P& start, double t_max, unsigned int);

  // arbitrary start time
  void ode_plot (P F(double, double),
		 const P& start, double t_min, double t_max, unsigned int);

  void ode_plot (P F(double, double, double),
		 const P& start, double t_min, double t_max, unsigned int);

  // flow x0 under field for specified time
  P flow (P F(double, double), const P& start, double t_max, unsigned int n=0);
  P flow (P F(double, double, double),
	  const P& start, double t_max, unsigned int n = 0);

  void riemann_sum(double f(double), double a, double b,
		   unsigned int n, epix_integral_type TYPE);


  // Jay Belanger's shaded plot functions -- December 1, 2002
  void shadeplot(double f1(double), double f2(double), double, double,
		 unsigned int);
  void shadeplot(double f(double), double t_min, double t_max, unsigned int);

  void blackplot(double f1(double), double f2(double), 
		 double t_min, double t_max, unsigned int);

  void blackplot(double f(double), double t_min, double t_max, unsigned int);

  void whiteplot(double f1(double), double f2(double), 
		 double t_min, double t_max, unsigned int);

  void whiteplot(double f(double), double t_min, double t_max, unsigned int);


  // wiremesh surface plotting
  void plot(P Phi(double, double),
	    const P&, const P&, const mesh& N, const mesh& num_pts);

  /*
  void clipplot(P Phi(double, double),
	    const P&, const P&, const mesh& N, const mesh& num_pts);

  void cropplot(P Phi(double, double),
		const P&, const P&, const mesh& N, const mesh& num_pts);
  */

} // end of namespace

#endif /* EPIX_PLOTS */
/*
 * surface.h -- Shaded surface plotting
 *
 * This file is part of ePiX, a C++ library for creating high-quality
 * figures in LaTeX
 *
 * Version 1.1.7
 * Last Change: July 08, 2007
 */

/*
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */



namespace ePiX 
{
  class domain;
  class domain_list;
  class facet;

  // manipulable collection of surface-like objects
  class scenery {
  public:
    scenery();
    scenery(P F(double, double), const domain& R);
    scenery(P F(double, double, double), const domain& R);

    // third arg is RGB-valued function of position
    scenery(P F(double, double), const domain& R,
	    P color(double, double, double));

    scenery(P F(double, double, double), const domain& R,
	    P color(double, double, double));

    // third arg is RGB-valued function of domain location
    scenery(P F(double, double), const domain& R,
	    P color(double, double));

    scenery(P F(double, double, double), const domain& R,
	    P color(double, double));

    scenery(const scenery&);
    scenery& operator= (const scenery&);
    ~scenery();

    scenery& add(P F(double, double), const domain& R);
    scenery& add(P F(double, double, double), const domain& R);

    scenery& add(P F(double, double), const domain_list&);
    scenery& add(P F(double, double, double), const domain_list&);

    // color-dependent
    // location-colored
    scenery& add(P F(double, double), const domain& R,
		 P color(double, double, double));
    scenery& add(P F(double, double, double), const domain& R,
		 P color(double, double, double));

    scenery& add(P F(double, double), const domain_list&,
		 P color(double, double, double));
    scenery& add(P F(double, double, double), const domain_list&,
		 P color(double, double, double));

    // domain-colored
    scenery& add(P F(double, double), const domain& R,
		 P color(double, double));
    scenery& add(P F(double, double, double), const domain& R,
		 P color(double, double));

    scenery& add(P F(double, double), const domain_list&,
		 P color(double, double));
    scenery& add(P F(double, double, double), const domain_list&,
		 P color(double, double));

    // remove backward-pointing faces? 0=no, -1=front, 1=back
    scenery& cull(int);
    void draw(int cull=0) const;

  private:
    int m_cull;

    std::list<facet*> m_data;
  }; // end of class scenery


  //// global functions ////
  // except as noted, dim(R) must be 2
  // cosine-shaded surface with fake z-buffered hidden object removal
  void surface(P F(double, double), const domain& R, int cull=0);

  // for slices of maps R^3 -> R^3
  void surface(P F(double, double, double), const domain& R, int cull=0);

  // plot multiple slices
  void surface(P F(double, double, double), const domain_list& R, int cull=0);

  // Surface from revolving the curve (f(t),g(t)) about the x-axis...
  void surface_rev(double f(double), double g(double),
		   double t_min, double t_max, 
		   int latitudes, int longitudes=24, int cull=0);

  void surface_rev(double f(double),
		   double t_min, double t_max, 
		   int latitudes, int longitudes=24, int cull=0);

  // or about the specified axis, with specified angle range
  void surface_rev(double f(double), double g(double), 
		   const domain& R, const frame& coords=frame(), int cull=0);



  //// location-colored versions ////
  void surface(P F(double, double), const domain& R,
	       P color(double, double, double), int cull=0);

  // for slices of maps R^3 -> R^3
  void surface(P F(double, double, double), const domain& R,
	       P color(double, double, double), int cull=0);

  // plot multiple slices
  void surface(P F(double, double, double), const domain_list& R,
	       P color(double, double, double), int cull=0);

  // Surface from revolving the curve (f(t),g(t)) about the x-axis...
  void surface_rev(double f(double), double g(double),
		   double t_min, double t_max, 
		   int latitudes, int longitudes,
		   P color(double, double, double), int cull=0);

  void surface_rev(double f(double),
		   double t_min, double t_max, 
		   int latitudes, int longitudes,
		   P color(double, double, double), int cull=0);

  // or about the specified axis, with specified angle range
  void surface_rev(double f(double), double g(double), 
		   const domain& R, P color(double, double, double),
		   const frame& coords=frame(), int cull=0);


  //// domain-colored versions ////
  void surface(P F(double, double), const domain& R,
	       P color(double, double), int cull=0);

  // for slices of maps R^3 -> R^3
  void surface(P F(double, double, double), const domain& R,
	       P color(double, double), int cull=0);

  // plot multiple slices
  void surface(P F(double, double), const domain_list& R,
	       P color(double, double, double), int cull=0);

  // Surface from revolving the curve (f(t),g(t)) about the x-axis...
  void surface_rev(double f(double), double g(double),
		   double t_min, double t_max, 
		   int latitudes, int longitudes,
		   P color(double, double), int cull=0);

  void surface_rev(double f(double),
		   double t_min, double t_max, 
		   int latitudes, int longitudes,
		   P color(double, double), int cull=0);

  // or about the specified axis, with specified angle range
  void surface_rev(double f(double), double g(double), 
		   const domain& R, P color(double, double),
		   const frame& coords=frame(), int cull=0);
} // end of namespace ePiX
/* 
 * data_mask.h -- ePiX::data_mask, for selective row removal from data_file
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.6
 * Last Change: July 3, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_DATA_MASK
#define EPIX_DATA_MASK



namespace ePiX {

  class data_mask {
  public:
    data_mask(std::string);
    data_mask(std::string, double f(double));

    data_mask(const interval& range);
    data_mask(const interval& range, double f(double));

    data_mask(double range_min, double range_max);
    data_mask(double range_min, double range_max,
	      double f(double));

    data_mask& reverse(); // negate mask criterion
    bool masks(double) const; // true if we mask arg 

  private:
    interval m_range;
    double (*m_filter)(double); // filter criterion
    bool m_reverse;

  }; // end of data_mask
} // end of namespace

#endif /* EPIX_DATA_MASK */
/* 
 * data_file.h -- ePiX::data_file class
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.6
 * Last Change: July 04, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */
#ifndef EPIX_DATA_FILE
#define EPIX_DATA_FILE



namespace ePiX {

  class data_bins;
  class data_mask;
  class interval;

  class data_file {
  public:
    explicit data_file(unsigned int n=2);

    // get size from first line of file;
    // provide two versions to avoid exposing default delim/commt strings
    data_file(const char* filename,
	      const std::string& delim, const std::string& commt);
    explicit data_file(const char* filename);

    // file made from components
    data_file(double f(double),
	      double t_min, double t_max, unsigned int num_pts);

    data_file(double f1(double), double f2(double), 
	      double t_min, double t_max, unsigned int num_pts);

    data_file(double f1(double), double f2(double), double f3(double),
	      double t_min, double t_max, unsigned int num_pts);

    // input
    data_file& read(const char*);

    // transform column(s)
    data_file& transform(double f(double), unsigned int col=0);

    // apply f to selected columns; components of image go back to columns
    data_file& transform(P f(double, double),
			 unsigned int col1=1, unsigned int col2=2);

    data_file& transform(P f(double, double, double),
			 unsigned int col1, unsigned int col2);

    data_file& transform(P f(double, double, double),
			 unsigned int col1,
			 unsigned int col2,
			 unsigned int col3);

    // remove rows where specified entry is masked or outside interval/range
    data_file& prune(const data_mask&, const unsigned int col);
    data_file& prune(const interval&, const unsigned int col);
    data_file& prune(double, double, const unsigned int col);

    // basic statistical operations on columns
    double dot(unsigned int col1, unsigned int col2) const; // dot product
    double avg(unsigned int col1) const;  // mean
    double var(unsigned int col1) const;  // |x|^2 - n*avg^2
    double covar(unsigned int col1, unsigned int col2) const;
    void   regression(unsigned int col1, unsigned int col2) const;


    // output functions
    // extract column
    std::vector<double> column(unsigned int) const;
    // apply f to values
    std::vector<double> column(double f(double), unsigned int) const;

    // set C++ output precision for write
    void precision(unsigned int n=6) const;

	// Setters and getters for delimiter and comment character
	void delimiter(const char* delim) { m_delim = delim; }
	std::string delimiter() const { return m_delim; }

	void comment(const char* commt) { m_commt = commt; }
	std::string comment() const { return m_commt; }

    // write raw data to file
    void write(const char* filename) const;

    // write selected columns formatted by string-valued function
    void write(const char* filename, std::string pt_out(double, double),
	       unsigned int col1=1, unsigned int col2=2) const;

    // LaTeX tabular environment
    void tabular(const char* filename,
		 const std::string& alignment,
		 const std::string& legend="") const;


    /*
    // (un)set cropping criterion for plotting
    void select(bool Fsel(P));
    void select(const selection&); // e.g. select(all());
    */

    // scatter plots; f applied to selected columns
    void plot(epix_mark_type TYPE,
	      unsigned int col1=1, unsigned int col2=2, unsigned int col3=0,
	      P f(double, double, double) = xyz) const;

    void plot(epix_mark_type TYPE, P f(double, double, double),
	      unsigned int col1=1, unsigned int col2=2,
	      unsigned int col3=0) const;

  private:
	std::vector<double> tokenise(std::string line);
	unsigned int entries(const char* filename);

    mutable unsigned int m_precision;
    // selection m_select;

    std::vector<std::vector<double> > m_data;
    std::string m_delim;		// Field delimiter
    std::string m_commt;		// Comment character
  }; // end of class data_file


  //// Global functions that use file_data, data_bins ////
  void plot(const char* filename, epix_mark_type TYPE,
	    unsigned int col1=1, unsigned int col2=2, unsigned int col3=0,
	    P f(double, double, double) = xyz);

  void plot(const char* filename, epix_mark_type TYPE,
	    P f(double, double, double),
	    unsigned int col1=1, unsigned int col2=2, unsigned int col3=0);

  // pass 3rd arg by value
  void histogram(const char* filename, unsigned int col, data_bins,
		 double scale=1);
  void bar_chart(const char* filename, unsigned int col, data_bins,
		 double scale=1);
} // end of namespace

#endif /* EPIX_DATA_FILE */
/* 
 * data_bins.h -- ePiX::data_bins class for histogram data
 *
 * This file is part of ePiX, a C++ library for creating high-quality 
 * figures in LaTeX 
 *
 * Version 1.1.19
 * Last Change: September 17, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * This file provides the "data_bins" class.
 *
 * An object of type data_bins models a closed interval [lo, hi]
 * divided into subintervals (not necessarily of equal length). Each
 * subinterval is a "d_bin", whose endpoints are "cuts". A data_bins
 * can read and sort numerical data, count how many numbers lie in
 * each bin, and print the result as a bar chart (rectangle height
 * proportional to population), histogram (area proportional to
 * population), or as a spline interpolation.
 *
 * Ideally no data lies outside (lo, hi) or on a cut. Data on an end
 * cut is "out of range", data on an interior cut contributes 1/2 to
 * the population of each adjacent bin. A summary statistic warning is
 * printed if any points are out of range or lie on a cut.
 */

#ifndef EPIX_DATA_BINS
#define EPIX_DATA_BINS


namespace ePiX {

  class d_bin;

  class data_bins {
  public:
    data_bins(double lo, double hi, unsigned int n=1);

    data_bins(const data_bins&);
    data_bins& operator= (const data_bins&);
    ~data_bins();

    data_bins&  cut(double); // add a cut
    data_bins& read(const std::vector<double>&);

    unsigned int pop() const; // current population

    // draw rectangles
    void histogram(double scale=1) const; // area prop to pop
    void bar_chart(double scale=1) const; // height prop to pop
    void plot(double scale=1) const;  // smooth interpolation

  private:
    double m_lo_val;
    double m_hi_val;

    unsigned int m_lo_ct; // population smaller than m_lo_val
    unsigned int m_hi_ct; // population larger than m_hi_val
    unsigned int m_pop;   // current total population

    unsigned int m_cut_hits; // number of data on a cut
    bool m_cuts_locked; // true once we read data

    std::list<double> m_cuts;
    std::list<d_bin*> m_bins;

    // bookkeeping
    void initialize();         // convert cuts to bins and lock
    void insert(double); // add data point
  }; // end of class data_bins

} // end of namespace

#endif /* EPIX_DATA_BINS */
/*
 * geometry.h 
 *
 * This file is part of ePiX, a preprocessor for creating high-quality 
 * line figures in LaTeX 
 *
 * Version 1.1.6
 * Last Change: June 28, 2007
 */

/* 
 * Copyright (C) 2001, 2002, 2003, 2004, 2005, 2006, 2007
 * Andrew D. Hwang <rot 13 nujnat at zngupf dot ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_GEOMETRY
#define EPIX_GEOMETRY


namespace ePiX {
  class P;

  // latitude, longitude lines with hiding
  void latitude(double lat, double long_min, double long_max,
		const Sphere& S=Sphere(), const frame& coords=frame());

  void longitude(double lngtd, double lat_min, double lat_max,
		 const Sphere& S=Sphere(), const frame& coords=frame());

  void back_latitude(double lat, double long_min, double long_max,
		     const Sphere& S=Sphere(), const frame& coords=frame());

  void back_longitude(double lngtd, double lat_min, double lat_max,
		      const Sphere& S=Sphere(), const frame& coords=frame());


  // stereographic projection; path first projected vertically
  void frontplot_N(double f1(double), double f2(double),
		   double t_min, double t_max, int num_pts, 
		   const Sphere& S = Sphere());

  void backplot_N(double f1(double), double f2(double),
		  double t_min, double t_max, int num_pts,
		  const Sphere& S = Sphere());

  void frontplot_S(double f1(double), double f2(double),
		   double t_min, double t_max, int num_pts, 
		   const Sphere& S = Sphere());

  void backplot_S(double f1(double), double f2(double),
		  double t_min, double t_max, int num_pts, 
		  const Sphere& S = Sphere());

  // Radial projection from center
  void frontplot_R(P phi(double), double t_min, double t_max, 
		   int num_pts, const Sphere& S = Sphere());

  void backplot_R(P phi(double), double t_min, double t_max, 
		  int num_pts, const Sphere& S = Sphere());


  // All sphere line functions treat point arguments as unscaled
  // displacements from the center of the sphere, not as absolute
  // spatial locations.  If the center or radius of the sphere
  // changes, the line automatically adjusts as expected.

  // front portion of spherical segment joining p1 to p2
  void front_arc(const P& p1, const P& p2, const Sphere& S = Sphere());
  void  back_arc(const P& p1, const P& p2, const Sphere& S = Sphere());

  // half-line joining p1 to -p1 through p2
  void front_arc2(const P&, const P&, const Sphere& S=Sphere());
  void  back_arc2(const P&, const P&, const Sphere& S=Sphere());

  // great circle through p1 and p2
  void front_line(const P&, const P&, const Sphere& S=Sphere());
  void  back_line(const P&, const P&, const Sphere& S=Sphere());

  void front_triangle(const P&, const P&, const P&, const Sphere& S=Sphere());
  void  back_triangle(const P&, const P&, const P&, const Sphere& S=Sphere());

  // spherical polyhedra
  void front_tetra(const Sphere& S=Sphere(), const frame& coords=frame());
  void  back_tetra(const Sphere& S=Sphere(), const frame& coords=frame());

  void front_cube(const Sphere& S=Sphere(), const frame& coords=frame());
  void  back_cube(const Sphere& S=Sphere(), const frame& coords=frame());

  void front_octa(const Sphere& S=Sphere(), const frame& coords=frame());
  void  back_octa(const Sphere& S=Sphere(), const frame& coords=frame());

  void front_dodeca(const Sphere& S=Sphere(), const frame& coords=frame());
  void  back_dodeca(const Sphere& S=Sphere(), const frame& coords=frame());

  void front_icosa(const Sphere& S=Sphere(), const frame& coords=frame());
  void  back_icosa(const Sphere& S=Sphere(), const frame& coords=frame());

  // Hyperbolic lines
  void hyperbolic_line (const P&, const P&);
  void disk_line (const P&, const P&);

} // end of namespace

#endif /* EPIX_GEOMETRY */
/*
 * Sline.h -- ePiX::Sline class (spherical arc)
 *
 * This file is part of ePiX, a preprocessor for creating high-quality 
 * line figures in LaTeX 
 *
 * Version 1.2.17
 * Last Change: June 29, 2017
 */

/* 
 * Copyright (C) 2017
 * Andrew D. Hwang <rot 13 nujnat at ubylpebff dot rqh>
 * Department of Mathematics and Computer Science
 * College of the Holy Cross
 * Worcester, MA, 01610-2395, USA
 */

/*
 * ePiX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * ePiX is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ePiX; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef EPIX_SLINE
#define EPIX_SLINE


namespace ePiX {
  class P;

  class Sline {
  public:
    Sline(const P& tail, const P& head);

    bool malformed() const;
    P pole() const; // normalization of tail * head
    double cosine() const; // cosine of subtended arc
    P operator * (const Sline&) const; // this.pole() * pole()

    P reflect(const P&) const;
    Sline reflect(const Sline&) const;

    bool collinear(const Sline&) const;
    bool cuts(const Sline&) const; // we cross arg

    void draw() const; // entire arc
    void draw_front() const; // front arc
    void draw_back() const; // back arc

    void draw_line() const; // entire line
    void line_front() const; // front line
    void line_back() const; // back line

  private:
    P m_tail;
    P m_head;
    P m_pole;
    bool m_malformed; // tail, head collinear
  }; // end of Sline class
} // end of namespace

#endif /* EPIX_SLINE */

#endif /* EPIX_H */
