package clippy

import (
	"github.com/carapace-sh/carapace"
	"github.com/carapace-sh/carapace/pkg/style"
)

// ActionCategories completes categories
//
//	clippy::all (all lints that are on by default)
//	clippy::correctness (code that is outright wrong or useless)
func ActionCategories() carapace.Action {
	return carapace.ActionValuesDescribed(
		"clippy::all", "all lints that are on by default",
		"clippy::correctness", "code that is outright wrong or useless",
		"clippy::suspicious", "code that is most likely wrong or useless",
		"clippy::style", "code that should be written in a more idiomatic way",
		"clippy::complexity", "code that does something simple but in a complex way",
		"clippy::perf", "code that can be written to run faster",
		"clippy::pedantic", "lints which are rather strict or have occasional false positives",
		"clippy::nursery", "new lints that are still under development",
		"clippy::cargo", "lints for the cargo manifest",
	).Style(style.Blue)
}

// ActionLints completes lints
//
//	clippy::absurd_extreme_comparisons (Checks for comparisons where one side of the relation is))
//	clippy::dbg_macro (Checks for usage of dbg!() macro.))
func ActionLints() carapace.Action {
	// curl https://rust-lang.github.io/rust-clippy/master/lints.json | jq --raw-output  '.[] | "\"clippy::\(.id)\", \"\(.docs | split("\n")[1]))\","'
	return carapace.ActionValuesDescribed(
		"clippy::absurd_extreme_comparisons", "Checks for comparisons where one side of the relation is)",
		"clippy::almost_swapped", "Checks for `foo = bar; bar = foo` sequences.)",
		"clippy::approx_constant", "Checks for floating point literals that approximate)",
		"clippy::as_conversions", "Checks for usage of `as` conversions.)",
		"clippy::assertions_on_constants", "Checks for `assert!(true)` and `assert!(false)` calls.)",
		"clippy::assign_op_pattern", "Checks for `a = a op b` or `a = b commutative_op a`)",
		"clippy::assign_ops", "Nothing. This lint has been deprecated.)",
		"clippy::async_yields_async", "Checks for async blocks that yield values of types)",
		"clippy::await_holding_lock", "Checks for calls to await while holding a)",
		"clippy::await_holding_refcell_ref", "Checks for calls to await while holding a)",
		"clippy::bad_bit_mask", "Checks for incompatible bit masks in comparisons.)",
		"clippy::bind_instead_of_map", "Checks for usage of `_.and_then(|x| Some(y))`, `_.and_then(|x| Ok(y))` or)",
		"clippy::blacklisted_name", "Checks for usage of blacklisted names for variables, such)",
		"clippy::blanket_clippy_restriction_lints", "Checks for `warn`/`deny`/`forbid` attributes targeting the whole clippy::restriction category.)",
		"clippy::blocks_in_if_conditions", "Checks for `if` conditions that use blocks containing an)",
		"clippy::bool_assert_comparison", "This lint warns about boolean comparisons in assert-like macros.)",
		"clippy::bool_comparison", "Checks for expressions of the form `x == true`,)",
		"clippy::borrow_interior_mutable_const", "Checks if `const` items which is interior mutable (e.g.,)",
		"clippy::borrowed_box", "Checks for use of `&Box<T>` anywhere in the code.)",
		"clippy::box_collection", "Checks for use of `Box<T>` where T is a collection such as Vec anywhere in the code.)",
		"clippy::boxed_local", "Checks for usage of `Box<T>` where an unboxed `T` would)",
		"clippy::branches_sharing_code", "Checks if the `if` and `else` block contain shared code that can be)",
		"clippy::builtin_type_shadow", "Warns if a generic shadows a built-in type.)",
		"clippy::bytes_nth", "Checks for the use of `.bytes().nth()`.)",
		"clippy::cargo_common_metadata", "Checks to see if all common metadata is defined in)",
		"clippy::case_sensitive_file_extension_comparisons", "Checks for calls to `ends_with` with possible file extensions)",
		"clippy::cast_lossless", "Checks for casts between numerical types that may)",
		"clippy::cast_possible_truncation", "Checks for casts between numerical types that may)",
		"clippy::cast_possible_wrap", "Checks for casts from an unsigned type to a signed type of)",
		"clippy::cast_precision_loss", "Checks for casts from any numerical to a float type where)",
		"clippy::cast_ptr_alignment", "Checks for casts, using `as` or `pointer::cast`,)",
		"clippy::cast_ref_to_mut", "Checks for casts of `&T` to `&mut T` anywhere in the code.)",
		"clippy::cast_sign_loss", "Checks for casts from a signed to an unsigned numerical)",
		"clippy::char_lit_as_u8", "Checks for expressions where a character literal is cast)",
		"clippy::chars_last_cmp", "Checks for usage of `_.chars().last()` or)",
		"clippy::chars_next_cmp", "Checks for usage of `.chars().next()` on a `str` to check)",
		"clippy::checked_conversions", "Checks for explicit bounds checking when casting.)",
		"clippy::clone_double_ref", "Checks for usage of `.clone()` on an `&&T`.)",
		"clippy::clone_on_copy", "Checks for usage of `.clone()` on a `Copy` type.)",
		"clippy::clone_on_ref_ptr", "Checks for usage of `.clone()` on a ref-counted pointer,)",
		"clippy::cloned_instead_of_copied", "Checks for usages of `cloned()` on an `Iterator` or `Option` where)",
		"clippy::cmp_nan", "Checks for comparisons to NaN.)",
		"clippy::cmp_null", "This lint checks for equality comparisons with `ptr::null`)",
		"clippy::cmp_owned", "Checks for conversions to owned values just for the sake)",
		"clippy::cognitive_complexity", "Checks for methods with high cognitive complexity.)",
		"clippy::collapsible_else_if", "Checks for collapsible `else { if ... }` expressions)",
		"clippy::collapsible_if", "Checks for nested `if` statements which can be collapsed)",
		"clippy::collapsible_match", "Finds nested `match` or `if let` expressions where the patterns may be \"collapsed\" together)",
		"clippy::comparison_chain", "Checks comparison chains written with `if` that can be)",
		"clippy::comparison_to_empty", "Checks for comparing to an empty slice such as `\"\"` or `[]`,)",
		"clippy::copy_iterator", "Checks for types that implement `Copy` as well as)",
		"clippy::create_dir", "Checks usage of `std::fs::create_dir` and suggest using `std::fs::create_dir_all` instead.)",
		"clippy::crosspointer_transmute", "Checks for transmutes between a type `T` and `*T`.)",
		"clippy::dbg_macro", "Checks for usage of dbg!() macro.)",
		"clippy::debug_assert_with_mut_call", "Checks for function/method calls with a mutable)",
		"clippy::decimal_literal_representation", "Warns if there is a better representation for a numeric literal.)",
		"clippy::declare_interior_mutable_const", "Checks for declaration of `const` items which is interior)",
		"clippy::default_numeric_fallback", "Checks for usage of unconstrained numeric literals which may cause default numeric fallback in type)",
		"clippy::default_trait_access", "Checks for literal calls to `Default::default()`.)",
		"clippy::deprecated_cfg_attr", "Checks for `#[cfg_attr(rustfmt, rustfmt_skip)]` and suggests to replace it)",
		"clippy::deprecated_semver", "Checks for `#[deprecated]` annotations with a `since`)",
		"clippy::deref_addrof", "Checks for usage of `*&` and `*&mut` in expressions.)",
		"clippy::derivable_impls", "Detects manual `std::default::Default` implementations that are identical to a derived implementation.)",
		"clippy::derive_hash_xor_eq", "Checks for deriving `Hash` but implementing `PartialEq`)",
		"clippy::derive_ord_xor_partial_ord", "Checks for deriving `Ord` but implementing `PartialOrd`)",
		"clippy::disallowed_methods", "Denies the configured methods and functions in clippy.toml)",
		"clippy::disallowed_script_idents", "Checks for usage of unicode scripts other than those explicitly allowed)",
		"clippy::disallowed_types", "Denies the configured types in clippy.toml.)",
		"clippy::diverging_sub_expression", "Checks for diverging calls that are not match arms or)",
		"clippy::doc_markdown", "Checks for the presence of `_`, `::` or camel-case words)",
		"clippy::double_comparisons", "Checks for double comparisons that could be simplified to a single expression.)",
		"clippy::double_must_use", "Checks for a `#[must_use]` attribute without)",
		"clippy::double_neg", "Detects expressions of the form `--x`.)",
		"clippy::double_parens", "Checks for unnecessary double parentheses.)",
		"clippy::drop_copy", "Checks for calls to `std::mem::drop` with a value)",
		"clippy::drop_ref", "Checks for calls to `std::mem::drop` with a reference)",
		"clippy::duplicate_underscore_argument", "Checks for function arguments having the similar names)",
		"clippy::duration_subsec", "Checks for calculation of subsecond microseconds or milliseconds)",
		"clippy::else_if_without_else", "Checks for usage of if expressions with an `else if` branch,)",
		"clippy::empty_enum", "Checks for `enum`s with no variants.)",
		"clippy::empty_line_after_outer_attr", "Checks for empty lines after outer attributes)",
		"clippy::empty_loop", "Checks for empty `loop` expressions.)",
		"clippy::enum_clike_unportable_variant", "Checks for C-like enumerations that are)",
		"clippy::enum_glob_use", "Checks for `use Enum::*`.)",
		"clippy::enum_variant_names", "Detects enumeration variants that are prefixed or suffixed)",
		"clippy::eq_op", "Checks for equal operands to comparison, logical and)",
		"clippy::equatable_if_let", "Checks for pattern matchings that can be expressed using equality.)",
		"clippy::erasing_op", "Checks for erasing operations, e.g., `x * 0`.)",
		"clippy::eval_order_dependence", "Checks for a read and a write to the same variable where)",
		"clippy::excessive_precision", "Checks for float literals with a precision greater)",
		"clippy::exhaustive_enums", "Warns on any exported `enum`s that are not tagged `#[non_exhaustive]`)",
		"clippy::exhaustive_structs", "Warns on any exported `structs`s that are not tagged `#[non_exhaustive]`)",
		"clippy::exit", "`exit()`  terminates the program and doesn't provide a)",
		"clippy::expect_fun_call", "Checks for calls to `.expect(&format!(...))`, `.expect(foo(..))`,)",
		"clippy::expect_used", "Checks for `.expect()` calls on `Option`s and `Result`s.)",
		"clippy::expl_impl_clone_on_copy", "Checks for explicit `Clone` implementations for `Copy`)",
		"clippy::explicit_counter_loop", "Checks `for` loops over slices with an explicit counter)",
		"clippy::explicit_deref_methods", "Checks for explicit `deref()` or `deref_mut()` method calls.)",
		"clippy::explicit_into_iter_loop", "Checks for loops on `y.into_iter()` where `y` will do, and)",
		"clippy::explicit_iter_loop", "Checks for loops on `x.iter()` where `&x` will do, and)",
		"clippy::explicit_write", "Checks for usage of `write!()` / `writeln()!` which can be)",
		"clippy::extend_from_slice", "Nothing. This lint has been deprecated.)",
		"clippy::extend_with_drain", "Checks for occurrences where one vector gets extended instead of append)",
		"clippy::extra_unused_lifetimes", "Checks for lifetimes in generics that are never used)",
		"clippy::fallible_impl_from", "Checks for impls of `From<..>` that contain `panic!()` or `unwrap()`)",
		"clippy::field_reassign_with_default", "Checks for immediate reassignment of fields initialized)",
		"clippy::filetype_is_file", "Checks for `FileType::is_file()`.)",
		"clippy::filter_map", "Nothing. This lint has been deprecated.)",
		"clippy::filter_map_identity", "Checks for usage of `filter_map(|x| x)`.)",
		"clippy::filter_map_next", "Checks for usage of `_.filter_map(_).next()`.)",
		"clippy::filter_next", "Checks for usage of `_.filter(_).next()`.)",
		"clippy::find_map", "Nothing. This lint has been deprecated.)",
		"clippy::flat_map_identity", "Checks for usage of `flat_map(|x| x)`.)",
		"clippy::flat_map_option", "Checks for usages of `Iterator::flat_map()` where `filter_map()` could be)",
		"clippy::float_arithmetic", "Checks for float arithmetic.)",
		"clippy::float_cmp", "Checks for (in-)equality comparisons on floating-point)",
		"clippy::float_cmp_const", "Checks for (in-)equality comparisons on floating-point)",
		"clippy::float_equality_without_abs", "Checks for statements of the form `(a - b) < f32::EPSILON` or)",
		"clippy::fn_address_comparisons", "Checks for comparisons with an address of a function item.)",
		"clippy::fn_params_excessive_bools", "Checks for excessive use of)",
		"clippy::fn_to_numeric_cast", "Checks for casts of function pointers to something other than usize)",
		"clippy::fn_to_numeric_cast_any", "Checks for casts of a function pointer to any integer type.)",
		"clippy::fn_to_numeric_cast_with_truncation", "Checks for casts of a function pointer to a numeric type not wide enough to)",
		"clippy::for_kv_map", "Checks for iterating a map (`HashMap` or `BTreeMap`) and)",
		"clippy::for_loops_over_fallibles", "Checks for `for` loops over `Option` or `Result` values.)",
		"clippy::forget_copy", "Checks for calls to `std::mem::forget` with a value that)",
		"clippy::forget_ref", "Checks for calls to `std::mem::forget` with a reference)",
		"clippy::format_in_format_args", "Detects `format!` within the arguments of another macro that does)",
		"clippy::from_iter_instead_of_collect", "Checks for `from_iter()` function calls on types that implement the `FromIterator`)",
		"clippy::from_over_into", "Searches for implementations of the `Into<..>` trait and suggests to implement `From<..>` instead.)",
		"clippy::from_str_radix_10", ")",
		"clippy::future_not_send", "This lint requires Future implementations returned from)",
		"clippy::get_last_with_len", "Checks for using `x.get(x.len() - 1)` instead of)",
		"clippy::get_unwrap", "Checks for use of `.get().unwrap()` (or)",
		"clippy::identity_op", "Checks for identity operations, e.g., `x + 0`.)",
		"clippy::if_let_mutex", "Checks for `Mutex::lock` calls in `if let` expression)",
		"clippy::if_let_redundant_pattern_matching", "Nothing. This lint has been deprecated.)",
		"clippy::if_not_else", "Checks for usage of `!` or `!=` in an if condition with an)",
		"clippy::if_same_then_else", "Checks for `if/else` with the same body as the *then* part)",
		"clippy::if_then_some_else_none", "Checks for if-else that could be written to `bool::then`.)",
		"clippy::ifs_same_cond", "Checks for consecutive `if`s with the same condition.)",
		"clippy::implicit_clone", "Checks for the usage of `_.to_owned()`, `vec.to_vec()`, or similar when calling `_.clone()` would be clearer.)",
		"clippy::implicit_hasher", "Checks for public `impl` or `fn` missing generalization)",
		"clippy::implicit_return", "Checks for missing return statements at the end of a block.)",
		"clippy::implicit_saturating_sub", "Checks for implicit saturating subtraction.)",
		"clippy::imprecise_flops", "Looks for floating-point expressions that)",
		"clippy::inconsistent_digit_grouping", "Warns if an integral or floating-point constant is)",
		"clippy::inconsistent_struct_constructor", "Checks for struct constructors where all fields are shorthand and)",
		"clippy::index_refutable_slice", "The lint checks for slice bindings in patterns that are only used to)",
		"clippy::indexing_slicing", "Checks for usage of indexing or slicing. Arrays are special cases, this lint)",
		"clippy::ineffective_bit_mask", "Checks for bit masks in comparisons which can be removed)",
		"clippy::inefficient_to_string", "Checks for usage of `.to_string()` on an `&&T` where)",
		"clippy::infallible_destructuring_match", "Checks for matches being used to destructure a single-variant enum)",
		"clippy::infinite_iter", "Checks for iteration that is guaranteed to be infinite.)",
		"clippy::inherent_to_string", "Checks for the definition of inherent methods with a signature of `to_string(&self) -> String`.)",
		"clippy::inherent_to_string_shadow_display", "Checks for the definition of inherent methods with a signature of `to_string(&self) -> String` and if the type implementing this method also implements the `Display` trait.)",
		"clippy::inline_always", "Checks for items annotated with `#[inline(always)]`,)",
		"clippy::inline_asm_x86_att_syntax", "Checks for usage of AT&T x86 assembly syntax.)",
		"clippy::inline_asm_x86_intel_syntax", "Checks for usage of Intel x86 assembly syntax.)",
		"clippy::inline_fn_without_body", "Checks for `#[inline]` on trait methods without bodies)",
		"clippy::inspect_for_each", "Checks for usage of `inspect().for_each()`.)",
		"clippy::int_plus_one", "Checks for usage of `x >= y + 1` or `x - 1 >= y` (and `<=`) in a block)",
		"clippy::integer_arithmetic", "Checks for integer arithmetic operations which could overflow or panic.)",
		"clippy::integer_division", "Checks for division of integers)",
		"clippy::into_iter_on_ref", "Checks for `into_iter` calls on references which should be replaced by `iter`)",
		"clippy::invalid_null_ptr_usage", "This lint checks for invalid usages of `ptr::null`.)",
		"clippy::invalid_regex", "Checks [regex](https://crates.io/crates/regex) creation)",
		"clippy::invalid_upcast_comparisons", "Checks for comparisons where the relation is always either)",
		"clippy::invisible_characters", "Checks for invisible Unicode characters in the code.)",
		"clippy::items_after_statements", "Checks for items declared after some statement in a block.)",
		"clippy::iter_cloned_collect", "Checks for the use of `.cloned().collect()` on slice to)",
		"clippy::iter_count", "Checks for the use of `.iter().count()`.)",
		"clippy::iter_next_loop", "Checks for loops on `x.next()`.)",
		"clippy::iter_next_slice", "Checks for usage of `iter().next()` on a Slice or an Array)",
		"clippy::iter_not_returning_iterator", "Detects methods named `iter` or `iter_mut` that do not have a return type that implements `Iterator`.)",
		"clippy::iter_nth", "Checks for use of `.iter().nth()` (and the related)",
		"clippy::iter_nth_zero", "Checks for the use of `iter.nth(0)`.)",
		"clippy::iter_skip_next", "Checks for use of `.skip(x).next()` on iterators.)",
		"clippy::iterator_step_by_zero", "Checks for calling `.step_by(0)` on iterators which panics.)",
		"clippy::just_underscores_and_digits", "Checks if you have variables whose name consists of just)",
		"clippy::large_const_arrays", "Checks for large `const` arrays that should)",
		"clippy::large_digit_groups", "Warns if the digits of an integral or floating-point)",
		"clippy::large_enum_variant", "Checks for large size differences between variants on)",
		"clippy::large_stack_arrays", "Checks for local arrays that may be too large.)",
		"clippy::large_types_passed_by_value", "Checks for functions taking arguments by value, where)",
		"clippy::len_without_is_empty", "Checks for items that implement `.len()` but not)",
		"clippy::len_zero", "Checks for getting the length of something via `.len()`)",
		"clippy::let_and_return", "Checks for `let`-bindings, which are subsequently)",
		"clippy::let_underscore_drop", "Checks for `let _ = <expr>`)",
		"clippy::let_underscore_lock", "Checks for `let _ = sync_lock`.)",
		"clippy::let_underscore_must_use", "Checks for `let _ = <expr>` where expr is `#[must_use]`)",
		"clippy::let_unit_value", "Checks for binding a unit value.)",
		"clippy::linkedlist", "Checks for usage of any `LinkedList`, suggesting to use a)",
		"clippy::logic_bug", "Checks for boolean expressions that contain terminals that)",
		"clippy::lossy_float_literal", "Checks for whole number float literals that)",
		"clippy::macro_use_imports", "Checks for `#[macro_use] use...`.)",
		"clippy::main_recursion", "Checks for recursion using the entrypoint.)",
		"clippy::manual_assert", "Detects `if`-then-`panic!` that can be replaced with `assert!`.)",
		"clippy::manual_async_fn", "It checks for manual implementations of `async` functions.)",
		"clippy::manual_filter_map", "Checks for usage of `_.filter(_).map(_)` that can be written more simply)",
		"clippy::manual_find_map", "Checks for usage of `_.find(_).map(_)` that can be written more simply)",
		"clippy::manual_flatten", "Check for unnecessary `if let` usage in a for loop)",
		"clippy::manual_map", "Checks for usages of `match` which could be implemented using `map`)",
		"clippy::manual_memcpy", "Checks for for-loops that manually copy items between)",
		"clippy::manual_non_exhaustive", "Checks for manual implementations of the non-exhaustive pattern.)",
		"clippy::manual_ok_or", ")",
		"clippy::manual_range_contains", "Checks for expressions like `x >= 3 && x < 8` that could)",
		"clippy::manual_saturating_arithmetic", "Checks for `.checked_add/sub(x).unwrap_or(MAX/MIN)`.)",
		"clippy::manual_split_once", "Checks for usages of `str::splitn(2, _)`)",
		"clippy::manual_str_repeat", "Checks for manual implementations of `str::repeat`)",
		"clippy::manual_strip", "Suggests using `strip_{prefix,suffix}` over `str::{starts,ends}_with` and slicing using)",
		"clippy::manual_swap", "Checks for manual swapping.)",
		"clippy::manual_unwrap_or", "Finds patterns that reimplement `Option::unwrap_or` or `Result::unwrap_or`.)",
		"clippy::many_single_char_names", "Checks for too many variables whose name consists of a)",
		"clippy::map_clone", "Checks for usage of `map(|x| x.clone())` or)",
		"clippy::map_collect_result_unit", "Checks for usage of `_.map(_).collect::<Result<(), _>()`.)",
		"clippy::map_entry", "Checks for uses of `contains_key` + `insert` on `HashMap`)",
		"clippy::map_err_ignore", "Checks for instances of `map_err(|_| Some::Enum)`)",
		"clippy::map_flatten", "Checks for usage of `_.map(_).flatten(_)` on `Iterator` and `Option`)",
		"clippy::map_identity", "Checks for instances of `map(f)` where `f` is the identity function.)",
		"clippy::map_unwrap_or", "Checks for usage of `option.map(_).unwrap_or(_)` or `option.map(_).unwrap_or_else(_)` or)",
		"clippy::match_as_ref", "Checks for match which is used to add a reference to an)",
		"clippy::match_bool", "Checks for matches where match expression is a `bool`. It)",
		"clippy::match_like_matches_macro", "Checks for `match`  or `if let` expressions producing a)",
		"clippy::match_on_vec_items", "Checks for `match vec[idx]` or `match vec[n..m]`.)",
		"clippy::match_overlapping_arm", "Checks for overlapping match arms.)",
		"clippy::match_ref_pats", "Checks for matches where all arms match a reference,)",
		"clippy::match_result_ok", "Checks for unnecessary `ok()` in `while let`.)",
		"clippy::match_same_arms", "Checks for `match` with identical arm bodies.)",
		"clippy::match_single_binding", "Checks for useless match that binds to only one value.)",
		"clippy::match_str_case_mismatch", "Checks for `match` expressions modifying the case of a string with non-compliant arms)",
		"clippy::match_wild_err_arm", "Checks for arm which matches all errors with `Err(_)`)",
		"clippy::match_wildcard_for_single_variants", "Checks for wildcard enum matches for a single variant.)",
		"clippy::maybe_infinite_iter", "Checks for iteration that may be infinite.)",
		"clippy::mem_forget", "Checks for usage of `std::mem::forget(t)` where `t` is)",
		"clippy::mem_replace_option_with_none", "Checks for `mem::replace()` on an `Option` with)",
		"clippy::mem_replace_with_default", "Checks for `std::mem::replace` on a value of type)",
		"clippy::mem_replace_with_uninit", "Checks for `mem::replace(&mut _, mem::uninitialized())`)",
		"clippy::min_max", "Checks for expressions where `std::cmp::min` and `max` are)",
		"clippy::misaligned_transmute", "Nothing. This lint has been deprecated.)",
		"clippy::mismatched_target_os", "Checks for cfg attributes having operating systems used in target family position.)",
		"clippy::misrefactored_assign_op", "Checks for `a op= a op b` or `a op= b op a` patterns.)",
		"clippy::missing_const_for_fn", "Suggests the use of `const` in functions and methods where possible.)",
		"clippy::missing_docs_in_private_items", "Warns if there is missing doc for any documentable item)",
		"clippy::missing_enforced_import_renames", "Checks for imports that do not rename the item as specified)",
		"clippy::missing_errors_doc", "Checks the doc comments of publicly visible functions that)",
		"clippy::missing_inline_in_public_items", "It lints if an exported function, method, trait method with default impl,)",
		"clippy::missing_panics_doc", "Checks the doc comments of publicly visible functions that)",
		"clippy::missing_safety_doc", "Checks for the doc comments of publicly visible)",
		"clippy::mistyped_literal_suffixes", "Warns for mistyped suffix in literals)",
		"clippy::mixed_case_hex_literals", "Warns on hexadecimal literals with mixed-case letter)",
		"clippy::mod_module_files", "Checks that module layout uses only self named module files, bans mod.rs files.)",
		"clippy::module_inception", "Checks for modules that have the same name as their)",
		"clippy::module_name_repetitions", "Detects type names that are prefixed or suffixed by the)",
		"clippy::modulo_arithmetic", "Checks for modulo arithmetic.)",
		"clippy::modulo_one", "Checks for getting the remainder of a division by one or minus)",
		"clippy::multiple_crate_versions", "Checks to see if multiple versions of a crate are being)",
		"clippy::multiple_inherent_impl", "Checks for multiple inherent implementations of a struct)",
		"clippy::must_use_candidate", "Checks for public functions that have no)",
		"clippy::must_use_unit", "Checks for a `#[must_use]` attribute on)",
		"clippy::mut_from_ref", "This lint checks for functions that take immutable)",
		"clippy::mut_mut", "Checks for instances of `mut mut` references.)",
		"clippy::mut_mutex_lock", "Checks for `&mut Mutex::lock` calls)",
		"clippy::mut_range_bound", "Checks for loops which have a range bound that is a mutable variable)",
		"clippy::mutable_key_type", "Checks for sets/maps with mutable key types.)",
		"clippy::mutex_atomic", "Checks for usages of `Mutex<X>` where an atomic will do.)",
		"clippy::mutex_integer", "Checks for usages of `Mutex<X>` where `X` is an integral)",
		"clippy::naive_bytecount", "Checks for naive byte counts)",
		"clippy::needless_arbitrary_self_type", "The lint checks for `self` in fn parameters that)",
		"clippy::needless_bitwise_bool", "Checks for uses of bitwise and/or operators between booleans, where performance may be improved by using)",
		"clippy::needless_bool", "Checks for expressions of the form `if c { true } else {)",
		"clippy::needless_borrow", "Checks for address of operations (`&`) that are going to)",
		"clippy::needless_borrowed_reference", "Checks for bindings that destructure a reference and borrow the inner)",
		"clippy::needless_collect", "Checks for functions collecting an iterator when collect)",
		"clippy::needless_continue", "The lint checks for `if`-statements appearing in loops)",
		"clippy::needless_doctest_main", "Checks for `fn main() { .. }` in doctests)",
		"clippy::needless_for_each", "Checks for usage of `for_each` that would be more simply written as a)",
		"clippy::needless_lifetimes", "Checks for lifetime annotations which can be removed by)",
		"clippy::needless_option_as_deref", "Checks for no-op uses of Option::{as_deref,as_deref_mut},)",
		"clippy::needless_pass_by_value", "Checks for functions taking arguments by value, but not)",
		"clippy::needless_question_mark", "Suggests alternatives for useless applications of `?` in terminating expressions)",
		"clippy::needless_range_loop", "Checks for looping over the range of `0..len` of some)",
		"clippy::needless_return", "Checks for return statements at the end of a block.)",
		"clippy::needless_splitn", "Checks for usages of `str::splitn` (or `str::rsplitn`) where using `str::split` would be the same.)",
		"clippy::needless_update", "Checks for needlessly including a base struct on update)",
		"clippy::neg_cmp_op_on_partial_ord", "Checks for the usage of negated comparison operators on types which only implement)",
		"clippy::neg_multiply", "Checks for multiplication by -1 as a form of negation.)",
		"clippy::negative_feature_names", "Checks for negative feature names with prefix `no-` or `not-`)",
		"clippy::never_loop", "Checks for loops that will always `break`, `return` or)",
		"clippy::new_ret_no_self", "Checks for `new` not returning a type that contains `Self`.)",
		"clippy::new_without_default", "Checks for types with a `fn new() -> Self` method and no)",
		"clippy::no_effect", "Checks for statements which have no effect.)",
		"clippy::no_effect_underscore_binding", "Checks for binding to underscore prefixed variable without side-effects.)",
		"clippy::non_ascii_literal", "Checks for non-ASCII characters in string literals.)",
		"clippy::non_octal_unix_permissions", "Checks for non-octal values used to set Unix file permissions.)",
		"clippy::non_send_fields_in_send_ty", "Warns about fields in struct implementing `Send` that are neither `Send` nor `Copy`.)",
		"clippy::nonminimal_bool", "Checks for boolean expressions that can be written more)",
		"clippy::nonsensical_open_options", "Checks for duplicate open options as well as combinations)",
		"clippy::nonstandard_macro_braces", "Checks that common macros are used with consistent bracing.)",
		"clippy::not_unsafe_ptr_arg_deref", "Checks for public functions that dereference raw pointer)",
		"clippy::octal_escapes", "Checks for `\\0` escapes in string and byte literals that look like octal)",
		"clippy::ok_expect", "Checks for usage of `ok().expect(..)`.)",
		"clippy::op_ref", "Checks for arguments to `==` which have their address)",
		"clippy::option_as_ref_deref", "Checks for usage of `_.as_ref().map(Deref::deref)` or it's aliases (such as String::as_str).)",
		"clippy::option_env_unwrap", "Checks for usage of `option_env!(...).unwrap()` and)",
		"clippy::option_filter_map", "Checks for indirect collection of populated `Option`)",
		"clippy::option_if_let_else", "Lints usage of `if let Some(v) = ... { y } else { x }` which is more)",
		"clippy::option_map_or_none", "Checks for usage of `_.map_or(None, _)`.)",
		"clippy::option_map_unit_fn", "Checks for usage of `option.map(f)` where f is a function)",
		"clippy::option_option", "Checks for use of `Option<Option<_>>` in function signatures and type)",
		"clippy::or_fun_call", "Checks for calls to `.or(foo(..))`, `.unwrap_or(foo(..))`,)",
		"clippy::out_of_bounds_indexing", "Checks for out of bounds array indexing with a constant)",
		"clippy::overflow_check_conditional", "Detects classic underflow/overflow checks.)",
		"clippy::panic", "Checks for usage of `panic!`.)",
		"clippy::panic_in_result_fn", "Checks for usage of `panic!`, `unimplemented!`, `todo!`, `unreachable!` or assertions in a function of type result.)",
		"clippy::panicking_unwrap", "Checks for calls of `unwrap[_err]()` that will always fail.)",
		"clippy::partialeq_ne_impl", "Checks for manual re-implementations of `PartialEq::ne`.)",
		"clippy::path_buf_push_overwrite", "* Checks for [push](https://doc.rust-lang.org/std/path/struct.PathBuf.html#method.push))",
		"clippy::pattern_type_mismatch", "Checks for patterns that aren't exact representations of the types)",
		"clippy::possible_missing_comma", "Checks for possible missing comma in an array. It lints if)",
		"clippy::precedence", "Checks for operations where precedence may be unclear)",
		"clippy::print_literal", "This lint warns about the use of literals as `print!`/`println!` args.)",
		"clippy::print_stderr", "Checks for printing on *stderr*. The purpose of this lint)",
		"clippy::print_stdout", "Checks for printing on *stdout*. The purpose of this lint)",
		"clippy::print_with_newline", "This lint warns when you use `print!()` with a format)",
		"clippy::println_empty_string", "This lint warns when you use `println!(\"\")` to)",
		"clippy::ptr_arg", "This lint checks for function arguments of type `&String`)",
		"clippy::ptr_as_ptr", "Checks for `as` casts between raw pointers without changing its mutability,)",
		"clippy::ptr_eq", "Use `std::ptr::eq` when applicable)",
		"clippy::ptr_offset_with_cast", "Checks for usage of the `offset` pointer method with a `usize` casted to an)",
		"clippy::pub_enum_variant_names", "Nothing. This lint has been deprecated.)",
		"clippy::question_mark", "Checks for expressions that could be replaced by the question mark operator.)",
		"clippy::range_minus_one", "Checks for inclusive ranges where 1 is subtracted from)",
		"clippy::range_plus_one", "Checks for exclusive ranges where 1 is added to the)",
		"clippy::range_step_by_zero", "Nothing. This lint has been deprecated.)",
		"clippy::range_zip_with_len", "Checks for zipping a collection with the range of)",
		"clippy::rc_buffer", "Checks for `Rc<T>` and `Arc<T>` when `T` is a mutable buffer type such as `String` or `Vec`.)",
		"clippy::rc_mutex", "Checks for `Rc<Mutex<T>>`.)",
		"clippy::redundant_allocation", "Checks for use of redundant allocations anywhere in the code.)",
		"clippy::redundant_clone", "Checks for a redundant `clone()` (and its relatives) which clones an owned)",
		"clippy::redundant_closure", "Checks for closures which just call another function where)",
		"clippy::redundant_closure_call", "Detects closures called in the same expression where they)",
		"clippy::redundant_closure_for_method_calls", "Checks for closures which only invoke a method on the closure)",
		"clippy::redundant_else", "Checks for `else` blocks that can be removed without changing semantics.)",
		"clippy::redundant_feature_names", "Checks for feature names with prefix `use-`, `with-` or suffix `-support`)",
		"clippy::redundant_field_names", "Checks for fields in struct literals where shorthands)",
		"clippy::redundant_pattern", "Checks for patterns in the form `name @ _`.)",
		"clippy::redundant_pattern_matching", "Lint for redundant pattern matching over `Result`, `Option`,)",
		"clippy::redundant_pub_crate", "Checks for items declared `pub(crate)` that are not crate visible because they)",
		"clippy::redundant_slicing", "Checks for redundant slicing expressions which use the full range, and)",
		"clippy::redundant_static_lifetimes", "Checks for constants and statics with an explicit `'static` lifetime.)",
		"clippy::ref_binding_to_reference", "Checks for `ref` bindings which create a reference to a reference.)",
		"clippy::ref_in_deref", "Checks for references in expressions that use)",
		"clippy::ref_option_ref", "Checks for usage of `&Option<&T>`.)",
		"clippy::regex_macro", "Nothing. This lint has been deprecated.)",
		"clippy::repeat_once", "Checks for usage of `.repeat(1)` and suggest the following method for each types.)",
		"clippy::replace_consts", "Nothing. This lint has been deprecated.)",
		"clippy::rest_pat_in_fully_bound_structs", "Checks for unnecessary '..' pattern binding on struct when all fields are explicitly matched.)",
		"clippy::result_map_or_into_option", "Checks for usage of `_.map_or(None, Some)`.)",
		"clippy::result_map_unit_fn", "Checks for usage of `result.map(f)` where f is a function)",
		"clippy::result_unit_err", "Checks for public functions that return a `Result`)",
		"clippy::reversed_empty_ranges", "Checks for range expressions `x..y` where both `x` and `y`)",
		"clippy::same_functions_in_if_condition", "Checks for consecutive `if`s with the same function call.)",
		"clippy::same_item_push", "Checks whether a for loop is being used to push a constant)",
		"clippy::same_name_method", "It lints if a struct has two method with same time:)",
		"clippy::search_is_some", "Checks for an iterator or string search (such as `find()`,)",
		"clippy::self_assignment", "Checks for explicit self-assignments.)",
		"clippy::self_named_constructors", "Warns when constructors have the same name as their types.)",
		"clippy::self_named_module_files", "Checks that module layout uses only mod.rs files.)",
		"clippy::semicolon_if_nothing_returned", "Looks for blocks of expressions and fires if the last expression returns)",
		"clippy::separated_literal_suffix", "Warns if literal suffixes are separated by an underscore.)",
		"clippy::serde_api_misuse", "Checks for mis-uses of the serde API.)",
		"clippy::shadow_reuse", "Checks for bindings that shadow other bindings already in)",
		"clippy::shadow_same", "Checks for bindings that shadow other bindings already in)",
		"clippy::shadow_unrelated", "Checks for bindings that shadow other bindings already in)",
		"clippy::short_circuit_statement", "Checks for the use of short circuit boolean conditions as)",
		"clippy::should_assert_eq", "Nothing. This lint has been deprecated.)",
		"clippy::should_implement_trait", "Checks for methods that should live in a trait)",
		"clippy::similar_names", "Checks for names that are very similar and thus confusing.)",
		"clippy::single_char_add_str", "Warns when using `push_str`/`insert_str` with a single-character string literal)",
		"clippy::single_char_pattern", "Checks for string methods that receive a single-character)",
		"clippy::single_component_path_imports", "Checking for imports with single component use path.)",
		"clippy::single_element_loop", "Checks whether a for loop has a single element.)",
		"clippy::single_match", "Checks for matches with a single arm where an `if let`)",
		"clippy::single_match_else", "Checks for matches with two arms where an `if let else` will)",
		"clippy::size_of_in_element_count", "Detects expressions where)",
		"clippy::skip_while_next", "Checks for usage of `_.skip_while(condition).next()`.)",
		"clippy::slow_vector_initialization", "Checks slow zero-filled vector initialization)",
		"clippy::stable_sort_primitive", "When sorting primitive values (integers, bools, chars, as well)",
		"clippy::str_to_string", "This lint checks for `.to_string()` method calls on values of type `&str`.)",
		"clippy::string_add", "Checks for all instances of `x + _` where `x` is of type)",
		"clippy::string_add_assign", "Checks for string appends of the form `x = x + y` (without)",
		"clippy::string_extend_chars", "Checks for the use of `.extend(s.chars())` where s is a)",
		"clippy::string_from_utf8_as_bytes", "Check if the string is transformed to byte array and casted back to string.)",
		"clippy::string_lit_as_bytes", "Checks for the `as_bytes` method called on string literals)",
		"clippy::string_slice", "Checks for slice operations on strings)",
		"clippy::string_to_string", "This lint checks for `.to_string()` method calls on values of type `String`.)",
		"clippy::strlen_on_c_strings", "Checks for usage of `libc::strlen` on a `CString` or `CStr` value,)",
		"clippy::struct_excessive_bools", "Checks for excessive)",
		"clippy::suboptimal_flops", "Looks for floating-point expressions that)",
		"clippy::suspicious_arithmetic_impl", "Lints for suspicious operations in impls of arithmetic operators, e.g.)",
		"clippy::suspicious_assignment_formatting", "Checks for use of the non-existent `=*`, `=!` and `=-`)",
		"clippy::suspicious_else_formatting", "Checks for formatting of `else`. It lints if the `else`)",
		"clippy::suspicious_map", "Checks for calls to `map` followed by a `count`.)",
		"clippy::suspicious_op_assign_impl", "Lints for suspicious operations in impls of OpAssign, e.g.)",
		"clippy::suspicious_operation_groupings", "Checks for unlikely usages of binary operators that are almost)",
		"clippy::suspicious_splitn", "Checks for calls to [`splitn`])",
		"clippy::suspicious_unary_op_formatting", "Checks the formatting of a unary operator on the right hand side)",
		"clippy::tabs_in_doc_comments", "Checks doc comments for usage of tab characters.)",
		"clippy::temporary_assignment", "Checks for construction of a structure or tuple just to)",
		"clippy::to_digit_is_some", "Checks for `.to_digit(..).is_some()` on `char`s.)",
		"clippy::to_string_in_display", "Checks for uses of `to_string()` in `Display` traits.)",
		"clippy::to_string_in_format_args", "Checks for [`ToString::to_string`](https://doc.rust-lang.org/std/string/trait.ToString.html#tymethod.to_string))",
		"clippy::todo", "Checks for usage of `todo!`.)",
		"clippy::too_many_arguments", "Checks for functions with too many parameters.)",
		"clippy::too_many_lines", "Checks for functions with a large amount of lines.)",
		"clippy::toplevel_ref_arg", "Checks for function arguments and let bindings denoted as)",
		"clippy::trailing_empty_array", "Displays a warning when a struct with a trailing zero-sized array is declared without a `repr` attribute.)",
		"clippy::trait_duplication_in_bounds", "Checks for cases where generics are being used and multiple)",
		"clippy::transmute_bytes_to_str", "Checks for transmutes from a `&[u8]` to a `&str`.)",
		"clippy::transmute_float_to_int", "Checks for transmutes from a float to an integer.)",
		"clippy::transmute_int_to_bool", "Checks for transmutes from an integer to a `bool`.)",
		"clippy::transmute_int_to_char", "Checks for transmutes from an integer to a `char`.)",
		"clippy::transmute_int_to_float", "Checks for transmutes from an integer to a float.)",
		"clippy::transmute_num_to_bytes", "Checks for transmutes from a number to an array of `u8`)",
		"clippy::transmute_ptr_to_ptr", "Checks for transmutes from a pointer to a pointer, or)",
		"clippy::transmute_ptr_to_ref", "Checks for transmutes from a pointer to a reference.)",
		"clippy::transmutes_expressible_as_ptr_casts", "Checks for transmutes that could be a pointer cast.)",
		"clippy::transmuting_null", "Checks for transmute calls which would receive a null pointer.)",
		"clippy::trivial_regex", "Checks for trivial [regex](https://crates.io/crates/regex))",
		"clippy::trivially_copy_pass_by_ref", "Checks for functions taking arguments by reference, where)",
		"clippy::try_err", "Checks for usages of `Err(x)?`.)",
		"clippy::type_complexity", "Checks for types used in structs, parameters and `let`)",
		"clippy::type_repetition_in_bounds", "This lint warns about unnecessary type repetitions in trait bounds)",
		"clippy::undocumented_unsafe_blocks", "Checks for `unsafe` blocks without a `// Safety: ` comment)",
		"clippy::undropped_manually_drops", "Prevents the safe `std::mem::drop` function from being called on `std::mem::ManuallyDrop`.)",
		"clippy::unicode_not_nfc", "Checks for string literals that contain Unicode in a form)",
		"clippy::unimplemented", "Checks for usage of `unimplemented!`.)",
		"clippy::uninit_assumed_init", "Checks for `MaybeUninit::uninit().assume_init()`.)",
		"clippy::uninit_vec", "Checks for `set_len()` call that creates `Vec` with uninitialized elements.)",
		"clippy::unit_arg", "Checks for passing a unit value as an argument to a function without using a)",
		"clippy::unit_cmp", "Checks for comparisons to unit. This includes all binary)",
		"clippy::unit_hash", "Detects `().hash(_)`.)",
		"clippy::unit_return_expecting_ord", "Checks for functions that expect closures of type)",
		"clippy::unnecessary_cast", "Checks for casts to the same type, casts of int literals to integer types)",
		"clippy::unnecessary_filter_map", "Checks for `filter_map` calls which could be replaced by `filter` or `map`.)",
		"clippy::unnecessary_fold", "Checks for using `fold` when a more succinct alternative exists.)",
		"clippy::unnecessary_lazy_evaluations", "As the counterpart to `or_fun_call`, this lint looks for unnecessary)",
		"clippy::unnecessary_mut_passed", "Detects passing a mutable reference to a function that only)",
		"clippy::unnecessary_operation", "Checks for expression statements that can be reduced to a)",
		"clippy::unnecessary_self_imports", "Checks for imports ending in `::{self}`.)",
		"clippy::unnecessary_sort_by", "Detects uses of `Vec::sort_by` passing in a closure)",
		"clippy::unnecessary_unwrap", "Checks for calls of `unwrap[_err]()` that cannot fail.)",
		"clippy::unnecessary_wraps", "Checks for private functions that only return `Ok` or `Some`.)",
		"clippy::unneeded_field_pattern", "Checks for structure field patterns bound to wildcards.)",
		"clippy::unneeded_wildcard_pattern", "Checks for tuple patterns with a wildcard)",
		"clippy::unnested_or_patterns", "Checks for unnested or-patterns, e.g., `Some(0) | Some(2)` and)",
		"clippy::unreachable", "Checks for usage of `unreachable!`.)",
		"clippy::unreadable_literal", "Warns if a long integral or floating-point constant does)",
		"clippy::unsafe_derive_deserialize", "Checks for deriving `serde::Deserialize` on a type that)",
		"clippy::unsafe_removed_from_name", "Checks for imports that remove \"unsafe\" from an item's)",
		"clippy::unsafe_vector_initialization", "Nothing. This lint has been deprecated.)",
		"clippy::unseparated_literal_suffix", "Warns if literal suffixes are not separated by an)",
		"clippy::unsound_collection_transmute", "Checks for transmutes between collections whose)",
		"clippy::unstable_as_mut_slice", "Nothing. This lint has been deprecated.)",
		"clippy::unstable_as_slice", "Nothing. This lint has been deprecated.)",
		"clippy::unused_async", "Checks for functions that are declared `async` but have no `.await`s inside of them.)",
		"clippy::unused_collect", "Nothing. This lint has been deprecated.)",
		"clippy::unused_io_amount", "Checks for unused written/read amount.)",
		"clippy::unused_self", "Checks methods that contain a `self` argument but don't use it)",
		"clippy::unused_unit", "Checks for unit (`()`) expressions that can be removed.)",
		"clippy::unusual_byte_groupings", "Warns if hexadecimal or binary literals are not grouped)",
		"clippy::unwrap_in_result", "Checks for functions of type `Result` that contain `expect()` or `unwrap()`)",
		"clippy::unwrap_or_else_default", "Checks for usages of `_.unwrap_or_else(Default::default)` on `Option` and)",
		"clippy::unwrap_used", "Checks for `.unwrap()` calls on `Option`s and on `Result`s.)",
		"clippy::upper_case_acronyms", "Checks for fully capitalized names and optionally names containing a capitalized acronym.)",
		"clippy::use_debug", "Checks for use of `Debug` formatting. The purpose of this)",
		"clippy::use_self", "Checks for unnecessary repetition of structure name when a)",
		"clippy::used_underscore_binding", "Checks for the use of bindings with a single leading)",
		"clippy::useless_asref", "Checks for usage of `.as_ref()` or `.as_mut()` where the)",
		"clippy::useless_attribute", "Checks for `extern crate` and `use` items annotated with)",
		"clippy::useless_conversion", "Checks for `Into`, `TryInto`, `From`, `TryFrom`, or `IntoIter` calls)",
		"clippy::useless_format", "Checks for the use of `format!(\"string literal with no)",
		"clippy::useless_let_if_seq", "Checks for variable declarations immediately followed by a)",
		"clippy::useless_transmute", "Checks for transmutes to the original type of the object)",
		"clippy::useless_vec", "Checks for usage of `&vec![..]` when using `&[..]` would)",
		"clippy::vec_box", "Checks for use of `Vec<Box<T>>` where T: Sized anywhere in the code.)",
		"clippy::vec_init_then_push", "Checks for calls to `push` immediately after creating a new `Vec`.)",
		"clippy::vec_resize_to_zero", "Finds occurrences of `Vec::resize(0, an_int)`)",
		"clippy::verbose_bit_mask", "Checks for bit masks that can be replaced by a call)",
		"clippy::verbose_file_reads", "Checks for use of File::read_to_end and File::read_to_string.)",
		"clippy::vtable_address_comparisons", "Checks for comparisons with an address of a trait vtable.)",
		"clippy::while_immutable_condition", "Checks whether variables used within while loop condition)",
		"clippy::while_let_loop", "Detects `loop + match` combinations that are easier)",
		"clippy::while_let_on_iterator", "Checks for `while let` expressions on iterators.)",
		"clippy::wildcard_dependencies", "Checks for wildcard dependencies in the `Cargo.toml`.)",
		"clippy::wildcard_enum_match_arm", "Checks for wildcard enum matches using `_`.)",
		"clippy::wildcard_imports", "Checks for wildcard imports `use _::*`.)",
		"clippy::wildcard_in_or_patterns", "Checks for wildcard pattern used with others patterns in same match arm.)",
		"clippy::write_literal", "This lint warns about the use of literals as `write!`/`writeln!` args.)",
		"clippy::write_with_newline", "This lint warns when you use `write!()` with a format)",
		"clippy::writeln_empty_string", "This lint warns when you use `writeln!(buf, \"\")` to)",
		"clippy::wrong_pub_self_convention", "Nothing. This lint has been deprecated.)",
		"clippy::wrong_self_convention", "Checks for methods with certain name prefixes and which)",
		"clippy::wrong_transmute", "Checks for transmutes that can't ever be correct on any)",
		"clippy::zero_divided_by_zero", "Checks for `0.0 / 0.0`.)",
		"clippy::zero_prefixed_literal", "Warns if an integral constant literal starts with `0`.)",
		"clippy::zero_ptr", "Catch casts from `0` to some pointer type)",
		"clippy::zero_sized_map_values", "Checks for maps with zero-sized value types anywhere in the code.)",
		"clippy::zst_offset", "Checks for `offset(_)`, `wrapping_`{`add`, `sub`}, etc. on raw pointers to)",
	)
}
