// Package icons provides file type icon mappings for Nerd Fonts.
// Icons sourced from eza (MIT) and superfile (MIT).
package icons

// Extensions maps file extensions to Nerd Font icons.
var Extensions = map[string]string{
	// Programming languages
	"go":     "\U0000e65e",
	"rs":     "\U0000e68b",
	"py":     "\U0000e606",
	"pyc":    "\U0000e606",
	"pyd":    "\U0000e606",
	"pyi":    "\U0000e606",
	"pyo":    "\U0000e606",
	"pyw":    "\U0000e606",
	"pyx":    "\U0000e606",
	"js":     "\U0000e74e",
	"mjs":    "\U0000e74e",
	"cjs":    "\U0000e74e",
	"ts":     "\U0000e628",
	"mts":    "\U0000e628",
	"cts":    "\U0000e628",
	"jsx":    "\U0000e7ba",
	"tsx":    "\U0000e7ba",
	"c":      "\U0000e61e",
	"h":      "\U0000e61e",
	"cpp":    "\U0000e61d",
	"cxx":    "\U0000e61d",
	"cc":     "\U0000e61d",
	"cp":     "\U0000e61d",
	"c++":    "\U0000e61d",
	"hpp":    "\U0000e61d",
	"hxx":    "\U0000e61d",
	"hh":     "\U0000e61d",
	"h++":    "\U0000e61d",
	"inl":    "\U0000e61e",
	"java":   "\U0000e256",
	"class":  "\U0000e256",
	"jar":    "\U0000e256",
	"jad":    "\U0000e256",
	"war":    "\U0000e256",
	"kt":     "\U0000e634",
	"kts":    "\U0000e634",
	"scala":  "\U0000e737",
	"rb":     "\U0000e739",
	"erb":    "\U0000e73b",
	"php":    "\U0000e73d",
	"phar":   "\U0000e73d",
	"swift":  "\U0000e755",
	"m":      "\U0000e61e",
	"mm":     "\U0000e61d",
	"cs":     "\U000f031b",
	"csx":    "\U000f031b",
	"csproj": "\U000f031b",
	"fs":     "\U0000e7a7",
	"fsi":    "\U0000e7a7",
	"fsx":    "\U0000e7a7",
	"fsproj": "\U0000e7a7",
	"vb":     "\U000f031b",
	"pl":     "\U0000e67e",
	"pm":     "\U0000e67e",
	"t":      "\U0000e67e",
	"pod":    "\U0000e67e",
	"plx":    "\U0000e67e",
	"lua":    "\U0000e620",
	"luac":   "\U0000e620",
	"luau":   "\U0000e620",
	"r":      "\U0000e68a",
	"rdata":  "\U0000e68a",
	"rds":    "\U0000e68a",
	"jl":     "\U0000e624",
	"hs":     "\U0000e777",
	"lhs":    "\U0000e777",
	"elm":    "\U0000e62c",
	"ex":     "\U0000e62d",
	"exs":    "\U0000e62d",
	"eex":    "\U0000e62d",
	"leex":   "\U0000e62d",
	"erl":    "\U0000e7b1",
	"hrl":    "\U0000e7b1",
	"clj":    "\U0000e768",
	"cljc":   "\U0000e768",
	"cljs":   "\U0000e76a",
	"edn":    "\U0000e76a",
	"dart":   "\U0000e798",
	"zig":    "\U0000e6a9",
	"nim":    "\U0000e677",
	"nimble": "\U0000e677",
	"nims":   "\U0000e677",
	"v":      "\U0000e6ac",
	"d":      "\U0000e7af",
	"di":     "\U0000e7af",
	"cr":     "\U0000e62f",
	"groovy": "\U0000e775",
	"gvy":    "\U0000e775",
	"gradle": "\U0000e660",
	"asm":    "\U0000e637",
	"s":      "\U0000e637",
	"f":      "\U000f121a",
	"for":    "\U000f121a",
	"f90":    "\U000f121a",
	"cob":    "\U0000e657",
	"cbl":    "\U0000e657",
	"lisp":   "\U0000f0172",
	"el":     "\U0000e632",
	"elc":    "\U0000e632",
	"scm":    "\U0000e6b1",
	"ss":     "\U0000e6b1",
	"rkt":    "\U0000e6b1",
	"sld":    "\U0000e6b1",
	"ml":     "\U0000e67a",
	"mli":    "\U0000e67a",
	"mll":    "\U0000e67a",
	"mly":    "\U0000e67a",
	"purs":   "\U0000e630",
	"re":     "\U0000e6b0",
	"rei":    "\U0000e6b0",
	"fnl":    "\U0000e6af",
	"gleam":  "\U0000f09a5",
	"cu":     "\U0000e64b",

	// Web
	"html":   "\U0000f13b",
	"htm":    "\U0000f13b",
	"xhtml":  "\U0000f13b",
	"shtml":  "\U0000f13b",
	"css":    "\U0000e749",
	"scss":   "\U0000e603",
	"sass":   "\U0000e603",
	"less":   "\U0000e758",
	"styl":   "\U0000e600",
	"vue":    "\U0000f0844",
	"svelte": "\U0000e697",
	"astro":  "\U0000e6b3",

	// Data/Config
	"json":    "\U0000e60b",
	"json5":   "\U0000e60b",
	"jsonc":   "\U0000e60b",
	"yaml":    "\U0000e6a8",
	"yml":     "\U0000e6a8",
	"toml":    "\U0000e6b2",
	"xml":     "\U0000f05c0",
	"xul":     "\U0000f05c0",
	"ini":     "\U000f107b",
	"cfg":     "\U000f107b",
	"conf":    "\U000f107b",
	"config":  "\U000f107b",
	"env":     "\U0000f462",
	"csv":     "\U0000f1c3",
	"tsv":     "\U0000f1c3",
	"sql":     "\U0000f1c0",
	"db":      "\U0000f1c0",
	"db3":     "\U0000e7c4",
	"sqlite":  "\U0000e7c4",
	"sqlite3": "\U0000e7c4",
	"s3db":    "\U0000e7c4",
	"mdb":     "\U0000f1c0",
	"ldb":     "\U0000f1c0",
	"rdb":     "\U0000e76d",
	"dump":    "\U0000f1c0",
	"graphql": "\U0000e662",
	"gql":     "\U0000e662",
	"avro":    "\U0000e60b",
	"proto":   "\U0000e60b",
	"prql":    "\U0000f1c0",

	// Shell/Scripts
	"sh":   "\U0000f489",
	"bash": "\U0000f489",
	"zsh":  "\U0000f489",
	"fish": "\U0000f489",
	"ksh":  "\U0000f489",
	"csh":  "\U0000f489",
	"nu":   "\U0000f489",
	"awk":  "\U0000f489",
	"bats": "\U0000f489",
	"ps1":  "\U0000ebc7",
	"psm1": "\U0000ebc7",
	"psd1": "\U0000ebc7",
	"bat":  "\U0000ebc4",
	"cmd":  "\U0000f17a",

	// Markup/Docs
	"md":       "\U0000f48a",
	"markdown": "\U0000f48a",
	"mdx":      "\U0000f48a",
	"mkd":      "\U0000f48a",
	"rmd":      "\U0000f48a",
	"jmd":      "\U0000f48a",
	"rdoc":     "\U0000f48a",
	"rst":      "\U0000f15c",
	"txt":      "\U0000f15c",
	"rtf":      "\U0000f15c",
	"tex":      "\U0000e69b",
	"ltx":      "\U0000e69b",
	"latex":    "\U0000e69b",
	"sty":      "\U0000e69b",
	"cls":      "\U0000e69b",
	"bst":      "\U0000e69b",
	"bib":      "\U0000e69b",
	"org":      "\U0000e633",
	"norg":     "\U0000e847",
	"typ":      "\U0000f37f",
	"adoc":     "\U0000f718",
	"asciidoc": "\U0000f718",

	// Documents
	"pdf":  "\U0000f1c1",
	"doc":  "\U0000f1c2",
	"docx": "\U0000f1c2",
	"docm": "\U0000f1c2",
	"odt":  "\U0000f37c",
	"fodt": "\U0000f37c",
	"xls":  "\U0000f1c3",
	"xlsx": "\U0000f1c3",
	"xlsm": "\U0000f1c3",
	"xlr":  "\U0000f1c3",
	"ods":  "\U0000f378",
	"fods": "\U0000f378",
	"ppt":  "\U0000f1c4",
	"pptx": "\U0000f1c4",
	"pps":  "\U0000f1c4",
	"ppsx": "\U0000f1c4",
	"odp":  "\U0000f37a",
	"fodp": "\U0000f37a",
	"odg":  "\U0000f379",
	"fodg": "\U0000f379",
	"odf":  "\U0000f37b",
	"epub": "\U0000e28b",
	"mobi": "\U0000e28b",
	"djvu": "\U0000f1c2",
	"djv":  "\U0000f1c2",

	// Images
	"png":  "\U0000f1c5",
	"jpg":  "\U0000f1c5",
	"jpeg": "\U0000f1c5",
	"jpe":  "\U0000f1c5",
	"jfi":  "\U0000f1c5",
	"jfif": "\U0000f1c5",
	"jif":  "\U0000f1c5",
	"gif":  "\U0000f1c5",
	"bmp":  "\U0000f1c5",
	"tif":  "\U0000f1c5",
	"tiff": "\U0000f1c5",
	"webp": "\U0000f1c5",
	"ico":  "\U0000f1c5",
	"svg":  "\U0000f0559",
	"eps":  "\U0000f0559",
	"ps":   "\U0000f0559",
	"ai":   "\U0000e7b4",
	"psd":  "\U0000e7b8",
	"psb":  "\U0000e7b8",
	"xcf":  "\U0000f338",
	"kra":  "\U0000f33d",
	"krz":  "\U0000f33d",
	"kpp":  "\U0000f33d",
	"raw":  "\U0000f1c5",
	"cr2":  "\U0000f1c5",
	"nef":  "\U0000f1c5",
	"orf":  "\U0000f1c5",
	"arw":  "\U0000f1c5",
	"heic": "\U0000f1c5",
	"heif": "\U0000f1c5",
	"avif": "\U0000f1c5",
	"jxl":  "\U0000f1c5",
	"jp2":  "\U0000f1c5",
	"j2k":  "\U0000f1c5",
	"j2c":  "\U0000f1c5",
	"jpf":  "\U0000f1c5",
	"jpx":  "\U0000f1c5",
	"pbm":  "\U0000f1c5",
	"pgm":  "\U0000f1c5",
	"ppm":  "\U0000f1c5",
	"pnm":  "\U0000f1c5",
	"pxm":  "\U0000f1c5",
	"xpm":  "\U0000f1c5",
	"dvi":  "\U0000f1c5",
	"cbr":  "\U0000f1c5",
	"cbz":  "\U0000f1c5",

	// Audio
	"mp3":  "\U0000f001",
	"m4a":  "\U0000f001",
	"aac":  "\U0000f001",
	"ogg":  "\U0000f001",
	"opus": "\U0000f001",
	"flac": "\U0000f001",
	"alac": "\U0000f001",
	"wav":  "\U0000f001",
	"aif":  "\U0000f001",
	"aiff": "\U0000f001",
	"aifc": "\U0000f001",
	"ape":  "\U0000f001",
	"wma":  "\U0000f001",
	"wv":   "\U0000f001",
	"mp2":  "\U0000f001",
	"mka":  "\U0000f001",
	"pcm":  "\U0000f001",
	"mid":  "\U0000f08f2",
	"sf2":  "\U0000f0f70",
	"sfz":  "\U0000f0f70",

	// Video
	"mp4":   "\U0000f03d",
	"m4v":   "\U0000f03d",
	"mkv":   "\U0000f03d",
	"webm":  "\U0000f03d",
	"avi":   "\U0000f03d",
	"mov":   "\U0000f03d",
	"wmv":   "\U0000f03d",
	"flv":   "\U0000f03d",
	"ogv":   "\U0000f03d",
	"ogm":   "\U0000f03d",
	"vob":   "\U0000f03d",
	"mpg":   "\U0000f03d",
	"mpeg":  "\U0000f03d",
	"m2v":   "\U0000f03d",
	"m2ts":  "\U0000f03d",
	"3gp":   "\U0000f03d",
	"3g2":   "\U0000f03d",
	"3gpp":  "\U0000f03d",
	"3gpp2": "\U0000f03d",
	"3gp2":  "\U0000f03d",
	"h264":  "\U0000f03d",
	"heics": "\U0000f03d",
	"cast":  "\U0000f03d",
	"swf":   "\U0000f001",

	// Archives
	"zip":  "\U0000f410",
	"tar":  "\U0000f410",
	"gz":   "\U0000f410",
	"gzip": "\U0000f410",
	"bz":   "\U0000f410",
	"bz2":  "\U0000f410",
	"bz3":  "\U0000f410",
	"xz":   "\U0000f410",
	"lz":   "\U0000f410",
	"lz4":  "\U0000f410",
	"lzma": "\U0000f410",
	"lzo":  "\U0000f410",
	"lzh":  "\U0000f410",
	"zst":  "\U0000f410",
	"7z":   "\U0000f410",
	"rar":  "\U0000f410",
	"tgz":  "\U0000f410",
	"tbz":  "\U0000f410",
	"tbz2": "\U0000f410",
	"txz":  "\U0000f410",
	"tlz":  "\U0000f410",
	"taz":  "\U0000f410",
	"tz":   "\U0000f410",
	"tzo":  "\U0000f410",
	"z":    "\U0000f410",
	"ar":   "\U0000f410",
	"arj":  "\U0000f410",
	"cpio": "\U0000f410",
	"par":  "\U0000f410",
	"br":   "\U0000f410",
	"cab":  "\U0000f17a",
	"deb":  "\U0000e77d",
	"rpm":  "\U0000e7bb",
	"apk":  "\U0000e70e",
	"pkg":  "\U0000eb29",
	"dmg":  "\U0000e271",
	"iso":  "\U0000e271",
	"img":  "\U0000e271",
	"msi":  "\U0000f17a",
	"exe":  "\U0000ebc4",

	// Fonts
	"ttf":   "\U0000f031",
	"ttc":   "\U0000f031",
	"otf":   "\U0000f031",
	"woff":  "\U0000f031",
	"woff2": "\U0000f031",
	"eot":   "\U0000f031",
	"fon":   "\U0000f031",
	"fnt":   "\U0000f031",
	"bdf":   "\U0000f031",
	"psf":   "\U0000f031",
	"flc":   "\U0000f031",
	"flf":   "\U0000f031",
	"lff":   "\U0000f031",
	"font":  "\U0000f031",

	// Templates
	"ejs":      "\U0000e618",
	"hbs":      "\U0000e60f",
	"mustache": "\U0000e60f",
	"pug":      "\U0000e686",
	"jade":     "\U0000e66c",
	"slim":     "\U0000e73b",
	"haml":     "\U0000e664",
	"twig":     "\U0000e61c",
	"blade":    "\U0000e73d",
	"razor":    "\U0000f1fa",
	"cshtml":   "\U0000f1fa",

	// DevOps/Infra
	"tf":      "\U0000f1062",
	"tfstate": "\U0000f1062",
	"tfvars":  "\U0000f1062",
	"hcl":     "\U0000f1062",
	"vagrant": "\U0000e21e",
	"ansible": "\U0000e77c",
	"puppet":  "\U0000e631",

	// Build/Project
	"cmake": "\U0000e794",
	"make":  "\U0000e673",
	"mk":    "\U0000e673",
	"ninja": "\U0000f0774",
	"gyp":   "\U0000e7a3",
	"sbt":   "\U0000f0a16",
	"maven": "\U0000e7b6",
	"ant":   "\U0000e7a0",
	"bazel": "\U0000e63a",

	// Lock files
	"lock": "\U0000f023",
	"lck":  "\U0000f023",

	// Logs
	"log": "\U0000f18d",

	// Keys/Certs
	"key":  "\U0000eb11",
	"pem":  "\U0000eb11",
	"crt":  "\U0000eafa",
	"cert": "\U0000eafa",
	"pfx":  "\U0000eb11",
	"p12":  "\U0000eb11",
	"gpg":  "\U000f099d",
	"asc":  "\U000f099d",
	"sig":  "\U0000f19c3",
	"pub":  "\U0000f0dd6",
	"kbx":  "\U0000f0bc4",
	"kdb":  "\U0000f23e",
	"kdbx": "\U0000f23e",
	"age":  "\U000f099d",

	// Misc
	"bin":      "\U0000eae8",
	"o":        "\U0000eae8",
	"a":        "\U0000f17c",
	"so":       "\U0000f17c",
	"ko":       "\U0000f17c",
	"dylib":    "\U0000f179",
	"dll":      "\U0000eb9c",
	"lib":      "\U0000eb9c",
	"out":      "\U0000eb2c",
	"elf":      "\U0000eae8",
	"hi":       "\U0000eae8",
	"rlib":     "\U0000e68b",
	"rmeta":    "\U0000e68b",
	"hex":      "\U0000f12a7",
	"diff":     "\U0000f440",
	"patch":    "\U0000f440",
	"torrent":  "\U0000e275",
	"magnet":   "\U0000f076",
	"desktop":  "\U0000ebd1",
	"service":  "\U0000eba2",
	"nfo":      "\U0000f129",
	"info":     "\U0000f129",
	"license":  "\U0000f02d",
	"eml":      "\U0000f003",
	"ics":      "\U0000eab0",
	"ical":     "\U0000eab0",
	"ifb":      "\U0000eab0",
	"cache":    "\U0000f49b",
	"tmp":      "\U0000f49b",
	"bak":      "\U0000f49b",
	"swp":      "\U0000f49b",
	"part":     "\U000f01da",
	"download": "\U000f01da",
	"qm":       "\U000f05ca",
	"po":       "\U000f05ca",
	"pot":      "\U000f05ca",
	"mo":       "\U000f05ca",

	// Game dev
	"unity":   "\U0000e721",
	"unity3d": "\U0000e721",
	"godot":   "\U0000e65f",
	"gd":      "\U0000e65f",
	"tscn":    "\U0000e65f",
	"tres":    "\U0000e65f",
	"blend":   "\U000f00ab",
	"fbx":     "\U0000f01a7",
	"obj":     "\U0000f01a7",
	"stl":     "\U0000f01a7",
	"ply":     "\U0000f01a7",
	"gltf":    "\U0000f01a7",
	"glb":     "\U0000f01a7",
	"wrl":     "\U0000f01a7",
	"wrz":     "\U0000f01a7",
	"3mf":     "\U0000f01a7",

	// CAD
	"dwg":        "\U000f0eeb",
	"dxf":        "\U000f0eeb",
	"step":       "\U000f0eeb",
	"stp":        "\U000f0eeb",
	"ste":        "\U000f0eeb",
	"iges":       "\U000f0eeb",
	"igs":        "\U000f0eeb",
	"ige":        "\U000f0eeb",
	"brep":       "\U000f0eeb",
	"fcstd":      "\U0000f336",
	"fcstd1":     "\U0000f336",
	"fcbak":      "\U0000f336",
	"scad":       "\U0000f34e",
	"skp":        "\U000f0eeb",
	"ipt":        "\U000f0eeb",
	"iam":        "\U000f0eeb",
	"catpart":    "\U000f0eeb",
	"catproduct": "\U000f0eeb",
	"sldprt":     "\U000f0eeb",
	"sldasm":     "\U000f0eeb",
	"slvs":       "\U000f0eeb",
	"psm":        "\U000f0eeb",
	"x_b":        "\U000f0eeb",
	"x_t":        "\U000f0eeb",
	"3dm":        "\U000f0eeb",
	"123dx":      "\U000f0eeb",
	"f3d":        "\U000f0eeb",
	"f3z":        "\U000f0eeb",
	"ifc":        "\U000f0eeb",

	// Electronics
	"sch":       "\U0000f0b45",
	"schdoc":    "\U0000f0b45",
	"brd":       "\U0000eabe",
	"pcbdoc":    "\U0000eabe",
	"prjpcb":    "\U0000eabe",
	"kicad_sch": "\U0000f34c",
	"kicad_pcb": "\U0000f34c",
	"kicad_mod": "\U0000f34c",
	"kicad_sym": "\U0000f34c",
	"kicad_pro": "\U0000f34c",
	"kicad_dru": "\U0000f34c",
	"kicad_wks": "\U0000f34c",
	"kicad_prl": "\U0000f34c",
	"gbr":       "\U0000eabe",
	"gbl":       "\U0000eabe",
	"gbo":       "\U0000eabe",
	"gbp":       "\U0000eabe",
	"gbs":       "\U0000eabe",
	"gtl":       "\U0000eabe",
	"gto":       "\U0000eabe",
	"gtp":       "\U0000eabe",
	"gts":       "\U0000eabe",
	"gm1":       "\U0000eabe",
	"gml":       "\U0000eabe",
	"lpp":       "\U0000eabe",
	"ino":       "\U0000f34b",
	"gcode":     "\U0000f0af4",

	// Notebooks
	"ipynb": "\U0000e678",

	// Subtitles
	"srt": "\U0000f0a16",
	"sub": "\U0000f0a16",
	"ass": "\U0000f0a16",
	"ssa": "\U0000f0a16",
	"lrc": "\U0000f0a16",

	// Playlists
	"m3u":  "\U0000f0cb9",
	"m3u8": "\U0000f0cb9",
	"pls":  "\U0000f0cb9",
	"cue":  "\U0000f0cb9",

	// Qt
	"qml": "\U0000f375",
	"qrc": "\U0000f375",
	"qss": "\U0000f375",
	"ui":  "\U0000f2d0",

	// Virtualization
	"qcow":  "\U0000e271",
	"qcow2": "\U0000e271",
	"vdi":   "\U0000e271",
	"vmdk":  "\U0000e271",
	"tc":    "\U0000e271",

	// Hashes
	"md5":    "\U0000f0565",
	"sha1":   "\U0000f0565",
	"sha224": "\U0000f0565",
	"sha256": "\U0000f0565",
	"sha384": "\U0000f0565",
	"sha512": "\U0000f0565",

	// Misc dev
	"nix":         "\U0000f313",
	"ebuild":      "\U0000f30d",
	"opam":        "\U0000f0627",
	"conda":       "\U0000e715",
	"whl":         "\U0000e606",
	"gem":         "\U0000e739",
	"gemspec":     "\U0000e739",
	"drawio":      "\U0000ebba",
	"dot":         "\U0000f1049",
	"gv":          "\U0000f1049",
	"rss":         "\U0000f09e",
	"opml":        "\U0000f05c0",
	"applescript": "\U0000f179",
	"vhs":         "\uF0A1B",

	// Verilog/VHDL
	"sv":   "\U000f035b",
	"svh":  "\U000f035b",
	"vhd":  "\U000f035b",
	"vhdl": "\U000f035b",
	"vi":   "\U0000e81e",

	// Vim
	"vim": "\U0000e7c5",

	// Tcl
	"tcl": "\U0000f06d3",
	"tbc": "\U0000f06d3",

	// Steam
	"acf": "\U0000f1b6",

	// ROM
	"gba": "\U0000f1393",
	"nsp": "\uF07E1",
	"xci": "\uF07E1",
	"z64": "\U0000f1393",

	// Salesforce
	"sal": "\U000f147b",
	"sr":  "\U000f147b",
}

// Filenames maps specific filenames to Nerd Font icons.
var Filenames = map[string]string{
	// Git
	".gitignore":             "\U0000f02a2",
	".gitattributes":         "\U0000f02a2",
	".gitmodules":            "\U0000f02a2",
	".gitconfig":             "\U0000f02a2",
	".gitkeep":               "\U0000f02a2",
	".git-blame-ignore-revs": "\U0000f02a2",

	// Docker
	"dockerfile":          "\U0000e650",
	"Dockerfile":          "\U0000e650",
	"docker-compose.yml":  "\U0000e650",
	"docker-compose.yaml": "\U0000e650",
	".dockerignore":       "\U0000e650",

	// Node/JS
	"package.json":      "\U0000e71e",
	"package-lock.json": "\U0000e71e",
	".npmrc":            "\U0000e71e",
	".npmignore":        "\U0000e71e",
	"yarn.lock":         "\U0000e718",
	".yarnrc":           "\U0000e718",
	".yarnrc.yml":       "\U0000e718",
	"pnpm-lock.yaml":    "\U0000e71e",
	".nvmrc":            "\U0000e718",
	"node_modules":      "\U0000e5fa",
	".node-version":     "\U0000e718",
	"tsconfig.json":     "\U0000e628",
	"jsconfig.json":     "\U0000e74e",
	".eslintrc":         "\U0000e655",
	".eslintrc.js":      "\U0000e655",
	".eslintrc.json":    "\U0000e655",
	".eslintrc.yml":     "\U0000e655",
	".eslintignore":     "\U0000e655",
	".prettierrc":       "\U0000e6b4",
	".prettierrc.js":    "\U0000e6b4",
	".prettierrc.json":  "\U0000e6b4",
	".prettierrc.yml":   "\U0000e6b4",
	".prettierignore":   "\U0000e6b4",
	"webpack.config.js": "\U0000e72e",
	"rollup.config.js":  "\U0000e72e",
	"vite.config.js":    "\U0000e72e",
	"vite.config.ts":    "\U0000e72e",
	"babel.config.js":   "\U0000e72e",
	".babelrc":          "\U0000e72e",
	"next.config.js":    "\U0000e7a3",
	"nuxt.config.js":    "\U0000e6a0",
	"svelte.config.js":  "\U0000e697",

	// Ruby
	"Gemfile":       "\U0000e739",
	"Gemfile.lock":  "\U0000e739",
	"Rakefile":      "\U0000e739",
	".rubocop.yml":  "\U0000e739",
	".ruby-version": "\U0000e739",
	"Guardfile":     "\U0000e739",
	"Procfile":      "\U0000e739",
	".rspec":        "\U0000e739",

	// Python
	"requirements.txt": "\U0000e606",
	"setup.py":         "\U0000e606",
	"setup.cfg":        "\U0000e606",
	"pyproject.toml":   "\U0000e606",
	"Pipfile":          "\U0000e606",
	"Pipfile.lock":     "\U0000e606",
	".python-version":  "\U0000e606",
	"tox.ini":          "\U0000e606",
	".flake8":          "\U0000e606",
	"pytest.ini":       "\U0000e606",
	"mypy.ini":         "\U0000e606",
	".pylintrc":        "\U0000e606",
	"poetry.lock":      "\U0000e606",

	// Go
	"go.mod":  "\U0000e65e",
	"go.sum":  "\U0000e65e",
	"go.work": "\U0000e65e",

	// Rust
	"Cargo.toml":          "\U0000e68b",
	"Cargo.lock":          "\U0000e68b",
	"rust-toolchain":      "\U0000e68b",
	"rust-toolchain.toml": "\U0000e68b",

	// Java/JVM
	"pom.xml":          "\U0000e7b6",
	"build.gradle":     "\U0000e660",
	"build.gradle.kts": "\U0000e660",
	"settings.gradle":  "\U0000e660",
	"gradlew":          "\U0000e660",
	"mvnw":             "\U0000e7b6",
	".java-version":    "\U0000e256",

	// Build
	"Makefile":        "\U0000e673",
	"makefile":        "\U0000e673",
	"GNUmakefile":     "\U0000e673",
	"CMakeLists.txt":  "\U0000e794",
	"Justfile":        "\U0000e673",
	"justfile":        "\U0000e673",
	"Taskfile.yml":    "\U0000e673",
	"Taskfile.yaml":   "\U0000e673",
	"BUILD":           "\U0000e63a",
	"BUILD.bazel":     "\U0000e63a",
	"WORKSPACE":       "\U0000e63a",
	"WORKSPACE.bazel": "\U0000e63a",
	"meson.build":     "\U0000e673",
	"SConstruct":      "\U0000e606",
	"SConscript":      "\U0000e606",
	"build.ninja":     "\U0000f0774",
	"configure":       "\U0000e673",
	"configure.ac":    "\U0000e673",
	"Makefile.am":     "\U0000e673",

	// CI/CD
	".travis.yml":         "\U0000e77e",
	".gitlab-ci.yml":      "\U0000f296",
	"Jenkinsfile":         "\U0000e767",
	"azure-pipelines.yml": "\U0000f3ca",
	".circleci":           "\U0000e765",
	"appveyor.yml":        "\U0000e77f",
	".github":             "\U0000e5fd",
	"codecov.yml":         "\U0000e769",
	".coveralls.yml":      "\U0000e769",

	// Editor/IDE
	".editorconfig":      "\U0000e652",
	".vscode":            "\U0000e70c",
	".idea":              "\U0000e7b5",
	".vim":               "\U0000e7c5",
	".vimrc":             "\U0000e7c5",
	"_vimrc":             "\U0000e7c5",
	".gvimrc":            "\U0000e7c5",
	".nvim":              "\U0000e7c5",
	".nvimrc":            "\U0000e7c5",
	"init.vim":           "\U0000e7c5",
	"init.lua":           "\U0000e7c5",
	".emacs":             "\U0000e632",
	".emacs.d":           "\U0000e632",
	".spacemacs":         "\U0000e632",
	".atom":              "\U0000e7b0",
	".sublime-project":   "\U0000e7aa",
	".sublime-workspace": "\U0000e7aa",

	// Shell
	".bashrc":       "\U0000f489",
	".bash_profile": "\U0000f489",
	".bash_history": "\U0000f489",
	".bash_logout":  "\U0000f489",
	".profile":      "\U0000f489",
	".zshrc":        "\U0000f489",
	".zshenv":       "\U0000f489",
	".zprofile":     "\U0000f489",
	".zsh_history":  "\U0000f489",
	".zsh_sessions": "\U0000f489",
	".fishrc":       "\U0000f489",
	"config.fish":   "\U0000f489",
	".inputrc":      "\U0000f489",
	".xinitrc":      "\U0000f489",
	".xprofile":     "\U0000f489",
	".Xresources":   "\U0000f489",
	".Xauthority":   "\U0000f489",

	// Docs
	"README":             "\U000f00ba",
	"README.md":          "\U000f00ba",
	"README.txt":         "\U000f00ba",
	"README.rst":         "\U000f00ba",
	"readme.md":          "\U000f00ba",
	"CHANGELOG":          "\U0000f4a2",
	"CHANGELOG.md":       "\U0000f4a2",
	"HISTORY":            "\U0000f4a2",
	"HISTORY.md":         "\U0000f4a2",
	"CHANGES":            "\U0000f4a2",
	"AUTHORS":            "\U0000f4a2",
	"CONTRIBUTORS":       "\U0000f4a2",
	"MAINTAINERS":        "\U0000f4a2",
	"LICENSE":            "\U0000f02d",
	"LICENSE.md":         "\U0000f02d",
	"LICENSE.txt":        "\U0000f02d",
	"LICENCE":            "\U0000f02d",
	"COPYING":            "\U0000f02d",
	"COPYRIGHT":          "\U0000f02d",
	"NOTICE":             "\U0000f02d",
	"PATENTS":            "\U0000f02d",
	"TODO":               "\U0000f0ae",
	"TODO.md":            "\U0000f0ae",
	"SECURITY":           "\U0000f49c",
	"SECURITY.md":        "\U0000f49c",
	"CODE_OF_CONDUCT":    "\U0000f49c",
	"CODE_OF_CONDUCT.md": "\U0000f49c",
	"CONTRIBUTING":       "\U0000f49c",
	"CONTRIBUTING.md":    "\U0000f49c",
	"CODEOWNERS":         "\U0000f0c0",
	"FUNDING.yml":        "\U0000f49c",

	// Ignore files (non-duplicates)
	".hgignore":        "\U0000f1c0",
	".stylelintignore": "\U0000e749",
	".vscodeignore":    "\U0000e70c",
	".ignore":          "\U0000f013",
	".fdignore":        "\U0000f013",
	".rgignore":        "\U0000f013",
	".agignore":        "\U0000f013",

	// Terraform
	"terraform.tf":        "\U0000f1062",
	"terraform.tfvars":    "\U0000f1062",
	"main.tf":             "\U0000f1062",
	"variables.tf":        "\U0000f1062",
	"outputs.tf":          "\U0000f1062",
	"providers.tf":        "\U0000f1062",
	"versions.tf":         "\U0000f1062",
	".terraform.lock.hcl": "\U0000f1062",

	// Kubernetes
	"Chart.yaml":         "\U0000fd31",
	"values.yaml":        "\U0000fd31",
	"kustomization.yaml": "\U0000fd31",

	// Nix
	"flake.nix":                  "\U0000f313",
	"flake.lock":                 "\U0000f313",
	"default.nix":                "\U0000f313",
	"shell.nix":                  "\U0000f313",
	"configuration.nix":          "\U0000f313",
	"hardware-configuration.nix": "\U0000f313",

	// Misc config
	".env":                  "\U0000f462",
	".env.local":            "\U0000f462",
	".env.example":          "\U0000f462",
	".env.development":      "\U0000f462",
	".env.production":       "\U0000f462",
	".env.test":             "\U0000f462",
	"netlify.toml":          "\U0000e7a3",
	"vercel.json":           "\U0000e7a3",
	"fly.toml":              "\U0000e7a3",
	"now.json":              "\U0000e7a3",
	"firebase.json":         "\U0000e7a3",
	"capacitor.config.json": "\U0000e7a3",
	"angular.json":          "\U0000e753",
	"browserslist":          "\U0000e796",
	".browserslistrc":       "\U0000e796",
	"robots.txt":            "\U0000f06e",
	"humans.txt":            "\U0000f06e",
	"sitemap.xml":           "\U0000f06e",
	"favicon.ico":           "\U0000e623",
	"manifest.json":         "\U0000e60b",
	"composer.json":         "\U0000e73d",
	"composer.lock":         "\U0000e73d",
	"phpunit.xml":           "\U0000e73d",
	".php_cs":               "\U0000e73d",
	".php-version":          "\U0000e73d",
	"mix.exs":               "\U0000e62d",
	"mix.lock":              "\U0000e62d",
	"rebar.config":          "\U0000e7b1",
	"erlang.mk":             "\U0000e7b1",
	"stack.yaml":            "\U0000e777",
	"cabal.project":         "\U0000e777",
	"dune":                  "\U0000e67a",
	"dune-project":          "\U0000e67a",
	".ocamlformat":          "\U0000e67a",
	"Vagrantfile":           "\U0000e21e",
	"Berksfile":             "\U0000e60b",
	"Cheffile":              "\U0000e60b",
	"Puppetfile":            "\U0000e631",
	"Brewfile":              "\U0000e7a4",
	"Brewfile.lock.json":    "\U0000e7a4",
	"Podfile":               "\U0000e7a4",
	"Podfile.lock":          "\U0000e7a4",
	"Cartfile":              "\U0000e7a4",
	"Cartfile.resolved":     "\U0000e7a4",
	"build.zig":             "\U0000e6a9",
	"build.zig.zon":         "\U0000e6a9",
}

// Directories maps directory names to Nerd Font icons.
var Directories = map[string]string{
	".git":             "\U0000f02a2",
	".github":          "\U0000e5fd",
	".gitlab":          "\U0000f296",
	".svn":             "\U0000e72d",
	".hg":              "\U0000e729",
	".bzr":             "\U0000e72d",
	"node_modules":     "\U0000e5fa",
	"vendor":           "\U0000f126",
	"bower_components": "\U0000e74d",
	".vscode":          "\U0000e70c",
	".idea":            "\U0000e7b5",
	".vim":             "\U0000e7c5",
	".emacs.d":         "\U0000e632",
	".atom":            "\U0000e7b0",
	".config":          "\U000f107b",
	".local":           "\U0000f07c",
	".cache":           "\U0000f49b",
	".npm":             "\U0000e71e",
	".yarn":            "\U0000e718",
	".cargo":           "\U0000e68b",
	".rustup":          "\U0000e68b",
	".go":              "\U0000e65e",
	".pyenv":           "\U0000e606",
	".rbenv":           "\U0000e739",
	".nvm":             "\U0000e718",
	".docker":          "\U0000e650",
	".kube":            "\U0000fd31",
	".terraform":       "\U0000f1062",
	".aws":             "\U0000e7ad",
	".azure":           "\U0000f3ca",
	".gcloud":          "\U0000e7b2",
	".ssh":             "\U0000f09c",
	".gnupg":           "\U0000f09c",
	"__pycache__":      "\U0000e606",
	".pytest_cache":    "\U0000e606",
	".mypy_cache":      "\U0000e606",
	".tox":             "\U0000e606",
	".eggs":            "\U0000e606",
	".venv":            "\U0000e606",
	"venv":             "\U0000e606",
	"env":              "\U0000e606",
	"virtualenv":       "\U0000e606",
	".nix-defexpr":     "\U0000f313",
	".nix-profile":     "\U0000f313",
	"target":           "\U0000e68b",
	"build":            "\U000f19fc",
	"dist":             "\U000f19fc",
	"out":              "\U000f19fc",
	"output":           "\U000f19fc",
	"bin":              "\U000f19fc",
	"obj":              "\U000f19fc",
	"lib":              "\U000f19fc",
	"libs":             "\U000f19fc",
	"include":          "\U000f19fc",
	"src":              "\U000f08de",
	"source":           "\U000f08de",
	"sources":          "\U000f08de",
	"app":              "\U0000f07c",
	"apps":             "\U0000f07c",
	"public":           "\U0000f07c",
	"static":           "\U0000f07c",
	"assets":           "\U0000f07c",
	"resources":        "\U0000f07c",
	"res":              "\U0000f07c",
	"images":           "\U0000f1c5",
	"img":              "\U0000f1c5",
	"icons":            "\U0000f1c5",
	"fonts":            "\U0000f031",
	"styles":           "\U0000e749",
	"css":              "\U0000e749",
	"scripts":          "\U0000f489",
	"js":               "\U0000e74e",
	"test":             "\U0000f0c3",
	"tests":            "\U0000f0c3",
	"spec":             "\U0000f0c3",
	"specs":            "\U0000f0c3",
	"__tests__":        "\U0000f0c3",
	"docs":             "\U0000f02d",
	"doc":              "\U0000f02d",
	"documentation":    "\U0000f02d",
	"man":              "\U0000f02d",
	"info":             "\U0000f02d",
	"examples":         "\U0000f02d",
	"example":          "\U0000f02d",
	"samples":          "\U0000f02d",
	"sample":           "\U0000f02d",
	"tmp":              "\U0000f49b",
	"temp":             "\U0000f49b",
	"logs":             "\U0000f18d",
	"log":              "\U0000f18d",
	"backup":           "\U0000f49b",
	"backups":          "\U0000f49b",
	"data":             "\U0000f1c0",
	"db":               "\U0000f1c0",
	"database":         "\U0000f1c0",
	"migrations":       "\U0000f1c0",
	"seeds":            "\U0000f1c0",
	"fixtures":         "\U0000f1c0",
	"config":           "\U000f107b",
	"configs":          "\U000f107b",
	"configuration":    "\U000f107b",
	"settings":         "\U000f107b",
	"etc":              "\U000f107b",
	"ci":               "\U0000e77e",
	".circleci":        "\U0000e765",
	"hooks":            "\U0000f069",
	"plugins":          "\U0000f069",
	"addons":           "\U0000f069",
	"modules":          "\U0000f069",
	"packages":         "\U0000f069",
	"components":       "\U0000f069",
	"views":            "\U0000f06e",
	"pages":            "\U0000f06e",
	"templates":        "\U0000f06e",
	"layouts":          "\U0000f06e",
	"partials":         "\U0000f06e",
	"helpers":          "\U0000f4a5",
	"utils":            "\U0000f4a5",
	"utilities":        "\U0000f4a5",
	"common":           "\U0000f4a5",
	"shared":           "\U0000f4a5",
	"core":             "\U0000f4a5",
	"internal":         "\U0000f4a5",
	"pkg":              "\U0000f4a5",
	"cmd":              "\U0000f4a5",
	"api":              "\U0000f4a5",
	"services":         "\U0000f4a5",
	"controllers":      "\U0000f4a5",
	"models":           "\U0000f4a5",
	"middleware":       "\U0000f4a5",
	"handlers":         "\U0000f4a5",
	"routes":           "\U0000f4a5",
	"locales":          "\U000f05ca",
	"i18n":             "\U000f05ca",
	"translations":     "\U000f05ca",
	"Desktop":          "\U0000f108",
	"Documents":        "\U0000f02d",
	"Downloads":        "\U0000f019",
	"Music":            "\U0000f001",
	"Pictures":         "\U0000f1c5",
	"Videos":           "\U0000f03d",
	"Movies":           "\U0000f03d",
	"home":             "\U0000f10b5",
	"Home":             "\U0000f10b5",
	"root":             "\U0000f023",
	"usr":              "\U0000f007",
	"var":              "\U0000f07c",
	"opt":              "\U0000f07c",
	"media":            "\U0000f07c",
	"mnt":              "\U0000f07c",
	"boot":             "\U000f107b",
	"dev":              "\U000f107b",
	"proc":             "\U000f107b",
	"sys":              "\U000f107b",
	"run":              "\U000f107b",
	"srv":              "\U000f107b",
	"snap":             "\U000f107b",
	"flatpak":          "\U000f107b",
	"Applications":     "\U0000eb44",
	"Library":          "\U0000f02d",
	"System":           "\U000f107b",
	"Trash":            "\U0000f014",
	"trash":            "\U0000f014",
	".Trash":           "\U0000f014",
	"lost+found":       "\U0000f071",
}

// DefaultFileIcon is used when no specific icon is found.
const DefaultFileIcon = "\U0000f15b"

// DefaultDirIcon is used for directories without a specific icon.
const DefaultDirIcon = "\U0000f07c"

// GetIcon returns the appropriate icon for a file based on its name.
// It checks filename first, then extension, then returns the default.
func GetIcon(filename string, isDir bool) string {
	if isDir {
		if icon, ok := Directories[filename]; ok {
			return icon
		}
		return DefaultDirIcon
	}

	// Check exact filename match
	if icon, ok := Filenames[filename]; ok {
		return icon
	}

	// Check extension
	ext := getExtension(filename)
	if ext != "" {
		if icon, ok := Extensions[ext]; ok {
			return icon
		}
	}

	return DefaultFileIcon
}

// getExtension returns the file extension without the leading dot.
func getExtension(filename string) string {
	for i := len(filename) - 1; i >= 0; i-- {
		if filename[i] == '.' {
			return filename[i+1:]
		}
		if filename[i] == '/' {
			break
		}
	}
	return ""
}
