// SPDX-FileCopyrightText: Copyright The Lima Authors
// SPDX-License-Identifier: Apache-2.0

package instance

import (
	"context"
	"errors"
	"fmt"
	"os"

	"github.com/lima-vm/lima/v2/pkg/driverutil"
	"github.com/lima-vm/lima/v2/pkg/limatype"
)

func Delete(ctx context.Context, inst *limatype.Instance, force bool) error {
	if inst.Protected {
		return errors.New("instance is protected to prohibit accidental removal (Hint: use `limactl unprotect`)")
	}
	if !force && inst.Status != limatype.StatusStopped {
		return fmt.Errorf("expected status %q, got %q", limatype.StatusStopped, inst.Status)
	}

	StopForcibly(inst)

	if len(inst.Errors) == 0 {
		if err := unregister(ctx, inst); err != nil {
			return fmt.Errorf("failed to unregister %q: %w", inst.Dir, err)
		}
	}
	if err := os.RemoveAll(inst.Dir); err != nil {
		return fmt.Errorf("failed to remove %q: %w", inst.Dir, err)
	}

	return nil
}

func unregister(ctx context.Context, inst *limatype.Instance) error {
	limaDriver, err := driverutil.CreateConfiguredDriver(inst, 0)
	if err != nil {
		return fmt.Errorf("failed to create driver instance: %w", err)
	}

	return limaDriver.Delete(ctx)
}
