subroutine mosaic_comm (line,error)
  use gkernel_interfaces
  use imager_interfaces, only : sub_mosaic, map_message, mosaic_show
  use clean_def
  use clean_default
  use clean_arrays
  use gbl_message
  !----------------------------------------------------------------------
  ! @  private
  !*
  ! IMAGER  -- Dispatching routine for MOSAIC
  !
  ! MOSAIC ON|OFF|GUETH|SAULT|?  or MOSAIC [Ra Dec Unit [ANGLE Angle]]  
  !     Activates or desactivates the mosaic mode, or 
  !   toggle to Sault et al mode for Mosaic imaging.
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(inout) :: line  !! Command line
  logical, intent(out) :: error            !! Logical error flag
  !
  ! Constants
  integer, parameter :: mvoc=4
  character(len=8) :: voc1(mvoc)
  data voc1/'OFF','ON','GUETH','SAULT'/
  !
  ! Local ---
  integer na,iv
  character(len=8) name,argum
  logical :: center
  !
  ! Code ----
  if (huv%loca%size.eq.0) then
    call map_message(seve%w,'MOSAIC','No UV data loaded') 
  endif
  !
  center = sic_present(0,2)
  !
  if (sic_narg(0).eq.1) then
    argum = 'ON'
    call sic_ke (line,0,1,argum,na,.false.,error)
    if (error) return
    if (argum.eq.'?') then
      call mosaic_show(huv%mos%fields)
    else
      call sic_ambigs ('MOSAIC',argum,name,iv,voc1,mvoc,error)
      if (error) return
      call sub_mosaic(name,error)
    endif
    call mosaic_set_header(line,center,error)
  else
    call mosaic_set_header(line,center,error)
  endif
end subroutine mosaic_comm
!
subroutine mosaic_show(hmos)
  use clean_arrays
  use clean_default
  use gbl_message
  use imager_interfaces, only : map_message
  use gkernel_interfaces, only : rad2sexa
  !---------------------------------------------------------------------
  ! @ private
  !!
  ! IMAGER -- Support for SHOW MOSAIC or MOSAIC ?
  !
  ! Show the MOSAIC Fields table values 
  !!
  !---------------------------------------------------------------------
  type(field_par), intent(in), allocatable :: hmos(:)  !! Mosaic Header
  !
  ! Constants
  character(len=*), parameter :: rname='SHOW'
  real, parameter :: sec=180.*3600./acos(-1.0)
  !
  ! Local
  character(len=14) :: chra,chde,cha0,chd0
  character(len=80) :: mess
  integer :: nf, if
  !
  ! Code
  nf = abs(themap%nfields)
  if (nf.eq.0) then
    call map_message(seve%i,rname,'Only a single field')
    return
  endif
  !
  if (.not.allocated(hmos)) then
    write(mess,'(A,I0,A)') 'Mosaic of ',themap%nfields,' fields with no Mosaic Header'
    call map_message(seve%i,rname,mess)
    return
  endif
  write(mess,'(A,I0,A)') 'Mosaic of ',nf,' fields.  Method '//mosaic_mode
  call map_message(seve%i,rname,mess)
  !
  write(6,'(A)') "Field    Ra              Dec           A0              D0           Nvisi"
  do if=1,nf
    call rad2sexa (hmos(if)%apoint(1),24,chra)
    call rad2sexa (hmos(if)%apoint(2),360,chde)
    call rad2sexa (hmos(if)%aphase(1),24,cha0)
    call rad2sexa (hmos(if)%aphase(2),360,chd0)
    write(6,'(I4,4(1X,A),I9)') if,chra,chde,cha0,chd0,hmos(if)%nvisi
  enddo
  write(6,'(A)') "Field     dRa     dDec     dA0      dD0    DeltaX   DeltaY    Jx    Jy  Noise"
  do if=1,nf
    write(6,'(I4,4(1X,F8.2),2(1X,F8.3),2(1X,I5),1X,1PG10.3)') if, &
      & hmos(if)%opoint(1)*sec, hmos(if)%opoint(2)*sec, &
      & hmos(if)%ophase(1)*sec, hmos(if)%ophase(2)*sec, &
      & hmos(if)%oxy(1)*sec, hmos(if)%oxy(2)*sec, &
      & hmos(if)%jxy(1), hmos(if)%jxy(2), hmos(if)%sigma
  enddo
end subroutine mosaic_show
!
subroutine sub_mosaic(name,error)
  use gkernel_interfaces
  use imager_interfaces, except_this=>sub_mosaic
  use clean_def
  use clean_default
  use clean_arrays
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !*
  ! IMAGER -- Support for MOSAIC ON|OFF  
  !             Activates or desactivates the mosaic mode
  !
  ! This should be changed to accept more keywords:  
  !     OFF, ON, GUETH, SAULT 
  !!
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: name  !! Command argument 
  logical, intent(out) :: error         !! Logical error flag
  !
  ! Constants
  real(8), parameter :: pi=3.14159265358979323846d0
  character(len=6), parameter :: rname = 'MOSAIC'
  !
  ! Local ---
  integer :: nf,nt
  logical :: old_mosaic
  real :: prim_beam(10) ! At most 10 telescopes
  character(len=message_length) :: mess
  !
  ! Code ----
  old_mosaic = user_method%mosaic
  !
  if (name.eq.'OFF') then
    ! Back to normal imaging mode
    if (old_mosaic) then
      call map_message(seve%i,rname,'Switch to NORMAL mode')
      call gprompt_set('IMAGER')
      user_method%trunca = 0.0
      call sic_delvariable('PRIMARY',.false.,error)
      hprim%gil%dim(1) = 1
      !      mosaic_mode = 'NONE'   ! Keep previous mode anyway
    endif
    user_method%mosaic = .false.
  else
    !
    if (name.ne.'ON') mosaic_mode = name  ! Specified Mosaic mode
    if (old_mosaic) then
      call map_message(seve%i,rname,'Already in MOSAIC mode, method '//mosaic_mode)
    else 
      call map_message(seve%i,rname,'Switch to MOSAIC mode, method '//mosaic_mode)
      call gprompt_set('MOSAIC')
      user_method%mosaic = .true.
    endif
    prim_beam = 0.0 !! hprim%gil%convert(3,4)   ! old convention
    if (hprim%loca%size.ne.0) then
      nf = hprim%gil%dim(1)
      ! This has been replaced by the Telescope Section 
      call get_bsize(hprim,rname,name,prim_beam,error) !! ,otrunc,btrunc)
    else
      nf = abs(themap%nfields)
    endif
    if (nf.eq.0) then
      call map_message(seve%w,rname,'No Mosaic loaded so far')
      return
    else
      write(mess,100) 'Last mosaic loaded: ', nf,' fields'
      call map_message(seve%i,rname,mess)
      nt = max(1,hprim%gil%nteles) 
      write(mess,101) 'Primary beam (arcsec) = ',prim_beam(1:nt)*180*3600/pi
      call map_message(seve%i,rname,mess)
      write(mess,101) 'Beam Truncation level MOSAIC_BEAM = ',user_method%trunca
      call map_message(seve%i,rname,mess)
    endif
    write(mess,101) 'Searching Clean Component down to MOSAIC_SEARCH = ',user_method%search
    call map_message(seve%i,rname,mess)
    write(mess,101) 'Restoring Sky brightness down to MOSAIC_TRUNCATE = ',user_method%restor
    call map_message(seve%i,rname,mess)
  endif
  !
100 format(a,i0,a)
101 format(a,10(f8.2))
end subroutine sub_mosaic
!
subroutine mosaic_set_header(line,center,error)
  use gkernel_interfaces
  use image_def
  use clean_arrays
  use clean_default
  use imager_interfaces, except_this => mosaic_set_header
  !---------------------------------------------------------------------
  ! @ private
  !*
  ! IMAGER --  Support for MOSAIC [Ra Dec Unit [ANGLE Angle]] command
  !
  ! Define the associated MOSAIC Table to enable use of the Sault et al
  ! method for Mosaic imaging, using a smaller image size for each field.
  !!
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: line      !! Command line
  logical, intent(in) :: center             !! Should we shift the Center ?
  logical, intent(inout) :: error           !! Logical Error flag
  !
  ! Constants
  character(len=*), parameter :: rname='MOSAIC'
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  !
  ! Local ---
  logical :: lshift, print
  real(8) :: newabs(3), freq
  real(4) :: uvmax, uvmin
  logical :: debug
  !
  ! Code ----
  error = .false.
  if (huv%loca%size.eq.0) return
  debug = .false.
  call sic_get_logi('DEBUG_MOSAIC',debug,error)
  error = .false.
      if (debug) Print *,'Into mosaic_set_header '
  !
  ! Code valid only if there is some UV data
  !
  ! 1) Define MAP_CENTER if any specified
  newabs = [huv%gil%a0,huv%gil%d0,huv%gil%pang]
      if (debug) Print *,'calling map_center ',print,newabs
  if (center) call map_center(line,rname,huv,lshift,newabs,error)
  if (error) return
  ! 2) Set the TABLE
  if (.not.allocated(huv%mos%fields)) then
      if (debug) Print *,'Doing init_field'
    call init_fields(huv,print,error)
      if (debug) Print *,'Done init_field'
  else
    huv%r2d => duv
      if (debug) Print *,'Doing load_fields ',themap%nfields
    call load_fields(rname,huv,abs(themap%nfields),error)
      if (debug) Print *,'Done load_fields'
  endif
  if (error) return
  !
  ! 3) Use MAP_CENTER
  call change_fields(rname,abs(themap%nfields),huv%mos%fields,newabs,error)    
      if (debug) Print *,'Done change_fields'
  if (error) return
  !
  ! 4) Define MAP_PARAMETERS
  call uvgmax(huv,huv%r2d,uvmax,uvmin)
  freq = gdf_uv_frequency(huv)
  uvmin = uvmin*freq*f_to_k
  uvmax = uvmax*freq*f_to_k
  call map_copy_par(default_map,themap)
  call map_parameters(rname,themap,huv,freq,uvmax,uvmin,error,print=print) 
      if (debug) Print *,'Done map_parameters'
  !
  if (.not.user_method%mosaic) then
    user_method%mosaic = .true.
    call gprompt_set('MOSAIC')
  endif
end subroutine mosaic_set_header
!
subroutine mosaic_uvmap(task,line,error)
  !$ use omp_lib
  use gkernel_interfaces
  use imager_interfaces, except_this=>mosaic_uvmap
  use clean_default
  !---------------------------------------------------------------------
  ! @ private
  !*
  ! IMAGER -- Support for command UV_MAP, Mosaic case
  !
  !   Compute a Mosaic from a UV Table with pointing offset information
  !   either as additional columns, or in an associated table.
  !
  !   Dispatches between the "Historical" version and the "Sault" version
  !   that uses smaller sub-images for each field.
  !!
  !----------------------------------------------------------------------
  character(len=*), intent(in) :: task   !! Caller (MOSAIC)
  character(len=*), intent(in) :: line   !! Command line
  logical, intent(out) :: error          !! Logical error flag
  !
  ! Code ----
  if (mosaic_mode.eq.'SAULT') then
    call mosaic_uvmap_sault(task,line,error)
  else
    call mosaic_uvmap_gueth(task,line,error)
  endif
end subroutine mosaic_uvmap
!
subroutine mosaic_headers (rname,map,huv,hbeam,hdirty,hprim,nb,nf,mcol)
  use gkernel_interfaces
  use imager_interfaces, except_this=>mosaic_headers
  use clean_def
  use image_def
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private
  !*
  ! IMAGER -- UV_MAP: Define the image headers for a Mosaic
  !!
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: rname     !! Calling Task name
  type (uvmap_par), intent(in) :: map       !! Mapping parameters
  type (gildas), intent(inout) :: huv       !! UV data set
  type (gildas), intent(inout) :: hbeam     !! Dirty beam data set
  type (gildas), intent(inout) :: hdirty    !! Dirty image data set
  type (gildas), intent(inout) :: hprim     !! Primary beam data set
  integer, intent(in) :: nb         !! Number of beams per field
  integer, intent(in) :: nf         !! Number of fields
  integer, intent(in) :: mcol(2)    !! First and last channel
  !
  ! Constants
  real(kind=8), parameter :: clight=299792458d-6 ! Frequency in  MHz
  !
  ! Local ---
  type(gildas) :: htmp
  integer :: nx   ! X size
  integer :: ny   ! Y size
  integer :: nc   ! Number of channels
  real vref,voff,vinc
  integer :: schunk ! Number of channels per beam
  real(kind=4) :: loff,boff
  integer :: fcol
  logical :: error
  character(len=4) :: code
  !
  ! Code ----
  nx = map%size(1)
  ny = map%size(2)
  !
  vref = huv%gil%ref(1)
  voff = huv%gil%voff
  vinc = huv%gil%vres
  !
  nc = mcol(2)-mcol(1)+1
  fcol = mcol(1)
  !
  ! Make beam, not normalized
  call gdf_copy_header(huv,hbeam,error)
  hbeam%gil%dopp = 0.0 ! Nullify the Doppler factor
  !
  ! Is that right ?
  schunk = nc/nb      ! No, of course
  ! This is the correct value
  schunk = (nc+nb-1)/nb
  !
  hbeam%gil%ndim = 4
  hbeam%gil%dim(1) = nx
  hbeam%gil%dim(2) = ny
  hbeam%gil%dim(3) = nb
  hbeam%gil%dim(4) = nf
  hbeam%gil%convert(1,1) = nx/2+1
  hbeam%gil%convert(1,2) = ny/2+1
  hbeam%gil%convert(2,1) = 0
  hbeam%gil%convert(2,2) = 0
  hbeam%gil%convert(3,1) = -map%xycell(1)  ! Assume EQUATORIAL system
  hbeam%gil%convert(3,2) = map%xycell(2)
  !
  ! Frequency axis 
  hbeam%gil%vres = hbeam%gil%vres*schunk
  hbeam%gil%fres = hbeam%gil%fres*schunk
  ! in Velocity
  hbeam%gil%convert(1,3) = (2.d0*(vref-fcol)+schunk+1.d0)/2/schunk ! Correct
  hbeam%gil%convert(2,3) = voff
  hbeam%gil%convert(3,3) = vinc*schunk    ! 
  hbeam%gil%faxi = 3
  ! We actually would like it in Frequency ...
  
  ! Field axis
  hbeam%gil%convert(:,4) = 1.d0
  hbeam%gil%blan_words = 0
  hbeam%gil%proj_words = 0
  hbeam%gil%extr_words = 0
  hbeam%gil%reso_words = 0
  hbeam%gil%uvda_words = 0
  hbeam%gil%type_gdf = code_gdf_image
  !
  hbeam%char%code(1) = 'ANGLE'
  hbeam%char%code(2) = 'ANGLE'
  hbeam%char%code(3) = 'VELOCITY'
  hbeam%char%code(4) = 'FIELD'
  hbeam%gil%majo = 0.0
  hbeam%loca%size = hbeam%gil%dim(1)*hbeam%gil%dim(2)*hbeam%gil%dim(3)*hbeam%gil%dim(4)
  !
  ! Prepare the dirty map header
  call gdf_copy_header(hbeam,hdirty,error)
  hdirty%gil%ndim = 3
  hdirty%gil%dim(1) = nx
  hdirty%gil%dim(2) = ny
  hdirty%gil%dim(3) = nc
  hdirty%gil%dim(4) = 1
  hdirty%gil%convert(1,3) = vref-fcol+1
  hdirty%gil%convert(2,3) = voff
  hdirty%gil%convert(3,3) = vinc
  !
  ! Frequency axis --- Caution resolution change compared to HBeam
  hdirty%gil%vres = hbeam%gil%vres/schunk
  hdirty%gil%fres = hbeam%gil%fres/schunk
  !
  hdirty%gil%blan_words = 0
  hdirty%gil%proj_words = def_proj_words
  hdirty%gil%uvda_words = 0
  hdirty%gil%type_gdf = code_gdf_image
  if (hbeam%char%syst.eq.' ') hbeam%char%syst = 'EQUATORIAL'
  hdirty%char%code(1) = 'RA'
  hdirty%char%code(2) = 'DEC'
  hdirty%char%code(3) = 'VELOCITY'
  call equ_to_gal(hdirty%gil%ra,hdirty%gil%dec,0.0,0.0,   &
                  hdirty%gil%epoc,hdirty%gil%lii,hdirty%gil%bii,loff,boff,error)
  if (huv%gil%ptyp.eq.p_none) then
    hdirty%gil%ptyp = p_azimuthal  ! Azimuthal (Sin)
    hdirty%gil%pang = 0.d0     ! Defined in table.
    hdirty%gil%a0 = hdirty%gil%ra
    hdirty%gil%d0 = hdirty%gil%dec
  else
    hdirty%gil%ptyp = p_azimuthal
    hdirty%gil%pang = huv%gil%pang ! Defined in table.
    hdirty%gil%a0 = huv%gil%a0
    hdirty%gil%d0 = huv%gil%d0
  endif
  hdirty%gil%xaxi = 1
  hdirty%gil%yaxi = 2
  hdirty%gil%faxi = 3
  hdirty%gil%extr_words = 0          ! extrema not computed
  hdirty%gil%reso_words = 0          ! no beam defined
  hdirty%gil%nois_words = 2
  hdirty%gil%majo = 0
  hdirty%char%unit = 'Jy/beam'
  hdirty%loca%size = hdirty%gil%dim(1)*hdirty%gil%dim(2)*hdirty%gil%dim(3)
  !
  call gildas_null(hprim)
  if (nf.ge.1) then
    call gildas_null(htmp)
    ! Prepare the primary beam cube header
    call gdf_copy_header(hdirty,htmp,error)
    htmp%gil%dim(4) = nf
    htmp%gil%convert(1:3,4) = 1.d0
    htmp%char%unit = ' '
    htmp%char%code(4) = 'FIELD'
    ! Also reset the Number of Beams in Frequency
    htmp%gil%dim(3) = nb
    code = '4123'
    call gdf_transpose_header(htmp,hprim,code,error)   
  endif
end subroutine mosaic_headers
!!
subroutine mosaic_sort (error,sorted,shift,newabs,uvmax,uvmin, &
  & ixoff,iyoff,idoff,nf,doff,voff)
  use gkernel_interfaces
  use imager_interfaces, except_this=>mosaic_sort
  use clean_def
  use clean_arrays
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER -- UV_MAP: Sort the input Mosaic UV table
  !!
  !---------------------------------------------------------------------
  logical, intent(inout) :: sorted           !! Is table sorted ?
  logical, intent(inout) :: shift            !! Do we shift phase center ?
  logical, intent(out) :: error              !! Logical error flag
  real(kind=8), intent(inout) :: newabs(3)   !! New phase center and PA
  real, intent(out) :: uvmin                 !! Min baseline
  real, intent(out) :: uvmax                 !! Max baseline
  integer, intent(in) :: ixoff, iyoff, idoff !! Offset pointers
  integer, intent(inout) :: nf               !! Number of fields
  real, intent(inout) :: doff(:,:)           !! Field offsets
  integer, intent(inout) :: voff(:)          !! Field visibility pointers
  !
  ! Constants
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  character(len=*), parameter :: rname='UV_MOSAIC'
  !
  ! Local ---
  real(kind=8) :: freq, off(3)
  real :: pos(2), cs(2)
  integer :: nu,nv
  real, pointer :: duv_previous(:,:), duv_next(:,:)
  !
  ! Code ----
  call imager_tree('MOSAIC_SORT',.false.)
  if (newabs(3).ne.0) then
    Print *,'Non zero angle, NEW ABS ',newabs
  endif
  ! The UV table is available in HUV%
  if (huv%loca%size.eq.0) then
    call map_message(seve%e,rname,'No UV data loaded')
    error = .true.
    return
  endif
  nu = huv%gil%dim(1)
  nv = huv%gil%nvisi ! not %dim(2)
  !
  ! Correct for new phase center if required
  if (shift) then
    if (huv%gil%ptyp.eq.p_none) then
      call map_message(seve%w,rname,'No previous phase center info')
      huv%gil%a0 = huv%gil%ra
      huv%gil%d0 = huv%gil%dec
      huv%gil%pang = 0.d0
      huv%gil%ptyp = p_azimuthal
    elseif (huv%gil%ptyp.ne.p_azimuthal) then
      call map_message(seve%w,rname,'Previous projection type not SIN')
      huv%gil%ptyp = p_azimuthal
    endif
    call uv_shift_header (newabs,huv%gil%a0,huv%gil%d0,huv%gil%pang,   &
        &      off,shift)
    huv%gil%posi_words = def_posi_words
    huv%gil%proj_words = def_proj_words
  endif
  !
  sorted = .false.
  if (.not.shift) then
    call check_order_mosaic (duv,nu,nv,ixoff,iyoff,sorted)
  endif
  !
  ! Get center frequency
  freq = gdf_uv_frequency(huv,huv%gil%ref(1))
  !
  if (sorted) then
    !
    ! If already sorted, use it
    call map_message(seve%i,rname,'UV table is already sorted',3)
    !
    ! Load Field coordinates and compute UVMAX
    call mosaic_loadfield (duv,nu,nv,ixoff,iyoff,nf,doff,voff,uvmax,uvmin)
  else
    !
    ! Else, create another copy
    call map_message(seve%i,rname,'Sorting UV table...')
    !
    ! Compute observing frequency, and new phase center in wavelengths
    if (shift) then
      huv%gil%a0 = newabs(1)
      huv%gil%d0 = newabs(2)
      huv%gil%pang = newabs(3)
      cs(1)  =  cos(off(3))
      cs(2)  = -sin(off(3))
      pos = off(1:2)  ! Real*8 to Real*4
    else
      pos(1) = 0.0
      pos(2) = 0.0
      cs(1) = 1.0
      cs(2) = 0.0
    endif
    !
    ! OK, rotate, shift, sort and copy...
    !
    nullify (duv_previous, duv_next)
    !
    call uv_find_buffers (rname,nu,nv,duv_previous,duv_next,error)
    if (error) return
    !!call uv_dump_buffers ('UV_MOSAIC - After Find')
    !
    ! DUV may NOT be associated to DUV_PREVIOUS, it must be used directly
    call mosaic_sortuv (nu,nv,huv%gil%ntrail,duv,duv_next,freq,   &
           &        pos,cs,uvmax,uvmin,error,ixoff,iyoff,idoff,nf,doff,voff)
    call uv_clean_buffers (duv_previous, duv_next, error)
    if (error) return
    !!call uv_dump_buffers ('UV_MOSAIC- After Clean')
  endif
  !
  ! Now transform UVMAX in kiloWavelength (including 2 pi factor)
  uvmax = uvmax*freq*f_to_k
  uvmin = uvmin*freq*f_to_k
  error = .false.
  call imager_tree('MOSAIC_SORT',.true.)
end subroutine mosaic_sort
!
subroutine mosaic_sortuv (np,nv,ntrail,vin,vout,freq,off,cs,uvmax,uvmin, &
  & error,ixoff,iyoff,idoff,nf,doff,voff)
  use gildas_def
  use gkernel_interfaces
  use gbl_message
  use clean_default
  use imager_interfaces, except_this=> mosaic_sortuv
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER -- UV_MAP:  Sort a UV table by fields  
  !     Rotate, Shift and Sort a UV table for map making
  !     Differential precession should have been applied before.
  !!
  !---------------------------------------------------------------------
  integer, intent(in) :: np           !! Size of a visibility
  integer, intent(in) :: nv           !! Number of visibilities
  integer, intent(in) :: ntrail       !! Number of trailing daps
  real, intent(in) :: vin(np,nv)      !! Input visibilities
  real, intent(out) :: vout(np,nv)    !! Output visibilities
  real(8), intent(in) :: freq         !! Reference Frequency
  real, intent(in) :: off(2)          !! Offset of Reference
  real, intent(in) :: cs(2)           !! Frame Rotation
  real, intent(out) :: uvmax          !! Max UV value
  real, intent(out) :: uvmin          !! Min UV value
  integer, intent(in) :: ixoff, iyoff, idoff  !! Offset pointers
  integer, intent(out) :: nf          !! Number of fields
  real, intent(out) :: doff(:,:)      !! Offsets of the fields
  integer, intent(out) :: voff(:)     !! Visibility offset - range
  logical, intent(out) :: error       !! Logical error flag
  !
  ! Constant
  real(8), parameter :: pi=3.14159265358979323846d0
  real(8), parameter :: f_to_k = 2.d0*pi/299792458.d-6
  integer, parameter :: mprint=0
  !
  ! Local ---
  logical, allocatable :: ips(:)       ! Sign of visibility
  real, allocatable :: rpu(:), rpv(:)  ! U,V coordinates
  real, allocatable :: spv(:)          ! Sorted V coordinates
  integer, allocatable :: ipi(:)       ! Index
  real(8), allocatable :: dtr(:)       ! Sorting number
  real, allocatable :: xy(:,:)         ! Phase Shifts
  logical :: sorted
  integer :: ier, ifi, iv, ioff, joff
  !
  ! Code ----
  call imager_tree('MOSAIC_SORTUV',.false.)
  !
  ! Load U,V coordinates, applying possible rotation (CS),
  ! and making all V negative
  allocate (ips(nv),rpu(nv),rpv(nv),ipi(nv),dtr(nv),stat=ier)
  if (ier.ne.0) then
    error = .true.
    return
  endif
  call loaduv (vin,np,nv,cs,rpu,rpv,ips,uvmax,uvmin)
  !TEST!Print *,'UVMIN ',uvmin,' UVMAX ',uvmax,' NP ',np,' NV ',nv
  !
  ! Modify the uv coordinates to minimize
  ! the projection errors ... See Sault et al 1996 Appendix 1
  ! Key question here
  ! - modification must be done before sorting
  ! - but should we use the modified or intrinsic UV coordinates ?
  !
  ! For the rotation above, it does not matter, actually: the
  ! matrix commutes (I think so - That can be check later...)
  !
  !  call remapuv(nv,cs,rpu,rpv,ixoff,iyoff,uvmax,uvmin)
  !
  ! Identify number of fields
  if (ixoff.ne.0 .and. iyoff.ne.0) then
    call loadfiuv_xy(vin,np,nv,dtr,ipi,sorted,ixoff,iyoff,rpv,nf,doff)
  else if (idoff.ne.0) then
    call map_message(seve%w,'LOADFIUV_ID', &
      & 'Sorting UV data set with associated Mosaic table is obsolescent')
    call loadfiuv_id(vin,np,nv,dtr,ipi,sorted,idoff,rpv,nf,doff)
  endif
  !
  allocate(xy(2,nf))
  ! Note that the new phase center is counter-rotated because rotations
  ! are applied before phase shift.
  if ((idoff.ne.0).and.(mosaic_mode.eq.'SAULT')) then
    !
    ! Should we Add the Global Offset ?
    ! We should also find the fractional pixel shift if needed here...
    ! The shifting should also always be done in this case, 
    ! and the initial UV table not changed ? 
    do ifi=1,nf
      xy(1,ifi) = - freq * f_to_k * ( doff(1,ifi)*cs(1) - doff(2,ifi)*cs(2) )
      xy(2,ifi) = - freq * f_to_k * ( doff(2,ifi)*cs(1) + doff(1,ifi)*cs(2) )
    enddo  
  else
    !
    ! In the GUETH method, add the global offset - CHECKPLEASE
    xy(1,:) = - freq * f_to_k * ( off(1)*cs(1) - off(2)*cs(2) )
    xy(2,:) = - freq * f_to_k * ( off(2)*cs(1) + off(1)*cs(2) )
  endif
  !
  ! Sort by fields (major number) then V (fractionary part)
  if (.not.sorted) then
    !!Print *,'Sorting UV data '
    call gr8_trie (dtr,ipi,nv,error)
    if (error) return
    deallocate (dtr,stat=ier)
    allocate (spv(nv),stat=ier)
    if (ier.ne.0) then
      error = .true.
      return
    endif
    !
    ! One must sort RPV here to use SORTUV later...
    do iv=1,nv
      spv(iv) = rpv(ipi(iv))
    enddo
    rpv(:) = spv(:)
    deallocate (spv,stat=ier)
  else
    deallocate (dtr,stat=ier)
    !! Print *,'UV Data is already sorted '
  endif
  ! !Read(5,*) ifi
  !
  ! Apply phase shift and copy to output visibilities
  !
  ! For Mosaics using the Sault et al method, this must be done
  ! per Pointing, as the phase shift to be applied is Pointing
  ! dependent. We may need an intermediate routine to do so, or
  ! to generalize the XY array to one per field.
  !
  call imager_tree('SUB_SORTUV',.false.)
  call sub_sortuv (vin,vout,np,nv,ntrail,xy,nf,idoff,rpu,rpv,ips,ipi)
  call imager_tree('SUB_SORTUV',.true.)
  !
  if (ixoff.ne.0 .and. iyoff.ne.0) then
    ifi = 1
    voff(ifi) = 1
    do iv=1,nv
      if ( (doff(1,ifi).ne.vout(ixoff,iv)) .or. &
        &  (doff(2,ifi).ne.vout(iyoff,iv)) ) then
        ifi = ifi+1
        voff(ifi) = iv
      endif
    enddo
    voff(nf+1) = nv+1
  else if (idoff.ne.0) then
    ifi = 0
    joff = -1 
    do iv=1,nv
      ioff = vout(idoff,iv)
      if (ioff.ne.joff) then
        ifi = ifi+1
        voff(ifi) = iv
        joff = ioff
      endif
    enddo
    voff(nf+1) = nv+1
  endif
  !
  if (mprint.eq.0) return
  !
  ! !Print *,'XOFF ',ixoff,' YOFF ',iyoff
  do ifi=1,min(nf,mprint)
    write(*,'(I4,A,2F12.4,2I10)') ifi,' DOFF ', &
      & doff(1,ifi)*180.*3600./pi, &
      & doff(2,ifi)*180.*3600./pi, &
      & voff(ifi), voff(ifi+1)-1
  enddo
  if (nf.gt.mprint) write(*,*) 'and ',nf-mprint,' more fields used but not printed above'
  !
  error = .false.
  !
  call imager_tree('MOSAIC_SORTUV',.true.)
end subroutine mosaic_sortuv
!
subroutine check_order_mosaic(visi,np,nv,ixoff,iyoff,sorted)
  !---------------------------------------------------------------------
  ! @ private
  !*
  ! IMAGER --  Check if visibilites are sorted.  
  !   Chksuv does a similar job, but using V values and an index.
  !   Here, we sort on field coordinates
  !!
  !---------------------------------------------------------------------
  integer, intent(in) :: np         !! Size of a visibility
  integer, intent(in) :: nv         !! Number of visibilities
  real, intent(in) :: visi(np,nv)   !! Visibilities
  integer, intent(in) :: ixoff      !! X pointing column
  integer, intent(in) :: iyoff      !! Y pointing column
  logical, intent(out) :: sorted    !! Is data sorted ?
  !
  ! Local ---
  real :: vmax,xoff,yoff
  integer iv
  !
  ! Code ----
  vmax = visi(2,1)
  xoff = visi(ixoff,1)
  yoff = visi(iyoff,1)
  !
  do iv=2,nv
    if (visi(2,iv).lt.vmax) then
      if (visi(ixoff,iv).eq.xoff .and. visi(iyoff,iv).eq.yoff) then
        !!Print *,'Unsorted V at ',iv,visi(2,iv),vmax
        sorted = .false.
        return
      endif
      ! else, this is a new offset
      xoff = visi(ixoff,iv)
      yoff = visi(iyoff,iv)
    else if (visi(ixoff,iv).eq.xoff .and. visi(iyoff,iv).eq.yoff) then
      ! ok, things progress normally
      continue
    else
      ! Unsorted offset
      Print *,'Unsorted Position offset at ',iv
      sorted = .false.
      return
    endif
    vmax = visi(2,iv)
  enddo
  sorted = .true.
end subroutine check_order_mosaic
!
subroutine loadfiuv_xy (visi,np,nv,dtr,it,sorted,ixoff,iyoff,rpv,nf,doff)
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER  -- Sorting routine  
  !     Load field numbers into work arrays for sorting.
  !!
  !---------------------------------------------------------------------
  integer, intent(in)  :: np                 !! Size of a visibility
  integer, intent(in)  :: nv                 !! Number of visibilities
  real, intent(in) :: visi(np,nv)            !! Input visibilities
  real(8), intent(out) :: dtr(nv)            !! Output field number
  integer, intent(out) :: it(nv)             !! Indexes
  logical, intent(out) :: sorted             !! Is data sorted ?
  integer, intent(in)  :: ixoff              !! X pointer
  integer, intent(in)  :: iyoff              !! Y pointer
  real(4), intent(in)  :: rpv(nv)            !! V Values
  integer, intent(inout) :: nf               !! Number of fields
  real(kind=4), intent(out) :: doff(:,:)     !! Fields offsets (radians)
  !
  ! Local ---
  integer :: iv, j
  integer :: ifi, mfi, kfi, nfi
  real(8) :: vmax
  !
  ! Code ----
  !
  ! Scan how many fields
  nfi = 1
  mfi = ubound(doff,2)
  if (nf.ne.mfi) Print *,'Warning Number of field mismatch ',nf, mfi
  !
  ! V are negative values, so this 1 + max(abs(V))
  vmax = 1.0d0-minval(rpv)
  !
  doff(1,1) = visi(ixoff,1)
  doff(2,1) = visi(iyoff,1)
  dtr(1) = 1.d0+rpv(1)/vmax ! We have here 0 =< dtr < 1
  !
  do iv=2,nv
    kfi = 0
    if (rpv(iv).gt.0) then
      Print *,'Unsorted Visibility with V > 0 ',iv,rpv(iv)
    endif
    do ifi=1,nfi
      if (visi(ixoff,iv).eq.doff(1,ifi) .and. &
      & visi(iyoff,iv).eq.doff(2,ifi) ) then
        dtr(iv) = dble(ifi)+rpv(iv)/vmax
        kfi = ifi
        exit
      endif
    enddo
    !
    ! New field
    if (kfi.eq.0) then
      if (nfi.eq.mfi) then
        Print *,'Programming error: More fields than expected ',mfi
        Print *,'Invalid number of Fields ',mfi,' at ',iv
        Print *,visi(ixoff,iv),visi(iyoff,iv)
        do j=1,mfi
          print *,doff(:,j)
        enddo
        return
      endif
      nfi = nfi+1
      doff(1,nfi) = visi(ixoff,iv)
      doff(2,nfi) = visi(iyoff,iv)
      dtr(iv) = dble(nfi)+rpv(iv)/vmax   ! nfi-1 =< dtr < nfi
      !TEST!Print *,'New field ',nfi,' at ',doff(1:2,nfi),' Visi ',iv,dtr(iv)
    endif
  enddo
  !
  nf = nfi
  !
  do iv=1,nv
    it(iv) = iv
  enddo
  !
  ! DTR must in the end be ordered and increasing.
  vmax = dtr(1)
  do iv = 1,nv
    if (dtr(iv).lt.vmax) then
      sorted = .false.
      return
    endif
    vmax = dtr(iv)
  enddo
  sorted = .true.
  !
end subroutine loadfiuv_xy
!
subroutine select_fields(rname,line,o_field,mp,np,fields,error)
  use gkernel_interfaces
  use imager_interfaces, only : get_i4list_fromsic, map_message
  use clean_arrays
  use gkernel_types
  use gbl_message
  !---------------------------------------------------------------------
  ! @ private-mandatory
  !*
  ! IMAGER -- UV_MAP -- Select a list of fields from a Mosaic
  !!
  !---------------------------------------------------------------------
  character(len=*), intent(in) :: rname !! Caller name
  character(len=*), intent(in) :: line  !! Command line
  integer, intent(in) :: o_field        !! /FIELD option number
  integer, intent(in) :: mp             !! Number of fields in UV data
  integer, intent(out) :: np            !! Number of fields selected 
  logical, intent(inout) :: error       !! Logical error flag
  integer, intent(inout), allocatable :: fields(:)  !! Selected Field numbers
  !
  ! Constants
  real, parameter :: rad_to_sec=180*3600/acos(-1.0)
  ! 
  ! Local ---
  type(sic_descriptor_t) :: desc
  integer :: ifield, jfield, n, i, ier
  character(len=80) :: chain
  logical :: found
  !
  ! Code ----
  np = sic_narg(o_field)
  if (np.le.1) then
    call sic_ch(line,o_field,1,chain,n,.true.,error)
    if (error) return
    call sic_descriptor(chain,desc,found)
    if (found) then
      np = 0
    else
      np = 1
    endif
  endif
  !
  if (np.ne.0) then
    allocate(fields(np), stat=ier)
    if (ier.ne.0) then
      call map_message(seve%e,rname,'Memory allocation error')
      error = .true.
      return
    endif
    do i=1,np
      call sic_i4(line,o_field,i,fields(i),.true.,error)
      if (error) return
    enddo
  else 
    call get_i4list_fromsic(rname,line,o_field,np,fields,error) 
  endif
  !
  if (np.gt.mp) then
    call map_message(seve%e,rname,'More selected fields than available')
    error = .true.
  else 
    do jfield=1,np
      ifield = fields(jfield)
      if (ifield.le.0 .or. ifield.gt.mp) then
        write(chain,'(A,I0,I0,A,I0,A)') 'Selected field ',jfield,& 
        & ifield,' out of range [1,',mp,']'
        call map_message(seve%e,rname,chain)
        error = .true.
      endif
    enddo
  endif
  if (error) return
  write(chain,'(I0,A,I0,A)') np,' fields selected:' 
  call map_message(seve%i,rname,chain)
  !
  if (allocated(huv%mos%fields)) then
    do jfield=1,np
      ifield = fields(jfield)
      write(*,'(I0,1X,F10.2,F10.2)') ifield, huv%mos%fields(ifield)%opoint(1)*rad_to_sec, &
        & huv%mos%fields(ifield)%opoint(1)*rad_to_sec
    enddo
  else  
    do jfield=1,np
      ifield = fields(jfield)
      write(*,'(I0,1X,F10.2,F10.2)') ifield, themap%offxy(1,ifield)*rad_to_sec, &
        & themap%offxy(2,ifield)*rad_to_sec
    enddo
  endif
  !
end subroutine select_fields
