import { Command } from 'commander';
import { createRequire } from 'module';
import ora from 'ora';
import path from 'path';
import { promises as fs } from 'fs';
import { AI_TOOLS } from '../core/config.js';
import { UpdateCommand } from '../core/update.js';
import { ListCommand } from '../core/list.js';
import { ArchiveCommand } from '../core/archive.js';
import { ViewCommand } from '../core/view.js';
import { registerSpecCommand } from '../commands/spec.js';
import { ChangeCommand } from '../commands/change.js';
import { ValidateCommand } from '../commands/validate.js';
import { ShowCommand } from '../commands/show.js';
import { CompletionCommand } from '../commands/completion.js';
import { registerConfigCommand } from '../commands/config.js';
const program = new Command();
const require = createRequire(import.meta.url);
const { version } = require('../../package.json');
program
    .name('openspec')
    .description('AI-native system for spec-driven development')
    .version(version);
// Global options
program.option('--no-color', 'Disable color output');
// Apply global flags before any command runs
program.hook('preAction', (thisCommand) => {
    const opts = thisCommand.opts();
    if (opts.color === false) {
        process.env.NO_COLOR = '1';
    }
});
const availableToolIds = AI_TOOLS.filter((tool) => tool.available).map((tool) => tool.value);
const toolsOptionDescription = `Configure AI tools non-interactively. Use "all", "none", or a comma-separated list of: ${availableToolIds.join(', ')}`;
program
    .command('init [path]')
    .description('Initialize OpenSpec in your project')
    .option('--tools <tools>', toolsOptionDescription)
    .action(async (targetPath = '.', options) => {
    try {
        // Validate that the path is a valid directory
        const resolvedPath = path.resolve(targetPath);
        try {
            const stats = await fs.stat(resolvedPath);
            if (!stats.isDirectory()) {
                throw new Error(`Path "${targetPath}" is not a directory`);
            }
        }
        catch (error) {
            if (error.code === 'ENOENT') {
                // Directory doesn't exist, but we can create it
                console.log(`Directory "${targetPath}" doesn't exist, it will be created.`);
            }
            else if (error.message && error.message.includes('not a directory')) {
                throw error;
            }
            else {
                throw new Error(`Cannot access path "${targetPath}": ${error.message}`);
            }
        }
        const { InitCommand } = await import('../core/init.js');
        const initCommand = new InitCommand({
            tools: options?.tools,
        });
        await initCommand.execute(targetPath);
    }
    catch (error) {
        console.log(); // Empty line for spacing
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
program
    .command('update [path]')
    .description('Update OpenSpec instruction files')
    .action(async (targetPath = '.') => {
    try {
        const resolvedPath = path.resolve(targetPath);
        const updateCommand = new UpdateCommand();
        await updateCommand.execute(resolvedPath);
    }
    catch (error) {
        console.log(); // Empty line for spacing
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
program
    .command('list')
    .description('List items (changes by default). Use --specs to list specs.')
    .option('--specs', 'List specs instead of changes')
    .option('--changes', 'List changes explicitly (default)')
    .action(async (options) => {
    try {
        const listCommand = new ListCommand();
        const mode = options?.specs ? 'specs' : 'changes';
        await listCommand.execute('.', mode);
    }
    catch (error) {
        console.log(); // Empty line for spacing
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
program
    .command('view')
    .description('Display an interactive dashboard of specs and changes')
    .action(async () => {
    try {
        const viewCommand = new ViewCommand();
        await viewCommand.execute('.');
    }
    catch (error) {
        console.log(); // Empty line for spacing
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
// Change command with subcommands
const changeCmd = program
    .command('change')
    .description('Manage OpenSpec change proposals');
// Deprecation notice for noun-based commands
changeCmd.hook('preAction', () => {
    console.error('Warning: The "openspec change ..." commands are deprecated. Prefer verb-first commands (e.g., "openspec list", "openspec validate --changes").');
});
changeCmd
    .command('show [change-name]')
    .description('Show a change proposal in JSON or markdown format')
    .option('--json', 'Output as JSON')
    .option('--deltas-only', 'Show only deltas (JSON only)')
    .option('--requirements-only', 'Alias for --deltas-only (deprecated)')
    .option('--no-interactive', 'Disable interactive prompts')
    .action(async (changeName, options) => {
    try {
        const changeCommand = new ChangeCommand();
        await changeCommand.show(changeName, options);
    }
    catch (error) {
        console.error(`Error: ${error.message}`);
        process.exitCode = 1;
    }
});
changeCmd
    .command('list')
    .description('List all active changes (DEPRECATED: use "openspec list" instead)')
    .option('--json', 'Output as JSON')
    .option('--long', 'Show id and title with counts')
    .action(async (options) => {
    try {
        console.error('Warning: "openspec change list" is deprecated. Use "openspec list".');
        const changeCommand = new ChangeCommand();
        await changeCommand.list(options);
    }
    catch (error) {
        console.error(`Error: ${error.message}`);
        process.exitCode = 1;
    }
});
changeCmd
    .command('validate [change-name]')
    .description('Validate a change proposal')
    .option('--strict', 'Enable strict validation mode')
    .option('--json', 'Output validation report as JSON')
    .option('--no-interactive', 'Disable interactive prompts')
    .action(async (changeName, options) => {
    try {
        const changeCommand = new ChangeCommand();
        await changeCommand.validate(changeName, options);
        if (typeof process.exitCode === 'number' && process.exitCode !== 0) {
            process.exit(process.exitCode);
        }
    }
    catch (error) {
        console.error(`Error: ${error.message}`);
        process.exitCode = 1;
    }
});
program
    .command('archive [change-name]')
    .description('Archive a completed change and update main specs')
    .option('-y, --yes', 'Skip confirmation prompts')
    .option('--skip-specs', 'Skip spec update operations (useful for infrastructure, tooling, or doc-only changes)')
    .option('--no-validate', 'Skip validation (not recommended, requires confirmation)')
    .action(async (changeName, options) => {
    try {
        const archiveCommand = new ArchiveCommand();
        await archiveCommand.execute(changeName, options);
    }
    catch (error) {
        console.log(); // Empty line for spacing
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
registerSpecCommand(program);
registerConfigCommand(program);
// Top-level validate command
program
    .command('validate [item-name]')
    .description('Validate changes and specs')
    .option('--all', 'Validate all changes and specs')
    .option('--changes', 'Validate all changes')
    .option('--specs', 'Validate all specs')
    .option('--type <type>', 'Specify item type when ambiguous: change|spec')
    .option('--strict', 'Enable strict validation mode')
    .option('--json', 'Output validation results as JSON')
    .option('--concurrency <n>', 'Max concurrent validations (defaults to env OPENSPEC_CONCURRENCY or 6)')
    .option('--no-interactive', 'Disable interactive prompts')
    .action(async (itemName, options) => {
    try {
        const validateCommand = new ValidateCommand();
        await validateCommand.execute(itemName, options);
    }
    catch (error) {
        console.log();
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
// Top-level show command
program
    .command('show [item-name]')
    .description('Show a change or spec')
    .option('--json', 'Output as JSON')
    .option('--type <type>', 'Specify item type when ambiguous: change|spec')
    .option('--no-interactive', 'Disable interactive prompts')
    // change-only flags
    .option('--deltas-only', 'Show only deltas (JSON only, change)')
    .option('--requirements-only', 'Alias for --deltas-only (deprecated, change)')
    // spec-only flags
    .option('--requirements', 'JSON only: Show only requirements (exclude scenarios)')
    .option('--no-scenarios', 'JSON only: Exclude scenario content')
    .option('-r, --requirement <id>', 'JSON only: Show specific requirement by ID (1-based)')
    // allow unknown options to pass-through to underlying command implementation
    .allowUnknownOption(true)
    .action(async (itemName, options) => {
    try {
        const showCommand = new ShowCommand();
        await showCommand.execute(itemName, options ?? {});
    }
    catch (error) {
        console.log();
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
// Completion command with subcommands
const completionCmd = program
    .command('completion')
    .description('Manage shell completions for OpenSpec CLI');
completionCmd
    .command('generate [shell]')
    .description('Generate completion script for a shell (outputs to stdout)')
    .action(async (shell) => {
    try {
        const completionCommand = new CompletionCommand();
        await completionCommand.generate({ shell });
    }
    catch (error) {
        console.log();
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
completionCmd
    .command('install [shell]')
    .description('Install completion script for a shell')
    .option('--verbose', 'Show detailed installation output')
    .action(async (shell, options) => {
    try {
        const completionCommand = new CompletionCommand();
        await completionCommand.install({ shell, verbose: options?.verbose });
    }
    catch (error) {
        console.log();
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
completionCmd
    .command('uninstall [shell]')
    .description('Uninstall completion script for a shell')
    .option('-y, --yes', 'Skip confirmation prompts')
    .action(async (shell, options) => {
    try {
        const completionCommand = new CompletionCommand();
        await completionCommand.uninstall({ shell, yes: options?.yes });
    }
    catch (error) {
        console.log();
        ora().fail(`Error: ${error.message}`);
        process.exit(1);
    }
});
// Hidden command for machine-readable completion data
program
    .command('__complete <type>', { hidden: true })
    .description('Output completion data in machine-readable format (internal use)')
    .action(async (type) => {
    try {
        const completionCommand = new CompletionCommand();
        await completionCommand.complete({ type });
    }
    catch (error) {
        // Silently fail for graceful shell completion experience
        process.exitCode = 1;
    }
});
program.parse();
//# sourceMappingURL=index.js.map