import * as nbformat from '@jupyterlab/nbformat';
import { IRenderMime } from '@jupyterlab/rendermime-interfaces';
import { PartialJSONObject, ReadonlyPartialJSONObject } from '@lumino/coreutils';
import { ISignal } from '@lumino/signaling';
/**
 * The interface for an output model.
 */
export interface IOutputModel extends IRenderMime.IMimeModel {
    /**
     * A signal emitted when the output model changes.
     */
    readonly changed: ISignal<this, void>;
    /**
     * The output type.
     */
    readonly type: string;
    /**
     * The execution count of the model.
     */
    readonly executionCount: nbformat.ExecutionCount;
    /**
     * Whether the output is trusted.
     */
    trusted: boolean;
    /**
     * Dispose of the resources used by the output model.
     */
    dispose(): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IOutput;
}
/**
 * The namespace for IOutputModel sub-interfaces.
 */
export declare namespace IOutputModel {
    /**
     * The options used to create a notebook output model.
     */
    interface IOptions {
        /**
         * The raw output value.
         */
        value: nbformat.IOutput;
        /**
         * Whether the output is trusted.  The default is false.
         */
        trusted?: boolean;
    }
}
/**
 * The default implementation of a notebook output model.
 */
export declare class OutputModel implements IOutputModel {
    /**
     * Construct a new output model.
     */
    constructor(options: IOutputModel.IOptions);
    /**
     * A signal emitted when the output model changes.
     */
    get changed(): ISignal<this, void>;
    /**
     * The output type.
     */
    readonly type: string;
    /**
     * The execution count.
     */
    readonly executionCount: nbformat.ExecutionCount;
    /**
     * Whether the model is trusted.
     */
    readonly trusted: boolean;
    /**
     * Dispose of the resources used by the output model.
     */
    dispose(): void;
    /**
     * The data associated with the model.
     */
    get data(): ReadonlyPartialJSONObject;
    /**
     * The metadata associated with the model.
     */
    get metadata(): ReadonlyPartialJSONObject;
    /**
     * Set the data associated with the model.
     *
     * #### Notes
     * Depending on the implementation of the mime model,
     * this call may or may not have deferred effects,
     */
    setData(options: IRenderMime.IMimeModel.ISetDataOptions): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IOutput;
    /**
     * Update an observable JSON object using a readonly JSON object.
     */
    private _updateObservable;
    private _changed;
    private _raw;
    private _rawMetadata;
    private _rawData;
    private _data;
    private _metadata;
}
/**
 * The namespace for OutputModel statics.
 */
export declare namespace OutputModel {
    /**
     * Get the data for an output.
     *
     * @param output - A kernel output message payload.
     *
     * @returns - The data for the payload.
     */
    function getData(output: nbformat.IOutput): PartialJSONObject;
    /**
     * Get the metadata from an output message.
     *
     * @param output - A kernel output message payload.
     *
     * @returns - The metadata for the payload.
     */
    function getMetadata(output: nbformat.IOutput): PartialJSONObject;
}
