/*
 * Copyright (c) Jupyter Development Team.
 * Distributed under the terms of the Modified BSD License.
 */
// Inspired by https://discuss.codemirror.net/t/how-to-implement-ruler/4616/
import { Facet } from '@codemirror/state';
import { EditorView, ViewPlugin } from '@codemirror/view';
import { JSONExt } from '@lumino/coreutils';
const RULERS_CLASSNAME = 'cm-rulers';
/**
 * Rulers style
 */
const baseTheme = EditorView.baseTheme({
    [`.${RULERS_CLASSNAME}`]: { borderRight: '1px dotted gray', opacity: 0.7 }
});
/**
 * Rulers facet
 */
const rulerConfig = Facet.define({
    // Merge all unique values
    combine(value) {
        const final = value.reduce((agg, arr) => agg.concat(
        // Check value is not in aggregate nor multiple time in the array.
        arr.filter((v, idx) => !agg.includes(v) && idx == arr.lastIndexOf(v))), []);
        return final;
    }
});
/**
 * View plugin displaying the rulers
 */
const plugin = ViewPlugin.fromClass(class {
    constructor(view) {
        var _a, _b;
        this.rulersContainer = view.dom.appendChild(document.createElement('div'));
        this.rulersContainer.style.cssText = `
                position: absolute;
                left: 0;
                top: 0;
                width: 100%;
                height: 100%;
                pointer-events: none;
                overflow: hidden;
            `;
        const defaultCharacterWidth = view.defaultCharacterWidth;
        const widths = view.state.facet(rulerConfig);
        const guttersWidths = (_b = (_a = view.scrollDOM.querySelector('.cm-gutters')) === null || _a === void 0 ? void 0 : _a.clientWidth) !== null && _b !== void 0 ? _b : 0;
        this.rulers = widths.map(width => {
            const ruler = this.rulersContainer.appendChild(document.createElement('div'));
            ruler.classList.add(RULERS_CLASSNAME);
            ruler.style.cssText = `
                position: absolute;
                left: ${guttersWidths + width * defaultCharacterWidth}px;
                height: 100%;
            `;
            // FIXME: This should be equal to the amount of padding on a line.
            // This value should be extracted from CodeMirror rather than hardcoded.
            ruler.style.width = '6px';
            return ruler;
        });
    }
    update(update) {
        var _a, _b;
        const widths = update.view.state.facet(rulerConfig);
        if (update.viewportChanged ||
            update.geometryChanged ||
            !JSONExt.deepEqual(widths, update.startState.facet(rulerConfig))) {
            const guttersWidth = (_b = (_a = update.view.scrollDOM.querySelector('.cm-gutters')) === null || _a === void 0 ? void 0 : _a.clientWidth) !== null && _b !== void 0 ? _b : 0;
            const defaultCharacterWidth = update.view.defaultCharacterWidth;
            this.rulers.forEach((ruler, rulerIdx) => {
                ruler.style.left = `${guttersWidth + widths[rulerIdx] * defaultCharacterWidth}px`;
            });
        }
    }
    destroy() {
        this.rulers.forEach(ruler => {
            ruler.remove();
        });
        this.rulersContainer.remove();
    }
});
/**
 * Extension for CodeMirror 6 displaying rulers.
 *
 * @param value Rulers position
 * @returns CodeMirror 6 extension
 */
export function rulers(value) {
    return [baseTheme, rulerConfig.of(value), plugin];
}
//# sourceMappingURL=rulers.js.map