import { Poll } from '@lumino/polling';
import { ISignal } from '@lumino/signaling';
import { ServerConnection } from '../serverconnection';
import * as Session from './session';
import { BaseManager } from '../basemanager';
import { Kernel } from '../kernel';
/**
 * An implementation of a session manager.
 */
export declare class SessionManager extends BaseManager implements Session.IManager {
    /**
     * Construct a new session manager.
     *
     * @param options - The default options for each session.
     */
    constructor(options: SessionManager.IOptions);
    /**
     * The server settings for the manager.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the manager is ready.
     */
    get isReady(): boolean;
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready(): Promise<void>;
    /**
     * A signal emitted when the running sessions change.
     */
    get runningChanged(): ISignal<this, Session.IModel[]>;
    /**
     * A signal emitted when there is a connection failure.
     */
    get connectionFailure(): ISignal<this, Error>;
    /**
     * Dispose of the resources used by the manager.
     */
    dispose(): void;
    connectTo(options: Omit<Session.ISessionConnection.IOptions, 'connectToKernel' | 'serverSettings'>): Session.ISessionConnection;
    /**
     * Create an iterator over the most recent running sessions.
     *
     * @returns A new iterator over the running sessions.
     */
    running(): IterableIterator<Session.IModel>;
    /**
     * Force a refresh of the running sessions.
     *
     * @returns A promise that with the list of running sessions.
     *
     * #### Notes
     * This is not typically meant to be called by the user, since the
     * manager maintains its own internal state.
     */
    refreshRunning(): Promise<void>;
    /**
     * Start a new session.  See also [[startNewSession]].
     *
     * @param createOptions - Options for creating the session
     *
     * @param connectOptions - Options for connecting to the session
     */
    startNew(createOptions: Session.ISessionOptions, connectOptions?: Omit<Session.ISessionConnection.IOptions, 'model' | 'connectToKernel' | 'serverSettings'>): Promise<Session.ISessionConnection>;
    /**
     * Shut down a session by id.
     */
    shutdown(id: string): Promise<void>;
    /**
     * Shut down all sessions.
     *
     * @returns A promise that resolves when all of the kernels are shut down.
     */
    shutdownAll(): Promise<void>;
    /**
     * Find a session associated with a path and stop it if it is the only session
     * using that kernel.
     *
     * @param path - The path in question.
     *
     * @returns A promise that resolves when the relevant sessions are stopped.
     */
    stopIfNeeded(path: string): Promise<void>;
    /**
     * Find a session by id.
     */
    findById(id: string): Promise<Session.IModel | undefined>;
    /**
     * Find a session by path.
     */
    findByPath(path: string): Promise<Session.IModel | undefined>;
    /**
     * Execute a request to the server to poll running kernels and update state.
     */
    protected requestRunning(): Promise<void>;
    /**
     * Handle a session starting.
     */
    private _onStarted;
    private _onDisposed;
    private _onChanged;
    private _isReady;
    private _sessionConnections;
    private _models;
    private _pollModels;
    private _ready;
    private _runningChanged;
    private _connectionFailure;
    private readonly _connectToKernel;
    private _kernelManager;
}
/**
 * The namespace for `SessionManager` class statics.
 */
export declare namespace SessionManager {
    /**
     * The options used to initialize a SessionManager.
     */
    interface IOptions extends BaseManager.IOptions {
        /**
         * When the manager stops polling the API. Defaults to `when-hidden`.
         */
        standby?: Poll.Standby | (() => boolean | Poll.Standby);
        /**
         * Kernel Manager
         */
        kernelManager: Kernel.IManager;
    }
    /**
     * A no-op session manager to be used when starting sessions is not supported.
     */
    class NoopManager extends SessionManager {
        /**
         * Whether the manager is active.
         */
        get isActive(): boolean;
        /**
         * Used for testing.
         */
        get parentReady(): Promise<void>;
        /**
         * Start a new session - throw an error since it is not supported.
         */
        startNew(createOptions: Session.ISessionOptions, connectOptions?: Omit<Session.ISessionConnection.IOptions, 'model' | 'connectToKernel' | 'serverSettings'>): Promise<Session.ISessionConnection>;
        connectTo(options: Omit<Session.ISessionConnection.IOptions, 'connectToKernel' | 'serverSettings'>): Session.ISessionConnection;
        /**
         * A promise that fulfills when the manager is ready (never).
         */
        get ready(): Promise<void>;
        /**
         * Shut down a session by id - throw an error since it is not supported.
         */
        shutdown(id: string): Promise<void>;
        /**
         * Execute a request to the server to poll running sessions and update state.
         */
        protected requestRunning(): Promise<void>;
        private _readyPromise;
    }
}
