"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.SettingManager = void 0;
const coreutils_1 = require("@jupyterlab/coreutils");
const statedb_1 = require("@jupyterlab/statedb");
const serverconnection_1 = require("../serverconnection");
/**
 * The url for the lab settings service.
 */
const SERVICE_SETTINGS_URL = 'api/settings';
/**
 * The settings API service manager.
 */
class SettingManager extends statedb_1.DataConnector {
    /**
     * Create a new setting manager.
     */
    constructor(options = {}) {
        var _a;
        super();
        this.serverSettings =
            (_a = options.serverSettings) !== null && _a !== void 0 ? _a : serverconnection_1.ServerConnection.makeSettings();
    }
    /**
     * Fetch a plugin's settings.
     *
     * @param id - The plugin's ID.
     *
     * @returns A promise that resolves if successful.
     */
    async fetch(id) {
        if (!id) {
            throw new Error('Plugin `id` parameter is required for settings fetch.');
        }
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        const response = await makeRequest(url, {}, serverSettings);
        if (response.status !== 200) {
            const err = await ResponseError.create(response);
            throw err;
        }
        // Assert what type the server response is returning.
        return response.json();
    }
    /**
     * Fetch the list of all plugin setting bundles.
     *
     * @returns A promise that resolves if successful.
     */
    async list(query) {
        var _a, _b, _c, _d;
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, '', query === 'ids');
        const response = await makeRequest(url, {}, serverSettings);
        if (response.status !== 200) {
            throw new ResponseError(response);
        }
        const json = await response.json();
        const ids = (_b = (_a = json === null || json === void 0 ? void 0 : json['settings']) === null || _a === void 0 ? void 0 : _a.map((plugin) => plugin.id)) !== null && _b !== void 0 ? _b : [];
        let values = [];
        if (!query) {
            values =
                (_d = (_c = json === null || json === void 0 ? void 0 : json['settings']) === null || _c === void 0 ? void 0 : _c.map((plugin) => {
                    plugin.data = { composite: {}, user: {} };
                    return plugin;
                })) !== null && _d !== void 0 ? _d : [];
        }
        return { ids, values };
    }
    /**
     * Save a plugin's settings.
     *
     * @param id - The plugin's ID.
     *
     * @param raw - The user setting values as a raw string of JSON with comments.
     *
     * @returns A promise that resolves if successful.
     */
    async save(id, raw) {
        const { serverSettings } = this;
        const { baseUrl, appUrl } = serverSettings;
        const { makeRequest, ResponseError } = serverconnection_1.ServerConnection;
        const base = baseUrl + appUrl;
        const url = Private.url(base, id);
        // NOTE: 'raw' is JSON5 (not valid JSON), so we encode it as a string in a valid JSON body
        const init = { body: JSON.stringify({ raw }), method: 'PUT' };
        const response = await makeRequest(url, init, serverSettings);
        if (response.status !== 204) {
            throw new ResponseError(response);
        }
    }
}
exports.SettingManager = SettingManager;
/**
 * A namespace for private data.
 */
var Private;
(function (Private) {
    /**
     * Get the url for a plugin's settings.
     */
    function url(base, id, idsOnly) {
        const idsOnlyParam = idsOnly
            ? coreutils_1.URLExt.objectToQueryString({ ids_only: true })
            : '';
        const settingsBase = coreutils_1.URLExt.join(base, SERVICE_SETTINGS_URL);
        const result = coreutils_1.URLExt.join(settingsBase, id);
        if (!result.startsWith(settingsBase)) {
            throw new Error('Can only be used for workspaces requests');
        }
        return `${result}${idsOnlyParam}`;
    }
    Private.url = url;
})(Private || (Private = {}));
//# sourceMappingURL=index.js.map