import { Widget } from '@lumino/widgets';
import { IHTMLSearchMatch, ISearchProvider, ISearchProviderRegistry } from '../tokens';
import { SearchProvider } from '../searchprovider';
import { ITranslator } from '@jupyterlab/translation';
export declare const FOUND_CLASSES: string[];
/**
 * HTML search engine
 */
export declare class HTMLSearchEngine {
    /**
     * We choose opt out as most node types should be searched (e.g. script).
     * Even nodes like <data>, could have textContent we care about.
     *
     * Note: nodeName is capitalized, so we do the same here
     */
    static UNSUPPORTED_ELEMENTS: {
        BASE: boolean;
        HEAD: boolean;
        LINK: boolean;
        META: boolean;
        STYLE: boolean;
        TITLE: boolean;
        BODY: boolean;
        AREA: boolean;
        AUDIO: boolean;
        IMG: boolean;
        MAP: boolean;
        TRACK: boolean;
        VIDEO: boolean;
        APPLET: boolean;
        EMBED: boolean;
        IFRAME: boolean;
        NOEMBED: boolean;
        OBJECT: boolean;
        PARAM: boolean;
        PICTURE: boolean;
        SOURCE: boolean;
        CANVAS: boolean;
        NOSCRIPT: boolean;
        SCRIPT: boolean;
        SVG: boolean;
    };
    /**
     * Search for a `query` in a DOM tree.
     *
     * @param query Regular expression to search
     * @param rootNode DOM root node to search in
     * @returns The list of matches
     */
    static search(query: RegExp, rootNode: Node): Promise<IHTMLSearchMatch[]>;
}
/**
 * Generic DOM tree search provider.
 */
export declare class GenericSearchProvider extends SearchProvider<Widget> {
    /**
     * Report whether or not this provider has the ability to search on the given object
     */
    static isApplicable(domain: Widget): boolean;
    /**
     * Instantiate a generic search provider for the widget.
     *
     * #### Notes
     * The widget provided is always checked using `isApplicable` before calling
     * this factory.
     *
     * @param widget The widget to search on
     * @param registry The search provider registry
     * @param translator [optional] The translator object
     *
     * @returns The search provider on the widget
     */
    static createNew(widget: Widget, registry: ISearchProviderRegistry, translator?: ITranslator): ISearchProvider;
    /**
     * The current index of the selected match.
     */
    get currentMatchIndex(): number | null;
    /**
     * The current match
     */
    get currentMatch(): IHTMLSearchMatch | null;
    /**
     * The current matches
     */
    get matches(): IHTMLSearchMatch[];
    /**
     * The number of matches.
     */
    get matchesCount(): number | null;
    /**
     * Set to true if the widget under search is read-only, false
     * if it is editable.  Will be used to determine whether to show
     * the replace option.
     */
    readonly isReadOnly = true;
    /**
     * Clear currently highlighted match.
     */
    clearHighlight(): Promise<void>;
    /**
     * Dispose of the resources held by the search provider.
     *
     * #### Notes
     * If the object's `dispose` method is called more than once, all
     * calls made after the first will be a no-op.
     *
     * #### Undefined Behavior
     * It is undefined behavior to use any functionality of the object
     * after it has been disposed unless otherwise explicitly noted.
     */
    dispose(): void;
    /**
     * Move the current match indicator to the next match.
     *
     * @param loop Whether to loop within the matches list.
     *
     * @returns A promise that resolves once the action has completed.
     */
    highlightNext(loop?: boolean): Promise<IHTMLSearchMatch | undefined>;
    /**
     * Move the current match indicator to the previous match.
     *
     * @param loop Whether to loop within the matches list.
     *
     * @returns A promise that resolves once the action has completed.
     */
    highlightPrevious(loop?: boolean): Promise<IHTMLSearchMatch | undefined>;
    /**
     * Replace the currently selected match with the provided text
     *
     * @param newText The replacement text
     * @param loop Whether to loop within the matches list.
     *
     * @returns A promise that resolves with a boolean indicating whether a replace occurred.
     */
    replaceCurrentMatch(newText: string, loop?: boolean): Promise<boolean>;
    /**
     * Replace all matches in the notebook with the provided text
     *
     * @param newText The replacement text
     *
     * @returns A promise that resolves with a boolean indicating whether a replace occurred.
     */
    replaceAllMatches(newText: string): Promise<boolean>;
    /**
     * Initialize the search using the provided options.  Should update the UI
     * to highlight all matches and "select" whatever the first match should be.
     *
     * @param query A RegExp to be use to perform the search
     * @param filters Filter parameters to pass to provider
     */
    startQuery(query: RegExp | null, filters?: {}): Promise<void>;
    /**
     * Clear the highlighted matches and any internal state.
     */
    endQuery(): Promise<void>;
    private _highlightNext;
    private _onWidgetChanged;
    private _query;
    private _currentMatchIndex;
    private _matches;
    private _mutationObserver;
    private _markNodes;
}
