// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
import { KernelMessage, ServiceManager } from '@jupyterlab/services';
import { changeKernel, KERNEL_MODELS, KernelMock, ServiceManagerMock, SessionConnectionMock } from '@jupyterlab/services/lib/testutils';
import { UUID } from '@lumino/coreutils';
import { AttachedProperty } from '@lumino/properties';
import { Signal } from '@lumino/signaling';
import { Context } from './context';
import { TextModelFactory } from './default';
/**
 * Create a context for a file.
 */
export function createFileContext(path = UUID.uuid4() + '.txt', manager = Private.getManager()) {
    const factory = Private.textFactory;
    return new Context({ manager, factory, path });
}
export async function createFileContextWithKernel(path = UUID.uuid4() + '.txt', manager = Private.getManager()) {
    var _a;
    const factory = Private.textFactory;
    const specsManager = manager.kernelspecs;
    await specsManager.ready;
    return new Context({
        manager,
        factory,
        path,
        kernelPreference: {
            shouldStart: true,
            canStart: true,
            name: (_a = specsManager.specs) === null || _a === void 0 ? void 0 : _a.default
        }
    });
}
export async function createFileContextWithMockedServices(startKernel = false, manager) {
    const path = UUID.uuid4() + '.txt';
    manager = manager || new ServiceManagerMock();
    const factory = new TextModelFactory();
    const context = new Context({
        manager: manager || new ServiceManagerMock(),
        factory,
        path,
        kernelPreference: {
            shouldStart: startKernel,
            canStart: startKernel,
            autoStartDefault: startKernel
        }
    });
    await context.initialize(true);
    await context.sessionContext.initialize();
    return context;
}
/**
 * Create a session and return a session connection.
 */
export async function createSession(options) {
    const manager = Private.getManager().sessions;
    await manager.ready;
    return manager.startNew(options);
}
/**
 * Create a session context given a partial session model.
 *
 * @param model The session model to use.
 */
export function createSimpleSessionContext(model = {}) {
    const kernel = new KernelMock({ model: (model === null || model === void 0 ? void 0 : model.kernel) || {} });
    const session = new SessionConnectionMock({ model }, kernel);
    return new SessionContextMock({}, session);
}
/**
 * Emit an iopub message on a session context.
 *
 * @param sessionContext The session context
 * @param msg Message created with `KernelMessage.createMessage`
 */
export function emitIopubMessage(context, msg) {
    const kernel = context.session.kernel;
    const msgId = Private.lastMessageProperty.get(kernel);
    msg.parent_header.session = kernel.clientId;
    msg.parent_header.msg_id = msgId;
    kernel.iopubMessage.emit(msg);
}
/**
 * Forceably change the status of a session context.
 * An iopub message is emitted for the change.
 *
 * @param sessionContext The session context of interest.
 * @param newStatus The new kernel status.
 */
export function updateKernelStatus(sessionContext, newStatus) {
    const kernel = sessionContext.session.kernel;
    kernel.status = newStatus;
    sessionContext.statusChanged.emit(newStatus);
    const msg = KernelMessage.createMessage({
        session: kernel.clientId,
        channel: 'iopub',
        msgType: 'status',
        content: { execution_state: newStatus }
    });
    emitIopubMessage(sessionContext, msg);
}
/**
 * A mock session context.
 *
 * @param session The session connection object to use
 */
export const SessionContextMock = jest.fn((options, connection) => {
    const session = connection ||
        new SessionConnectionMock({
            model: {
                path: options.path || '',
                type: options.type || '',
                name: options.name || ''
            }
        }, null);
    const thisObject = {
        ...jest.requireActual('@jupyterlab/apputils'),
        ...options,
        path: session.path,
        type: session.type,
        name: session.name,
        session,
        dispose: jest.fn(),
        initialize: jest.fn(() => Promise.resolve(false)),
        ready: Promise.resolve(),
        changeKernel: jest.fn(partialModel => {
            return changeKernel(session.kernel || Private.RUNNING_KERNELS[0], partialModel);
        }),
        shutdown: jest.fn(() => Promise.resolve())
    };
    const disposedSignal = new Signal(thisObject);
    const propertyChangedSignal = new Signal(thisObject);
    const statusChangedSignal = new Signal(thisObject);
    const kernelChangedSignal = new Signal(thisObject);
    const iopubMessageSignal = new Signal(thisObject);
    session.statusChanged.connect((_, args) => {
        statusChangedSignal.emit(args);
    }, thisObject);
    session.iopubMessage.connect((_, args) => {
        iopubMessageSignal.emit(args);
    });
    session.kernelChanged.connect((_, args) => {
        kernelChangedSignal.emit(args);
    });
    session.pendingInput.connect((_, args) => {
        thisObject.pendingInput = args;
    });
    thisObject.statusChanged = statusChangedSignal;
    thisObject.kernelChanged = kernelChangedSignal;
    thisObject.iopubMessage = iopubMessageSignal;
    thisObject.propertyChanged = propertyChangedSignal;
    thisObject.disposed = disposedSignal;
    thisObject.session = session;
    thisObject.pendingInput = false;
    return thisObject;
});
/**
 * A namespace for module private data.
 */
var Private;
(function (Private) {
    let manager;
    Private.textFactory = new TextModelFactory();
    /**
     * Get or create the service manager singleton.
     */
    function getManager() {
        if (!manager) {
            manager = new ServiceManager({ standby: 'never' });
        }
        return manager;
    }
    Private.getManager = getManager;
    Private.lastMessageProperty = new AttachedProperty({
        name: 'lastMessageId',
        create: () => ''
    });
    // This list of running kernels simply mirrors the KERNEL_MODELS and KERNELSPECS lists
    Private.RUNNING_KERNELS = KERNEL_MODELS.map((model, _) => {
        return new KernelMock({ model });
    });
})(Private || (Private = {}));
/**
 * A mock document widget opener.
 */
export class DocumentWidgetOpenerMock {
    constructor() {
        this._opened = new Signal(this);
    }
    get opened() {
        return this._opened;
    }
    open(widget) {
        // no-op, just emit the signal
        this._opened.emit(widget);
    }
}
//# sourceMappingURL=testutils.js.map