import type * as nbformat from '@jupyterlab/nbformat';
import { PartialJSONValue } from '@lumino/coreutils';
import { ISignal, Signal } from '@lumino/signaling';
import { Awareness } from 'y-protocols/awareness';
import * as Y from 'yjs';
import type { CellChange, IMapChange, ISharedAttachmentsCell, ISharedBaseCell, ISharedCodeCell, ISharedMarkdownCell, ISharedRawCell, SharedCell } from './api.js';
import { IYText } from './ytext.js';
import { YNotebook } from './ynotebook.js';
/**
 * Cell type.
 */
export type YCellType = YRawCell | YCodeCell | YMarkdownCell;
/**
 * Create a new shared cell model given the YJS shared type.
 */
export declare const createCellModelFromSharedType: (type: Y.Map<any>, options?: SharedCell.IOptions) => YCellType;
/**
 * Create a new cell that can be inserted in an existing shared model.
 *
 * If no notebook is specified the cell will be standalone.
 *
 * @param cell Cell JSON representation
 * @param notebook Notebook to which the cell will be added
 */
export declare const createCell: (cell: SharedCell.Cell, notebook?: YNotebook) => YCodeCell | YMarkdownCell | YRawCell;
/**
 * Create a new cell that cannot be inserted in an existing shared model.
 *
 * @param cell Cell JSON representation
 */
export declare const createStandaloneCell: (cell: SharedCell.Cell) => YCellType;
export declare class YBaseCell<Metadata extends nbformat.IBaseCellMetadata> implements ISharedBaseCell<Metadata>, IYText {
    /**
     * Create a new YCell that works standalone. It cannot be
     * inserted into a YNotebook because the Yjs model is already
     * attached to an anonymous Y.Doc instance.
     */
    static create(id?: string): YBaseCell<any>;
    /**
     * Base cell constructor
     *
     * ### Notes
     * Don't use the constructor directly - prefer using ``YNotebook.insertCell``
     *
     * The ``ysource`` is needed because ``ymodel.get('source')`` will
     * not return the real source if the model is not yet attached to
     * a document. Requesting it explicitly allows to introspect a non-empty
     * source before the cell is attached to the document.
     *
     * @param ymodel Cell map
     * @param ysource Cell source
     * @param options \{ notebook?: The notebook the cell is attached to \}
     * @param ymetadata Cell metadata
     */
    constructor(ymodel: Y.Map<any>, ysource: Y.Text, options?: SharedCell.IOptions, ymetadata?: Y.Map<any>);
    /**
     * Cell notebook awareness or null if the cell is standalone.
     */
    get awareness(): Awareness | null;
    /**
     * The type of the cell.
     */
    get cell_type(): any;
    /**
     * The changed signal.
     */
    get changed(): ISignal<this, CellChange>;
    /**
     * Signal emitted when the cell is disposed.
     */
    get disposed(): ISignal<this, void>;
    /**
     * Cell id
     */
    get id(): string;
    /**
     * Whether the model has been disposed or not.
     */
    get isDisposed(): boolean;
    /**
     * Whether the cell is standalone or not.
     *
     * If the cell is standalone. It cannot be
     * inserted into a YNotebook because the Yjs model is already
     * attached to an anonymous Y.Doc instance.
     */
    get isStandalone(): boolean;
    /**
     * Cell metadata.
     *
     * #### Notes
     * You should prefer to access and modify the specific key of interest.
     */
    get metadata(): Partial<Metadata>;
    set metadata(v: Partial<Metadata>);
    /**
     * Signal triggered when the cell metadata changes.
     */
    get metadataChanged(): ISignal<this, IMapChange>;
    /**
     * The notebook that this cell belongs to.
     */
    get notebook(): YNotebook | null;
    /**
     * Cell input content.
     */
    get source(): string;
    set source(v: string);
    /**
     * The cell undo manager.
     */
    get undoManager(): Y.UndoManager | null;
    /**
     * Defer setting the undo manager as it requires the
     * cell to be attached to the notebook Y document.
     */
    setUndoManager(): void;
    readonly ymodel: Y.Map<any>;
    get ysource(): Y.Text;
    /**
     * Whether the object can undo changes.
     */
    canUndo(): boolean;
    /**
     * Whether the object can redo changes.
     */
    canRedo(): boolean;
    /**
     * Clear the change stack.
     */
    clearUndoHistory(): void;
    /**
     * Undo an operation.
     */
    undo(): void;
    /**
     * Redo an operation.
     */
    redo(): void;
    /**
     * Dispose of the resources.
     */
    dispose(): void;
    /**
     * Get cell id.
     *
     * @returns Cell id
     */
    getId(): string;
    /**
     * Gets cell's source.
     *
     * @returns Cell's source.
     */
    getSource(): string;
    /**
     * Sets cell's source.
     *
     * @param value: New source.
     */
    setSource(value: string): void;
    /**
     * Replace content from `start' to `end` with `value`.
     *
     * @param start: The start index of the range to replace (inclusive).
     *
     * @param end: The end index of the range to replace (exclusive).
     *
     * @param value: New source (optional).
     */
    updateSource(start: number, end: number, value?: string): void;
    /**
     * Delete a metadata cell.
     *
     * @param key The key to delete
     */
    deleteMetadata(key: string): void;
    /**
     * Returns all or a single metadata associated with the cell.
     *
     * @param key The metadata key
     * @returns cell's metadata.
     */
    getMetadata(): Partial<Metadata>;
    getMetadata(key: string): PartialJSONValue | undefined;
    /**
     * Sets all or a single cell metadata.
     *
     * If only one argument is provided, it will override all cell metadata.
     * Otherwise a single key will be set to a new value.
     *
     * @param metadata Cell's metadata key or cell's metadata.
     * @param value Metadata value
     */
    setMetadata(metadata: Partial<Metadata>): void;
    setMetadata(metadata: string, value: PartialJSONValue): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IBaseCell;
    /**
     * Perform a transaction. While the function f is called, all changes to the shared
     * document are bundled into a single event.
     *
     * @param f Transaction to execute
     * @param undoable Whether to track the change in the action history or not (default `true`)
     */
    transact(f: () => void, undoable?: boolean): void;
    /**
     * Extract changes from YJS events
     *
     * @param events YJS events
     * @returns Cell changes
     */
    protected getChanges(events: Y.YEvent<any>[]): Partial<CellChange>;
    /**
     * Handle a change to the ymodel.
     */
    private _modelObserver;
    protected _metadataChanged: Signal<this, IMapChange<any>>;
    /**
     * The notebook that this cell belongs to.
     */
    protected _notebook: YNotebook | null;
    private _awareness;
    private _changed;
    private _disposed;
    private _isDisposed;
    private _prevSourceLength;
    private _undoManager;
    private _ymetadata;
    private _ysource;
}
/**
 * Shareable code cell.
 */
export declare class YCodeCell extends YBaseCell<nbformat.IBaseCellMetadata> implements ISharedCodeCell {
    /**
     * Create a new YCodeCell that works standalone. It cannot be
     * inserted into a YNotebook because the Yjs model is already
     * attached to an anonymous Y.Doc instance.
     */
    static create(id?: string): YCodeCell;
    /**
     * Code cell constructor
     *
     * ### Notes
     * Don't use the constructor directly - prefer using ``YNotebook.insertCell``
     *
     * The ``ysource`` is needed because ``ymodel.get('source')`` will
     * not return the real source if the model is not yet attached to
     * a document. Requesting it explicitly allows to introspect a non-empty
     * source before the cell is attached to the document.
     *
     * @param ymodel Cell map
     * @param ysource Cell source
     * @param youtputs Code cell outputs
     * @param options \{ notebook?: The notebook the cell is attached to \}
     * @param ymetadata Cell metadata
     */
    constructor(ymodel: Y.Map<any>, ysource: Y.Text, youtputs: Y.Array<any>, options?: SharedCell.IOptions, ymetadata?: Y.Map<any>);
    /**
     * The type of the cell.
     */
    get cell_type(): 'code';
    /**
     * The code cell's prompt number. Will be null if the cell has not been run.
     */
    get execution_count(): number | null;
    set execution_count(count: number | null);
    /**
     * Cell outputs.
     */
    get outputs(): Array<nbformat.IOutput>;
    set outputs(v: Array<nbformat.IOutput>);
    /**
     * Execution, display, or stream outputs.
     */
    getOutputs(): Array<nbformat.IOutput>;
    /**
     * Replace all outputs.
     */
    setOutputs(outputs: Array<nbformat.IOutput>): void;
    /**
     * Replace content from `start' to `end` with `outputs`.
     *
     * @param start: The start index of the range to replace (inclusive).
     *
     * @param end: The end index of the range to replace (exclusive).
     *
     * @param outputs: New outputs (optional).
     */
    updateOutputs(start: number, end: number, outputs?: Array<nbformat.IOutput>): void;
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.ICodeCell;
    /**
     * Extract changes from YJS events
     *
     * @param events YJS events
     * @returns Cell changes
     */
    protected getChanges(events: Y.YEvent<any>[]): Partial<CellChange>;
    private _youtputs;
}
declare class YAttachmentCell extends YBaseCell<nbformat.IBaseCellMetadata> implements ISharedAttachmentsCell {
    /**
     * Cell attachments
     */
    get attachments(): nbformat.IAttachments | undefined;
    set attachments(v: nbformat.IAttachments | undefined);
    /**
     * Gets the cell attachments.
     *
     * @returns The cell attachments.
     */
    getAttachments(): nbformat.IAttachments | undefined;
    /**
     * Sets the cell attachments
     *
     * @param attachments: The cell attachments.
     */
    setAttachments(attachments: nbformat.IAttachments | undefined): void;
    /**
     * Extract changes from YJS events
     *
     * @param events YJS events
     * @returns Cell changes
     */
    protected getChanges(events: Y.YEvent<any>[]): Partial<CellChange>;
}
/**
 * Shareable raw cell.
 */
export declare class YRawCell extends YAttachmentCell implements ISharedRawCell {
    /**
     * Create a new YRawCell that works standalone. It cannot be
     * inserted into a YNotebook because the Yjs model is already
     * attached to an anonymous Y.Doc instance.
     */
    static create(id?: string): YRawCell;
    /**
     * String identifying the type of cell.
     */
    get cell_type(): 'raw';
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IRawCell;
}
/**
 * Shareable markdown cell.
 */
export declare class YMarkdownCell extends YAttachmentCell implements ISharedMarkdownCell {
    /**
     * Create a new YMarkdownCell that works standalone. It cannot be
     * inserted into a YNotebook because the Yjs model is already
     * attached to an anonymous Y.Doc instance.
     */
    static create(id?: string): YMarkdownCell;
    /**
     * String identifying the type of cell.
     */
    get cell_type(): 'markdown';
    /**
     * Serialize the model to JSON.
     */
    toJSON(): nbformat.IMarkdownCell;
}
export {};
