/****************************************************************************/
// Eclipse SUMO, Simulation of Urban MObility; see https://eclipse.dev/sumo
// Copyright (C) 2001-2026 German Aerospace Center (DLR) and others.
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0/
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License 2.0 are satisfied: GNU General Public License, version 2
// or later which is available at
// https://www.gnu.org/licenses/old-licenses/gpl-2.0-standalone.html
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0-or-later
/****************************************************************************/
/// @file    TraCIServerAPI_VehicleType.cpp
/// @author  Daniel Krajzewicz
/// @author  Jakob Erdmann
/// @author  Michael Behrisch
/// @author  Laura Bieker
/// @author  Robert Hilbrich
/// @author  Gregor Laemmel
/// @date    07.05.2009
///
// APIs for getting/setting vehicle type values via TraCI
/****************************************************************************/
#include <config.h>

#include <limits>
#include <utils/emissions/PollutantsInterface.h>
#include <microsim/MSNet.h>
#include <microsim/MSVehicleType.h>
#include <libsumo/StorageHelper.h>
#include <libsumo/TraCIConstants.h>
#include <libsumo/VehicleType.h>
#include "TraCIServerAPI_VehicleType.h"


// ===========================================================================
// method definitions
// ===========================================================================
bool
TraCIServerAPI_VehicleType::processSet(TraCIServer& server, tcpip::Storage& inputStorage,
                                       tcpip::Storage& outputStorage) {
    std::string warning = ""; // additional description for response
    // variable
    int variable = inputStorage.readUnsignedByte();
    if (variable != libsumo::VAR_LENGTH && variable != libsumo::VAR_MAXSPEED && variable != libsumo::VAR_VEHICLECLASS
            && variable != libsumo::VAR_SPEED_FACTOR && variable != libsumo::VAR_SPEED_DEVIATION && variable != libsumo::VAR_EMISSIONCLASS
            && variable != libsumo::VAR_WIDTH && variable != libsumo::VAR_MINGAP && variable != libsumo::VAR_SHAPECLASS
            && variable != libsumo::VAR_ACCEL && variable != libsumo::VAR_IMPERFECTION
            && variable != libsumo::VAR_DECEL && variable != libsumo::VAR_EMERGENCY_DECEL && variable != libsumo::VAR_APPARENT_DECEL
            && variable != libsumo::VAR_TAU && variable != libsumo::VAR_COLOR && variable != libsumo::VAR_ACTIONSTEPLENGTH
            && variable != libsumo::VAR_SCALE
            && variable != libsumo::VAR_HEIGHT
            && variable != libsumo::VAR_MASS
            && variable != libsumo::VAR_MINGAP_LAT
            && variable != libsumo::VAR_MAXSPEED_LAT
            && variable != libsumo::VAR_LATALIGNMENT
            && variable != libsumo::VAR_BOARDING_DURATION
            && variable != libsumo::VAR_IMPATIENCE
            && variable != libsumo::VAR_PARAMETER
            && variable != libsumo::COPY
       ) {
        return server.writeErrorStatusCmd(libsumo::CMD_SET_VEHICLETYPE_VARIABLE,
                                          "Change Vehicle Type State: unsupported variable " + toHex(variable, 2)
                                          + " specified", outputStorage);
    }
    // id
    std::string id = inputStorage.readString();
//    MSVehicleType* v = libsumo::VehicleType::getVType(id);
//    if (v == 0) {
//        return server.writeErrorStatusCmd(libsumo::CMD_SET_VEHICLETYPE_VARIABLE, "Vehicle type '" + id + "' is not known",
//                                          outputStorage);
//    }
    // process
    try {
        if (setVariable(libsumo::CMD_SET_VEHICLETYPE_VARIABLE, variable, id, server, inputStorage, outputStorage)) {
            server.writeStatusCmd(libsumo::CMD_SET_VEHICLETYPE_VARIABLE, libsumo::RTYPE_OK, warning, outputStorage);
            return true;
        }
    } catch (ProcessError& e) {
        return server.writeErrorStatusCmd(libsumo::CMD_SET_VEHICLETYPE_VARIABLE, e.what(), outputStorage);
    } catch (libsumo::TraCIException& e) {
        return server.writeErrorStatusCmd(libsumo::CMD_SET_VEHICLETYPE_VARIABLE, e.what(), outputStorage);
    }
    return false;
}


bool
TraCIServerAPI_VehicleType::setVariable(const int cmd, const int variable,
                                        const std::string& id, TraCIServer& server,
                                        tcpip::Storage& inputStorage, tcpip::Storage& outputStorage) {
    switch (variable) {
        case libsumo::VAR_LENGTH: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting length requires a double.");
            if (value <= 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid length.", outputStorage);
            }
            libsumo::VehicleType::setLength(id, value);
            break;
        }
        case libsumo::VAR_HEIGHT: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting height requires a double.");
            if (value <= 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid height.", outputStorage);
            }
            libsumo::VehicleType::setHeight(id, value);
            break;
        }
        case libsumo::VAR_MASS: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting mass requires a double.");
            if (value <= 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid mass.", outputStorage);
            }
            libsumo::VehicleType::setMass(id, value);
            break;
        }
        case libsumo::VAR_MAXSPEED: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting maximum speed requires a double.");
            if (value <= 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid maximum speed.", outputStorage);
            }
            libsumo::VehicleType::setMaxSpeed(id, value);
            break;
        }
        case libsumo::VAR_VEHICLECLASS: {
            libsumo::VehicleType::setVehicleClass(id, StoHelp::readTypedString(inputStorage, "Setting vehicle class requires a string."));
            break;
        }
        case libsumo::VAR_SPEED_FACTOR: {
            double value = StoHelp::readTypedDouble(inputStorage, "Setting speed factor requires a double.");
            if (value <= 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid speed factor.", outputStorage);
            }
            libsumo::VehicleType::setSpeedFactor(id, value);
            break;
        }
        case libsumo::VAR_SPEED_DEVIATION: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting speed deviation requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid speed deviation.", outputStorage);
            }
            libsumo::VehicleType::setSpeedDeviation(id, value);
            break;
        }
        case libsumo::VAR_EMISSIONCLASS: {
            libsumo::VehicleType::setEmissionClass(id, StoHelp::readTypedString(inputStorage, "Setting emission class requires a string."));
            break;
        }
        case libsumo::VAR_WIDTH: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting width requires a double.");
            if (value <= 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid width.", outputStorage);
            }
            libsumo::VehicleType::setWidth(id, value);
            break;
        }
        case libsumo::VAR_MINGAP: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting minimum gap requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid minimum gap.", outputStorage);
            }
            libsumo::VehicleType::setMinGap(id, value);
            break;
        }
        case libsumo::VAR_MINGAP_LAT: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting minimum lateral gap requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid minimum lateral gap.", outputStorage);
            }
            libsumo::VehicleType::setMinGapLat(id, value);
            break;
        }
        case libsumo::VAR_MAXSPEED_LAT: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting maximum lateral speed requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid maximum lateral speed.", outputStorage);
            }
            libsumo::VehicleType::setMaxSpeedLat(id, value);
            break;
        }
        case libsumo::VAR_LATALIGNMENT: {
            libsumo::VehicleType::setLateralAlignment(id, StoHelp::readTypedString(inputStorage, "Setting preferred lateral alignment requires a string."));
            break;
        }
        case libsumo::VAR_SHAPECLASS: {
            libsumo::VehicleType::setShapeClass(id, StoHelp::readTypedString(inputStorage, "Setting vehicle shape requires a string."));
            break;
        }
        case libsumo::VAR_ACCEL: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting acceleration requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid acceleration.", outputStorage);
            }
            libsumo::VehicleType::setAccel(id, value);
            break;
        }
        case libsumo::VAR_DECEL: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting deceleration requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid deceleration.", outputStorage);
            }
            libsumo::VehicleType::setDecel(id, value);
            break;
        }
        case libsumo::VAR_EMERGENCY_DECEL: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting emergency deceleration requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid emergency deceleration.", outputStorage);
            }
            libsumo::VehicleType::setEmergencyDecel(id, value);
            break;
        }
        case libsumo::VAR_APPARENT_DECEL: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting apparent deceleration requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid apparent deceleration.", outputStorage);
            }
            libsumo::VehicleType::setApparentDecel(id, value);
            break;
        }
        case libsumo::VAR_SCALE: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Traffic scale requires a double.");
            if (value < 0.0) {
                return server.writeErrorStatusCmd(cmd, "Invalid traffic scale.", outputStorage);
            }
            libsumo::VehicleType::setScale(id, value);
            break;
        }
        case libsumo::VAR_ACTIONSTEPLENGTH: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting action step length requires a double.");
            if (fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(libsumo::CMD_SET_VEHICLE_VARIABLE, "Invalid action step length.", outputStorage);
            }
            bool resetActionOffset = value >= 0.0;
            libsumo::VehicleType::setActionStepLength(id, fabs(value), resetActionOffset);
            break;
        }
        case libsumo::VAR_IMPERFECTION: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting driver imperfection requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid driver imperfection.", outputStorage);
            }
            libsumo::VehicleType::setImperfection(id, value);
            break;
        }
        case libsumo::VAR_TAU: {
            const double value = StoHelp::readTypedDouble(inputStorage, "Setting headway time requires a double.");
            if (value < 0.0 || fabs(value) == std::numeric_limits<double>::infinity()) {
                return server.writeErrorStatusCmd(cmd, "Invalid headway time.", outputStorage);
            }
            libsumo::VehicleType::setTau(id, value);
            break;
        }
        case libsumo::VAR_IMPATIENCE: {
            libsumo::VehicleType::setImpatience(id, StoHelp::readTypedDouble(inputStorage, "Setting impatience requires a double."));
            break;
        }
        case libsumo::VAR_BOARDING_DURATION: {
            libsumo::VehicleType::setBoardingDuration(id, StoHelp::readTypedDouble(inputStorage, "Setting boarding duration requires a double."));
            break;
        }
        case libsumo::VAR_COLOR: {
            libsumo::VehicleType::setColor(id, StoHelp::readTypedColor(inputStorage, "The color must be given using the according type."));
            break;
        }
        case libsumo::COPY: {
            libsumo::VehicleType::copy(id, StoHelp::readTypedString(inputStorage, "Copying a vehicle type requires a string."));
            break;
        }
        case libsumo::VAR_PARAMETER: {
            StoHelp::readCompound(inputStorage, 2, "A compound object of size 2 is needed for setting a parameter.");
            const std::string name = StoHelp::readTypedString(inputStorage, "The name of the parameter must be given as a string.");
            const std::string value = StoHelp::readTypedString(inputStorage, "The value of the parameter must be given as a string.");
            libsumo::VehicleType::setParameter(id, name, value);
            break;
        }
        default:
            break;
    }
    return true;
}


/****************************************************************************/
