"""MIME type definitions based on [IANA media types](https://www.iana.org/assignments/media-types/media-types.xhtml)."""

from __future__ import annotations

import sys
from dataclasses import dataclass
from functools import lru_cache

if sys.version_info >= (3, 11):
    from enum import StrEnum
else:
    from enum import Enum

    class StrEnum(str, Enum):
        """A backport of StrEnum for Python < 3.11."""

        pass


from typing import Any

from pydantic import GetCoreSchemaHandler, GetJsonSchemaHandler
from pydantic_core import PydanticCustomError, core_schema


@dataclass
class MimeTypeInfo:
    mime_type: str
    category: str


@lru_cache
def _all_mime_types() -> list[MimeTypeInfo]:
    """Build a list of all MIME types from the enum classes."""
    mime_types = []
    for enum_class, category in [
        (_ApplicationEnum, 'application'),
        (_AudioEnum, 'audio'),
        (_FontEnum, 'font'),
        (_HapticsEnum, 'haptics'),
        (_ImageEnum, 'image'),
        (_MessageEnum, 'message'),
        (_ModelEnum, 'model'),
        (_MultipartEnum, 'multipart'),
        (_TextEnum, 'text'),
        (_VideoEnum, 'video'),
    ]:
        for mime in enum_class:
            mime_types.append(MimeTypeInfo(mime_type=mime.value, category=category))
    return mime_types


@lru_cache
def _index_by_mime_type() -> dict[str, MimeTypeInfo]:
    """Create an index of MIME types by their full type string (case-insensitive)."""
    return {mime.mime_type.lower(): mime for mime in _all_mime_types()}


@lru_cache
def _index_by_category() -> dict[str, list[str]]:
    """Create an index of MIME types grouped by category."""
    result: dict[str, list[str]] = {}
    for mime in _all_mime_types():
        if mime.category not in result:
            result[mime.category] = []
        result[mime.category].append(mime.mime_type)
    return result


class _ApplicationEnum(StrEnum):  # pragma: no cover
    APPLICATION_1D_INTERLEAVED_PARITYFEC = 'application/1d-interleaved-parityfec'
    APPLICATION_3GPDASH_QOE_REPORT_XML = 'application/3gpdash-qoe-report+xml'
    APPLICATION_3GPPHAL_JSON = 'application/3gppHal+json'
    APPLICATION_3GPPHALFORMS_JSON = 'application/3gppHalForms+json'
    APPLICATION_3GPP_IMS_XML = 'application/3gpp-ims+xml'
    APPLICATION_3GPP_MBS_OBJECT_MANIFEST_JSON = 'application/3gpp-mbs-object-manifest+json'
    APPLICATION_3GPP_MBS_USER_SERVICE_DESCRIPTIONS_JSON = 'application/3gpp-mbs-user-service-descriptions+json'
    APPLICATION_A2L = 'application/A2L'
    APPLICATION_ACE_GROUPCOMM_CBOR = 'application/ace-groupcomm+cbor'
    APPLICATION_ACE_TRL_CBOR = 'application/ace-trl+cbor'
    APPLICATION_ACE_CBOR = 'application/ace+cbor'
    APPLICATION_ACE_JSON = 'application/ace+json'
    APPLICATION_ACTIVEMESSAGE = 'application/activemessage'
    APPLICATION_ACTIVITY_JSON = 'application/activity+json'
    APPLICATION_AIF_CBOR = 'application/aif+cbor'
    APPLICATION_AIF_JSON = 'application/aif+json'
    APPLICATION_ALTO_CDNI_JSON = 'application/alto-cdni+json'
    APPLICATION_ALTO_CDNIFILTER_JSON = 'application/alto-cdnifilter+json'
    APPLICATION_ALTO_COSTMAP_JSON = 'application/alto-costmap+json'
    APPLICATION_ALTO_COSTMAPFILTER_JSON = 'application/alto-costmapfilter+json'
    APPLICATION_ALTO_DIRECTORY_JSON = 'application/alto-directory+json'
    APPLICATION_ALTO_ENDPOINTPROP_JSON = 'application/alto-endpointprop+json'
    APPLICATION_ALTO_ENDPOINTPROPPARAMS_JSON = 'application/alto-endpointpropparams+json'
    APPLICATION_ALTO_ENDPOINTCOST_JSON = 'application/alto-endpointcost+json'
    APPLICATION_ALTO_ENDPOINTCOSTPARAMS_JSON = 'application/alto-endpointcostparams+json'
    APPLICATION_ALTO_ERROR_JSON = 'application/alto-error+json'
    APPLICATION_ALTO_NETWORKMAPFILTER_JSON = 'application/alto-networkmapfilter+json'
    APPLICATION_ALTO_NETWORKMAP_JSON = 'application/alto-networkmap+json'
    APPLICATION_ALTO_PROPMAP_JSON = 'application/alto-propmap+json'
    APPLICATION_ALTO_PROPMAPPARAMS_JSON = 'application/alto-propmapparams+json'
    APPLICATION_ALTO_TIPS_JSON = 'application/alto-tips+json'
    APPLICATION_ALTO_TIPSPARAMS_JSON = 'application/alto-tipsparams+json'
    APPLICATION_ALTO_UPDATESTREAMCONTROL_JSON = 'application/alto-updatestreamcontrol+json'
    APPLICATION_ALTO_UPDATESTREAMPARAMS_JSON = 'application/alto-updatestreamparams+json'
    APPLICATION_AML = 'application/AML'
    APPLICATION_ANDREW_INSET = 'application/andrew-inset'
    APPLICATION_APPLEFILE = 'application/applefile'
    APPLICATION_AT_JWT = 'application/at+jwt'
    APPLICATION_ATF = 'application/ATF'
    APPLICATION_ATFX = 'application/ATFX'
    APPLICATION_ATOM_XML = 'application/atom+xml'
    APPLICATION_ATOMCAT_XML = 'application/atomcat+xml'
    APPLICATION_ATOMDELETED_XML = 'application/atomdeleted+xml'
    APPLICATION_ATOMICMAIL = 'application/atomicmail'
    APPLICATION_ATOMSVC_XML = 'application/atomsvc+xml'
    APPLICATION_ATSC_DWD_XML = 'application/atsc-dwd+xml'
    APPLICATION_ATSC_DYNAMIC_EVENT_MESSAGE = 'application/atsc-dynamic-event-message'
    APPLICATION_ATSC_HELD_XML = 'application/atsc-held+xml'
    APPLICATION_ATSC_RDT_JSON = 'application/atsc-rdt+json'
    APPLICATION_ATSC_RSAT_XML = 'application/atsc-rsat+xml'
    APPLICATION_ATXML = 'application/ATXML'
    APPLICATION_AUTH_POLICY_XML = 'application/auth-policy+xml'
    APPLICATION_AUTOMATIONML_AML_XML = 'application/automationml-aml+xml'
    APPLICATION_AUTOMATIONML_AMLX_ZIP = 'application/automationml-amlx+zip'
    APPLICATION_BACNET_XDD_ZIP = 'application/bacnet-xdd+zip'
    APPLICATION_BATCH_SMTP = 'application/batch-SMTP'
    APPLICATION_BEEP_XML = 'application/beep+xml'
    APPLICATION_BUFR = 'application/bufr'
    APPLICATION_C2PA = 'application/c2pa'
    APPLICATION_CALENDAR_JSON = 'application/calendar+json'
    APPLICATION_CALENDAR_XML = 'application/calendar+xml'
    APPLICATION_CALL_COMPLETION = 'application/call-completion'
    APPLICATION_CALS_1840 = 'application/CALS-1840'
    APPLICATION_CAPTIVE_JSON = 'application/captive+json'
    APPLICATION_CBOR = 'application/cbor'
    APPLICATION_CBOR_SEQ = 'application/cbor-seq'
    APPLICATION_CCCEX = 'application/cccex'
    APPLICATION_CCMP_XML = 'application/ccmp+xml'
    APPLICATION_CCXML_XML = 'application/ccxml+xml'
    APPLICATION_CDA_XML = 'application/cda+xml'
    APPLICATION_CDFX_XML = 'application/CDFX+XML'
    APPLICATION_CDMI_CAPABILITY = 'application/cdmi-capability'
    APPLICATION_CDMI_CONTAINER = 'application/cdmi-container'
    APPLICATION_CDMI_DOMAIN = 'application/cdmi-domain'
    APPLICATION_CDMI_OBJECT = 'application/cdmi-object'
    APPLICATION_CDMI_QUEUE = 'application/cdmi-queue'
    APPLICATION_CDNI = 'application/cdni'
    APPLICATION_CE_CBOR = 'application/ce+cbor'
    APPLICATION_CEA = 'application/CEA'
    APPLICATION_CEA_2018_XML = 'application/cea-2018+xml'
    APPLICATION_CELLML_XML = 'application/cellml+xml'
    APPLICATION_CFW = 'application/cfw'
    APPLICATION_CID = 'application/cid'
    APPLICATION_CID_EDHOC_CBOR_SEQ = 'application/cid-edhoc+cbor-seq'
    APPLICATION_CITY_JSON = 'application/city+json'
    APPLICATION_CITY_JSON_SEQ = 'application/city+json-seq'
    APPLICATION_CLR = 'application/clr'
    APPLICATION_CLUE_INFO_XML = 'application/clue_info+xml'
    APPLICATION_CLUE_XML = 'application/clue+xml'
    APPLICATION_CMS = 'application/cms'
    APPLICATION_CNRP_XML = 'application/cnrp+xml'
    APPLICATION_COAP_EAP = 'application/coap-eap'
    APPLICATION_COAP_GROUP_JSON = 'application/coap-group+json'
    APPLICATION_COAP_PAYLOAD = 'application/coap-payload'
    APPLICATION_COMMONGROUND = 'application/commonground'
    APPLICATION_CONCISE_PROBLEM_DETAILS_CBOR = 'application/concise-problem-details+cbor'
    APPLICATION_CONFERENCE_INFO_XML = 'application/conference-info+xml'
    APPLICATION_CPL_XML = 'application/cpl+xml'
    APPLICATION_COSE = 'application/cose'
    APPLICATION_COSE_KEY = 'application/cose-key'
    APPLICATION_COSE_KEY_SET = 'application/cose-key-set'
    APPLICATION_COSE_X509 = 'application/cose-x509'
    APPLICATION_CSRATTRS = 'application/csrattrs'
    APPLICATION_CSTA_XML = 'application/csta+xml'
    APPLICATION_CSTADATA_XML = 'application/CSTAdata+xml'
    APPLICATION_CSVM_JSON = 'application/csvm+json'
    APPLICATION_CWL = 'application/cwl'
    APPLICATION_CWL_JSON = 'application/cwl+json'
    APPLICATION_CWL_YAML = 'application/cwl+yaml'
    APPLICATION_CWT = 'application/cwt'
    APPLICATION_CYBERCASH = 'application/cybercash'
    APPLICATION_DASH_XML = 'application/dash+xml'
    APPLICATION_DASH_PATCH_XML = 'application/dash-patch+xml'
    APPLICATION_DASHDELTA = 'application/dashdelta'
    APPLICATION_DAVMOUNT_XML = 'application/davmount+xml'
    APPLICATION_DCA_RFT = 'application/dca-rft'
    APPLICATION_DCD = 'application/DCD'
    APPLICATION_DEC_DX = 'application/dec-dx'
    APPLICATION_DIALOG_INFO_XML = 'application/dialog-info+xml'
    APPLICATION_DICOM = 'application/dicom'
    APPLICATION_DICOM_JSON = 'application/dicom+json'
    APPLICATION_DICOM_XML = 'application/dicom+xml'
    APPLICATION_DID = 'application/did'
    APPLICATION_DII = 'application/DII'
    APPLICATION_DIT = 'application/DIT'
    APPLICATION_DNS = 'application/dns'
    APPLICATION_DNS_JSON = 'application/dns+json'
    APPLICATION_DNS_MESSAGE = 'application/dns-message'
    APPLICATION_DOTS_CBOR = 'application/dots+cbor'
    APPLICATION_DPOP_JWT = 'application/dpop+jwt'
    APPLICATION_DSKPP_XML = 'application/dskpp+xml'
    APPLICATION_DSSC_DER = 'application/dssc+der'
    APPLICATION_DSSC_XML = 'application/dssc+xml'
    APPLICATION_DVCS = 'application/dvcs'
    APPLICATION_EAT_CWT = 'application/eat+cwt'
    APPLICATION_EAT_JWT = 'application/eat+jwt'
    APPLICATION_EAT_BUN_CBOR = 'application/eat-bun+cbor'
    APPLICATION_EAT_BUN_JSON = 'application/eat-bun+json'
    APPLICATION_EAT_UCS_CBOR = 'application/eat-ucs+cbor'
    APPLICATION_EAT_UCS_JSON = 'application/eat-ucs+json'
    APPLICATION_ECMASCRIPT = 'application/ecmascript'  # OBSOLETED in favor of text/javascript
    APPLICATION_EDHOC_CBOR_SEQ = 'application/edhoc+cbor-seq'
    APPLICATION_EDI_CONSENT = 'application/EDI-consent'
    APPLICATION_EDIFACT = 'application/EDIFACT'
    APPLICATION_EDI_X12 = 'application/EDI-X12'
    APPLICATION_EFI = 'application/efi'
    APPLICATION_ELM_JSON = 'application/elm+json'
    APPLICATION_ELM_XML = 'application/elm+xml'
    APPLICATION_EMERGENCYCALLDATA_CAP_XML = 'application/EmergencyCallData.cap+xml'
    APPLICATION_EMERGENCYCALLDATA_COMMENT_XML = 'application/EmergencyCallData.Comment+xml'
    APPLICATION_EMERGENCYCALLDATA_CONTROL_XML = 'application/EmergencyCallData.Control+xml'
    APPLICATION_EMERGENCYCALLDATA_DEVICEINFO_XML = 'application/EmergencyCallData.DeviceInfo+xml'
    APPLICATION_EMERGENCYCALLDATA_ECALL_MSD = 'application/EmergencyCallData.eCall.MSD'
    APPLICATION_EMERGENCYCALLDATA_LEGACYESN_JSON = 'application/EmergencyCallData.LegacyESN+json'
    APPLICATION_EMERGENCYCALLDATA_PROVIDERINFO_XML = 'application/EmergencyCallData.ProviderInfo+xml'
    APPLICATION_EMERGENCYCALLDATA_SERVICEINFO_XML = 'application/EmergencyCallData.ServiceInfo+xml'
    APPLICATION_EMERGENCYCALLDATA_SUBSCRIBERINFO_XML = 'application/EmergencyCallData.SubscriberInfo+xml'
    APPLICATION_EMERGENCYCALLDATA_VEDS_XML = 'application/EmergencyCallData.VEDS+xml'
    APPLICATION_EMMA_XML = 'application/emma+xml'
    APPLICATION_EMOTIONML_XML = 'application/emotionml+xml'
    APPLICATION_ENCAPRTP = 'application/encaprtp'
    APPLICATION_ENTITY_STATEMENT_JWT = 'application/entity-statement+jwt'
    APPLICATION_EPP_XML = 'application/epp+xml'
    APPLICATION_EPUB_ZIP = 'application/epub+zip'
    APPLICATION_ESHOP = 'application/eshop'
    APPLICATION_EXAMPLE = 'application/example'
    APPLICATION_EXI = 'application/exi'
    APPLICATION_EXPECT_CT_REPORT_JSON = 'application/expect-ct-report+json'
    APPLICATION_EXPRESS = 'application/express'
    APPLICATION_FASTINFOSET = 'application/fastinfoset'
    APPLICATION_FASTSOAP = 'application/fastsoap'
    APPLICATION_FDF = 'application/fdf'
    APPLICATION_FDT_XML = 'application/fdt+xml'
    APPLICATION_FHIR_JSON = 'application/fhir+json'
    APPLICATION_FHIR_XML = 'application/fhir+xml'
    APPLICATION_FITS = 'application/fits'
    APPLICATION_FLEXFEC = 'application/flexfec'
    APPLICATION_FONT_SFNT_DEPRECATED_IN_FAVOR_OF_FONT_SFNT = 'application/font-sfnt'
    APPLICATION_FONT_TDPFR = 'application/font-tdpfr'
    APPLICATION_FONT_WOFF_DEPRECATED_IN_FAVOR_OF_FONT_WOFF = 'application/font-woff'
    APPLICATION_FRAMEWORK_ATTRIBUTES_XML = 'application/framework-attributes+xml'
    APPLICATION_GEO_JSON = 'application/geo+json'
    APPLICATION_GEO_JSON_SEQ = 'application/geo+json-seq'
    APPLICATION_GEOFEED_CSV = 'application/geofeed+csv'
    APPLICATION_GEOPACKAGE_SQLITE3 = 'application/geopackage+sqlite3'
    APPLICATION_GEOPOSE_JSON = 'application/geopose+json'
    APPLICATION_GEOXACML_JSON = 'application/geoxacml+json'
    APPLICATION_GEOXACML_XML = 'application/geoxacml+xml'
    APPLICATION_GLTF_BUFFER = 'application/gltf-buffer'
    APPLICATION_GML_XML = 'application/gml+xml'
    APPLICATION_GNAP_BINDING_JWS = 'application/gnap-binding-jws'
    APPLICATION_GNAP_BINDING_JWSD = 'application/gnap-binding-jwsd'
    APPLICATION_GNAP_BINDING_ROTATION_JWS = 'application/gnap-binding-rotation-jws'
    APPLICATION_GNAP_BINDING_ROTATION_JWSD = 'application/gnap-binding-rotation-jwsd'
    APPLICATION_GRIB = 'application/grib'
    APPLICATION_GZIP = 'application/gzip'
    APPLICATION_H224 = 'application/H224'
    APPLICATION_HELD_XML = 'application/held+xml'
    APPLICATION_HL7V2_XML = 'application/hl7v2+xml'
    APPLICATION_HTTP = 'application/http'
    APPLICATION_HYPERSTUDIO = 'application/hyperstudio'
    APPLICATION_IBE_KEY_REQUEST_XML = 'application/ibe-key-request+xml'
    APPLICATION_IBE_PKG_REPLY_XML = 'application/ibe-pkg-reply+xml'
    APPLICATION_IBE_PP_DATA = 'application/ibe-pp-data'
    APPLICATION_IGES = 'application/iges'
    APPLICATION_IM_ISCOMPOSING_XML = 'application/im-iscomposing+xml'
    APPLICATION_INDEX = 'application/index'
    APPLICATION_INDEX_CMD = 'application/index.cmd'
    APPLICATION_INDEX_OBJ = 'application/index.obj'
    APPLICATION_INDEX_RESPONSE = 'application/index.response'
    APPLICATION_INDEX_VND = 'application/index.vnd'
    APPLICATION_INKML_XML = 'application/inkml+xml'
    APPLICATION_IOTP = 'application/IOTP'
    APPLICATION_IPFIX = 'application/ipfix'
    APPLICATION_IPP = 'application/ipp'
    APPLICATION_ISUP = 'application/ISUP'
    APPLICATION_ITS_XML = 'application/its+xml'
    APPLICATION_JAVA_ARCHIVE = 'application/java-archive'
    APPLICATION_JAVASCRIPT = 'application/javascript'  # OBSOLETED in favor of text/javascript
    APPLICATION_JF2FEED_JSON = 'application/jf2feed+json'
    APPLICATION_JOSE = 'application/jose'
    APPLICATION_JOSE_JSON = 'application/jose+json'
    APPLICATION_JRD_JSON = 'application/jrd+json'
    APPLICATION_JSCALENDAR_JSON = 'application/jscalendar+json'
    APPLICATION_JSCONTACT_JSON = 'application/jscontact+json'
    APPLICATION_JSON = 'application/json'
    APPLICATION_JSON_PATCH_JSON = 'application/json-patch+json'
    APPLICATION_JSON_PATCH_QUERY_JSON = 'application/json-patch-query+json'
    APPLICATION_JSON_SEQ = 'application/json-seq'
    APPLICATION_JSONPATH = 'application/jsonpath'
    APPLICATION_JWK_JSON = 'application/jwk+json'
    APPLICATION_JWK_SET_JSON = 'application/jwk-set+json'
    APPLICATION_JWK_SET_JWT = 'application/jwk-set+jwt'
    APPLICATION_JWT = 'application/jwt'
    APPLICATION_KB_JWT = 'application/kb+jwt'
    APPLICATION_KBL_XML = 'application/kbl+xml'
    APPLICATION_KPML_REQUEST_XML = 'application/kpml-request+xml'
    APPLICATION_KPML_RESPONSE_XML = 'application/kpml-response+xml'
    APPLICATION_LD_JSON = 'application/ld+json'
    APPLICATION_LGR_XML = 'application/lgr+xml'
    APPLICATION_LINK_FORMAT = 'application/link-format'
    APPLICATION_LINKSET = 'application/linkset'
    APPLICATION_LINKSET_JSON = 'application/linkset+json'
    APPLICATION_LOAD_CONTROL_XML = 'application/load-control+xml'
    APPLICATION_LOGOUT_JWT = 'application/logout+jwt'
    APPLICATION_LOST_XML = 'application/lost+xml'
    APPLICATION_LOSTSYNC_XML = 'application/lostsync+xml'
    APPLICATION_LPF_ZIP = 'application/lpf+zip'
    APPLICATION_LXF = 'application/LXF'
    APPLICATION_MAC_BINHEX40 = 'application/mac-binhex40'
    APPLICATION_MACWRITEII = 'application/macwriteii'
    APPLICATION_MADS_XML = 'application/mads+xml'
    APPLICATION_MANIFEST_JSON = 'application/manifest+json'
    APPLICATION_MARC = 'application/marc'
    APPLICATION_MARCXML_XML = 'application/marcxml+xml'
    APPLICATION_MATHEMATICA = 'application/mathematica'
    APPLICATION_MATHML_XML = 'application/mathml+xml'
    APPLICATION_MATHML_CONTENT_XML = 'application/mathml-content+xml'
    APPLICATION_MATHML_PRESENTATION_XML = 'application/mathml-presentation+xml'
    APPLICATION_MBMS_ASSOCIATED_PROCEDURE_DESCRIPTION_XML = 'application/mbms-associated-procedure-description+xml'
    APPLICATION_MBMS_DEREGISTER_XML = 'application/mbms-deregister+xml'
    APPLICATION_MBMS_ENVELOPE_XML = 'application/mbms-envelope+xml'
    APPLICATION_MBMS_MSK_RESPONSE_XML = 'application/mbms-msk-response+xml'
    APPLICATION_MBMS_MSK_XML = 'application/mbms-msk+xml'
    APPLICATION_MBMS_PROTECTION_DESCRIPTION_XML = 'application/mbms-protection-description+xml'
    APPLICATION_MBMS_RECEPTION_REPORT_XML = 'application/mbms-reception-report+xml'
    APPLICATION_MBMS_REGISTER_RESPONSE_XML = 'application/mbms-register-response+xml'
    APPLICATION_MBMS_REGISTER_XML = 'application/mbms-register+xml'
    APPLICATION_MBMS_SCHEDULE_XML = 'application/mbms-schedule+xml'
    APPLICATION_MBMS_USER_SERVICE_DESCRIPTION_XML = 'application/mbms-user-service-description+xml'
    APPLICATION_MBOX = 'application/mbox'
    APPLICATION_MEDIA_CONTROL_XML = 'application/media_control+xml'
    APPLICATION_MEDIA_POLICY_DATASET_XML = 'application/media-policy-dataset+xml'
    APPLICATION_MEDIASERVERCONTROL_XML = 'application/mediaservercontrol+xml'
    APPLICATION_MERGE_PATCH_JSON = 'application/merge-patch+json'
    APPLICATION_METALINK4_XML = 'application/metalink4+xml'
    APPLICATION_METS_XML = 'application/mets+xml'
    APPLICATION_MF4 = 'application/MF4'
    APPLICATION_MIKEY = 'application/mikey'
    APPLICATION_MIPC = 'application/mipc'
    APPLICATION_MISSING_BLOCKS_CBOR_SEQ = 'application/missing-blocks+cbor-seq'
    APPLICATION_MMT_AEI_XML = 'application/mmt-aei+xml'
    APPLICATION_MMT_USD_XML = 'application/mmt-usd+xml'
    APPLICATION_MODS_XML = 'application/mods+xml'
    APPLICATION_MOSS_KEYS = 'application/moss-keys'
    APPLICATION_MOSS_SIGNATURE = 'application/moss-signature'
    APPLICATION_MOSSKEY_DATA = 'application/mosskey-data'
    APPLICATION_MOSSKEY_REQUEST = 'application/mosskey-request'
    APPLICATION_MP21 = 'application/mp21'
    APPLICATION_MP4 = 'application/mp4'
    APPLICATION_MPEG4_GENERIC = 'application/mpeg4-generic'
    APPLICATION_MPEG4_IOD = 'application/mpeg4-iod'
    APPLICATION_MPEG4_IOD_XMT = 'application/mpeg4-iod-xmt'
    APPLICATION_MRB_CONSUMER_XML = 'application/mrb-consumer+xml'
    APPLICATION_MRB_PUBLISH_XML = 'application/mrb-publish+xml'
    APPLICATION_MSC_IVR_XML = 'application/msc-ivr+xml'
    APPLICATION_MSC_MIXER_XML = 'application/msc-mixer+xml'
    APPLICATION_MSWORD = 'application/msword'
    APPLICATION_MUD_JSON = 'application/mud+json'
    APPLICATION_MULTIPART_CORE = 'application/multipart-core'
    APPLICATION_MXF = 'application/mxf'
    APPLICATION_N_QUADS = 'application/n-quads'
    APPLICATION_N_TRIPLES = 'application/n-triples'
    APPLICATION_NASDATA = 'application/nasdata'
    APPLICATION_NEWS_CHECKGROUPS = 'application/news-checkgroups'
    APPLICATION_NEWS_GROUPINFO = 'application/news-groupinfo'
    APPLICATION_NEWS_TRANSMISSION = 'application/news-transmission'
    APPLICATION_NLSML_XML = 'application/nlsml+xml'
    APPLICATION_NODE = 'application/node'
    APPLICATION_NSS = 'application/nss'
    APPLICATION_OAUTH_AUTHZ_REQ_JWT = 'application/oauth-authz-req+jwt'
    APPLICATION_OBLIVIOUS_DNS_MESSAGE = 'application/oblivious-dns-message'
    APPLICATION_OCSP_REQUEST = 'application/ocsp-request'
    APPLICATION_OCSP_RESPONSE = 'application/ocsp-response'
    APPLICATION_OCTET_STREAM = 'application/octet-stream'
    APPLICATION_ODA = 'application/ODA'
    APPLICATION_ODM_XML = 'application/odm+xml'
    APPLICATION_ODX = 'application/ODX'
    APPLICATION_OEBPS_PACKAGE_XML = 'application/oebps-package+xml'
    APPLICATION_OGG = 'application/ogg'
    APPLICATION_OHTTP_KEYS = 'application/ohttp-keys'
    APPLICATION_OPC_NODESET_XML = 'application/opc-nodeset+xml'
    APPLICATION_OSCORE = 'application/oscore'
    APPLICATION_OXPS = 'application/oxps'
    APPLICATION_P21 = 'application/p21'
    APPLICATION_P21_ZIP = 'application/p21+zip'
    APPLICATION_P2P_OVERLAY_XML = 'application/p2p-overlay+xml'
    APPLICATION_PARITYFEC = 'application/parityfec'
    APPLICATION_PASSPORT = 'application/passport'
    APPLICATION_PATCH_OPS_ERROR_XML = 'application/patch-ops-error+xml'
    APPLICATION_PDF = 'application/pdf'
    APPLICATION_PDX = 'application/PDX'
    APPLICATION_PEM_CERTIFICATE_CHAIN = 'application/pem-certificate-chain'
    APPLICATION_PGP_ENCRYPTED = 'application/pgp-encrypted'
    APPLICATION_PGP_KEYS = 'application/pgp-keys'
    APPLICATION_PGP_SIGNATURE = 'application/pgp-signature'
    APPLICATION_PIDF_DIFF_XML = 'application/pidf-diff+xml'
    APPLICATION_PIDF_XML = 'application/pidf+xml'
    APPLICATION_PKCS10 = 'application/pkcs10'
    APPLICATION_PKCS7_MIME = 'application/pkcs7-mime'
    APPLICATION_PKCS7_SIGNATURE = 'application/pkcs7-signature'
    APPLICATION_PKCS8 = 'application/pkcs8'
    APPLICATION_PKCS8_ENCRYPTED = 'application/pkcs8-encrypted'
    APPLICATION_PKCS12 = 'application/pkcs12'
    APPLICATION_PKIX_ATTR_CERT = 'application/pkix-attr-cert'
    APPLICATION_PKIX_CERT = 'application/pkix-cert'
    APPLICATION_PKIX_CRL = 'application/pkix-crl'
    APPLICATION_PKIX_PKIPATH = 'application/pkix-pkipath'
    APPLICATION_PKIXCMP = 'application/pkixcmp'
    APPLICATION_PLS_XML = 'application/pls+xml'
    APPLICATION_POC_SETTINGS_XML = 'application/poc-settings+xml'
    APPLICATION_POSTSCRIPT = 'application/postscript'
    APPLICATION_PPSP_TRACKER_JSON = 'application/ppsp-tracker+json'
    APPLICATION_PRIVATE_TOKEN_ISSUER_DIRECTORY = 'application/private-token-issuer-directory'
    APPLICATION_PRIVATE_TOKEN_REQUEST = 'application/private-token-request'
    APPLICATION_PRIVATE_TOKEN_RESPONSE = 'application/private-token-response'
    APPLICATION_PROBLEM_JSON = 'application/problem+json'
    APPLICATION_PROBLEM_XML = 'application/problem+xml'
    APPLICATION_PROVENANCE_XML = 'application/provenance+xml'
    APPLICATION_PROVIDED_CLAIMS_JWT = 'application/provided-claims+jwt'
    APPLICATION_PRS_ALVESTRAND_TITRAX_SHEET = 'application/prs.alvestrand.titrax-sheet'
    APPLICATION_PRS_CWW = 'application/prs.cww'
    APPLICATION_PRS_CYN = 'application/prs.cyn'
    APPLICATION_PRS_HPUB_ZIP = 'application/prs.hpub+zip'
    APPLICATION_PRS_IMPLIED_DOCUMENT_XML = 'application/prs.implied-document+xml'
    APPLICATION_PRS_IMPLIED_EXECUTABLE = 'application/prs.implied-executable'
    APPLICATION_PRS_IMPLIED_OBJECT_JSON = 'application/prs.implied-object+json'
    APPLICATION_PRS_IMPLIED_OBJECT_JSON_SEQ = 'application/prs.implied-object+json-seq'
    APPLICATION_PRS_IMPLIED_OBJECT_YAML = 'application/prs.implied-object+yaml'
    APPLICATION_PRS_IMPLIED_STRUCTURE = 'application/prs.implied-structure'
    APPLICATION_PRS_MAYFILE = 'application/prs.mayfile'
    APPLICATION_PRS_NPREND = 'application/prs.nprend'
    APPLICATION_PRS_PLUCKER = 'application/prs.plucker'
    APPLICATION_PRS_RDF_XML_CRYPT = 'application/prs.rdf-xml-crypt'
    APPLICATION_PRS_VCFBZIP2 = 'application/prs.vcfbzip2'
    APPLICATION_PRS_XSF_XML = 'application/prs.xsf+xml'
    APPLICATION_PSKC_XML = 'application/pskc+xml'
    APPLICATION_PVD_JSON = 'application/pvd+json'
    APPLICATION_RDF_XML = 'application/rdf+xml'
    APPLICATION_ROUTE_APD_XML = 'application/route-apd+xml'
    APPLICATION_ROUTE_S_TSID_XML = 'application/route-s-tsid+xml'
    APPLICATION_ROUTE_USD_XML = 'application/route-usd+xml'
    APPLICATION_QSIG = 'application/QSIG'
    APPLICATION_RAPTORFEC = 'application/raptorfec'
    APPLICATION_RDAP_JSON = 'application/rdap+json'
    APPLICATION_REGINFO_XML = 'application/reginfo+xml'
    APPLICATION_RELAX_NG_COMPACT_SYNTAX = 'application/relax-ng-compact-syntax'
    APPLICATION_REMOTE_PRINTING = 'application/remote-printing'  # OBSOLETE
    APPLICATION_REPUTON_JSON = 'application/reputon+json'
    APPLICATION_RESOLVE_RESPONSE_JWT = 'application/resolve-response+jwt'
    APPLICATION_RESOURCE_LISTS_DIFF_XML = 'application/resource-lists-diff+xml'
    APPLICATION_RESOURCE_LISTS_XML = 'application/resource-lists+xml'
    APPLICATION_RFC_XML = 'application/rfc+xml'
    APPLICATION_RISCOS = 'application/riscos'
    APPLICATION_RLMI_XML = 'application/rlmi+xml'
    APPLICATION_RLS_SERVICES_XML = 'application/rls-services+xml'
    APPLICATION_RPKI_CHECKLIST = 'application/rpki-checklist'
    APPLICATION_RPKI_GHOSTBUSTERS = 'application/rpki-ghostbusters'
    APPLICATION_RPKI_MANIFEST = 'application/rpki-manifest'
    APPLICATION_RPKI_PUBLICATION = 'application/rpki-publication'
    APPLICATION_RPKI_ROA = 'application/rpki-roa'
    APPLICATION_RPKI_SIGNED_TAL = 'application/rpki-signed-tal'
    APPLICATION_RPKI_UPDOWN = 'application/rpki-updown'
    APPLICATION_RS_METADATA_XML = 'application/rs-metadata+xml'
    APPLICATION_RTF = 'application/rtf'
    APPLICATION_RTPLOOPBACK = 'application/rtploopback'
    APPLICATION_RTX = 'application/rtx'
    APPLICATION_SAMLASSERTION_XML = 'application/samlassertion+xml'
    APPLICATION_SAMLMETADATA_XML = 'application/samlmetadata+xml'
    APPLICATION_SARIF_EXTERNAL_PROPERTIES_JSON = 'application/sarif-external-properties+json'
    APPLICATION_SARIF_JSON = 'application/sarif+json'
    APPLICATION_SBE = 'application/sbe'
    APPLICATION_SBML_XML = 'application/sbml+xml'
    APPLICATION_SCAIP_XML = 'application/scaip+xml'
    APPLICATION_SCIM_JSON = 'application/scim+json'
    APPLICATION_SCITT_RECEIPT_COSE = 'application/scitt-receipt+cose'
    APPLICATION_SCITT_STATEMENT_COSE = 'application/scitt-statement+cose'
    APPLICATION_SCVP_CV_REQUEST = 'application/scvp-cv-request'
    APPLICATION_SCVP_CV_RESPONSE = 'application/scvp-cv-response'
    APPLICATION_SCVP_VP_REQUEST = 'application/scvp-vp-request'
    APPLICATION_SCVP_VP_RESPONSE = 'application/scvp-vp-response'
    APPLICATION_SD_JWT = 'application/sd-jwt'
    APPLICATION_SD_JWT_JSON = 'application/sd-jwt+json'
    APPLICATION_SDF_JSON = 'application/sdf+json'
    APPLICATION_SDP = 'application/sdp'
    APPLICATION_SECEVENT_JWT = 'application/secevent+jwt'
    APPLICATION_SENML_ETCH_CBOR = 'application/senml-etch+cbor'
    APPLICATION_SENML_ETCH_JSON = 'application/senml-etch+json'
    APPLICATION_SENML_EXI = 'application/senml-exi'
    APPLICATION_SENML_CBOR = 'application/senml+cbor'
    APPLICATION_SENML_JSON = 'application/senml+json'
    APPLICATION_SENML_XML = 'application/senml+xml'
    APPLICATION_SENSML_EXI = 'application/sensml-exi'
    APPLICATION_SENSML_CBOR = 'application/sensml+cbor'
    APPLICATION_SENSML_JSON = 'application/sensml+json'
    APPLICATION_SENSML_XML = 'application/sensml+xml'
    APPLICATION_SEP_EXI = 'application/sep-exi'
    APPLICATION_SEP_XML = 'application/sep+xml'
    APPLICATION_SESSION_INFO = 'application/session-info'
    APPLICATION_SET_PAYMENT = 'application/set-payment'
    APPLICATION_SET_PAYMENT_INITIATION = 'application/set-payment-initiation'
    APPLICATION_SET_REGISTRATION = 'application/set-registration'
    APPLICATION_SET_REGISTRATION_INITIATION = 'application/set-registration-initiation'
    APPLICATION_SGML = 'application/SGML'
    APPLICATION_SGML_OPEN_CATALOG = 'application/sgml-open-catalog'
    APPLICATION_SHF_XML = 'application/shf+xml'
    APPLICATION_SIEVE = 'application/sieve'
    APPLICATION_SIMPLE_FILTER_XML = 'application/simple-filter+xml'
    APPLICATION_SIMPLE_MESSAGE_SUMMARY = 'application/simple-message-summary'
    APPLICATION_SIMPLESYMBOLCONTAINER = 'application/simpleSymbolContainer'
    APPLICATION_SIPC = 'application/sipc'
    APPLICATION_SLATE = 'application/slate'
    APPLICATION_SMIL = 'application/smil'  # OBSOLETED in favor of application/smil+xml
    APPLICATION_SMIL_XML = 'application/smil+xml'
    APPLICATION_SMPTE336M = 'application/smpte336m'
    APPLICATION_SOAP_FASTINFOSET = 'application/soap+fastinfoset'
    APPLICATION_SOAP_XML = 'application/soap+xml'
    APPLICATION_SPARQL_QUERY = 'application/sparql-query'
    APPLICATION_SPDX_JSON = 'application/spdx+json'
    APPLICATION_SPARQL_RESULTS_XML = 'application/sparql-results+xml'
    APPLICATION_SPIRITS_EVENT_XML = 'application/spirits-event+xml'
    APPLICATION_SQL = 'application/sql'
    APPLICATION_SRGS = 'application/srgs'
    APPLICATION_SRGS_XML = 'application/srgs+xml'
    APPLICATION_SRU_XML = 'application/sru+xml'
    APPLICATION_SSLKEYLOGFILE = 'application/sslkeylogfile'
    APPLICATION_SSML_XML = 'application/ssml+xml'
    APPLICATION_ST2110_41 = 'application/ST2110-41'
    APPLICATION_STIX_JSON = 'application/stix+json'
    APPLICATION_STRATUM = 'application/stratum'
    APPLICATION_SUIT_ENVELOPE_COSE = 'application/suit-envelope+cose'
    APPLICATION_SWID_CBOR = 'application/swid+cbor'
    APPLICATION_SWID_XML = 'application/swid+xml'
    APPLICATION_TAMP_APEX_UPDATE = 'application/tamp-apex-update'
    APPLICATION_TAMP_APEX_UPDATE_CONFIRM = 'application/tamp-apex-update-confirm'
    APPLICATION_TAMP_COMMUNITY_UPDATE = 'application/tamp-community-update'
    APPLICATION_TAMP_COMMUNITY_UPDATE_CONFIRM = 'application/tamp-community-update-confirm'
    APPLICATION_TAMP_ERROR = 'application/tamp-error'
    APPLICATION_TAMP_SEQUENCE_ADJUST = 'application/tamp-sequence-adjust'
    APPLICATION_TAMP_SEQUENCE_ADJUST_CONFIRM = 'application/tamp-sequence-adjust-confirm'
    APPLICATION_TAMP_STATUS_QUERY = 'application/tamp-status-query'
    APPLICATION_TAMP_STATUS_RESPONSE = 'application/tamp-status-response'
    APPLICATION_TAMP_UPDATE = 'application/tamp-update'
    APPLICATION_TAMP_UPDATE_CONFIRM = 'application/tamp-update-confirm'
    APPLICATION_TAXII_JSON = 'application/taxii+json'
    APPLICATION_TD_JSON = 'application/td+json'
    APPLICATION_TEI_XML = 'application/tei+xml'
    APPLICATION_TETRA_ISI = 'application/TETRA_ISI'
    APPLICATION_TEXINFO = 'application/texinfo'
    APPLICATION_THRAUD_XML = 'application/thraud+xml'
    APPLICATION_TIMESTAMP_QUERY = 'application/timestamp-query'
    APPLICATION_TIMESTAMP_REPLY = 'application/timestamp-reply'
    APPLICATION_TIMESTAMPED_DATA = 'application/timestamped-data'
    APPLICATION_TLSRPT_GZIP = 'application/tlsrpt+gzip'
    APPLICATION_TLSRPT_JSON = 'application/tlsrpt+json'
    APPLICATION_TM_JSON = 'application/tm+json'
    APPLICATION_TNAUTHLIST = 'application/tnauthlist'
    APPLICATION_TOC_CBOR = 'application/toc+cbor'
    APPLICATION_TOKEN_INTROSPECTION_JWT = 'application/token-introspection+jwt'
    APPLICATION_TOML = 'application/toml'
    APPLICATION_TRICKLE_ICE_SDPFRAG = 'application/trickle-ice-sdpfrag'
    APPLICATION_TRIG = 'application/trig'
    APPLICATION_TRUST_CHAIN_JSON = 'application/trust-chain+json'
    APPLICATION_TRUST_MARK_JWT = 'application/trust-mark+jwt'
    APPLICATION_TRUST_MARK_DELEGATION_JWT = 'application/trust-mark-delegation+jwt'
    APPLICATION_TTML_XML = 'application/ttml+xml'
    APPLICATION_TVE_TRIGGER = 'application/tve-trigger'
    APPLICATION_TZIF = 'application/tzif'
    APPLICATION_TZIF_LEAP = 'application/tzif-leap'
    APPLICATION_UCCS_CBOR = 'application/uccs+cbor'
    APPLICATION_UJCS_JSON = 'application/ujcs+json'
    APPLICATION_ULPFEC = 'application/ulpfec'
    APPLICATION_URC_GRPSHEET_XML = 'application/urc-grpsheet+xml'
    APPLICATION_URC_RESSHEET_XML = 'application/urc-ressheet+xml'
    APPLICATION_URC_TARGETDESC_XML = 'application/urc-targetdesc+xml'
    APPLICATION_URC_UISOCKETDESC_XML = 'application/urc-uisocketdesc+xml'
    APPLICATION_VC = 'application/vc'
    APPLICATION_VC_COSE = 'application/vc+cose'
    APPLICATION_VC_JWT = 'application/vc+jwt'
    APPLICATION_VCARD_JSON = 'application/vcard+json'
    APPLICATION_VCARD_XML = 'application/vcard+xml'
    APPLICATION_VEC_XML = 'application/vec+xml'
    APPLICATION_VEMMI = 'application/vemmi'
    APPLICATION_VND_1000MINDS_DECISION_MODEL_XML = 'application/vnd.1000minds.decision-model+xml'
    APPLICATION_VND_1OB = 'application/vnd.1ob'
    APPLICATION_VND_3GPP_5GNAS = 'application/vnd.3gpp.5gnas'
    APPLICATION_VND_3GPP_5GSA2X = 'application/vnd.3gpp.5gsa2x'
    APPLICATION_VND_3GPP_5GSA2X_LOCAL_SERVICE_INFORMATION = 'application/vnd.3gpp.5gsa2x-local-service-information'
    APPLICATION_VND_3GPP_5GSV2X = 'application/vnd.3gpp.5gsv2x'
    APPLICATION_VND_3GPP_5GSV2X_LOCAL_SERVICE_INFORMATION = 'application/vnd.3gpp.5gsv2x-local-service-information'
    APPLICATION_VND_3GPP_ACCESS_TRANSFER_EVENTS_XML = 'application/vnd.3gpp.access-transfer-events+xml'
    APPLICATION_VND_3GPP_BSF_XML = 'application/vnd.3gpp.bsf+xml'
    APPLICATION_VND_3GPP_CRS_XML = 'application/vnd.3gpp.crs+xml'
    APPLICATION_VND_3GPP_CURRENT_LOCATION_DISCOVERY_XML = 'application/vnd.3gpp.current-location-discovery+xml'
    APPLICATION_VND_3GPP_GMOP_XML = 'application/vnd.3gpp.GMOP+xml'
    APPLICATION_VND_3GPP_GTPC = 'application/vnd.3gpp.gtpc'
    APPLICATION_VND_3GPP_INTERWORKING_DATA = 'application/vnd.3gpp.interworking-data'
    APPLICATION_VND_3GPP_LPP = 'application/vnd.3gpp.lpp'
    APPLICATION_VND_3GPP_MC_SIGNALLING_EAR = 'application/vnd.3gpp.mc-signalling-ear'
    APPLICATION_VND_3GPP_MCDATA_AFFILIATION_COMMAND_XML = 'application/vnd.3gpp.mcdata-affiliation-command+xml'
    APPLICATION_VND_3GPP_MCDATA_INFO_XML = 'application/vnd.3gpp.mcdata-info+xml'
    APPLICATION_VND_3GPP_MCDATA_MSGSTORE_CTRL_REQUEST_XML = 'application/vnd.3gpp.mcdata-msgstore-ctrl-request+xml'
    APPLICATION_VND_3GPP_MCDATA_PAYLOAD = 'application/vnd.3gpp.mcdata-payload'
    APPLICATION_VND_3GPP_MCDATA_REGROUP_XML = 'application/vnd.3gpp.mcdata-regroup+xml'
    APPLICATION_VND_3GPP_MCDATA_SERVICE_CONFIG_XML = 'application/vnd.3gpp.mcdata-service-config+xml'
    APPLICATION_VND_3GPP_MCDATA_SIGNALLING = 'application/vnd.3gpp.mcdata-signalling'
    APPLICATION_VND_3GPP_MCDATA_UE_CONFIG_XML = 'application/vnd.3gpp.mcdata-ue-config+xml'
    APPLICATION_VND_3GPP_MCDATA_USER_PROFILE_XML = 'application/vnd.3gpp.mcdata-user-profile+xml'
    APPLICATION_VND_3GPP_MCPTT_AFFILIATION_COMMAND_XML = 'application/vnd.3gpp.mcptt-affiliation-command+xml'
    APPLICATION_VND_3GPP_MCPTT_FLOOR_REQUEST_XML = 'application/vnd.3gpp.mcptt-floor-request+xml'
    APPLICATION_VND_3GPP_MCPTT_INFO_XML = 'application/vnd.3gpp.mcptt-info+xml'
    APPLICATION_VND_3GPP_MCPTT_LOCATION_INFO_XML = 'application/vnd.3gpp.mcptt-location-info+xml'
    APPLICATION_VND_3GPP_MCPTT_MBMS_USAGE_INFO_XML = 'application/vnd.3gpp.mcptt-mbms-usage-info+xml'
    APPLICATION_VND_3GPP_MCPTT_REGROUP_XML = 'application/vnd.3gpp.mcptt-regroup+xml'
    APPLICATION_VND_3GPP_MCPTT_SERVICE_CONFIG_XML = 'application/vnd.3gpp.mcptt-service-config+xml'
    APPLICATION_VND_3GPP_MCPTT_SIGNED_XML = 'application/vnd.3gpp.mcptt-signed+xml'
    APPLICATION_VND_3GPP_MCPTT_UE_CONFIG_XML = 'application/vnd.3gpp.mcptt-ue-config+xml'
    APPLICATION_VND_3GPP_MCPTT_UE_INIT_CONFIG_XML = 'application/vnd.3gpp.mcptt-ue-init-config+xml'
    APPLICATION_VND_3GPP_MCPTT_USER_PROFILE_XML = 'application/vnd.3gpp.mcptt-user-profile+xml'
    APPLICATION_VND_3GPP_MCVIDEO_AFFILIATION_COMMAND_XML = 'application/vnd.3gpp.mcvideo-affiliation-command+xml'
    APPLICATION_VND_3GPP_MCVIDEO_AFFILIATION_INFO_XML = 'application/vnd.3gpp.mcvideo-affiliation-info+xml'  # OBSOLETED in favor of application/vnd.3gpp.mcvideo-info+xml
    APPLICATION_VND_3GPP_MCVIDEO_INFO_XML = 'application/vnd.3gpp.mcvideo-info+xml'
    APPLICATION_VND_3GPP_MCVIDEO_LOCATION_INFO_XML = 'application/vnd.3gpp.mcvideo-location-info+xml'
    APPLICATION_VND_3GPP_MCVIDEO_MBMS_USAGE_INFO_XML = 'application/vnd.3gpp.mcvideo-mbms-usage-info+xml'
    APPLICATION_VND_3GPP_MCVIDEO_REGROUP_XML = 'application/vnd.3gpp.mcvideo-regroup+xml'
    APPLICATION_VND_3GPP_MCVIDEO_SERVICE_CONFIG_XML = 'application/vnd.3gpp.mcvideo-service-config+xml'
    APPLICATION_VND_3GPP_MCVIDEO_TRANSMISSION_REQUEST_XML = 'application/vnd.3gpp.mcvideo-transmission-request+xml'
    APPLICATION_VND_3GPP_MCVIDEO_UE_CONFIG_XML = 'application/vnd.3gpp.mcvideo-ue-config+xml'
    APPLICATION_VND_3GPP_MCVIDEO_USER_PROFILE_XML = 'application/vnd.3gpp.mcvideo-user-profile+xml'
    APPLICATION_VND_3GPP_MID_CALL_XML = 'application/vnd.3gpp.mid-call+xml'
    APPLICATION_VND_3GPP_NGAP = 'application/vnd.3gpp.ngap'
    APPLICATION_VND_3GPP_PFCP = 'application/vnd.3gpp.pfcp'
    APPLICATION_VND_3GPP_PIC_BW_LARGE = 'application/vnd.3gpp.pic-bw-large'
    APPLICATION_VND_3GPP_PIC_BW_SMALL = 'application/vnd.3gpp.pic-bw-small'
    APPLICATION_VND_3GPP_PIC_BW_VAR = 'application/vnd.3gpp.pic-bw-var'
    APPLICATION_VND_3GPP_PINAPP_INFO_XML = 'application/vnd.3gpp.pinapp-info+xml'
    APPLICATION_VND_3GPP_PROSE_PC3A_XML = 'application/vnd.3gpp-prose-pc3a+xml'
    APPLICATION_VND_3GPP_PROSE_PC3ACH_XML = 'application/vnd.3gpp-prose-pc3ach+xml'
    APPLICATION_VND_3GPP_PROSE_PC3CH_XML = 'application/vnd.3gpp-prose-pc3ch+xml'
    APPLICATION_VND_3GPP_PROSE_PC8_XML = 'application/vnd.3gpp-prose-pc8+xml'
    APPLICATION_VND_3GPP_PROSE_XML = 'application/vnd.3gpp-prose+xml'
    APPLICATION_VND_3GPP_S1AP = 'application/vnd.3gpp.s1ap'
    APPLICATION_VND_3GPP_SEAL_APP_COMM_REQUIREMENTS_INFO_XML = (
        'application/vnd.3gpp.seal-app-comm-requirements-info+xml'
    )
    APPLICATION_VND_3GPP_SEAL_DATA_DELIVERY_INFO_CBOR = 'application/vnd.3gpp.seal-data-delivery-info+cbor'
    APPLICATION_VND_3GPP_SEAL_DATA_DELIVERY_INFO_XML = 'application/vnd.3gpp.seal-data-delivery-info+xml'
    APPLICATION_VND_3GPP_SEAL_GROUP_DOC_XML = 'application/vnd.3gpp.seal-group-doc+xml'
    APPLICATION_VND_3GPP_SEAL_INFO_XML = 'application/vnd.3gpp.seal-info+xml'
    APPLICATION_VND_3GPP_SEAL_LOCATION_INFO_CBOR = 'application/vnd.3gpp.seal-location-info+cbor'
    APPLICATION_VND_3GPP_SEAL_LOCATION_INFO_XML = 'application/vnd.3gpp.seal-location-info+xml'
    APPLICATION_VND_3GPP_SEAL_MBMS_USAGE_INFO_XML = 'application/vnd.3gpp.seal-mbms-usage-info+xml'
    APPLICATION_VND_3GPP_SEAL_MBS_USAGE_INFO_XML = 'application/vnd.3gpp.seal-mbs-usage-info+xml'
    APPLICATION_VND_3GPP_SEAL_NETWORK_QOS_MANAGEMENT_INFO_XML = (
        'application/vnd.3gpp.seal-network-QoS-management-info+xml'
    )
    APPLICATION_VND_3GPP_SEAL_NETWORK_RESOURCE_INFO_CBOR = 'application/vnd.3gpp.seal-network-resource-info+cbor'
    APPLICATION_VND_3GPP_SEAL_UE_CONFIG_INFO_XML = 'application/vnd.3gpp.seal-ue-config-info+xml'
    APPLICATION_VND_3GPP_SEAL_UNICAST_INFO_XML = 'application/vnd.3gpp.seal-unicast-info+xml'
    APPLICATION_VND_3GPP_SEAL_USER_PROFILE_INFO_XML = 'application/vnd.3gpp.seal-user-profile-info+xml'
    APPLICATION_VND_3GPP_SMS = 'application/vnd.3gpp.sms'
    APPLICATION_VND_3GPP_SMS_XML = 'application/vnd.3gpp.sms+xml'
    APPLICATION_VND_3GPP_SRVCC_EXT_XML = 'application/vnd.3gpp.srvcc-ext+xml'
    APPLICATION_VND_3GPP_SRVCC_INFO_XML = 'application/vnd.3gpp.SRVCC-info+xml'
    APPLICATION_VND_3GPP_STATE_AND_EVENT_INFO_XML = 'application/vnd.3gpp.state-and-event-info+xml'
    APPLICATION_VND_3GPP_USSD_XML = 'application/vnd.3gpp.ussd+xml'
    APPLICATION_VND_3GPP_VAE_INFO_XML = 'application/vnd.3gpp.vae-info+xml'
    APPLICATION_VND_3GPP_V2X_LOCAL_SERVICE_INFORMATION = 'application/vnd.3gpp-v2x-local-service-information'
    APPLICATION_VND_3GPP2_BCMCSINFO_XML = 'application/vnd.3gpp2.bcmcsinfo+xml'
    APPLICATION_VND_3GPP2_SMS = 'application/vnd.3gpp2.sms'
    APPLICATION_VND_3GPP2_TCAP = 'application/vnd.3gpp2.tcap'
    APPLICATION_VND_3GPP_V2X = 'application/vnd.3gpp.v2x'
    APPLICATION_VND_3LIGHTSSOFTWARE_IMAGESCAL = 'application/vnd.3lightssoftware.imagescal'
    APPLICATION_VND_3M_POST_IT_NOTES = 'application/vnd.3M.Post-it-Notes'
    APPLICATION_VND_ACCPAC_SIMPLY_ASO = 'application/vnd.accpac.simply.aso'
    APPLICATION_VND_ACCPAC_SIMPLY_IMP = 'application/vnd.accpac.simply.imp'
    APPLICATION_VND_ACM_ADDRESSXFER_JSON = 'application/vnd.acm.addressxfer+json'
    APPLICATION_VND_ACM_CHATBOT_JSON = 'application/vnd.acm.chatbot+json'
    APPLICATION_VND_ACUCOBOL = 'application/vnd.acucobol'
    APPLICATION_VND_ACUCORP = 'application/vnd.acucorp'
    APPLICATION_VND_ADOBE_FLASH_MOVIE = 'application/vnd.adobe.flash.movie'
    APPLICATION_VND_ADOBE_FORMSCENTRAL_FCDT = 'application/vnd.adobe.formscentral.fcdt'
    APPLICATION_VND_ADOBE_FXP = 'application/vnd.adobe.fxp'
    APPLICATION_VND_ADOBE_PARTIAL_UPLOAD = 'application/vnd.adobe.partial-upload'
    APPLICATION_VND_ADOBE_XDP_XML = 'application/vnd.adobe.xdp+xml'
    APPLICATION_VND_AETHER_IMP = 'application/vnd.aether.imp'
    APPLICATION_VND_AFPC_AFPLINEDATA = 'application/vnd.afpc.afplinedata'
    APPLICATION_VND_AFPC_AFPLINEDATA_PAGEDEF = 'application/vnd.afpc.afplinedata-pagedef'
    APPLICATION_VND_AFPC_CMOCA_CMRESOURCE = 'application/vnd.afpc.cmoca-cmresource'
    APPLICATION_VND_AFPC_FOCA_CHARSET = 'application/vnd.afpc.foca-charset'
    APPLICATION_VND_AFPC_FOCA_CODEDFONT = 'application/vnd.afpc.foca-codedfont'
    APPLICATION_VND_AFPC_FOCA_CODEPAGE = 'application/vnd.afpc.foca-codepage'
    APPLICATION_VND_AFPC_MODCA = 'application/vnd.afpc.modca'
    APPLICATION_VND_AFPC_MODCA_CMTABLE = 'application/vnd.afpc.modca-cmtable'
    APPLICATION_VND_AFPC_MODCA_FORMDEF = 'application/vnd.afpc.modca-formdef'
    APPLICATION_VND_AFPC_MODCA_MEDIUMMAP = 'application/vnd.afpc.modca-mediummap'
    APPLICATION_VND_AFPC_MODCA_OBJECTCONTAINER = 'application/vnd.afpc.modca-objectcontainer'
    APPLICATION_VND_AFPC_MODCA_OVERLAY = 'application/vnd.afpc.modca-overlay'
    APPLICATION_VND_AFPC_MODCA_PAGESEGMENT = 'application/vnd.afpc.modca-pagesegment'
    APPLICATION_VND_AGE = 'application/vnd.age'
    APPLICATION_VND_AH_BARCODE = 'application/vnd.ah-barcode'
    APPLICATION_VND_AHEAD_SPACE = 'application/vnd.ahead.space'
    APPLICATION_VND_AIRZIP_FILESECURE_AZF = 'application/vnd.airzip.filesecure.azf'
    APPLICATION_VND_AIRZIP_FILESECURE_AZS = 'application/vnd.airzip.filesecure.azs'
    APPLICATION_VND_AMADEUS_JSON = 'application/vnd.amadeus+json'
    APPLICATION_VND_AMAZON_MOBI8_EBOOK = 'application/vnd.amazon.mobi8-ebook'
    APPLICATION_VND_AMERICANDYNAMICS_ACC = 'application/vnd.americandynamics.acc'
    APPLICATION_VND_AMIGA_AMI = 'application/vnd.amiga.ami'
    APPLICATION_VND_AMUNDSEN_MAZE_XML = 'application/vnd.amundsen.maze+xml'
    APPLICATION_VND_ANDROID_OTA = 'application/vnd.android.ota'
    APPLICATION_VND_ANKI = 'application/vnd.anki'
    APPLICATION_VND_ANSER_WEB_CERTIFICATE_ISSUE_INITIATION = 'application/vnd.anser-web-certificate-issue-initiation'
    APPLICATION_VND_ANTIX_GAME_COMPONENT = 'application/vnd.antix.game-component'
    APPLICATION_VND_APACHE_ARROW_FILE = 'application/vnd.apache.arrow.file'
    APPLICATION_VND_APACHE_ARROW_STREAM = 'application/vnd.apache.arrow.stream'
    APPLICATION_VND_APACHE_PARQUET = 'application/vnd.apache.parquet'
    APPLICATION_VND_APACHE_THRIFT_BINARY = 'application/vnd.apache.thrift.binary'
    APPLICATION_VND_APACHE_THRIFT_COMPACT = 'application/vnd.apache.thrift.compact'
    APPLICATION_VND_APACHE_THRIFT_JSON = 'application/vnd.apache.thrift.json'
    APPLICATION_VND_APEXLANG = 'application/vnd.apexlang'
    APPLICATION_VND_API_JSON = 'application/vnd.api+json'
    APPLICATION_VND_APLEXTOR_WARRP_JSON = 'application/vnd.aplextor.warrp+json'
    APPLICATION_VND_APOTHEKENDE_RESERVATION_JSON = 'application/vnd.apothekende.reservation+json'
    APPLICATION_VND_APPLE_INSTALLER_XML = 'application/vnd.apple.installer+xml'
    APPLICATION_VND_APPLE_KEYNOTE = 'application/vnd.apple.keynote'
    APPLICATION_VND_APPLE_MPEGURL = 'application/vnd.apple.mpegurl'
    APPLICATION_VND_APPLE_NUMBERS = 'application/vnd.apple.numbers'
    APPLICATION_VND_APPLE_PAGES = 'application/vnd.apple.pages'
    APPLICATION_VND_ARASTRA_SWI = (
        'application/vnd.arastra.swi'  # OBSOLETED in favor of application/vnd.aristanetworks.swi
    )
    APPLICATION_VND_ARISTANETWORKS_SWI = 'application/vnd.aristanetworks.swi'
    APPLICATION_VND_ARTISAN_JSON = 'application/vnd.artisan+json'
    APPLICATION_VND_ARTSQUARE = 'application/vnd.artsquare'
    APPLICATION_VND_AS207960_VAS_CONFIG_JER = 'application/vnd.as207960.vas.config+jer'
    APPLICATION_VND_AS207960_VAS_CONFIG_UPER = 'application/vnd.as207960.vas.config+uper'
    APPLICATION_VND_AS207960_VAS_TAP_JER = 'application/vnd.as207960.vas.tap+jer'
    APPLICATION_VND_AS207960_VAS_TAP_UPER = 'application/vnd.as207960.vas.tap+uper'
    APPLICATION_VND_ASTRAEA_SOFTWARE_IOTA = 'application/vnd.astraea-software.iota'
    APPLICATION_VND_AUDIOGRAPH = 'application/vnd.audiograph'
    APPLICATION_VND_AUTOPACKAGE = 'application/vnd.autopackage'
    APPLICATION_VND_AVALON_JSON = 'application/vnd.avalon+json'
    APPLICATION_VND_AVISTAR_XML = 'application/vnd.avistar+xml'
    APPLICATION_VND_BALSAMIQ_BMML_XML = 'application/vnd.balsamiq.bmml+xml'
    APPLICATION_VND_BANANA_ACCOUNTING = 'application/vnd.banana-accounting'
    APPLICATION_VND_BBF_USP_ERROR = 'application/vnd.bbf.usp.error'
    APPLICATION_VND_BBF_USP_MSG = 'application/vnd.bbf.usp.msg'
    APPLICATION_VND_BBF_USP_MSG_JSON = 'application/vnd.bbf.usp.msg+json'
    APPLICATION_VND_BALSAMIQ_BMPR = 'application/vnd.balsamiq.bmpr'
    APPLICATION_VND_BEKITZUR_STECH_JSON = 'application/vnd.bekitzur-stech+json'
    APPLICATION_VND_BELIGHTSOFT_LHZD_ZIP = 'application/vnd.belightsoft.lhzd+zip'
    APPLICATION_VND_BELIGHTSOFT_LHZL_ZIP = 'application/vnd.belightsoft.lhzl+zip'
    APPLICATION_VND_BINT_MED_CONTENT = 'application/vnd.bint.med-content'
    APPLICATION_VND_BIOPAX_RDF_XML = 'application/vnd.biopax.rdf+xml'
    APPLICATION_VND_BLINK_IDB_VALUE_WRAPPER = 'application/vnd.blink-idb-value-wrapper'
    APPLICATION_VND_BLUEICE_MULTIPASS = 'application/vnd.blueice.multipass'
    APPLICATION_VND_BLUETOOTH_EP_OOB = 'application/vnd.bluetooth.ep.oob'
    APPLICATION_VND_BLUETOOTH_LE_OOB = 'application/vnd.bluetooth.le.oob'
    APPLICATION_VND_BMI = 'application/vnd.bmi'
    APPLICATION_VND_BPF = 'application/vnd.bpf'
    APPLICATION_VND_BPF3 = 'application/vnd.bpf3'
    APPLICATION_VND_BUSINESSOBJECTS = 'application/vnd.businessobjects'
    APPLICATION_VND_BYU_UAPI_JSON = 'application/vnd.byu.uapi+json'
    APPLICATION_VND_BZIP3 = 'application/vnd.bzip3'
    APPLICATION_VND_C3VOC_SCHEDULE_XML = 'application/vnd.c3voc.schedule+xml'
    APPLICATION_VND_CAB_JSCRIPT = 'application/vnd.cab-jscript'
    APPLICATION_VND_CANON_CPDL = 'application/vnd.canon-cpdl'
    APPLICATION_VND_CANON_LIPS = 'application/vnd.canon-lips'
    APPLICATION_VND_CAPASYSTEMS_PG_JSON = 'application/vnd.capasystems-pg+json'
    APPLICATION_VND_CEL = 'application/vnd.cel'
    APPLICATION_VND_CENDIO_THINLINC_CLIENTCONF = 'application/vnd.cendio.thinlinc.clientconf'
    APPLICATION_VND_CENTURY_SYSTEMS_TCP_STREAM = 'application/vnd.century-systems.tcp_stream'
    APPLICATION_VND_CHEMDRAW_XML = 'application/vnd.chemdraw+xml'
    APPLICATION_VND_CHESS_PGN = 'application/vnd.chess-pgn'
    APPLICATION_VND_CHIPNUTS_KARAOKE_MMD = 'application/vnd.chipnuts.karaoke-mmd'
    APPLICATION_VND_CIEDI = 'application/vnd.ciedi'
    APPLICATION_VND_CINDERELLA = 'application/vnd.cinderella'
    APPLICATION_VND_CIRPACK_ISDN_EXT = 'application/vnd.cirpack.isdn-ext'
    APPLICATION_VND_CITATIONSTYLES_STYLE_XML = 'application/vnd.citationstyles.style+xml'
    APPLICATION_VND_CLAYMORE = 'application/vnd.claymore'
    APPLICATION_VND_CLOANTO_RP9 = 'application/vnd.cloanto.rp9'
    APPLICATION_VND_CLONK_C4GROUP = 'application/vnd.clonk.c4group'
    APPLICATION_VND_CLUETRUST_CARTOMOBILE_CONFIG = 'application/vnd.cluetrust.cartomobile-config'
    APPLICATION_VND_CLUETRUST_CARTOMOBILE_CONFIG_PKG = 'application/vnd.cluetrust.cartomobile-config-pkg'
    APPLICATION_VND_CNCF_HELM_CHART_CONTENT_V1_TAR_GZIP = 'application/vnd.cncf.helm.chart.content.v1.tar+gzip'
    APPLICATION_VND_CNCF_HELM_CHART_PROVENANCE_V1_PROV = 'application/vnd.cncf.helm.chart.provenance.v1.prov'
    APPLICATION_VND_CNCF_HELM_CONFIG_V1_JSON = 'application/vnd.cncf.helm.config.v1+json'
    APPLICATION_VND_COFFEESCRIPT = 'application/vnd.coffeescript'
    APPLICATION_VND_COLLABIO_XODOCUMENTS_DOCUMENT = 'application/vnd.collabio.xodocuments.document'
    APPLICATION_VND_COLLABIO_XODOCUMENTS_DOCUMENT_TEMPLATE = 'application/vnd.collabio.xodocuments.document-template'
    APPLICATION_VND_COLLABIO_XODOCUMENTS_PRESENTATION = 'application/vnd.collabio.xodocuments.presentation'
    APPLICATION_VND_COLLABIO_XODOCUMENTS_PRESENTATION_TEMPLATE = (
        'application/vnd.collabio.xodocuments.presentation-template'
    )
    APPLICATION_VND_COLLABIO_XODOCUMENTS_SPREADSHEET = 'application/vnd.collabio.xodocuments.spreadsheet'
    APPLICATION_VND_COLLABIO_XODOCUMENTS_SPREADSHEET_TEMPLATE = (
        'application/vnd.collabio.xodocuments.spreadsheet-template'
    )
    APPLICATION_VND_COLLECTION_DOC_JSON = 'application/vnd.collection.doc+json'
    APPLICATION_VND_COLLECTION_JSON = 'application/vnd.collection+json'
    APPLICATION_VND_COLLECTION_NEXT_JSON = 'application/vnd.collection.next+json'
    APPLICATION_VND_COMICBOOK_RAR = 'application/vnd.comicbook-rar'
    APPLICATION_VND_COMICBOOK_ZIP = 'application/vnd.comicbook+zip'
    APPLICATION_VND_COMMERCE_BATTELLE = 'application/vnd.commerce-battelle'
    APPLICATION_VND_COMMONSPACE = 'application/vnd.commonspace'
    APPLICATION_VND_COREOS_IGNITION_JSON = 'application/vnd.coreos.ignition+json'
    APPLICATION_VND_COSMOCALLER = 'application/vnd.cosmocaller'
    APPLICATION_VND_CONTACT_CMSG = 'application/vnd.contact.cmsg'
    APPLICATION_VND_CRICK_CLICKER = 'application/vnd.crick.clicker'
    APPLICATION_VND_CRICK_CLICKER_KEYBOARD = 'application/vnd.crick.clicker.keyboard'
    APPLICATION_VND_CRICK_CLICKER_PALETTE = 'application/vnd.crick.clicker.palette'
    APPLICATION_VND_CRICK_CLICKER_TEMPLATE = 'application/vnd.crick.clicker.template'
    APPLICATION_VND_CRICK_CLICKER_WORDBANK = 'application/vnd.crick.clicker.wordbank'
    APPLICATION_VND_CRITICALTOOLS_WBS_XML = 'application/vnd.criticaltools.wbs+xml'
    APPLICATION_VND_CRYPTII_PIPE_JSON = 'application/vnd.cryptii.pipe+json'
    APPLICATION_VND_CRYPTO_SHADE_FILE = 'application/vnd.crypto-shade-file'
    APPLICATION_VND_CRYPTOMATOR_ENCRYPTED = 'application/vnd.cryptomator.encrypted'
    APPLICATION_VND_CRYPTOMATOR_VAULT = 'application/vnd.cryptomator.vault'
    APPLICATION_VND_CTC_POSML = 'application/vnd.ctc-posml'
    APPLICATION_VND_CTCT_WS_XML = 'application/vnd.ctct.ws+xml'
    APPLICATION_VND_CUPS_PDF = 'application/vnd.cups-pdf'
    APPLICATION_VND_CUPS_POSTSCRIPT = 'application/vnd.cups-postscript'
    APPLICATION_VND_CUPS_PPD = 'application/vnd.cups-ppd'
    APPLICATION_VND_CUPS_RASTER = 'application/vnd.cups-raster'
    APPLICATION_VND_CUPS_RAW = 'application/vnd.cups-raw'
    APPLICATION_VND_CURL = 'application/vnd.curl'
    APPLICATION_VND_CYAN_DEAN_ROOT_XML = 'application/vnd.cyan.dean.root+xml'
    APPLICATION_VND_CYBANK = 'application/vnd.cybank'
    APPLICATION_VND_CYCLONEDX_JSON = 'application/vnd.cyclonedx+json'
    APPLICATION_VND_CYCLONEDX_XML = 'application/vnd.cyclonedx+xml'
    APPLICATION_VND_D2L_COURSEPACKAGE1P0_ZIP = 'application/vnd.d2l.coursepackage1p0+zip'
    APPLICATION_VND_D3M_DATASET = 'application/vnd.d3m-dataset'
    APPLICATION_VND_D3M_PROBLEM = 'application/vnd.d3m-problem'
    APPLICATION_VND_DART = 'application/vnd.dart'
    APPLICATION_VND_DATA_VISION_RDZ = 'application/vnd.data-vision.rdz'
    APPLICATION_VND_DATALOG = 'application/vnd.datalog'
    APPLICATION_VND_DATAPACKAGE_JSON = 'application/vnd.datapackage+json'
    APPLICATION_VND_DATARESOURCE_JSON = 'application/vnd.dataresource+json'
    APPLICATION_VND_DBF = 'application/vnd.dbf'
    APPLICATION_VND_DCMP_XML = 'application/vnd.dcmp+xml'
    APPLICATION_VND_DEBIAN_BINARY_PACKAGE = 'application/vnd.debian.binary-package'
    APPLICATION_VND_DECE_DATA = 'application/vnd.dece.data'
    APPLICATION_VND_DECE_TTML_XML = 'application/vnd.dece.ttml+xml'
    APPLICATION_VND_DECE_UNSPECIFIED = 'application/vnd.dece.unspecified'
    APPLICATION_VND_DECE_ZIP = 'application/vnd.dece.zip'
    APPLICATION_VND_DENOVO_FCSELAYOUT_LINK = 'application/vnd.denovo.fcselayout-link'
    APPLICATION_VND_DESMUME_MOVIE = 'application/vnd.desmume.movie'
    APPLICATION_VND_DIR_BI_PLATE_DL_NOSUFFIX = 'application/vnd.dir-bi.plate-dl-nosuffix'
    APPLICATION_VND_DM_DELEGATION_XML = 'application/vnd.dm.delegation+xml'
    APPLICATION_VND_DNA = 'application/vnd.dna'
    APPLICATION_VND_DOCUMENT_JSON = 'application/vnd.document+json'
    APPLICATION_VND_DOLBY_MOBILE_1 = 'application/vnd.dolby.mobile.1'
    APPLICATION_VND_DOLBY_MOBILE_2 = 'application/vnd.dolby.mobile.2'
    APPLICATION_VND_DOREMIR_SCORECLOUD_BINARY_DOCUMENT = 'application/vnd.doremir.scorecloud-binary-document'
    APPLICATION_VND_DPGRAPH = 'application/vnd.dpgraph'
    APPLICATION_VND_DREAMFACTORY = 'application/vnd.dreamfactory'
    APPLICATION_VND_DRIVE_JSON = 'application/vnd.drive+json'
    APPLICATION_VND_DTG_LOCAL = 'application/vnd.dtg.local'
    APPLICATION_VND_DTG_LOCAL_FLASH = 'application/vnd.dtg.local.flash'
    APPLICATION_VND_DTG_LOCAL_HTML = 'application/vnd.dtg.local.html'
    APPLICATION_VND_DVB_AIT = 'application/vnd.dvb.ait'
    APPLICATION_VND_DVB_DVBISL_XML = 'application/vnd.dvb.dvbisl+xml'
    APPLICATION_VND_DVB_DVBJ = 'application/vnd.dvb.dvbj'
    APPLICATION_VND_DVB_ESGCONTAINER = 'application/vnd.dvb.esgcontainer'
    APPLICATION_VND_DVB_IPDCDFTNOTIFACCESS = 'application/vnd.dvb.ipdcdftnotifaccess'
    APPLICATION_VND_DVB_IPDCESGACCESS = 'application/vnd.dvb.ipdcesgaccess'
    APPLICATION_VND_DVB_IPDCESGACCESS2 = 'application/vnd.dvb.ipdcesgaccess2'
    APPLICATION_VND_DVB_IPDCESGPDD = 'application/vnd.dvb.ipdcesgpdd'
    APPLICATION_VND_DVB_IPDCROAMING = 'application/vnd.dvb.ipdcroaming'
    APPLICATION_VND_DVB_IPTV_ALFEC_BASE = 'application/vnd.dvb.iptv.alfec-base'
    APPLICATION_VND_DVB_IPTV_ALFEC_ENHANCEMENT = 'application/vnd.dvb.iptv.alfec-enhancement'
    APPLICATION_VND_DVB_NOTIF_AGGREGATE_ROOT_XML = 'application/vnd.dvb.notif-aggregate-root+xml'
    APPLICATION_VND_DVB_NOTIF_CONTAINER_XML = 'application/vnd.dvb.notif-container+xml'
    APPLICATION_VND_DVB_NOTIF_GENERIC_XML = 'application/vnd.dvb.notif-generic+xml'
    APPLICATION_VND_DVB_NOTIF_IA_MSGLIST_XML = 'application/vnd.dvb.notif-ia-msglist+xml'
    APPLICATION_VND_DVB_NOTIF_IA_REGISTRATION_REQUEST_XML = 'application/vnd.dvb.notif-ia-registration-request+xml'
    APPLICATION_VND_DVB_NOTIF_IA_REGISTRATION_RESPONSE_XML = 'application/vnd.dvb.notif-ia-registration-response+xml'
    APPLICATION_VND_DVB_NOTIF_INIT_XML = 'application/vnd.dvb.notif-init+xml'
    APPLICATION_VND_DVB_PFR = 'application/vnd.dvb.pfr'
    APPLICATION_VND_DVB_SERVICE = 'application/vnd.dvb.service'
    APPLICATION_VND_DXR = 'application/vnd.dxr'
    APPLICATION_VND_DYNAGEO = 'application/vnd.dynageo'
    APPLICATION_VND_DZR = 'application/vnd.dzr'
    APPLICATION_VND_EASYKARAOKE_CDGDOWNLOAD = 'application/vnd.easykaraoke.cdgdownload'
    APPLICATION_VND_ECIP_RLP = 'application/vnd.ecip.rlp'
    APPLICATION_VND_ECDIS_UPDATE = 'application/vnd.ecdis-update'
    APPLICATION_VND_ECLIPSE_DITTO_JSON = 'application/vnd.eclipse.ditto+json'
    APPLICATION_VND_ECOWIN_CHART = 'application/vnd.ecowin.chart'
    APPLICATION_VND_ECOWIN_FILEREQUEST = 'application/vnd.ecowin.filerequest'
    APPLICATION_VND_ECOWIN_FILEUPDATE = 'application/vnd.ecowin.fileupdate'
    APPLICATION_VND_ECOWIN_SERIES = 'application/vnd.ecowin.series'
    APPLICATION_VND_ECOWIN_SERIESREQUEST = 'application/vnd.ecowin.seriesrequest'
    APPLICATION_VND_ECOWIN_SERIESUPDATE = 'application/vnd.ecowin.seriesupdate'
    APPLICATION_VND_EFI_IMG = 'application/vnd.efi.img'
    APPLICATION_VND_EFI_ISO = 'application/vnd.efi.iso'
    APPLICATION_VND_ELN_ZIP = 'application/vnd.eln+zip'
    APPLICATION_VND_EMCLIENT_ACCESSREQUEST_XML = 'application/vnd.emclient.accessrequest+xml'
    APPLICATION_VND_ENLIVEN = 'application/vnd.enliven'
    APPLICATION_VND_ENPHASE_ENVOY = 'application/vnd.enphase.envoy'
    APPLICATION_VND_EPRINTS_DATA_XML = 'application/vnd.eprints.data+xml'
    APPLICATION_VND_EPSON_ESF = 'application/vnd.epson.esf'
    APPLICATION_VND_EPSON_MSF = 'application/vnd.epson.msf'
    APPLICATION_VND_EPSON_QUICKANIME = 'application/vnd.epson.quickanime'
    APPLICATION_VND_EPSON_SALT = 'application/vnd.epson.salt'
    APPLICATION_VND_EPSON_SSF = 'application/vnd.epson.ssf'
    APPLICATION_VND_ERICSSON_QUICKCALL = 'application/vnd.ericsson.quickcall'
    APPLICATION_VND_EROFS = 'application/vnd.erofs'
    APPLICATION_VND_ESPASS_ESPASS_ZIP = 'application/vnd.espass-espass+zip'
    APPLICATION_VND_ESZIGNO3_XML = 'application/vnd.eszigno3+xml'
    APPLICATION_VND_ETSI_AOC_XML = 'application/vnd.etsi.aoc+xml'
    APPLICATION_VND_ETSI_ASIC_S_ZIP = 'application/vnd.etsi.asic-s+zip'
    APPLICATION_VND_ETSI_ASIC_E_ZIP = 'application/vnd.etsi.asic-e+zip'
    APPLICATION_VND_ETSI_CUG_XML = 'application/vnd.etsi.cug+xml'
    APPLICATION_VND_ETSI_IPTVCOMMAND_XML = 'application/vnd.etsi.iptvcommand+xml'
    APPLICATION_VND_ETSI_IPTVDISCOVERY_XML = 'application/vnd.etsi.iptvdiscovery+xml'
    APPLICATION_VND_ETSI_IPTVPROFILE_XML = 'application/vnd.etsi.iptvprofile+xml'
    APPLICATION_VND_ETSI_IPTVSAD_BC_XML = 'application/vnd.etsi.iptvsad-bc+xml'
    APPLICATION_VND_ETSI_IPTVSAD_COD_XML = 'application/vnd.etsi.iptvsad-cod+xml'
    APPLICATION_VND_ETSI_IPTVSAD_NPVR_XML = 'application/vnd.etsi.iptvsad-npvr+xml'
    APPLICATION_VND_ETSI_IPTVSERVICE_XML = 'application/vnd.etsi.iptvservice+xml'
    APPLICATION_VND_ETSI_IPTVSYNC_XML = 'application/vnd.etsi.iptvsync+xml'
    APPLICATION_VND_ETSI_IPTVUEPROFILE_XML = 'application/vnd.etsi.iptvueprofile+xml'
    APPLICATION_VND_ETSI_MCID_XML = 'application/vnd.etsi.mcid+xml'
    APPLICATION_VND_ETSI_MHEG5 = 'application/vnd.etsi.mheg5'
    APPLICATION_VND_ETSI_OVERLOAD_CONTROL_POLICY_DATASET_XML = (
        'application/vnd.etsi.overload-control-policy-dataset+xml'
    )
    APPLICATION_VND_ETSI_PSTN_XML = 'application/vnd.etsi.pstn+xml'
    APPLICATION_VND_ETSI_SCI_XML = 'application/vnd.etsi.sci+xml'
    APPLICATION_VND_ETSI_SIMSERVS_XML = 'application/vnd.etsi.simservs+xml'
    APPLICATION_VND_ETSI_TIMESTAMP_TOKEN = 'application/vnd.etsi.timestamp-token'
    APPLICATION_VND_ETSI_TSL_XML = 'application/vnd.etsi.tsl+xml'
    APPLICATION_VND_ETSI_TSL_DER = 'application/vnd.etsi.tsl.der'
    APPLICATION_VND_EU_KASPARIAN_CAR_JSON = 'application/vnd.eu.kasparian.car+json'
    APPLICATION_VND_EUDORA_DATA = 'application/vnd.eudora.data'
    APPLICATION_VND_EVOLV_ECIG_PROFILE = 'application/vnd.evolv.ecig.profile'
    APPLICATION_VND_EVOLV_ECIG_SETTINGS = 'application/vnd.evolv.ecig.settings'
    APPLICATION_VND_EVOLV_ECIG_THEME = 'application/vnd.evolv.ecig.theme'
    APPLICATION_VND_EXSTREAM_EMPOWER_ZIP = 'application/vnd.exstream-empower+zip'
    APPLICATION_VND_EXSTREAM_PACKAGE = 'application/vnd.exstream-package'
    APPLICATION_VND_EZPIX_ALBUM = 'application/vnd.ezpix-album'
    APPLICATION_VND_EZPIX_PACKAGE = 'application/vnd.ezpix-package'
    APPLICATION_VND_F_SECURE_MOBILE = 'application/vnd.f-secure.mobile'
    APPLICATION_VND_FASTCOPY_DISK_IMAGE = 'application/vnd.fastcopy-disk-image'
    APPLICATION_VND_FAMILYSEARCH_GEDCOM_ZIP = 'application/vnd.familysearch.gedcom+zip'
    APPLICATION_VND_FDSN_MSEED = 'application/vnd.fdsn.mseed'
    APPLICATION_VND_FDSN_SEED = 'application/vnd.fdsn.seed'
    APPLICATION_VND_FDSN_STATIONXML_XML = 'application/vnd.fdsn.stationxml+xml'
    APPLICATION_VND_FFSNS = 'application/vnd.ffsns'
    APPLICATION_VND_FGB = 'application/vnd.fgb'
    APPLICATION_VND_FICLAB_FLB_ZIP = 'application/vnd.ficlab.flb+zip'
    APPLICATION_VND_FILMIT_ZFC = 'application/vnd.filmit.zfc'
    APPLICATION_VND_FINTS = 'application/vnd.fints'
    APPLICATION_VND_FIREMONKEYS_CLOUDCELL = 'application/vnd.firemonkeys.cloudcell'
    APPLICATION_VND_FLOGRAPHIT = 'application/vnd.FloGraphIt'
    APPLICATION_VND_FLUXTIME_CLIP = 'application/vnd.fluxtime.clip'
    APPLICATION_VND_FONT_FONTFORGE_SFD = 'application/vnd.font-fontforge-sfd'
    APPLICATION_VND_FRAMEMAKER = 'application/vnd.framemaker'
    APPLICATION_VND_FREELOG_COMIC = 'application/vnd.freelog.comic'
    APPLICATION_VND_FROGANS_FNC = 'application/vnd.frogans.fnc'  # OBSOLETE
    APPLICATION_VND_FROGANS_LTF = 'application/vnd.frogans.ltf'  # OBSOLETE
    APPLICATION_VND_FSC_WEBLAUNCH = 'application/vnd.fsc.weblaunch'
    APPLICATION_VND_FUJIFILM_FB_DOCUWORKS = 'application/vnd.fujifilm.fb.docuworks'
    APPLICATION_VND_FUJIFILM_FB_DOCUWORKS_BINDER = 'application/vnd.fujifilm.fb.docuworks.binder'
    APPLICATION_VND_FUJIFILM_FB_DOCUWORKS_CONTAINER = 'application/vnd.fujifilm.fb.docuworks.container'
    APPLICATION_VND_FUJIFILM_FB_JFI_XML = 'application/vnd.fujifilm.fb.jfi+xml'
    APPLICATION_VND_FUJITSU_OASYS = 'application/vnd.fujitsu.oasys'
    APPLICATION_VND_FUJITSU_OASYS2 = 'application/vnd.fujitsu.oasys2'
    APPLICATION_VND_FUJITSU_OASYS3 = 'application/vnd.fujitsu.oasys3'
    APPLICATION_VND_FUJITSU_OASYSGP = 'application/vnd.fujitsu.oasysgp'
    APPLICATION_VND_FUJITSU_OASYSPRS = 'application/vnd.fujitsu.oasysprs'
    APPLICATION_VND_FUJIXEROX_ART4 = 'application/vnd.fujixerox.ART4'
    APPLICATION_VND_FUJIXEROX_ART_EX = 'application/vnd.fujixerox.ART-EX'
    APPLICATION_VND_FUJIXEROX_DDD = 'application/vnd.fujixerox.ddd'
    APPLICATION_VND_FUJIXEROX_DOCUWORKS = 'application/vnd.fujixerox.docuworks'
    APPLICATION_VND_FUJIXEROX_DOCUWORKS_BINDER = 'application/vnd.fujixerox.docuworks.binder'
    APPLICATION_VND_FUJIXEROX_DOCUWORKS_CONTAINER = 'application/vnd.fujixerox.docuworks.container'
    APPLICATION_VND_FUJIXEROX_HBPL = 'application/vnd.fujixerox.HBPL'
    APPLICATION_VND_FUT_MISNET = 'application/vnd.fut-misnet'
    APPLICATION_VND_FUTOIN_CBOR = 'application/vnd.futoin+cbor'
    APPLICATION_VND_FUTOIN_JSON = 'application/vnd.futoin+json'
    APPLICATION_VND_FUZZYSHEET = 'application/vnd.fuzzysheet'
    APPLICATION_VND_G3PIX_G3FC = 'application/vnd.g3pix.g3fc'
    APPLICATION_VND_GA4GH_PASSPORT_JWT = 'application/vnd.ga4gh.passport+jwt'
    APPLICATION_VND_GENOMATIX_TUXEDO = 'application/vnd.genomatix.tuxedo'
    APPLICATION_VND_GENOZIP = 'application/vnd.genozip'
    APPLICATION_VND_GENTICS_GRD_JSON = 'application/vnd.gentics.grd+json'
    APPLICATION_VND_GENTOO_CATMETADATA_XML = 'application/vnd.gentoo.catmetadata+xml'
    APPLICATION_VND_GENTOO_EBUILD = 'application/vnd.gentoo.ebuild'
    APPLICATION_VND_GENTOO_ECLASS = 'application/vnd.gentoo.eclass'
    APPLICATION_VND_GENTOO_GPKG = 'application/vnd.gentoo.gpkg'
    APPLICATION_VND_GENTOO_MANIFEST = 'application/vnd.gentoo.manifest'
    APPLICATION_VND_GENTOO_XPAK = 'application/vnd.gentoo.xpak'
    APPLICATION_VND_GENTOO_PKGMETADATA_XML = 'application/vnd.gentoo.pkgmetadata+xml'
    APPLICATION_VND_GEO_JSON = 'application/vnd.geo+json'  # OBSOLETED by [RFC7946] in favor of application/geo+json
    APPLICATION_VND_GEOCUBE_XML = 'application/vnd.geocube+xml'  # OBSOLETED by request
    APPLICATION_VND_GEOGEBRA_FILE = 'application/vnd.geogebra.file'
    APPLICATION_VND_GEOGEBRA_PINBOARD = 'application/vnd.geogebra.pinboard'
    APPLICATION_VND_GEOGEBRA_SLIDES = 'application/vnd.geogebra.slides'
    APPLICATION_VND_GEOGEBRA_TOOL = 'application/vnd.geogebra.tool'
    APPLICATION_VND_GEOMETRY_EXPLORER = 'application/vnd.geometry-explorer'
    APPLICATION_VND_GEONEXT = 'application/vnd.geonext'
    APPLICATION_VND_GEOPLAN = 'application/vnd.geoplan'
    APPLICATION_VND_GEOSPACE = 'application/vnd.geospace'
    APPLICATION_VND_GERBER = 'application/vnd.gerber'
    APPLICATION_VND_GLOBALPLATFORM_CARD_CONTENT_MGT = 'application/vnd.globalplatform.card-content-mgt'
    APPLICATION_VND_GLOBALPLATFORM_CARD_CONTENT_MGT_RESPONSE = (
        'application/vnd.globalplatform.card-content-mgt-response'
    )
    APPLICATION_VND_GMX_DEPRECATED = 'application/vnd.gmx'
    APPLICATION_VND_GNU_TALER_EXCHANGE_JSON = 'application/vnd.gnu.taler.exchange+json'
    APPLICATION_VND_GNU_TALER_MERCHANT_JSON = 'application/vnd.gnu.taler.merchant+json'
    APPLICATION_VND_GOOGLE_EARTH_KML_XML = 'application/vnd.google-earth.kml+xml'
    APPLICATION_VND_GOOGLE_EARTH_KMZ = 'application/vnd.google-earth.kmz'
    APPLICATION_VND_GOV_SK_E_FORM_XML = 'application/vnd.gov.sk.e-form+xml'  # OBSOLETED by request
    APPLICATION_VND_GOV_SK_E_FORM_ZIP = 'application/vnd.gov.sk.e-form+zip'
    APPLICATION_VND_GOV_SK_XMLDATACONTAINER_XML = 'application/vnd.gov.sk.xmldatacontainer+xml'
    APPLICATION_VND_GPXSEE_MAP_XML = 'application/vnd.gpxsee.map+xml'
    APPLICATION_VND_GRAFEQ = 'application/vnd.grafeq'
    APPLICATION_VND_GRIDMP = 'application/vnd.gridmp'
    APPLICATION_VND_GROOVE_ACCOUNT = 'application/vnd.groove-account'
    APPLICATION_VND_GROOVE_HELP = 'application/vnd.groove-help'
    APPLICATION_VND_GROOVE_IDENTITY_MESSAGE = 'application/vnd.groove-identity-message'
    APPLICATION_VND_GROOVE_INJECTOR = 'application/vnd.groove-injector'
    APPLICATION_VND_GROOVE_TOOL_MESSAGE = 'application/vnd.groove-tool-message'
    APPLICATION_VND_GROOVE_TOOL_TEMPLATE = 'application/vnd.groove-tool-template'
    APPLICATION_VND_GROOVE_VCARD = 'application/vnd.groove-vcard'
    APPLICATION_VND_HAL_JSON = 'application/vnd.hal+json'
    APPLICATION_VND_HAL_XML = 'application/vnd.hal+xml'
    APPLICATION_VND_HANDHELD_ENTERTAINMENT_XML = 'application/vnd.HandHeld-Entertainment+xml'
    APPLICATION_VND_HBCI = 'application/vnd.hbci'
    APPLICATION_VND_HC_JSON = 'application/vnd.hc+json'
    APPLICATION_VND_HCL_BIREPORTS = 'application/vnd.hcl-bireports'
    APPLICATION_VND_HDT = 'application/vnd.hdt'
    APPLICATION_VND_HEROKU_JSON = 'application/vnd.heroku+json'
    APPLICATION_VND_HHE_LESSON_PLAYER = 'application/vnd.hhe.lesson-player'
    APPLICATION_VND_HP_HPGL = 'application/vnd.hp-HPGL'
    APPLICATION_VND_HP_HPID = 'application/vnd.hp-hpid'
    APPLICATION_VND_HP_HPS = 'application/vnd.hp-hps'
    APPLICATION_VND_HP_JLYT = 'application/vnd.hp-jlyt'
    APPLICATION_VND_HP_PCL = 'application/vnd.hp-PCL'
    APPLICATION_VND_HP_PCLXL = 'application/vnd.hp-PCLXL'
    APPLICATION_VND_HSL = 'application/vnd.hsl'
    APPLICATION_VND_HTTPHONE = 'application/vnd.httphone'
    APPLICATION_VND_HYDROSTATIX_SOF_DATA = 'application/vnd.hydrostatix.sof-data'
    APPLICATION_VND_HYPER_ITEM_JSON = 'application/vnd.hyper-item+json'
    APPLICATION_VND_HYPER_JSON = 'application/vnd.hyper+json'
    APPLICATION_VND_HYPERDRIVE_JSON = 'application/vnd.hyperdrive+json'
    APPLICATION_VND_HZN_3D_CROSSWORD = 'application/vnd.hzn-3d-crossword'
    APPLICATION_VND_IBM_AFPLINEDATA = 'application/vnd.ibm.afplinedata'  # OBSOLETED in favor of vnd.afpc.afplinedata
    APPLICATION_VND_IBM_ELECTRONIC_MEDIA = 'application/vnd.ibm.electronic-media'
    APPLICATION_VND_IBM_MINIPAY = 'application/vnd.ibm.MiniPay'
    APPLICATION_VND_IBM_MODCAP = 'application/vnd.ibm.modcap'  # OBSOLETED in favor of application/vnd.afpc.modca
    APPLICATION_VND_IBM_RIGHTS_MANAGEMENT = 'application/vnd.ibm.rights-management'
    APPLICATION_VND_IBM_SECURE_CONTAINER = 'application/vnd.ibm.secure-container'
    APPLICATION_VND_ICCPROFILE = 'application/vnd.iccprofile'
    APPLICATION_VND_IEEE_1905 = 'application/vnd.ieee.1905'
    APPLICATION_VND_IGLOADER = 'application/vnd.igloader'
    APPLICATION_VND_IMAGEMETER_FOLDER_ZIP = 'application/vnd.imagemeter.folder+zip'
    APPLICATION_VND_IMAGEMETER_IMAGE_ZIP = 'application/vnd.imagemeter.image+zip'
    APPLICATION_VND_IMMERVISION_IVP = 'application/vnd.immervision-ivp'
    APPLICATION_VND_IMMERVISION_IVU = 'application/vnd.immervision-ivu'
    APPLICATION_VND_IMS_IMSCCV1P1 = 'application/vnd.ims.imsccv1p1'
    APPLICATION_VND_IMS_IMSCCV1P2 = 'application/vnd.ims.imsccv1p2'
    APPLICATION_VND_IMS_IMSCCV1P3 = 'application/vnd.ims.imsccv1p3'
    APPLICATION_VND_IMS_LIS_V2_RESULT_JSON = 'application/vnd.ims.lis.v2.result+json'
    APPLICATION_VND_IMS_LTI_V2_TOOLCONSUMERPROFILE_JSON = 'application/vnd.ims.lti.v2.toolconsumerprofile+json'
    APPLICATION_VND_IMS_LTI_V2_TOOLPROXY_ID_JSON = 'application/vnd.ims.lti.v2.toolproxy.id+json'
    APPLICATION_VND_IMS_LTI_V2_TOOLPROXY_JSON = 'application/vnd.ims.lti.v2.toolproxy+json'
    APPLICATION_VND_IMS_LTI_V2_TOOLSETTINGS_JSON = 'application/vnd.ims.lti.v2.toolsettings+json'
    APPLICATION_VND_IMS_LTI_V2_TOOLSETTINGS_SIMPLE_JSON = 'application/vnd.ims.lti.v2.toolsettings.simple+json'
    APPLICATION_VND_INFORMEDCONTROL_RMS_XML = 'application/vnd.informedcontrol.rms+xml'
    APPLICATION_VND_INFOTECH_PROJECT = 'application/vnd.infotech.project'
    APPLICATION_VND_INFOTECH_PROJECT_XML = 'application/vnd.infotech.project+xml'
    APPLICATION_VND_INFORMIX_VISIONARY = (
        'application/vnd.informix-visionary'  # OBSOLETED in favor of application/vnd.visionary
    )
    APPLICATION_VND_INNOPATH_WAMP_NOTIFICATION = 'application/vnd.innopath.wamp.notification'
    APPLICATION_VND_INSORS_IGM = 'application/vnd.insors.igm'
    APPLICATION_VND_INTERCON_FORMNET = 'application/vnd.intercon.formnet'
    APPLICATION_VND_INTERGEO = 'application/vnd.intergeo'
    APPLICATION_VND_INTERTRUST_DIGIBOX = 'application/vnd.intertrust.digibox'
    APPLICATION_VND_INTERTRUST_NNCP = 'application/vnd.intertrust.nncp'
    APPLICATION_VND_INTU_QBO = 'application/vnd.intu.qbo'
    APPLICATION_VND_INTU_QFX = 'application/vnd.intu.qfx'
    APPLICATION_VND_IPFS_IPNS_RECORD = 'application/vnd.ipfs.ipns-record'
    APPLICATION_VND_IPLD_CAR = 'application/vnd.ipld.car'
    APPLICATION_VND_IPLD_DAG_CBOR = 'application/vnd.ipld.dag-cbor'
    APPLICATION_VND_IPLD_DAG_JSON = 'application/vnd.ipld.dag-json'
    APPLICATION_VND_IPLD_RAW = 'application/vnd.ipld.raw'
    APPLICATION_VND_IPTC_G2_CATALOGITEM_XML = 'application/vnd.iptc.g2.catalogitem+xml'
    APPLICATION_VND_IPTC_G2_CONCEPTITEM_XML = 'application/vnd.iptc.g2.conceptitem+xml'
    APPLICATION_VND_IPTC_G2_KNOWLEDGEITEM_XML = 'application/vnd.iptc.g2.knowledgeitem+xml'
    APPLICATION_VND_IPTC_G2_NEWSITEM_XML = 'application/vnd.iptc.g2.newsitem+xml'
    APPLICATION_VND_IPTC_G2_NEWSMESSAGE_XML = 'application/vnd.iptc.g2.newsmessage+xml'
    APPLICATION_VND_IPTC_G2_PACKAGEITEM_XML = 'application/vnd.iptc.g2.packageitem+xml'
    APPLICATION_VND_IPTC_G2_PLANNINGITEM_XML = 'application/vnd.iptc.g2.planningitem+xml'
    APPLICATION_VND_IPUNPLUGGED_RCPROFILE = 'application/vnd.ipunplugged.rcprofile'
    APPLICATION_VND_IREPOSITORY_PACKAGE_XML = 'application/vnd.irepository.package+xml'
    APPLICATION_VND_IS_XPR = 'application/vnd.is-xpr'
    APPLICATION_VND_ISAC_FCS = 'application/vnd.isac.fcs'
    APPLICATION_VND_JAM = 'application/vnd.jam'
    APPLICATION_VND_ISO11783_10_ZIP = 'application/vnd.iso11783-10+zip'
    APPLICATION_VND_JAPANNET_DIRECTORY_SERVICE = 'application/vnd.japannet-directory-service'
    APPLICATION_VND_JAPANNET_JPNSTORE_WAKEUP = 'application/vnd.japannet-jpnstore-wakeup'
    APPLICATION_VND_JAPANNET_PAYMENT_WAKEUP = 'application/vnd.japannet-payment-wakeup'
    APPLICATION_VND_JAPANNET_REGISTRATION = 'application/vnd.japannet-registration'
    APPLICATION_VND_JAPANNET_REGISTRATION_WAKEUP = 'application/vnd.japannet-registration-wakeup'
    APPLICATION_VND_JAPANNET_SETSTORE_WAKEUP = 'application/vnd.japannet-setstore-wakeup'
    APPLICATION_VND_JAPANNET_VERIFICATION = 'application/vnd.japannet-verification'
    APPLICATION_VND_JAPANNET_VERIFICATION_WAKEUP = 'application/vnd.japannet-verification-wakeup'
    APPLICATION_VND_JCP_JAVAME_MIDLET_RMS = 'application/vnd.jcp.javame.midlet-rms'
    APPLICATION_VND_JISP = 'application/vnd.jisp'
    APPLICATION_VND_JOOST_JODA_ARCHIVE = 'application/vnd.joost.joda-archive'
    APPLICATION_VND_JSK_ISDN_NGN = 'application/vnd.jsk.isdn-ngn'
    APPLICATION_VND_KAHOOTZ = 'application/vnd.kahootz'
    APPLICATION_VND_KDE_KARBON = 'application/vnd.kde.karbon'
    APPLICATION_VND_KDE_KCHART = 'application/vnd.kde.kchart'
    APPLICATION_VND_KDE_KFORMULA = 'application/vnd.kde.kformula'
    APPLICATION_VND_KDE_KIVIO = 'application/vnd.kde.kivio'
    APPLICATION_VND_KDE_KONTOUR = 'application/vnd.kde.kontour'
    APPLICATION_VND_KDE_KPRESENTER = 'application/vnd.kde.kpresenter'
    APPLICATION_VND_KDE_KSPREAD = 'application/vnd.kde.kspread'
    APPLICATION_VND_KDE_KWORD = 'application/vnd.kde.kword'
    APPLICATION_VND_KDL = 'application/vnd.kdl'
    APPLICATION_VND_KENAMEAAPP = 'application/vnd.kenameaapp'
    APPLICATION_VND_KEYMAN_KMP_ZIP = 'application/vnd.keyman.kmp+zip'
    APPLICATION_VND_KEYMAN_KMX = 'application/vnd.keyman.kmx'
    APPLICATION_VND_KIDSPIRATION = 'application/vnd.kidspiration'
    APPLICATION_VND_KINAR = 'application/vnd.Kinar'
    APPLICATION_VND_KOAN = 'application/vnd.koan'
    APPLICATION_VND_KODAK_DESCRIPTOR = 'application/vnd.kodak-descriptor'
    APPLICATION_VND_LAS = 'application/vnd.las'
    APPLICATION_VND_LAS_LAS_JSON = 'application/vnd.las.las+json'
    APPLICATION_VND_LAS_LAS_XML = 'application/vnd.las.las+xml'
    APPLICATION_VND_LASZIP = 'application/vnd.laszip'
    APPLICATION_VND_LDEV_PRODUCTLICENSING = 'application/vnd.ldev.productlicensing'
    APPLICATION_VND_LEAP_JSON = 'application/vnd.leap+json'
    APPLICATION_VND_LIBERTY_REQUEST_XML = 'application/vnd.liberty-request+xml'
    APPLICATION_VND_LLAMAGRAPHICS_LIFE_BALANCE_DESKTOP = 'application/vnd.llamagraphics.life-balance.desktop'
    APPLICATION_VND_LLAMAGRAPHICS_LIFE_BALANCE_EXCHANGE_XML = 'application/vnd.llamagraphics.life-balance.exchange+xml'
    APPLICATION_VND_LOGIPIPE_CIRCUIT_ZIP = 'application/vnd.logipipe.circuit+zip'
    APPLICATION_VND_LOOM = 'application/vnd.loom'
    APPLICATION_VND_LOTUS_1_2_3 = 'application/vnd.lotus-1-2-3'
    APPLICATION_VND_LOTUS_APPROACH = 'application/vnd.lotus-approach'
    APPLICATION_VND_LOTUS_FREELANCE = 'application/vnd.lotus-freelance'
    APPLICATION_VND_LOTUS_NOTES = 'application/vnd.lotus-notes'
    APPLICATION_VND_LOTUS_ORGANIZER = 'application/vnd.lotus-organizer'
    APPLICATION_VND_LOTUS_SCREENCAM = 'application/vnd.lotus-screencam'
    APPLICATION_VND_LOTUS_WORDPRO = 'application/vnd.lotus-wordpro'
    APPLICATION_VND_MACPORTS_PORTPKG = 'application/vnd.macports.portpkg'
    APPLICATION_VND_MAML = 'application/vnd.maml'
    APPLICATION_VND_MAPBOX_VECTOR_TILE = 'application/vnd.mapbox-vector-tile'
    APPLICATION_VND_MARLIN_DRM_ACTIONTOKEN_XML = 'application/vnd.marlin.drm.actiontoken+xml'
    APPLICATION_VND_MARLIN_DRM_CONFTOKEN_XML = 'application/vnd.marlin.drm.conftoken+xml'
    APPLICATION_VND_MARLIN_DRM_LICENSE_XML = 'application/vnd.marlin.drm.license+xml'
    APPLICATION_VND_MARLIN_DRM_MDCF = 'application/vnd.marlin.drm.mdcf'
    APPLICATION_VND_MASON_JSON = 'application/vnd.mason+json'
    APPLICATION_VND_MAXAR_ARCHIVE_3TZ_ZIP = 'application/vnd.maxar.archive.3tz+zip'
    APPLICATION_VND_MAXMIND_MAXMIND_DB = 'application/vnd.maxmind.maxmind-db'
    APPLICATION_VND_MCD = 'application/vnd.mcd'
    APPLICATION_VND_MDL = 'application/vnd.mdl'
    APPLICATION_VND_MDL_MBSDF = 'application/vnd.mdl-mbsdf'
    APPLICATION_VND_MEDCALCDATA = 'application/vnd.medcalcdata'
    APPLICATION_VND_MEDIASTATION_CDKEY = 'application/vnd.mediastation.cdkey'
    APPLICATION_VND_MEDICALHOLODECK_RECORDXR = 'application/vnd.medicalholodeck.recordxr'
    APPLICATION_VND_MERIDIAN_SLINGSHOT = 'application/vnd.meridian-slingshot'
    APPLICATION_VND_MERMAID = 'application/vnd.mermaid'
    APPLICATION_VND_MFER = 'application/vnd.MFER'
    APPLICATION_VND_MFMP = 'application/vnd.mfmp'
    APPLICATION_VND_MICRO_JSON = 'application/vnd.micro+json'
    APPLICATION_VND_MICROGRAFX_FLO = 'application/vnd.micrografx.flo'
    APPLICATION_VND_MICROGRAFX_IGX = 'application/vnd.micrografx.igx'
    APPLICATION_VND_MICROSOFT_PORTABLE_EXECUTABLE = 'application/vnd.microsoft.portable-executable'
    APPLICATION_VND_MICROSOFT_WINDOWS_THUMBNAIL_CACHE = 'application/vnd.microsoft.windows.thumbnail-cache'
    APPLICATION_VND_MIELE_JSON = 'application/vnd.miele+json'
    APPLICATION_VND_MIF = 'application/vnd.mif'
    APPLICATION_VND_MINISOFT_HP3000_SAVE = 'application/vnd.minisoft-hp3000-save'
    APPLICATION_VND_MITSUBISHI_MISTY_GUARD_TRUSTWEB = 'application/vnd.mitsubishi.misty-guard.trustweb'
    APPLICATION_VND_MOBIUS_DAF = 'application/vnd.Mobius.DAF'
    APPLICATION_VND_MOBIUS_DIS = 'application/vnd.Mobius.DIS'
    APPLICATION_VND_MOBIUS_MBK = 'application/vnd.Mobius.MBK'
    APPLICATION_VND_MOBIUS_MQY = 'application/vnd.Mobius.MQY'
    APPLICATION_VND_MOBIUS_MSL = 'application/vnd.Mobius.MSL'
    APPLICATION_VND_MOBIUS_PLC = 'application/vnd.Mobius.PLC'
    APPLICATION_VND_MOBIUS_TXF = 'application/vnd.Mobius.TXF'
    APPLICATION_VND_MODL = 'application/vnd.modl'
    APPLICATION_VND_MOPHUN_APPLICATION = 'application/vnd.mophun.application'
    APPLICATION_VND_MOPHUN_CERTIFICATE = 'application/vnd.mophun.certificate'
    APPLICATION_VND_MOTOROLA_FLEXSUITE = 'application/vnd.motorola.flexsuite'
    APPLICATION_VND_MOTOROLA_FLEXSUITE_ADSI = 'application/vnd.motorola.flexsuite.adsi'
    APPLICATION_VND_MOTOROLA_FLEXSUITE_FIS = 'application/vnd.motorola.flexsuite.fis'
    APPLICATION_VND_MOTOROLA_FLEXSUITE_GOTAP = 'application/vnd.motorola.flexsuite.gotap'
    APPLICATION_VND_MOTOROLA_FLEXSUITE_KMR = 'application/vnd.motorola.flexsuite.kmr'
    APPLICATION_VND_MOTOROLA_FLEXSUITE_TTC = 'application/vnd.motorola.flexsuite.ttc'
    APPLICATION_VND_MOTOROLA_FLEXSUITE_WEM = 'application/vnd.motorola.flexsuite.wem'
    APPLICATION_VND_MOTOROLA_IPRM = 'application/vnd.motorola.iprm'
    APPLICATION_VND_MOZILLA_XUL_XML = 'application/vnd.mozilla.xul+xml'
    APPLICATION_VND_MS_ARTGALRY = 'application/vnd.ms-artgalry'
    APPLICATION_VND_MS_ASF = 'application/vnd.ms-asf'
    APPLICATION_VND_MS_CAB_COMPRESSED = 'application/vnd.ms-cab-compressed'
    APPLICATION_VND_MS_3MFDOCUMENT = 'application/vnd.ms-3mfdocument'
    APPLICATION_VND_MS_EXCEL = 'application/vnd.ms-excel'
    APPLICATION_VND_MS_EXCEL_ADDIN_MACROENABLED_12 = 'application/vnd.ms-excel.addin.macroEnabled.12'
    APPLICATION_VND_MS_EXCEL_SHEET_BINARY_MACROENABLED_12 = 'application/vnd.ms-excel.sheet.binary.macroEnabled.12'
    APPLICATION_VND_MS_EXCEL_SHEET_MACROENABLED_12 = 'application/vnd.ms-excel.sheet.macroEnabled.12'
    APPLICATION_VND_MS_EXCEL_TEMPLATE_MACROENABLED_12 = 'application/vnd.ms-excel.template.macroEnabled.12'
    APPLICATION_VND_MS_FONTOBJECT = 'application/vnd.ms-fontobject'
    APPLICATION_VND_MS_HTMLHELP = 'application/vnd.ms-htmlhelp'
    APPLICATION_VND_MS_IMS = 'application/vnd.ms-ims'
    APPLICATION_VND_MS_LRM = 'application/vnd.ms-lrm'
    APPLICATION_VND_MS_OFFICE_ACTIVEX_XML = 'application/vnd.ms-office.activeX+xml'
    APPLICATION_VND_MS_OFFICETHEME = 'application/vnd.ms-officetheme'
    APPLICATION_VND_MS_PLAYREADY_INITIATOR_XML = 'application/vnd.ms-playready.initiator+xml'
    APPLICATION_VND_MS_POWERPOINT = 'application/vnd.ms-powerpoint'
    APPLICATION_VND_MS_POWERPOINT_ADDIN_MACROENABLED_12 = 'application/vnd.ms-powerpoint.addin.macroEnabled.12'
    APPLICATION_VND_MS_POWERPOINT_PRESENTATION_MACROENABLED_12 = (
        'application/vnd.ms-powerpoint.presentation.macroEnabled.12'
    )
    APPLICATION_VND_MS_POWERPOINT_SLIDE_MACROENABLED_12 = 'application/vnd.ms-powerpoint.slide.macroEnabled.12'
    APPLICATION_VND_MS_POWERPOINT_SLIDESHOW_MACROENABLED_12 = 'application/vnd.ms-powerpoint.slideshow.macroEnabled.12'
    APPLICATION_VND_MS_POWERPOINT_TEMPLATE_MACROENABLED_12 = 'application/vnd.ms-powerpoint.template.macroEnabled.12'
    APPLICATION_VND_MS_PRINTDEVICECAPABILITIES_XML = 'application/vnd.ms-PrintDeviceCapabilities+xml'
    APPLICATION_VND_MS_PRINTSCHEMATICKET_XML = 'application/vnd.ms-PrintSchemaTicket+xml'
    APPLICATION_VND_MS_PROJECT = 'application/vnd.ms-project'
    APPLICATION_VND_MS_TNEF = 'application/vnd.ms-tnef'
    APPLICATION_VND_MS_WINDOWS_DEVICEPAIRING = 'application/vnd.ms-windows.devicepairing'
    APPLICATION_VND_MS_WINDOWS_NWPRINTING_OOB = 'application/vnd.ms-windows.nwprinting.oob'
    APPLICATION_VND_MS_WINDOWS_PRINTERPAIRING = 'application/vnd.ms-windows.printerpairing'
    APPLICATION_VND_MS_WINDOWS_WSD_OOB = 'application/vnd.ms-windows.wsd.oob'
    APPLICATION_VND_MS_WMDRM_LIC_CHLG_REQ = 'application/vnd.ms-wmdrm.lic-chlg-req'
    APPLICATION_VND_MS_WMDRM_LIC_RESP = 'application/vnd.ms-wmdrm.lic-resp'
    APPLICATION_VND_MS_WMDRM_METER_CHLG_REQ = 'application/vnd.ms-wmdrm.meter-chlg-req'
    APPLICATION_VND_MS_WMDRM_METER_RESP = 'application/vnd.ms-wmdrm.meter-resp'
    APPLICATION_VND_MS_WORD_DOCUMENT_MACROENABLED_12 = 'application/vnd.ms-word.document.macroEnabled.12'
    APPLICATION_VND_MS_WORD_TEMPLATE_MACROENABLED_12 = 'application/vnd.ms-word.template.macroEnabled.12'
    APPLICATION_VND_MS_WORKS = 'application/vnd.ms-works'
    APPLICATION_VND_MS_WPL = 'application/vnd.ms-wpl'
    APPLICATION_VND_MS_XPSDOCUMENT = 'application/vnd.ms-xpsdocument'
    APPLICATION_VND_MSA_DISK_IMAGE = 'application/vnd.msa-disk-image'
    APPLICATION_VND_MSEQ = 'application/vnd.mseq'
    APPLICATION_VND_MSGPACK = 'application/vnd.msgpack'
    APPLICATION_VND_MSIGN = 'application/vnd.msign'
    APPLICATION_VND_MULTIAD_CREATOR = 'application/vnd.multiad.creator'
    APPLICATION_VND_MULTIAD_CREATOR_CIF = 'application/vnd.multiad.creator.cif'
    APPLICATION_VND_MUSICIAN = 'application/vnd.musician'
    APPLICATION_VND_MUSIC_NIFF = 'application/vnd.music-niff'
    APPLICATION_VND_MUVEE_STYLE = 'application/vnd.muvee.style'
    APPLICATION_VND_MYNFC = 'application/vnd.mynfc'
    APPLICATION_VND_NACAMAR_YBRID_JSON = 'application/vnd.nacamar.ybrid+json'
    APPLICATION_VND_NATO_BINDINGDATAOBJECT_CBOR = 'application/vnd.nato.bindingdataobject+cbor'
    APPLICATION_VND_NATO_BINDINGDATAOBJECT_JSON = 'application/vnd.nato.bindingdataobject+json'
    APPLICATION_VND_NATO_BINDINGDATAOBJECT_XML = 'application/vnd.nato.bindingdataobject+xml'
    APPLICATION_VND_NATO_OPENXMLFORMATS_PACKAGE_IEPD_ZIP = 'application/vnd.nato.openxmlformats-package.iepd+zip'
    APPLICATION_VND_NCD_CONTROL = 'application/vnd.ncd.control'
    APPLICATION_VND_NCD_REFERENCE = 'application/vnd.ncd.reference'
    APPLICATION_VND_NEARST_INV_JSON = 'application/vnd.nearst.inv+json'
    APPLICATION_VND_NEBUMIND_LINE = 'application/vnd.nebumind.line'
    APPLICATION_VND_NERVANA = 'application/vnd.nervana'
    APPLICATION_VND_NETFPX = 'application/vnd.netfpx'
    APPLICATION_VND_NEUROLANGUAGE_NLU = 'application/vnd.neurolanguage.nlu'
    APPLICATION_VND_NIMN = 'application/vnd.nimn'
    APPLICATION_VND_NINTENDO_SNES_ROM = 'application/vnd.nintendo.snes.rom'
    APPLICATION_VND_NINTENDO_NITRO_ROM = 'application/vnd.nintendo.nitro.rom'
    APPLICATION_VND_NITF = 'application/vnd.nitf'
    APPLICATION_VND_NOBLENET_DIRECTORY = 'application/vnd.noblenet-directory'
    APPLICATION_VND_NOBLENET_SEALER = 'application/vnd.noblenet-sealer'
    APPLICATION_VND_NOBLENET_WEB = 'application/vnd.noblenet-web'
    APPLICATION_VND_NOKIA_CATALOGS = 'application/vnd.nokia.catalogs'
    APPLICATION_VND_NOKIA_CONML_WBXML = 'application/vnd.nokia.conml+wbxml'
    APPLICATION_VND_NOKIA_CONML_XML = 'application/vnd.nokia.conml+xml'
    APPLICATION_VND_NOKIA_IPTV_CONFIG_XML = 'application/vnd.nokia.iptv.config+xml'
    APPLICATION_VND_NOKIA_ISDS_RADIO_PRESETS = 'application/vnd.nokia.iSDS-radio-presets'
    APPLICATION_VND_NOKIA_LANDMARK_WBXML = 'application/vnd.nokia.landmark+wbxml'
    APPLICATION_VND_NOKIA_LANDMARK_XML = 'application/vnd.nokia.landmark+xml'
    APPLICATION_VND_NOKIA_LANDMARKCOLLECTION_XML = 'application/vnd.nokia.landmarkcollection+xml'
    APPLICATION_VND_NOKIA_NCD = 'application/vnd.nokia.ncd'
    APPLICATION_VND_NOKIA_N_GAGE_AC_XML = 'application/vnd.nokia.n-gage.ac+xml'
    APPLICATION_VND_NOKIA_N_GAGE_DATA = 'application/vnd.nokia.n-gage.data'
    APPLICATION_VND_NOKIA_N_GAGE_SYMBIAN_INSTALL = (
        'application/vnd.nokia.n-gage.symbian.install'  # OBSOLETE; no replacement given
    )
    APPLICATION_VND_NOKIA_PCD_WBXML = 'application/vnd.nokia.pcd+wbxml'
    APPLICATION_VND_NOKIA_PCD_XML = 'application/vnd.nokia.pcd+xml'
    APPLICATION_VND_NOKIA_RADIO_PRESET = 'application/vnd.nokia.radio-preset'
    APPLICATION_VND_NOKIA_RADIO_PRESETS = 'application/vnd.nokia.radio-presets'
    APPLICATION_VND_NOVADIGM_EDM = 'application/vnd.novadigm.EDM'
    APPLICATION_VND_NOVADIGM_EDX = 'application/vnd.novadigm.EDX'
    APPLICATION_VND_NOVADIGM_EXT = 'application/vnd.novadigm.EXT'
    APPLICATION_VND_NTT_LOCAL_CONTENT_SHARE = 'application/vnd.ntt-local.content-share'
    APPLICATION_VND_NTT_LOCAL_FILE_TRANSFER = 'application/vnd.ntt-local.file-transfer'
    APPLICATION_VND_NTT_LOCAL_OGW_REMOTE_ACCESS = 'application/vnd.ntt-local.ogw_remote-access'
    APPLICATION_VND_NTT_LOCAL_SIP_TA_REMOTE = 'application/vnd.ntt-local.sip-ta_remote'
    APPLICATION_VND_NTT_LOCAL_SIP_TA_TCP_STREAM = 'application/vnd.ntt-local.sip-ta_tcp_stream'
    APPLICATION_VND_NUBALTEC_NUDOKU_GAME = 'application/vnd.nubaltec.nudoku-game'
    APPLICATION_VND_OAI_WORKFLOWS = 'application/vnd.oai.workflows'
    APPLICATION_VND_OAI_WORKFLOWS_JSON = 'application/vnd.oai.workflows+json'
    APPLICATION_VND_OAI_WORKFLOWS_YAML = 'application/vnd.oai.workflows+yaml'
    APPLICATION_VND_OASIS_OPENDOCUMENT_BASE = 'application/vnd.oasis.opendocument.base'
    APPLICATION_VND_OASIS_OPENDOCUMENT_CHART = 'application/vnd.oasis.opendocument.chart'
    APPLICATION_VND_OASIS_OPENDOCUMENT_CHART_TEMPLATE = 'application/vnd.oasis.opendocument.chart-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_DATABASE = (
        'application/vnd.oasis.opendocument.database'  # OBSOLETED in favor of application/vnd.oasis.opendocument.base
    )
    APPLICATION_VND_OASIS_OPENDOCUMENT_FORMULA = 'application/vnd.oasis.opendocument.formula'
    APPLICATION_VND_OASIS_OPENDOCUMENT_FORMULA_TEMPLATE = 'application/vnd.oasis.opendocument.formula-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_GRAPHICS = 'application/vnd.oasis.opendocument.graphics'
    APPLICATION_VND_OASIS_OPENDOCUMENT_GRAPHICS_TEMPLATE = 'application/vnd.oasis.opendocument.graphics-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_IMAGE = 'application/vnd.oasis.opendocument.image'
    APPLICATION_VND_OASIS_OPENDOCUMENT_IMAGE_TEMPLATE = 'application/vnd.oasis.opendocument.image-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_PRESENTATION = 'application/vnd.oasis.opendocument.presentation'
    APPLICATION_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE = (
        'application/vnd.oasis.opendocument.presentation-template'
    )
    APPLICATION_VND_OASIS_OPENDOCUMENT_SPREADSHEET = 'application/vnd.oasis.opendocument.spreadsheet'
    APPLICATION_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE = 'application/vnd.oasis.opendocument.spreadsheet-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_TEXT = 'application/vnd.oasis.opendocument.text'
    APPLICATION_VND_OASIS_OPENDOCUMENT_TEXT_MASTER = 'application/vnd.oasis.opendocument.text-master'
    APPLICATION_VND_OASIS_OPENDOCUMENT_TEXT_MASTER_TEMPLATE = 'application/vnd.oasis.opendocument.text-master-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE = 'application/vnd.oasis.opendocument.text-template'
    APPLICATION_VND_OASIS_OPENDOCUMENT_TEXT_WEB = 'application/vnd.oasis.opendocument.text-web'
    APPLICATION_VND_OBN = 'application/vnd.obn'
    APPLICATION_VND_OCF_CBOR = 'application/vnd.ocf+cbor'
    APPLICATION_VND_OCI_IMAGE_MANIFEST_V1_JSON = 'application/vnd.oci.image.manifest.v1+json'
    APPLICATION_VND_OFTN_L10N_JSON = 'application/vnd.oftn.l10n+json'
    APPLICATION_VND_OIPF_CONTENTACCESSDOWNLOAD_XML = 'application/vnd.oipf.contentaccessdownload+xml'
    APPLICATION_VND_OIPF_CONTENTACCESSSTREAMING_XML = 'application/vnd.oipf.contentaccessstreaming+xml'
    APPLICATION_VND_OIPF_CSPG_HEXBINARY = 'application/vnd.oipf.cspg-hexbinary'
    APPLICATION_VND_OIPF_DAE_SVG_XML = 'application/vnd.oipf.dae.svg+xml'
    APPLICATION_VND_OIPF_DAE_XHTML_XML = 'application/vnd.oipf.dae.xhtml+xml'
    APPLICATION_VND_OIPF_MIPPVCONTROLMESSAGE_XML = 'application/vnd.oipf.mippvcontrolmessage+xml'
    APPLICATION_VND_OIPF_PAE_GEM = 'application/vnd.oipf.pae.gem'
    APPLICATION_VND_OIPF_SPDISCOVERY_XML = 'application/vnd.oipf.spdiscovery+xml'
    APPLICATION_VND_OIPF_SPDLIST_XML = 'application/vnd.oipf.spdlist+xml'
    APPLICATION_VND_OIPF_UEPROFILE_XML = 'application/vnd.oipf.ueprofile+xml'
    APPLICATION_VND_OIPF_USERPROFILE_XML = 'application/vnd.oipf.userprofile+xml'
    APPLICATION_VND_OLPC_SUGAR = 'application/vnd.olpc-sugar'
    APPLICATION_VND_OMA_BCAST_ASSOCIATED_PROCEDURE_PARAMETER_XML = (
        'application/vnd.oma.bcast.associated-procedure-parameter+xml'
    )
    APPLICATION_VND_OMA_BCAST_DRM_TRIGGER_XML = 'application/vnd.oma.bcast.drm-trigger+xml'
    APPLICATION_VND_OMA_BCAST_IMD_XML = 'application/vnd.oma.bcast.imd+xml'
    APPLICATION_VND_OMA_BCAST_LTKM = 'application/vnd.oma.bcast.ltkm'
    APPLICATION_VND_OMA_BCAST_NOTIFICATION_XML = 'application/vnd.oma.bcast.notification+xml'
    APPLICATION_VND_OMA_BCAST_PROVISIONINGTRIGGER = 'application/vnd.oma.bcast.provisioningtrigger'
    APPLICATION_VND_OMA_BCAST_SGBOOT = 'application/vnd.oma.bcast.sgboot'
    APPLICATION_VND_OMA_BCAST_SGDD_XML = 'application/vnd.oma.bcast.sgdd+xml'
    APPLICATION_VND_OMA_BCAST_SGDU = 'application/vnd.oma.bcast.sgdu'
    APPLICATION_VND_OMA_BCAST_SIMPLE_SYMBOL_CONTAINER = 'application/vnd.oma.bcast.simple-symbol-container'
    APPLICATION_VND_OMA_BCAST_SMARTCARD_TRIGGER_XML = 'application/vnd.oma.bcast.smartcard-trigger+xml'
    APPLICATION_VND_OMA_BCAST_SPROV_XML = 'application/vnd.oma.bcast.sprov+xml'
    APPLICATION_VND_OMA_BCAST_STKM = 'application/vnd.oma.bcast.stkm'
    APPLICATION_VND_OMA_CAB_ADDRESS_BOOK_XML = 'application/vnd.oma.cab-address-book+xml'
    APPLICATION_VND_OMA_CAB_FEATURE_HANDLER_XML = 'application/vnd.oma.cab-feature-handler+xml'
    APPLICATION_VND_OMA_CAB_PCC_XML = 'application/vnd.oma.cab-pcc+xml'
    APPLICATION_VND_OMA_CAB_SUBS_INVITE_XML = 'application/vnd.oma.cab-subs-invite+xml'
    APPLICATION_VND_OMA_CAB_USER_PREFS_XML = 'application/vnd.oma.cab-user-prefs+xml'
    APPLICATION_VND_OMA_DCD = 'application/vnd.oma.dcd'
    APPLICATION_VND_OMA_DCDC = 'application/vnd.oma.dcdc'
    APPLICATION_VND_OMA_DD2_XML = 'application/vnd.oma.dd2+xml'
    APPLICATION_VND_OMA_DRM_RISD_XML = 'application/vnd.oma.drm.risd+xml'
    APPLICATION_VND_OMA_GROUP_USAGE_LIST_XML = 'application/vnd.oma.group-usage-list+xml'
    APPLICATION_VND_OMA_LWM2M_CBOR = 'application/vnd.oma.lwm2m+cbor'
    APPLICATION_VND_OMA_LWM2M_JSON = 'application/vnd.oma.lwm2m+json'
    APPLICATION_VND_OMA_LWM2M_TLV = 'application/vnd.oma.lwm2m+tlv'
    APPLICATION_VND_OMA_PAL_XML = 'application/vnd.oma.pal+xml'
    APPLICATION_VND_OMA_POC_DETAILED_PROGRESS_REPORT_XML = 'application/vnd.oma.poc.detailed-progress-report+xml'
    APPLICATION_VND_OMA_POC_FINAL_REPORT_XML = 'application/vnd.oma.poc.final-report+xml'
    APPLICATION_VND_OMA_POC_GROUPS_XML = 'application/vnd.oma.poc.groups+xml'
    APPLICATION_VND_OMA_POC_INVOCATION_DESCRIPTOR_XML = 'application/vnd.oma.poc.invocation-descriptor+xml'
    APPLICATION_VND_OMA_POC_OPTIMIZED_PROGRESS_REPORT_XML = 'application/vnd.oma.poc.optimized-progress-report+xml'
    APPLICATION_VND_OMA_PUSH = 'application/vnd.oma.push'
    APPLICATION_VND_OMA_SCIDM_MESSAGES_XML = 'application/vnd.oma.scidm.messages+xml'
    APPLICATION_VND_OMA_XCAP_DIRECTORY_XML = 'application/vnd.oma.xcap-directory+xml'
    APPLICATION_VND_OMADS_EMAIL_XML = 'application/vnd.omads-email+xml'
    APPLICATION_VND_OMADS_FILE_XML = 'application/vnd.omads-file+xml'
    APPLICATION_VND_OMADS_FOLDER_XML = 'application/vnd.omads-folder+xml'
    APPLICATION_VND_OMALOC_SUPL_INIT = 'application/vnd.omaloc-supl-init'
    APPLICATION_VND_OMA_SCWS_CONFIG = 'application/vnd.oma-scws-config'
    APPLICATION_VND_OMA_SCWS_HTTP_REQUEST = 'application/vnd.oma-scws-http-request'
    APPLICATION_VND_OMA_SCWS_HTTP_RESPONSE = 'application/vnd.oma-scws-http-response'
    APPLICATION_VND_OMS_CELLULAR_COSE_CONTENT_CBOR = 'application/vnd.oms.cellular-cose-content+cbor'
    APPLICATION_VND_ONEPAGER = 'application/vnd.onepager'
    APPLICATION_VND_ONEPAGERTAMP = 'application/vnd.onepagertamp'
    APPLICATION_VND_ONEPAGERTAMX = 'application/vnd.onepagertamx'
    APPLICATION_VND_ONEPAGERTAT = 'application/vnd.onepagertat'
    APPLICATION_VND_ONEPAGERTATP = 'application/vnd.onepagertatp'
    APPLICATION_VND_ONEPAGERTATX = 'application/vnd.onepagertatx'
    APPLICATION_VND_ONVIF_METADATA = 'application/vnd.onvif.metadata'
    APPLICATION_VND_OPENBLOX_GAME_BINARY = 'application/vnd.openblox.game-binary'
    APPLICATION_VND_OPENBLOX_GAME_XML = 'application/vnd.openblox.game+xml'
    APPLICATION_VND_OPENEYE_OEB = 'application/vnd.openeye.oeb'
    APPLICATION_VND_OPENSTREETMAP_DATA_XML = 'application/vnd.openstreetmap.data+xml'
    APPLICATION_VND_OPENTIMESTAMPS_OTS = 'application/vnd.opentimestamps.ots'
    APPLICATION_VND_OPENVPI_DSPX_JSON = 'application/vnd.openvpi.dspx+json'
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_CUSTOM_PROPERTIES_XML = (
        'application/vnd.openxmlformats-officedocument.custom-properties+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_CUSTOMXMLPROPERTIES_XML = (
        'application/vnd.openxmlformats-officedocument.customXmlProperties+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWING_XML = (
        'application/vnd.openxmlformats-officedocument.drawing+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWINGML_CHART_XML = (
        'application/vnd.openxmlformats-officedocument.drawingml.chart+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWINGML_CHARTSHAPES_XML = (
        'application/vnd.openxmlformats-officedocument.drawingml.chartshapes+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWINGML_DIAGRAMCOLORS_XML = (
        'application/vnd.openxmlformats-officedocument.drawingml.diagramColors+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWINGML_DIAGRAMDATA_XML = (
        'application/vnd.openxmlformats-officedocument.drawingml.diagramData+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWINGML_DIAGRAMLAYOUT_XML = (
        'application/vnd.openxmlformats-officedocument.drawingml.diagramLayout+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_DRAWINGML_DIAGRAMSTYLE_XML = (
        'application/vnd.openxmlformats-officedocument.drawingml.diagramStyle+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_EXTENDED_PROPERTIES_XML = (
        'application/vnd.openxmlformats-officedocument.extended-properties+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_COMMENTAUTHORS_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.commentAuthors+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_COMMENTS_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.comments+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_HANDOUTMASTER_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.handoutMaster+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_NOTESMASTER_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.notesMaster+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_NOTESSLIDE_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.notesSlide+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_PRESENTATION = (
        'application/vnd.openxmlformats-officedocument.presentationml.presentation'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_PRESENTATION_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.presentation.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_PRESPROPS_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.presProps+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDE = (
        'application/vnd.openxmlformats-officedocument.presentationml.slide'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDE_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.slide+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDELAYOUT_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.slideLayout+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDEMASTER_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.slideMaster+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDESHOW = (
        'application/vnd.openxmlformats-officedocument.presentationml.slideshow'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDESHOW_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.slideshow.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_SLIDEUPDATEINFO_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.slideUpdateInfo+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_TABLESTYLES_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.tableStyles+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_TAGS_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.tags+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_TEMPLATE = (
        'application/vnd.openxmlformats-officedocument.presentationml.template'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_TEMPLATE_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.template.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATIONML_VIEWPROPS_XML = (
        'application/vnd.openxmlformats-officedocument.presentationml.viewProps+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_CALCCHAIN_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.calcChain+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_CHARTSHEET_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.chartsheet+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_COMMENTS_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.comments+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_CONNECTIONS_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.connections+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_DIALOGSHEET_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.dialogsheet+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_EXTERNALLINK_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.externalLink+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_PIVOTCACHEDEFINITION_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheDefinition+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_PIVOTCACHERECORDS_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheRecords+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_PIVOTTABLE_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.pivotTable+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_QUERYTABLE_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.queryTable+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_REVISIONHEADERS_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.revisionHeaders+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_REVISIONLOG_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.revisionLog+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_SHAREDSTRINGS_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_SHEET = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_SHEET_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_SHEETMETADATA_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheetMetadata+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_STYLES_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_TABLE_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.table+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_TABLESINGLECELLS_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.tableSingleCells+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_TEMPLATE = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.template'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_TEMPLATE_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.template.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_USERNAMES_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.userNames+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_VOLATILEDEPENDENCIES_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.volatileDependencies+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEETML_WORKSHEET_XML = (
        'application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_THEME_XML = 'application/vnd.openxmlformats-officedocument.theme+xml'
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_THEMEOVERRIDE_XML = (
        'application/vnd.openxmlformats-officedocument.themeOverride+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_VMLDRAWING = (
        'application/vnd.openxmlformats-officedocument.vmlDrawing'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_COMMENTS_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.comments+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_DOCUMENT = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_DOCUMENT_GLOSSARY_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document.glossary+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_DOCUMENT_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_ENDNOTES_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.endnotes+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_FONTTABLE_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.fontTable+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_FOOTER_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.footer+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_FOOTNOTES_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.footnotes+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_NUMBERING_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.numbering+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_SETTINGS_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.settings+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_STYLES_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.styles+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_TEMPLATE = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.template'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_TEMPLATE_MAIN_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.template.main+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSINGML_WEBSETTINGS_XML = (
        'application/vnd.openxmlformats-officedocument.wordprocessingml.webSettings+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_PACKAGE_CORE_PROPERTIES_XML = (
        'application/vnd.openxmlformats-package.core-properties+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_PACKAGE_DIGITAL_SIGNATURE_XMLSIGNATURE_XML = (
        'application/vnd.openxmlformats-package.digital-signature-xmlsignature+xml'
    )
    APPLICATION_VND_OPENXMLFORMATS_PACKAGE_RELATIONSHIPS_XML = (
        'application/vnd.openxmlformats-package.relationships+xml'
    )
    APPLICATION_VND_ORACLE_RESOURCE_JSON = 'application/vnd.oracle.resource+json'
    APPLICATION_VND_ORANGE_INDATA = 'application/vnd.orange.indata'
    APPLICATION_VND_OSA_NETDEPLOY = 'application/vnd.osa.netdeploy'
    APPLICATION_VND_OSGEO_MAPGUIDE_PACKAGE = 'application/vnd.osgeo.mapguide.package'
    APPLICATION_VND_OSGI_BUNDLE = 'application/vnd.osgi.bundle'
    APPLICATION_VND_OSGI_DP = 'application/vnd.osgi.dp'
    APPLICATION_VND_OSGI_SUBSYSTEM = 'application/vnd.osgi.subsystem'
    APPLICATION_VND_OTPS_CT_KIP_XML = 'application/vnd.otps.ct-kip+xml'
    APPLICATION_VND_OXLI_COUNTGRAPH = 'application/vnd.oxli.countgraph'
    APPLICATION_VND_PAGERDUTY_JSON = 'application/vnd.pagerduty+json'
    APPLICATION_VND_PALM = 'application/vnd.palm'
    APPLICATION_VND_PANOPLY = 'application/vnd.panoply'
    APPLICATION_VND_PAOS_XML = 'application/vnd.paos.xml'
    APPLICATION_VND_PATENTDIVE = 'application/vnd.patentdive'
    APPLICATION_VND_PATIENTECOMMSDOC = 'application/vnd.patientecommsdoc'
    APPLICATION_VND_PAWAAFILE = 'application/vnd.pawaafile'
    APPLICATION_VND_PCOS = 'application/vnd.pcos'
    APPLICATION_VND_PG_FORMAT = 'application/vnd.pg.format'
    APPLICATION_VND_PG_OSASLI = 'application/vnd.pg.osasli'
    APPLICATION_VND_PIACCESS_APPLICATION_LICENCE = 'application/vnd.piaccess.application-licence'
    APPLICATION_VND_PICSEL = 'application/vnd.picsel'
    APPLICATION_VND_PMI_WIDGET = 'application/vnd.pmi.widget'
    APPLICATION_VND_PMTILES = 'application/vnd.pmtiles'
    APPLICATION_VND_POC_GROUP_ADVERTISEMENT_XML = 'application/vnd.poc.group-advertisement+xml'
    APPLICATION_VND_POCKETLEARN = 'application/vnd.pocketlearn'
    APPLICATION_VND_POWERBUILDER6 = 'application/vnd.powerbuilder6'
    APPLICATION_VND_POWERBUILDER6_S = 'application/vnd.powerbuilder6-s'
    APPLICATION_VND_POWERBUILDER7 = 'application/vnd.powerbuilder7'
    APPLICATION_VND_POWERBUILDER75 = 'application/vnd.powerbuilder75'
    APPLICATION_VND_POWERBUILDER75_S = 'application/vnd.powerbuilder75-s'
    APPLICATION_VND_POWERBUILDER7_S = 'application/vnd.powerbuilder7-s'
    APPLICATION_VND_PP_SYSTEMVERIFY_XML = 'application/vnd.pp.systemverify+xml'
    APPLICATION_VND_PREMINET = 'application/vnd.preminet'
    APPLICATION_VND_PREVIEWSYSTEMS_BOX = 'application/vnd.previewsystems.box'
    APPLICATION_VND_PROJECT_GRAPH = 'application/vnd.project-graph'
    APPLICATION_VND_PROTEUS_MAGAZINE = 'application/vnd.proteus.magazine'
    APPLICATION_VND_PSFS = 'application/vnd.psfs'
    APPLICATION_VND_PT_MUNDUSMUNDI = 'application/vnd.pt.mundusmundi'
    APPLICATION_VND_PUBLISHARE_DELTA_TREE = 'application/vnd.publishare-delta-tree'
    APPLICATION_VND_PVI_PTID1 = 'application/vnd.pvi.ptid1'
    APPLICATION_VND_PWG_MULTIPLEXED = 'application/vnd.pwg-multiplexed'
    APPLICATION_VND_PWG_XHTML_PRINT_XML = 'application/vnd.pwg-xhtml-print+xml'
    APPLICATION_VND_PYON_JSON = 'application/vnd.pyon+json'
    APPLICATION_VND_QUALCOMM_BREW_APP_RES = 'application/vnd.qualcomm.brew-app-res'
    APPLICATION_VND_QUARANTAINENET = 'application/vnd.quarantainenet'
    APPLICATION_VND_QUARK_QUARKXPRESS = 'application/vnd.Quark.QuarkXPress'
    APPLICATION_VND_QUOBJECT_QUOXDOCUMENT = 'application/vnd.quobject-quoxdocument'
    APPLICATION_VND_R74N_SANDBOXELS_JSON = 'application/vnd.R74n.sandboxels+json'
    APPLICATION_VND_RADISYS_MOML_XML = 'application/vnd.radisys.moml+xml'
    APPLICATION_VND_RADISYS_MSML_AUDIT_CONF_XML = 'application/vnd.radisys.msml-audit-conf+xml'
    APPLICATION_VND_RADISYS_MSML_AUDIT_CONN_XML = 'application/vnd.radisys.msml-audit-conn+xml'
    APPLICATION_VND_RADISYS_MSML_AUDIT_DIALOG_XML = 'application/vnd.radisys.msml-audit-dialog+xml'
    APPLICATION_VND_RADISYS_MSML_AUDIT_STREAM_XML = 'application/vnd.radisys.msml-audit-stream+xml'
    APPLICATION_VND_RADISYS_MSML_AUDIT_XML = 'application/vnd.radisys.msml-audit+xml'
    APPLICATION_VND_RADISYS_MSML_CONF_XML = 'application/vnd.radisys.msml-conf+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_BASE_XML = 'application/vnd.radisys.msml-dialog-base+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_FAX_DETECT_XML = 'application/vnd.radisys.msml-dialog-fax-detect+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_FAX_SENDRECV_XML = 'application/vnd.radisys.msml-dialog-fax-sendrecv+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_GROUP_XML = 'application/vnd.radisys.msml-dialog-group+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_SPEECH_XML = 'application/vnd.radisys.msml-dialog-speech+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_TRANSFORM_XML = 'application/vnd.radisys.msml-dialog-transform+xml'
    APPLICATION_VND_RADISYS_MSML_DIALOG_XML = 'application/vnd.radisys.msml-dialog+xml'
    APPLICATION_VND_RADISYS_MSML_XML = 'application/vnd.radisys.msml+xml'
    APPLICATION_VND_RAINSTOR_DATA = 'application/vnd.rainstor.data'
    APPLICATION_VND_RAPID = 'application/vnd.rapid'
    APPLICATION_VND_RAR = 'application/vnd.rar'
    APPLICATION_VND_REALVNC_BED = 'application/vnd.realvnc.bed'
    APPLICATION_VND_RECORDARE_MUSICXML = 'application/vnd.recordare.musicxml'
    APPLICATION_VND_RECORDARE_MUSICXML_XML = 'application/vnd.recordare.musicxml+xml'
    APPLICATION_VND_RELPIPE = 'application/vnd.relpipe'
    APPLICATION_VND_RENLEARN_RLPRINT = 'application/vnd.RenLearn.rlprint'
    APPLICATION_VND_RESILIENT_LOGIC = 'application/vnd.resilient.logic'
    APPLICATION_VND_RESTFUL_JSON = 'application/vnd.restful+json'
    APPLICATION_VND_RIG_CRYPTONOTE = 'application/vnd.rig.cryptonote'
    APPLICATION_VND_ROUTE66_LINK66_XML = 'application/vnd.route66.link66+xml'
    APPLICATION_VND_RS_274X = 'application/vnd.rs-274x'
    APPLICATION_VND_RUCKUS_DOWNLOAD = 'application/vnd.ruckus.download'
    APPLICATION_VND_S3SMS = 'application/vnd.s3sms'
    APPLICATION_VND_SAILINGTRACKER_TRACK = 'application/vnd.sailingtracker.track'
    APPLICATION_VND_SAR = 'application/vnd.sar'
    APPLICATION_VND_SBM_CID = 'application/vnd.sbm.cid'
    APPLICATION_VND_SBM_MID2 = 'application/vnd.sbm.mid2'
    APPLICATION_VND_SCRIBUS = 'application/vnd.scribus'
    APPLICATION_VND_SEALED_3DF = 'application/vnd.sealed.3df'
    APPLICATION_VND_SEALED_CSF = 'application/vnd.sealed.csf'
    APPLICATION_VND_SEALED_DOC = 'application/vnd.sealed.doc'
    APPLICATION_VND_SEALED_EML = 'application/vnd.sealed.eml'
    APPLICATION_VND_SEALED_MHT = 'application/vnd.sealed.mht'
    APPLICATION_VND_SEALED_NET = 'application/vnd.sealed.net'
    APPLICATION_VND_SEALED_PPT = 'application/vnd.sealed.ppt'
    APPLICATION_VND_SEALED_TIFF = 'application/vnd.sealed.tiff'
    APPLICATION_VND_SEALED_XLS = 'application/vnd.sealed.xls'
    APPLICATION_VND_SEALEDMEDIA_SOFTSEAL_HTML = 'application/vnd.sealedmedia.softseal.html'
    APPLICATION_VND_SEALEDMEDIA_SOFTSEAL_PDF = 'application/vnd.sealedmedia.softseal.pdf'
    APPLICATION_VND_SEEMAIL = 'application/vnd.seemail'
    APPLICATION_VND_SEIS_JSON = 'application/vnd.seis+json'
    APPLICATION_VND_SEMA = 'application/vnd.sema'
    APPLICATION_VND_SEMD = 'application/vnd.semd'
    APPLICATION_VND_SEMF = 'application/vnd.semf'
    APPLICATION_VND_SHADE_SAVE_FILE = 'application/vnd.shade-save-file'
    APPLICATION_VND_SHANA_INFORMED_FORMDATA = 'application/vnd.shana.informed.formdata'
    APPLICATION_VND_SHANA_INFORMED_FORMTEMPLATE = 'application/vnd.shana.informed.formtemplate'
    APPLICATION_VND_SHANA_INFORMED_INTERCHANGE = 'application/vnd.shana.informed.interchange'
    APPLICATION_VND_SHANA_INFORMED_PACKAGE = 'application/vnd.shana.informed.package'
    APPLICATION_VND_SHOOTPROOF_JSON = 'application/vnd.shootproof+json'
    APPLICATION_VND_SHOPKICK_JSON = 'application/vnd.shopkick+json'
    APPLICATION_VND_SHP = 'application/vnd.shp'
    APPLICATION_VND_SHX = 'application/vnd.shx'
    APPLICATION_VND_SIGROK_SESSION = 'application/vnd.sigrok.session'
    APPLICATION_VND_SIMTECH_MINDMAPPER = 'application/vnd.SimTech-MindMapper'
    APPLICATION_VND_SIREN_JSON = 'application/vnd.siren+json'
    APPLICATION_VND_SIRTX_VMV0 = 'application/vnd.sirtx.vmv0'
    APPLICATION_VND_SKETCHOMETRY = 'application/vnd.sketchometry'
    APPLICATION_VND_SMAF = 'application/vnd.smaf'
    APPLICATION_VND_SMART_NOTEBOOK = 'application/vnd.smart.notebook'
    APPLICATION_VND_SMART_TEACHER = 'application/vnd.smart.teacher'
    APPLICATION_VND_SMINTIO_PORTALS_ARCHIVE = 'application/vnd.smintio.portals.archive'
    APPLICATION_VND_SNESDEV_PAGE_TABLE = 'application/vnd.snesdev-page-table'
    APPLICATION_VND_SOFTWARE602_FILLER_FORM_XML = 'application/vnd.software602.filler.form+xml'
    APPLICATION_VND_SOFTWARE602_FILLER_FORM_XML_ZIP = 'application/vnd.software602.filler.form-xml-zip'
    APPLICATION_VND_SOLENT_SDKM_XML = 'application/vnd.solent.sdkm+xml'
    APPLICATION_VND_SPOTFIRE_DXP = 'application/vnd.spotfire.dxp'
    APPLICATION_VND_SPOTFIRE_SFS = 'application/vnd.spotfire.sfs'
    APPLICATION_VND_SQLITE3 = 'application/vnd.sqlite3'
    APPLICATION_VND_SSS_COD = 'application/vnd.sss-cod'
    APPLICATION_VND_SSS_DTF = 'application/vnd.sss-dtf'
    APPLICATION_VND_SSS_NTF = 'application/vnd.sss-ntf'
    APPLICATION_VND_STEPMANIA_PACKAGE = 'application/vnd.stepmania.package'
    APPLICATION_VND_STEPMANIA_STEPCHART = 'application/vnd.stepmania.stepchart'
    APPLICATION_VND_STREET_STREAM = 'application/vnd.street-stream'
    APPLICATION_VND_SUN_WADL_XML = 'application/vnd.sun.wadl+xml'
    APPLICATION_VND_SUPERFILE_SUPER = 'application/vnd.superfile.super'
    APPLICATION_VND_SUS_CALENDAR = 'application/vnd.sus-calendar'
    APPLICATION_VND_SVD = 'application/vnd.svd'
    APPLICATION_VND_SWIFTVIEW_ICS = 'application/vnd.swiftview-ics'
    APPLICATION_VND_SYBYL_MOL2 = 'application/vnd.sybyl.mol2'
    APPLICATION_VND_SYCLE_XML = 'application/vnd.sycle+xml'
    APPLICATION_VND_SYFT_JSON = 'application/vnd.syft+json'
    APPLICATION_VND_SYNCML_DM_NOTIFICATION = 'application/vnd.syncml.dm.notification'
    APPLICATION_VND_SYNCML_DMDDF_XML = 'application/vnd.syncml.dmddf+xml'
    APPLICATION_VND_SYNCML_DMTNDS_WBXML = 'application/vnd.syncml.dmtnds+wbxml'
    APPLICATION_VND_SYNCML_DMTNDS_XML = 'application/vnd.syncml.dmtnds+xml'
    APPLICATION_VND_SYNCML_DMDDF_WBXML = 'application/vnd.syncml.dmddf+wbxml'
    APPLICATION_VND_SYNCML_DM_WBXML = 'application/vnd.syncml.dm+wbxml'
    APPLICATION_VND_SYNCML_DM_XML = 'application/vnd.syncml.dm+xml'
    APPLICATION_VND_SYNCML_DS_NOTIFICATION = 'application/vnd.syncml.ds.notification'
    APPLICATION_VND_SYNCML_XML = 'application/vnd.syncml+xml'
    APPLICATION_VND_TABLESCHEMA_JSON = 'application/vnd.tableschema+json'
    APPLICATION_VND_TAO_INTENT_MODULE_ARCHIVE = 'application/vnd.tao.intent-module-archive'
    APPLICATION_VND_TCPDUMP_PCAP = 'application/vnd.tcpdump.pcap'
    APPLICATION_VND_THINK_CELL_PPTTC_JSON = 'application/vnd.think-cell.ppttc+json'
    APPLICATION_VND_TML = 'application/vnd.tml'
    APPLICATION_VND_TMD_MEDIAFLEX_API_XML = 'application/vnd.tmd.mediaflex.api+xml'
    APPLICATION_VND_TMOBILE_LIVETV = 'application/vnd.tmobile-livetv'
    APPLICATION_VND_TRI_ONESOURCE = 'application/vnd.tri.onesource'
    APPLICATION_VND_TRID_TPT = 'application/vnd.trid.tpt'
    APPLICATION_VND_TRISCAPE_MXS = 'application/vnd.triscape.mxs'
    APPLICATION_VND_TRUEAPP = 'application/vnd.trueapp'
    APPLICATION_VND_TRUEDOC = 'application/vnd.truedoc'
    APPLICATION_VND_UBISOFT_WEBPLAYER = 'application/vnd.ubisoft.webplayer'
    APPLICATION_VND_UFDL = 'application/vnd.ufdl'
    APPLICATION_VND_UIC_OSDM_JSON = 'application/vnd.uic.osdm+json'
    APPLICATION_VND_UIQ_THEME = 'application/vnd.uiq.theme'
    APPLICATION_VND_UMAJIN = 'application/vnd.umajin'
    APPLICATION_VND_UNITY = 'application/vnd.unity'
    APPLICATION_VND_UOML_XML = 'application/vnd.uoml+xml'
    APPLICATION_VND_UPLANET_ALERT = 'application/vnd.uplanet.alert'
    APPLICATION_VND_UPLANET_ALERT_WBXML = 'application/vnd.uplanet.alert-wbxml'
    APPLICATION_VND_UPLANET_BEARER_CHOICE = 'application/vnd.uplanet.bearer-choice'
    APPLICATION_VND_UPLANET_BEARER_CHOICE_WBXML = 'application/vnd.uplanet.bearer-choice-wbxml'
    APPLICATION_VND_UPLANET_CACHEOP = 'application/vnd.uplanet.cacheop'
    APPLICATION_VND_UPLANET_CACHEOP_WBXML = 'application/vnd.uplanet.cacheop-wbxml'
    APPLICATION_VND_UPLANET_CHANNEL = 'application/vnd.uplanet.channel'
    APPLICATION_VND_UPLANET_CHANNEL_WBXML = 'application/vnd.uplanet.channel-wbxml'
    APPLICATION_VND_UPLANET_LIST = 'application/vnd.uplanet.list'
    APPLICATION_VND_UPLANET_LISTCMD = 'application/vnd.uplanet.listcmd'
    APPLICATION_VND_UPLANET_LISTCMD_WBXML = 'application/vnd.uplanet.listcmd-wbxml'
    APPLICATION_VND_UPLANET_LIST_WBXML = 'application/vnd.uplanet.list-wbxml'
    APPLICATION_VND_URI_MAP = 'application/vnd.uri-map'
    APPLICATION_VND_UPLANET_SIGNAL = 'application/vnd.uplanet.signal'
    APPLICATION_VND_VALVE_SOURCE_MATERIAL = 'application/vnd.valve.source.material'
    APPLICATION_VND_VCX = 'application/vnd.vcx'
    APPLICATION_VND_VD_STUDY = 'application/vnd.vd-study'
    APPLICATION_VND_VECTORWORKS = 'application/vnd.vectorworks'
    APPLICATION_VND_VEL_JSON = 'application/vnd.vel+json'
    APPLICATION_VND_VERAISON_TSM_REPORT_CBOR = 'application/vnd.veraison.tsm-report+cbor'
    APPLICATION_VND_VERAISON_TSM_REPORT_JSON = 'application/vnd.veraison.tsm-report+json'
    APPLICATION_VND_VERIFIER_ATTESTATION_JWT = 'application/vnd.verifier-attestation+jwt'
    APPLICATION_VND_VERIMATRIX_VCAS = 'application/vnd.verimatrix.vcas'
    APPLICATION_VND_VERITONE_AION_JSON = 'application/vnd.veritone.aion+json'
    APPLICATION_VND_VERYANT_THIN = 'application/vnd.veryant.thin'
    APPLICATION_VND_VES_ENCRYPTED = 'application/vnd.ves.encrypted'
    APPLICATION_VND_VIDSOFT_VIDCONFERENCE = 'application/vnd.vidsoft.vidconference'
    APPLICATION_VND_VISIO = 'application/vnd.visio'
    APPLICATION_VND_VISIONARY = 'application/vnd.visionary'
    APPLICATION_VND_VIVIDENCE_SCRIPTFILE = 'application/vnd.vividence.scriptfile'
    APPLICATION_VND_VOCALSHAPER_VSP4 = 'application/vnd.vocalshaper.vsp4'
    APPLICATION_VND_VSF = 'application/vnd.vsf'
    APPLICATION_VND_VUQ = 'application/vnd.vuq'
    APPLICATION_VND_WANTVERSE = 'application/vnd.wantverse'
    APPLICATION_VND_WAP_SIC = 'application/vnd.wap.sic'
    APPLICATION_VND_WAP_SLC = 'application/vnd.wap.slc'
    APPLICATION_VND_WAP_WBXML = 'application/vnd.wap.wbxml'
    APPLICATION_VND_WAP_WMLC = 'application/vnd.wap.wmlc'
    APPLICATION_VND_WAP_WMLSCRIPTC = 'application/vnd.wap.wmlscriptc'
    APPLICATION_VND_WASMFLOW_WAFL = 'application/vnd.wasmflow.wafl'
    APPLICATION_VND_WEBTURBO = 'application/vnd.webturbo'
    APPLICATION_VND_WFA_DPP = 'application/vnd.wfa.dpp'
    APPLICATION_VND_WFA_P2P = 'application/vnd.wfa.p2p'
    APPLICATION_VND_WFA_WSC = 'application/vnd.wfa.wsc'
    APPLICATION_VND_WINDOWS_DEVICEPAIRING = 'application/vnd.windows.devicepairing'
    APPLICATION_VND_WMAP = 'application/vnd.wmap'
    APPLICATION_VND_WMC = 'application/vnd.wmc'
    APPLICATION_VND_WMF_BOOTSTRAP = 'application/vnd.wmf.bootstrap'
    APPLICATION_VND_WOLFRAM_MATHEMATICA = 'application/vnd.wolfram.mathematica'
    APPLICATION_VND_WOLFRAM_MATHEMATICA_PACKAGE = 'application/vnd.wolfram.mathematica.package'
    APPLICATION_VND_WOLFRAM_PLAYER = 'application/vnd.wolfram.player'
    APPLICATION_VND_WORDLIFT = 'application/vnd.wordlift'
    APPLICATION_VND_WORDPERFECT = 'application/vnd.wordperfect'
    APPLICATION_VND_WQD = 'application/vnd.wqd'
    APPLICATION_VND_WRQ_HP3000_LABELLED = 'application/vnd.wrq-hp3000-labelled'
    APPLICATION_VND_WT_STF = 'application/vnd.wt.stf'
    APPLICATION_VND_WV_CSP_XML = 'application/vnd.wv.csp+xml'
    APPLICATION_VND_WV_CSP_WBXML = 'application/vnd.wv.csp+wbxml'
    APPLICATION_VND_WV_SSP_XML = 'application/vnd.wv.ssp+xml'
    APPLICATION_VND_XACML_JSON = 'application/vnd.xacml+json'
    APPLICATION_VND_XARA = 'application/vnd.xara'
    APPLICATION_VND_XARIN_CPJ = 'application/vnd.xarin.cpj'
    APPLICATION_VND_XECRETS_ENCRYPTED = 'application/vnd.xecrets-encrypted'
    APPLICATION_VND_XFDL = 'application/vnd.xfdl'
    APPLICATION_VND_XFDL_WEBFORM = 'application/vnd.xfdl.webform'
    APPLICATION_VND_XMI_XML = 'application/vnd.xmi+xml'
    APPLICATION_VND_XMPIE_CPKG = 'application/vnd.xmpie.cpkg'
    APPLICATION_VND_XMPIE_DPKG = 'application/vnd.xmpie.dpkg'
    APPLICATION_VND_XMPIE_PLAN = 'application/vnd.xmpie.plan'
    APPLICATION_VND_XMPIE_PPKG = 'application/vnd.xmpie.ppkg'
    APPLICATION_VND_XMPIE_XLIM = 'application/vnd.xmpie.xlim'
    APPLICATION_VND_YAMAHA_HV_DIC = 'application/vnd.yamaha.hv-dic'
    APPLICATION_VND_YAMAHA_HV_SCRIPT = 'application/vnd.yamaha.hv-script'
    APPLICATION_VND_YAMAHA_HV_VOICE = 'application/vnd.yamaha.hv-voice'
    APPLICATION_VND_YAMAHA_OPENSCOREFORMAT_OSFPVG_XML = 'application/vnd.yamaha.openscoreformat.osfpvg+xml'
    APPLICATION_VND_YAMAHA_OPENSCOREFORMAT = 'application/vnd.yamaha.openscoreformat'
    APPLICATION_VND_YAMAHA_REMOTE_SETUP = 'application/vnd.yamaha.remote-setup'
    APPLICATION_VND_YAMAHA_SMAF_AUDIO = 'application/vnd.yamaha.smaf-audio'
    APPLICATION_VND_YAMAHA_SMAF_PHRASE = 'application/vnd.yamaha.smaf-phrase'
    APPLICATION_VND_YAMAHA_THROUGH_NGN = 'application/vnd.yamaha.through-ngn'
    APPLICATION_VND_YAMAHA_TUNNEL_UDPENCAP = 'application/vnd.yamaha.tunnel-udpencap'
    APPLICATION_VND_YAOWEME = 'application/vnd.yaoweme'
    APPLICATION_VND_YELLOWRIVER_CUSTOM_MENU = 'application/vnd.yellowriver-custom-menu'
    APPLICATION_VND_YOUTUBE_YT = 'application/vnd.youtube.yt'  # OBSOLETED in favor of video/vnd.youtube.yt
    APPLICATION_VND_ZUL = 'application/vnd.zul'
    APPLICATION_VND_ZZAZZ_DECK_XML = 'application/vnd.zzazz.deck+xml'
    APPLICATION_VOICEXML_XML = 'application/voicexml+xml'
    APPLICATION_VOUCHER_CMS_JSON = 'application/voucher-cms+json'
    APPLICATION_VOUCHER_JWS_JSON = 'application/voucher-jws+json'
    APPLICATION_VP = 'application/vp'
    APPLICATION_VP_COSE = 'application/vp+cose'
    APPLICATION_VP_JWT = 'application/vp+jwt'
    APPLICATION_VQ_RTCPXR = 'application/vq-rtcpxr'
    APPLICATION_WASM = 'application/wasm'
    APPLICATION_WATCHERINFO_XML = 'application/watcherinfo+xml'
    APPLICATION_WEBPUSH_OPTIONS_JSON = 'application/webpush-options+json'
    APPLICATION_WHOISPP_QUERY = 'application/whoispp-query'
    APPLICATION_WHOISPP_RESPONSE = 'application/whoispp-response'
    APPLICATION_WIDGET = 'application/widget'
    APPLICATION_WITA = 'application/wita'
    APPLICATION_WORDPERFECT5_1 = 'application/wordperfect5.1'
    APPLICATION_WSDL_XML = 'application/wsdl+xml'
    APPLICATION_WSPOLICY_XML = 'application/wspolicy+xml'
    APPLICATION_X_PKI_MESSAGE = 'application/x-pki-message'
    APPLICATION_X_WWW_FORM_URLENCODED = 'application/x-www-form-urlencoded'
    APPLICATION_X_X509_CA_CERT = 'application/x-x509-ca-cert'
    APPLICATION_X_X509_CA_RA_CERT = 'application/x-x509-ca-ra-cert'
    APPLICATION_X_X509_NEXT_CA_CERT = 'application/x-x509-next-ca-cert'
    APPLICATION_X400_BP = 'application/x400-bp'
    APPLICATION_XACML_XML = 'application/xacml+xml'
    APPLICATION_XCAP_ATT_XML = 'application/xcap-att+xml'
    APPLICATION_XCAP_CAPS_XML = 'application/xcap-caps+xml'
    APPLICATION_XCAP_DIFF_XML = 'application/xcap-diff+xml'
    APPLICATION_XCAP_EL_XML = 'application/xcap-el+xml'
    APPLICATION_XCAP_ERROR_XML = 'application/xcap-error+xml'
    APPLICATION_XCAP_NS_XML = 'application/xcap-ns+xml'
    APPLICATION_XCON_CONFERENCE_INFO_DIFF_XML = 'application/xcon-conference-info-diff+xml'
    APPLICATION_XCON_CONFERENCE_INFO_XML = 'application/xcon-conference-info+xml'
    APPLICATION_XENC_XML = 'application/xenc+xml'
    APPLICATION_XFDF = 'application/xfdf'
    APPLICATION_XHTML_XML = 'application/xhtml+xml'
    APPLICATION_XLIFF_XML = 'application/xliff+xml'
    APPLICATION_XML = 'application/xml'
    APPLICATION_XML_DTD = 'application/xml-dtd'
    APPLICATION_XML_EXTERNAL_PARSED_ENTITY = 'application/xml-external-parsed-entity'
    APPLICATION_XML_PATCH_XML = 'application/xml-patch+xml'
    APPLICATION_XMPP_XML = 'application/xmpp+xml'
    APPLICATION_XOP_XML = 'application/xop+xml'
    APPLICATION_XSLT_XML = 'application/xslt+xml'
    APPLICATION_XV_XML = 'application/xv+xml'
    APPLICATION_YAML = 'application/yaml'
    APPLICATION_YANG = 'application/yang'
    APPLICATION_YANG_DATA_CBOR = 'application/yang-data+cbor'
    APPLICATION_YANG_DATA_JSON = 'application/yang-data+json'
    APPLICATION_YANG_DATA_XML = 'application/yang-data+xml'
    APPLICATION_YANG_PATCH_JSON = 'application/yang-patch+json'
    APPLICATION_YANG_PATCH_XML = 'application/yang-patch+xml'
    APPLICATION_YANG_SID_JSON = 'application/yang-sid+json'
    APPLICATION_YIN_XML = 'application/yin+xml'
    APPLICATION_ZIP = 'application/zip'
    APPLICATION_ZLIB = 'application/zlib'
    APPLICATION_ZSTD = 'application/zstd'


class _AudioEnum(StrEnum):  # pragma: no cover
    AUDIO_1D_INTERLEAVED_PARITYFEC = 'audio/1d-interleaved-parityfec'
    AUDIO_32KADPCM = 'audio/32kadpcm'
    AUDIO_3GPP = 'audio/3gpp'
    AUDIO_3GPP2 = 'audio/3gpp2'
    AUDIO_AAC = 'audio/aac'
    AUDIO_AC3 = 'audio/ac3'
    AUDIO_AMR = 'audio/AMR'
    AUDIO_AMR_WB = 'audio/amr-wb+'
    AUDIO_APTX = 'audio/aptx'
    AUDIO_ASC = 'audio/asc'
    AUDIO_ATRAC_ADVANCED_LOSSLESS = 'audio/ATRAC-ADVANCED-LOSSLESS'
    AUDIO_ATRAC_X = 'audio/ATRAC-X'
    AUDIO_ATRAC3 = 'audio/ATRAC3'
    AUDIO_BASIC = 'audio/basic'
    AUDIO_BV16 = 'audio/BV16'
    AUDIO_BV32 = 'audio/BV32'
    AUDIO_CLEARMODE = 'audio/clearmode'
    AUDIO_CN = 'audio/CN'
    AUDIO_DAT12 = 'audio/DAT12'
    AUDIO_DLS = 'audio/dls'
    AUDIO_DSR_ES201108 = 'audio/dsr-es201108'
    AUDIO_DSR_ES202050 = 'audio/dsr-es202050'
    AUDIO_DSR_ES202211 = 'audio/dsr-es202211'
    AUDIO_DSR_ES202212 = 'audio/dsr-es202212'
    AUDIO_DV = 'audio/DV'
    AUDIO_DVI4 = 'audio/DVI4'
    AUDIO_EAC3 = 'audio/eac3'
    AUDIO_ENCAPRTP = 'audio/encaprtp'
    AUDIO_EVRC = 'audio/EVRC'
    AUDIO_EVRC_QCP = 'audio/EVRC-QCP'
    AUDIO_EVRC0 = 'audio/EVRC0'
    AUDIO_EVRC1 = 'audio/EVRC1'
    AUDIO_EVRCB = 'audio/EVRCB'
    AUDIO_EVRCB0 = 'audio/EVRCB0'
    AUDIO_EVRCB1 = 'audio/EVRCB1'
    AUDIO_EVRCNW = 'audio/EVRCNW'
    AUDIO_EVRCNW0 = 'audio/EVRCNW0'
    AUDIO_EVRCNW1 = 'audio/EVRCNW1'
    AUDIO_EVRCWB = 'audio/EVRCWB'
    AUDIO_EVRCWB0 = 'audio/EVRCWB0'
    AUDIO_EVRCWB1 = 'audio/EVRCWB1'
    AUDIO_EVS = 'audio/EVS'
    AUDIO_EXAMPLE = 'audio/example'
    AUDIO_FLAC = 'audio/flac'
    AUDIO_FLEXFEC = 'audio/flexfec'
    AUDIO_FWDRED = 'audio/fwdred'
    AUDIO_G711_0 = 'audio/G711-0'
    AUDIO_G719 = 'audio/G719'
    AUDIO_G7221 = 'audio/G7221'
    AUDIO_G722 = 'audio/G722'
    AUDIO_G723 = 'audio/G723'
    AUDIO_G726_16 = 'audio/G726-16'
    AUDIO_G726_24 = 'audio/G726-24'
    AUDIO_G726_32 = 'audio/G726-32'
    AUDIO_G726_40 = 'audio/G726-40'
    AUDIO_G728 = 'audio/G728'
    AUDIO_G729 = 'audio/G729'
    AUDIO_G7291 = 'audio/G7291'
    AUDIO_G729D = 'audio/G729D'
    AUDIO_G729E = 'audio/G729E'
    AUDIO_GSM = 'audio/GSM'
    AUDIO_GSM_EFR = 'audio/GSM-EFR'
    AUDIO_GSM_HR_08 = 'audio/GSM-HR-08'
    AUDIO_ILBC = 'audio/iLBC'
    AUDIO_IP_MR_V2_5 = 'audio/ip-mr_v2.5'
    AUDIO_L8 = 'audio/L8'
    AUDIO_L16 = 'audio/L16'
    AUDIO_L20 = 'audio/L20'
    AUDIO_L24 = 'audio/L24'
    AUDIO_LPC = 'audio/LPC'
    AUDIO_MATROSKA = 'audio/matroska'
    AUDIO_MELP = 'audio/MELP'
    AUDIO_MELP600 = 'audio/MELP600'
    AUDIO_MELP1200 = 'audio/MELP1200'
    AUDIO_MELP2400 = 'audio/MELP2400'
    AUDIO_MHAS = 'audio/mhas'
    AUDIO_MIDI_CLIP = 'audio/midi-clip'
    AUDIO_MOBILE_XMF = 'audio/mobile-xmf'
    AUDIO_MPA = 'audio/MPA'
    AUDIO_MP4 = 'audio/mp4'
    AUDIO_MP4A_LATM = 'audio/MP4A-LATM'
    AUDIO_MPA_ROBUST = 'audio/mpa-robust'
    AUDIO_MPEG = 'audio/mpeg'
    AUDIO_MPEG4_GENERIC = 'audio/mpeg4-generic'
    AUDIO_OGG = 'audio/ogg'
    AUDIO_OPUS = 'audio/opus'
    AUDIO_PARITYFEC = 'audio/parityfec'
    AUDIO_PCMA = 'audio/PCMA'
    AUDIO_PCMA_WB = 'audio/PCMA-WB'
    AUDIO_PCMU = 'audio/PCMU'
    AUDIO_PCMU_WB = 'audio/PCMU-WB'
    AUDIO_PRS_SID = 'audio/prs.sid'
    AUDIO_QCELP = 'audio/QCELP'
    AUDIO_RAPTORFEC = 'audio/raptorfec'
    AUDIO_RED = 'audio/RED'
    AUDIO_RTP_ENC_AESCM128 = 'audio/rtp-enc-aescm128'
    AUDIO_RTPLOOPBACK = 'audio/rtploopback'
    AUDIO_RTP_MIDI = 'audio/rtp-midi'
    AUDIO_RTX = 'audio/rtx'
    AUDIO_SCIP = 'audio/scip'
    AUDIO_SMV = 'audio/SMV'
    AUDIO_SMV0 = 'audio/SMV0'
    AUDIO_SMV_QCP = 'audio/SMV-QCP'
    AUDIO_SOFA = 'audio/sofa'
    AUDIO_SP_MIDI = 'audio/sp-midi'
    AUDIO_SPEEX = 'audio/speex'
    AUDIO_T140C = 'audio/t140c'
    AUDIO_T38 = 'audio/t38'
    AUDIO_TELEPHONE_EVENT = 'audio/telephone-event'
    AUDIO_TETRA_ACELP = 'audio/TETRA_ACELP'
    AUDIO_TETRA_ACELP_BB = 'audio/TETRA_ACELP_BB'
    AUDIO_TONE = 'audio/tone'
    AUDIO_TSVCIS = 'audio/TSVCIS'
    AUDIO_UEMCLIP = 'audio/UEMCLIP'
    AUDIO_ULPFEC = 'audio/ulpfec'
    AUDIO_USAC = 'audio/usac'
    AUDIO_VDVI = 'audio/VDVI'
    AUDIO_VMR_WB = 'audio/VMR-WB'
    AUDIO_VND_3GPP_IUFP = 'audio/vnd.3gpp.iufp'
    AUDIO_VND_4SB = 'audio/vnd.4SB'
    AUDIO_VND_AUDIOKOZ = 'audio/vnd.audiokoz'
    AUDIO_VND_BLOCKFACT_FACTA = 'audio/vnd.blockfact.facta'
    AUDIO_VND_CELP = 'audio/vnd.CELP'
    AUDIO_VND_CISCO_NSE = 'audio/vnd.cisco.nse'
    AUDIO_VND_CMLES_RADIO_EVENTS = 'audio/vnd.cmles.radio-events'
    AUDIO_VND_CNS_ANP1 = 'audio/vnd.cns.anp1'
    AUDIO_VND_CNS_INF1 = 'audio/vnd.cns.inf1'
    AUDIO_VND_DECE_AUDIO = 'audio/vnd.dece.audio'
    AUDIO_VND_DIGITAL_WINDS = 'audio/vnd.digital-winds'
    AUDIO_VND_DLNA_ADTS = 'audio/vnd.dlna.adts'
    AUDIO_VND_DOLBY_HEAAC_1 = 'audio/vnd.dolby.heaac.1'
    AUDIO_VND_DOLBY_HEAAC_2 = 'audio/vnd.dolby.heaac.2'
    AUDIO_VND_DOLBY_MLP = 'audio/vnd.dolby.mlp'
    AUDIO_VND_DOLBY_MPS = 'audio/vnd.dolby.mps'
    AUDIO_VND_DOLBY_PL2 = 'audio/vnd.dolby.pl2'
    AUDIO_VND_DOLBY_PL2X = 'audio/vnd.dolby.pl2x'
    AUDIO_VND_DOLBY_PL2Z = 'audio/vnd.dolby.pl2z'
    AUDIO_VND_DOLBY_PULSE_1 = 'audio/vnd.dolby.pulse.1'
    AUDIO_VND_DRA = 'audio/vnd.dra'
    AUDIO_VND_DTS = 'audio/vnd.dts'
    AUDIO_VND_DTS_HD = 'audio/vnd.dts.hd'
    AUDIO_VND_DTS_UHD = 'audio/vnd.dts.uhd'
    AUDIO_VND_DVB_FILE = 'audio/vnd.dvb.file'
    AUDIO_VND_EVERAD_PLJ = 'audio/vnd.everad.plj'
    AUDIO_VND_HNS_AUDIO = 'audio/vnd.hns.audio'
    AUDIO_VND_LUCENT_VOICE = 'audio/vnd.lucent.voice'
    AUDIO_VND_MS_PLAYREADY_MEDIA_PYA = 'audio/vnd.ms-playready.media.pya'
    AUDIO_VND_NOKIA_MOBILE_XMF = 'audio/vnd.nokia.mobile-xmf'
    AUDIO_VND_NORTEL_VBK = 'audio/vnd.nortel.vbk'
    AUDIO_VND_NUERA_ECELP4800 = 'audio/vnd.nuera.ecelp4800'
    AUDIO_VND_NUERA_ECELP7470 = 'audio/vnd.nuera.ecelp7470'
    AUDIO_VND_NUERA_ECELP9600 = 'audio/vnd.nuera.ecelp9600'
    AUDIO_VND_OCTEL_SBC = 'audio/vnd.octel.sbc'
    AUDIO_VND_PRESONUS_MULTITRACK = 'audio/vnd.presonus.multitrack'
    AUDIO_VND_QCELP_DEPRECATED_IN_FAVOR_OF_AUDIO_QCELP = 'audio/vnd.qcelp'
    AUDIO_VND_RHETOREX_32KADPCM = 'audio/vnd.rhetorex.32kadpcm'
    AUDIO_VND_RIP = 'audio/vnd.rip'
    AUDIO_VND_SEALEDMEDIA_SOFTSEAL_MPEG = 'audio/vnd.sealedmedia.softseal.mpeg'
    AUDIO_VND_VMX_CVSD = 'audio/vnd.vmx.cvsd'
    AUDIO_VORBIS = 'audio/vorbis'
    AUDIO_VORBIS_CONFIG = 'audio/vorbis-config'


class _FontEnum(StrEnum):  # pragma: no cover
    FONT_COLLECTION = 'font/collection'
    FONT_OTF = 'font/otf'
    FONT_SFNT = 'font/sfnt'
    FONT_TTF = 'font/ttf'
    FONT_WOFF = 'font/woff'
    FONT_WOFF2 = 'font/woff2'


class _HapticsEnum(StrEnum):  # pragma: no cover
    HAPTICS_IVS = 'haptics/ivs'
    HAPTICS_HJIF = 'haptics/hjif'
    HAPTICS_HMPG = 'haptics/hmpg'


class _ImageEnum(StrEnum):  # pragma: no cover
    IMAGE_ACES = 'image/aces'
    IMAGE_APNG = 'image/apng'
    IMAGE_AVCI = 'image/avci'
    IMAGE_AVCS = 'image/avcs'
    IMAGE_AVIF = 'image/avif'
    IMAGE_BMP = 'image/bmp'
    IMAGE_CGM = 'image/cgm'
    IMAGE_DICOM_RLE = 'image/dicom-rle'
    IMAGE_DPX = 'image/dpx'
    IMAGE_EMF = 'image/emf'
    IMAGE_EXAMPLE = 'image/example'
    IMAGE_FITS = 'image/fits'
    IMAGE_G3FAX = 'image/g3fax'
    IMAGE_GIF = 'image/gif'
    IMAGE_HEIC = 'image/heic'
    IMAGE_HEIC_SEQUENCE = 'image/heic-sequence'
    IMAGE_HEIF = 'image/heif'
    IMAGE_HEIF_SEQUENCE = 'image/heif-sequence'
    IMAGE_HEJ2K = 'image/hej2k'
    IMAGE_HSJ2 = 'image/hsj2'  # OBSOLETE
    IMAGE_IEF = 'image/ief'
    IMAGE_J2C = 'image/j2c'
    IMAGE_JAII = 'image/jaii'
    IMAGE_JAIS = 'image/jais'
    IMAGE_JLS = 'image/jls'
    IMAGE_JP2 = 'image/jp2'
    IMAGE_JPEG = 'image/jpeg'
    IMAGE_JPH = 'image/jph'
    IMAGE_JPHC = 'image/jphc'
    IMAGE_JPM = 'image/jpm'
    IMAGE_JPX = 'image/jpx'
    IMAGE_JXL = 'image/jxl'
    IMAGE_JXR = 'image/jxr'
    IMAGE_JXRA = 'image/jxrA'
    IMAGE_JXRS = 'image/jxrS'
    IMAGE_JXS = 'image/jxs'
    IMAGE_JXSC = 'image/jxsc'
    IMAGE_JXSI = 'image/jxsi'
    IMAGE_JXSS = 'image/jxss'
    IMAGE_KTX = 'image/ktx'
    IMAGE_KTX2 = 'image/ktx2'
    IMAGE_NAPLPS = 'image/naplps'
    IMAGE_PNG = 'image/png'
    IMAGE_PRS_BTIF = 'image/prs.btif'
    IMAGE_PRS_PTI = 'image/prs.pti'
    IMAGE_PWG_RASTER = 'image/pwg-raster'
    IMAGE_SVG_XML = 'image/svg+xml'
    IMAGE_T38 = 'image/t38'
    IMAGE_TIFF = 'image/tiff'
    IMAGE_TIFF_FX = 'image/tiff-fx'
    IMAGE_VND_ADOBE_PHOTOSHOP = 'image/vnd.adobe.photoshop'
    IMAGE_VND_AIRZIP_ACCELERATOR_AZV = 'image/vnd.airzip.accelerator.azv'
    IMAGE_VND_BLOCKFACT_FACTI = 'image/vnd.blockfact.facti'
    IMAGE_VND_CLIP = 'image/vnd.clip'
    IMAGE_VND_CNS_INF2 = 'image/vnd.cns.inf2'
    IMAGE_VND_DECE_GRAPHIC = 'image/vnd.dece.graphic'
    IMAGE_VND_DJVU = 'image/vnd.djvu'
    IMAGE_VND_DWG = 'image/vnd.dwg'
    IMAGE_VND_DXF = 'image/vnd.dxf'
    IMAGE_VND_DVB_SUBTITLE = 'image/vnd.dvb.subtitle'
    IMAGE_VND_FASTBIDSHEET = 'image/vnd.fastbidsheet'
    IMAGE_VND_FPX = 'image/vnd.fpx'
    IMAGE_VND_FST = 'image/vnd.fst'
    IMAGE_VND_FUJIXEROX_EDMICS_MMR = 'image/vnd.fujixerox.edmics-mmr'
    IMAGE_VND_FUJIXEROX_EDMICS_RLC = 'image/vnd.fujixerox.edmics-rlc'
    IMAGE_VND_GLOBALGRAPHICS_PGB = 'image/vnd.globalgraphics.pgb'
    IMAGE_VND_MICROSOFT_ICON = 'image/vnd.microsoft.icon'
    IMAGE_VND_MIX = 'image/vnd.mix'
    IMAGE_VND_MS_MODI = 'image/vnd.ms-modi'
    IMAGE_VND_MOZILLA_APNG = 'image/vnd.mozilla.apng'
    IMAGE_VND_NET_FPX = 'image/vnd.net-fpx'
    IMAGE_VND_PCO_B16 = 'image/vnd.pco.b16'
    IMAGE_VND_RADIANCE = 'image/vnd.radiance'
    IMAGE_VND_SEALED_PNG = 'image/vnd.sealed.png'
    IMAGE_VND_SEALEDMEDIA_SOFTSEAL_GIF = 'image/vnd.sealedmedia.softseal.gif'
    IMAGE_VND_SEALEDMEDIA_SOFTSEAL_JPG = 'image/vnd.sealedmedia.softseal.jpg'
    IMAGE_VND_SVF = 'image/vnd.svf'
    IMAGE_VND_TENCENT_TAP = 'image/vnd.tencent.tap'
    IMAGE_VND_VALVE_SOURCE_TEXTURE = 'image/vnd.valve.source.texture'
    IMAGE_VND_WAP_WBMP = 'image/vnd.wap.wbmp'
    IMAGE_VND_XIFF = 'image/vnd.xiff'
    IMAGE_VND_ZBRUSH_PCX = 'image/vnd.zbrush.pcx'
    IMAGE_WEBP = 'image/webp'
    IMAGE_WMF = 'image/wmf'
    IMAGE_X_EMF_DEPRECATED_IN_FAVOR_OF_IMAGE_EMF = 'image/x-emf'
    IMAGE_X_WMF_DEPRECATED_IN_FAVOR_OF_IMAGE_WMF = 'image/x-wmf'


class _MessageEnum(StrEnum):  # pragma: no cover
    MESSAGE_BHTTP = 'message/bhttp'
    MESSAGE_CPIM = 'message/CPIM'
    MESSAGE_DELIVERY_STATUS = 'message/delivery-status'
    MESSAGE_DISPOSITION_NOTIFICATION = 'message/disposition-notification'
    MESSAGE_EXAMPLE = 'message/example'
    MESSAGE_EXTERNAL_BODY = 'message/external-body'
    MESSAGE_FEEDBACK_REPORT = 'message/feedback-report'
    MESSAGE_GLOBAL = 'message/global'
    MESSAGE_GLOBAL_DELIVERY_STATUS = 'message/global-delivery-status'
    MESSAGE_GLOBAL_DISPOSITION_NOTIFICATION = 'message/global-disposition-notification'
    MESSAGE_GLOBAL_HEADERS = 'message/global-headers'
    MESSAGE_HTTP = 'message/http'
    MESSAGE_IMDN_XML = 'message/imdn+xml'
    MESSAGE_MLS = 'message/mls'
    MESSAGE_NEWS = 'message/news'  # OBSOLETED by [RFC5537]
    MESSAGE_OHTTP_REQ = 'message/ohttp-req'
    MESSAGE_OHTTP_RES = 'message/ohttp-res'
    MESSAGE_PARTIAL = 'message/partial'
    MESSAGE_RFC822 = 'message/rfc822'
    MESSAGE_S_HTTP = 'message/s-http'  # OBSOLETE
    MESSAGE_SIP = 'message/sip'
    MESSAGE_SIPFRAG = 'message/sipfrag'
    MESSAGE_TRACKING_STATUS = 'message/tracking-status'
    MESSAGE_VND_SI_SIMP = 'message/vnd.si.simp'  # OBSOLETED by request
    MESSAGE_VND_WFA_WSC = 'message/vnd.wfa.wsc'


class _ModelEnum(StrEnum):  # pragma: no cover
    MODEL_3MF = 'model/3mf'
    MODEL_E57 = 'model/e57'
    MODEL_EXAMPLE = 'model/example'
    MODEL_GLTF_BINARY = 'model/gltf-binary'
    MODEL_GLTF_JSON = 'model/gltf+json'
    MODEL_JT = 'model/JT'
    MODEL_IGES = 'model/iges'
    MODEL_MESH = 'model/mesh'
    MODEL_MTL = 'model/mtl'
    MODEL_OBJ = 'model/obj'
    MODEL_PRC = 'model/prc'
    MODEL_STEP = 'model/step'
    MODEL_STEP_XML = 'model/step+xml'
    MODEL_STEP_ZIP = 'model/step+zip'
    MODEL_STEP_XML_ZIP = 'model/step-xml+zip'
    MODEL_STL = 'model/stl'
    MODEL_U3D = 'model/u3d'
    MODEL_VND_BARY = 'model/vnd.bary'
    MODEL_VND_CLD = 'model/vnd.cld'
    MODEL_VND_COLLADA_XML = 'model/vnd.collada+xml'
    MODEL_VND_DWF = 'model/vnd.dwf'
    MODEL_VND_FLATLAND_3DML = 'model/vnd.flatland.3dml'
    MODEL_VND_GDL = 'model/vnd.gdl'
    MODEL_VND_GS_GDL = 'model/vnd.gs-gdl'
    MODEL_VND_GTW = 'model/vnd.gtw'
    MODEL_VND_MOML_XML = 'model/vnd.moml+xml'
    MODEL_VND_MTS = 'model/vnd.mts'
    MODEL_VND_OPENGEX = 'model/vnd.opengex'
    MODEL_VND_PARASOLID_TRANSMIT_BINARY = 'model/vnd.parasolid.transmit.binary'
    MODEL_VND_PARASOLID_TRANSMIT_TEXT = 'model/vnd.parasolid.transmit.text'
    MODEL_VND_PYTHA_PYOX = 'model/vnd.pytha.pyox'
    MODEL_VND_ROSETTE_ANNOTATED_DATA_MODEL = 'model/vnd.rosette.annotated-data-model'
    MODEL_VND_SAP_VDS = 'model/vnd.sap.vds'
    MODEL_VND_USDA = 'model/vnd.usda'
    MODEL_VND_USDZ_ZIP = 'model/vnd.usdz+zip'
    MODEL_VND_VALVE_SOURCE_COMPILED_MAP = 'model/vnd.valve.source.compiled-map'
    MODEL_VND_VTU = 'model/vnd.vtu'
    MODEL_VRML = 'model/vrml'
    MODEL_X3D_VRML = 'model/x3d-vrml'
    MODEL_X3D_FASTINFOSET = 'model/x3d+fastinfoset'
    MODEL_X3D_XML = 'model/x3d+xml'


class _MultipartEnum(StrEnum):  # pragma: no cover
    MULTIPART_ALTERNATIVE = 'multipart/alternative'
    MULTIPART_APPLEDOUBLE = 'multipart/appledouble'
    MULTIPART_BYTERANGES = 'multipart/byteranges'
    MULTIPART_DIGEST = 'multipart/digest'
    MULTIPART_ENCRYPTED = 'multipart/encrypted'
    MULTIPART_EXAMPLE = 'multipart/example'
    MULTIPART_FORM_DATA = 'multipart/form-data'
    MULTIPART_HEADER_SET = 'multipart/header-set'
    MULTIPART_MIXED = 'multipart/mixed'
    MULTIPART_MULTILINGUAL = 'multipart/multilingual'
    MULTIPART_PARALLEL = 'multipart/parallel'
    MULTIPART_RELATED = 'multipart/related'
    MULTIPART_REPORT = 'multipart/report'
    MULTIPART_SIGNED = 'multipart/signed'
    MULTIPART_VND_BINT_MED_PLUS = 'multipart/vnd.bint.med-plus'
    MULTIPART_VOICE_MESSAGE = 'multipart/voice-message'
    MULTIPART_X_MIXED_REPLACE = 'multipart/x-mixed-replace'


class _TextEnum(StrEnum):  # pragma: no cover
    TEXT_1D_INTERLEAVED_PARITYFEC = 'text/1d-interleaved-parityfec'
    TEXT_CACHE_MANIFEST = 'text/cache-manifest'
    TEXT_CALENDAR = 'text/calendar'
    TEXT_CQL = 'text/cql'
    TEXT_CQL_EXPRESSION = 'text/cql-expression'
    TEXT_CQL_IDENTIFIER = 'text/cql-identifier'
    TEXT_CSS = 'text/css'
    TEXT_CSV = 'text/csv'
    TEXT_CSV_SCHEMA = 'text/csv-schema'
    TEXT_DIRECTORY_DEPRECATED_BY_RFC6350 = 'text/directory'
    TEXT_DNS = 'text/dns'
    TEXT_ECMASCRIPT = 'text/ecmascript'  # OBSOLETED in favor of text/javascript
    TEXT_ENCAPRTP = 'text/encaprtp'
    TEXT_ENRICHED = 'text/enriched'
    TEXT_EXAMPLE = 'text/example'
    TEXT_FHIRPATH = 'text/fhirpath'
    TEXT_FLEXFEC = 'text/flexfec'
    TEXT_FWDRED = 'text/fwdred'
    TEXT_GFF3 = 'text/gff3'
    TEXT_GRAMMAR_REF_LIST = 'text/grammar-ref-list'
    TEXT_HL7V2 = 'text/hl7v2'
    TEXT_HTML = 'text/html'
    TEXT_JAVASCRIPT = 'text/javascript'
    TEXT_JCR_CND = 'text/jcr-cnd'
    TEXT_MARKDOWN = 'text/markdown'
    TEXT_MIZAR = 'text/mizar'
    TEXT_N3 = 'text/n3'
    TEXT_ORG = 'text/org'
    TEXT_PARAMETERS = 'text/parameters'
    TEXT_PARITYFEC = 'text/parityfec'
    TEXT_PLAIN = 'text/plain'
    TEXT_PROVENANCE_NOTATION = 'text/provenance-notation'
    TEXT_PRS_FALLENSTEIN_RST = 'text/prs.fallenstein.rst'
    TEXT_PRS_LINES_TAG = 'text/prs.lines.tag'
    TEXT_PRS_PROP_LOGIC = 'text/prs.prop.logic'
    TEXT_PRS_TEXI = 'text/prs.texi'
    TEXT_RAPTORFEC = 'text/raptorfec'
    TEXT_RED = 'text/RED'
    TEXT_RFC822_HEADERS = 'text/rfc822-headers'
    TEXT_RICHTEXT = 'text/richtext'
    TEXT_RTF = 'text/rtf'
    TEXT_RTP_ENC_AESCM128 = 'text/rtp-enc-aescm128'
    TEXT_RTPLOOPBACK = 'text/rtploopback'
    TEXT_RTX = 'text/rtx'
    TEXT_SGML = 'text/SGML'
    TEXT_SHACLC = 'text/shaclc'
    TEXT_SHEX = 'text/shex'
    TEXT_SPDX = 'text/spdx'
    TEXT_STRINGS = 'text/strings'
    TEXT_T140 = 'text/t140'
    TEXT_TAB_SEPARATED_VALUES = 'text/tab-separated-values'
    TEXT_TROFF = 'text/troff'
    TEXT_TURTLE = 'text/turtle'
    TEXT_ULPFEC = 'text/ulpfec'
    TEXT_URI_LIST = 'text/uri-list'
    TEXT_VCARD = 'text/vcard'
    TEXT_VND_A = 'text/vnd.a'
    TEXT_VND_ABC = 'text/vnd.abc'
    TEXT_VND_ASCII_ART = 'text/vnd.ascii-art'
    TEXT_VND_CURL = 'text/vnd.curl'
    TEXT_VND_DEBIAN_COPYRIGHT = 'text/vnd.debian.copyright'
    TEXT_VND_DMCLIENTSCRIPT = 'text/vnd.DMClientScript'
    TEXT_VND_DVB_SUBTITLE = 'text/vnd.dvb.subtitle'
    TEXT_VND_ESMERTEC_THEME_DESCRIPTOR = 'text/vnd.esmertec.theme-descriptor'
    TEXT_VND_EXCHANGEABLE = 'text/vnd.exchangeable'
    TEXT_VND_FAMILYSEARCH_GEDCOM = 'text/vnd.familysearch.gedcom'
    TEXT_VND_FICLAB_FLT = 'text/vnd.ficlab.flt'
    TEXT_VND_FLY = 'text/vnd.fly'
    TEXT_VND_FMI_FLEXSTOR = 'text/vnd.fmi.flexstor'
    TEXT_VND_GML = 'text/vnd.gml'
    TEXT_VND_GRAPHVIZ = 'text/vnd.graphviz'
    TEXT_VND_HANS = 'text/vnd.hans'
    TEXT_VND_HGL = 'text/vnd.hgl'
    TEXT_VND_IN3D_3DML = 'text/vnd.in3d.3dml'
    TEXT_VND_IN3D_SPOT = 'text/vnd.in3d.spot'
    TEXT_VND_IPTC_NEWSML = 'text/vnd.IPTC.NewsML'
    TEXT_VND_IPTC_NITF = 'text/vnd.IPTC.NITF'
    TEXT_VND_LATEX_Z = 'text/vnd.latex-z'
    TEXT_VND_MOTOROLA_REFLEX = 'text/vnd.motorola.reflex'
    TEXT_VND_MS_MEDIAPACKAGE = 'text/vnd.ms-mediapackage'
    TEXT_VND_NET2PHONE_COMMCENTER_COMMAND = 'text/vnd.net2phone.commcenter.command'
    TEXT_VND_RADISYS_MSML_BASIC_LAYOUT = 'text/vnd.radisys.msml-basic-layout'
    TEXT_VND_SENX_WARPSCRIPT = 'text/vnd.senx.warpscript'
    TEXT_VND_SI_URICATALOGUE = 'text/vnd.si.uricatalogue'  # OBSOLETED by request
    TEXT_VND_SUN_J2ME_APP_DESCRIPTOR = 'text/vnd.sun.j2me.app-descriptor'
    TEXT_VND_SOSI = 'text/vnd.sosi'
    TEXT_VND_TYPST = 'text/vnd.typst'
    TEXT_VND_TROLLTECH_LINGUIST = 'text/vnd.trolltech.linguist'
    TEXT_VND_VCF = 'text/vnd.vcf'
    TEXT_VND_WAP_SI = 'text/vnd.wap.si'
    TEXT_VND_WAP_SL = 'text/vnd.wap.sl'
    TEXT_VND_WAP_WML = 'text/vnd.wap.wml'
    TEXT_VND_WAP_WMLSCRIPT = 'text/vnd.wap.wmlscript'
    TEXT_VND_ZOO_KCL = 'text/vnd.zoo.kcl'
    TEXT_VTT = 'text/vtt'
    TEXT_WGSL = 'text/wgsl'
    TEXT_XML = 'text/xml'
    TEXT_XML_EXTERNAL_PARSED_ENTITY = 'text/xml-external-parsed-entity'


class _VideoEnum(StrEnum):  # pragma: no cover
    VIDEO_1D_INTERLEAVED_PARITYFEC = 'video/1d-interleaved-parityfec'
    VIDEO_3GPP = 'video/3gpp'
    VIDEO_3GPP2 = 'video/3gpp2'
    VIDEO_3GPP_TT = 'video/3gpp-tt'
    VIDEO_AV1 = 'video/AV1'
    VIDEO_BMPEG = 'video/BMPEG'
    VIDEO_BT656 = 'video/BT656'
    VIDEO_CELB = 'video/CelB'
    VIDEO_DV = 'video/DV'
    VIDEO_ENCAPRTP = 'video/encaprtp'
    VIDEO_EVC = 'video/evc'
    VIDEO_EXAMPLE = 'video/example'
    VIDEO_FFV1 = 'video/FFV1'
    VIDEO_FLEXFEC = 'video/flexfec'
    VIDEO_H261 = 'video/H261'
    VIDEO_H263 = 'video/H263'
    VIDEO_H263_1998 = 'video/H263-1998'
    VIDEO_H263_2000 = 'video/H263-2000'
    VIDEO_H264 = 'video/H264'
    VIDEO_H264_RCDO = 'video/H264-RCDO'
    VIDEO_H264_SVC = 'video/H264-SVC'
    VIDEO_H265 = 'video/H265'
    VIDEO_H266 = 'video/H266'
    VIDEO_ISO_SEGMENT = 'video/iso.segment'
    VIDEO_JPEG = 'video/JPEG'
    VIDEO_JPEG2000 = 'video/jpeg2000'
    VIDEO_JPEG2000_SCL = 'video/jpeg2000-scl'
    VIDEO_JXSV = 'video/jxsv'
    VIDEO_LOTTIE_JSON = 'video/lottie+json'
    VIDEO_MATROSKA = 'video/matroska'
    VIDEO_MATROSKA_3D = 'video/matroska-3d'
    VIDEO_MJ2 = 'video/mj2'
    VIDEO_MP1S = 'video/MP1S'
    VIDEO_MP2P = 'video/MP2P'
    VIDEO_MP2T = 'video/MP2T'
    VIDEO_MP4 = 'video/mp4'
    VIDEO_MP4V_ES = 'video/MP4V-ES'
    VIDEO_MPV = 'video/MPV'
    VIDEO_MPEG = 'video/mpeg'
    VIDEO_MPEG4_GENERIC = 'video/mpeg4-generic'
    VIDEO_NV = 'video/nv'
    VIDEO_OGG = 'video/ogg'
    VIDEO_PARITYFEC = 'video/parityfec'
    VIDEO_POINTER = 'video/pointer'
    VIDEO_QUICKTIME = 'video/quicktime'
    VIDEO_RAPTORFEC = 'video/raptorfec'
    VIDEO_RAW = 'video/raw'
    VIDEO_RTP_ENC_AESCM128 = 'video/rtp-enc-aescm128'
    VIDEO_RTPLOOPBACK = 'video/rtploopback'
    VIDEO_RTX = 'video/rtx'
    VIDEO_SCIP = 'video/scip'
    VIDEO_SMPTE291 = 'video/smpte291'
    VIDEO_SMPTE292M = 'video/SMPTE292M'
    VIDEO_ULPFEC = 'video/ulpfec'
    VIDEO_VC1 = 'video/vc1'
    VIDEO_VC2 = 'video/vc2'
    VIDEO_VND_BLOCKFACT_FACTV = 'video/vnd.blockfact.factv'
    VIDEO_VND_CCTV = 'video/vnd.CCTV'
    VIDEO_VND_DECE_HD = 'video/vnd.dece.hd'
    VIDEO_VND_DECE_MOBILE = 'video/vnd.dece.mobile'
    VIDEO_VND_DECE_MP4 = 'video/vnd.dece.mp4'
    VIDEO_VND_DECE_PD = 'video/vnd.dece.pd'
    VIDEO_VND_DECE_SD = 'video/vnd.dece.sd'
    VIDEO_VND_DECE_VIDEO = 'video/vnd.dece.video'
    VIDEO_VND_DIRECTV_MPEG = 'video/vnd.directv.mpeg'
    VIDEO_VND_DIRECTV_MPEG_TTS = 'video/vnd.directv.mpeg-tts'
    VIDEO_VND_DLNA_MPEG_TTS = 'video/vnd.dlna.mpeg-tts'
    VIDEO_VND_DVB_FILE = 'video/vnd.dvb.file'
    VIDEO_VND_FVT = 'video/vnd.fvt'
    VIDEO_VND_HNS_VIDEO = 'video/vnd.hns.video'
    VIDEO_VND_IPTVFORUM_1DPARITYFEC_1010 = 'video/vnd.iptvforum.1dparityfec-1010'
    VIDEO_VND_IPTVFORUM_1DPARITYFEC_2005 = 'video/vnd.iptvforum.1dparityfec-2005'
    VIDEO_VND_IPTVFORUM_2DPARITYFEC_1010 = 'video/vnd.iptvforum.2dparityfec-1010'
    VIDEO_VND_IPTVFORUM_2DPARITYFEC_2005 = 'video/vnd.iptvforum.2dparityfec-2005'
    VIDEO_VND_IPTVFORUM_TTSAVC = 'video/vnd.iptvforum.ttsavc'
    VIDEO_VND_IPTVFORUM_TTSMPEG2 = 'video/vnd.iptvforum.ttsmpeg2'
    VIDEO_VND_MOTOROLA_VIDEO = 'video/vnd.motorola.video'
    VIDEO_VND_MOTOROLA_VIDEOP = 'video/vnd.motorola.videop'
    VIDEO_VND_MPEGURL = 'video/vnd.mpegurl'
    VIDEO_VND_MS_PLAYREADY_MEDIA_PYV = 'video/vnd.ms-playready.media.pyv'
    VIDEO_VND_NOKIA_INTERLEAVED_MULTIMEDIA = 'video/vnd.nokia.interleaved-multimedia'
    VIDEO_VND_NOKIA_MP4VR = 'video/vnd.nokia.mp4vr'
    VIDEO_VND_NOKIA_VIDEOVOIP = 'video/vnd.nokia.videovoip'
    VIDEO_VND_OBJECTVIDEO = 'video/vnd.objectvideo'
    VIDEO_VND_PLANAR = 'video/vnd.planar'
    VIDEO_VND_RADGAMETTOOLS_BINK = 'video/vnd.radgamettools.bink'
    VIDEO_VND_RADGAMETTOOLS_SMACKER = 'video/vnd.radgamettools.smacker'
    VIDEO_VND_SEALED_MPEG1 = 'video/vnd.sealed.mpeg1'
    VIDEO_VND_SEALED_MPEG4 = 'video/vnd.sealed.mpeg4'
    VIDEO_VND_SEALED_SWF = 'video/vnd.sealed.swf'
    VIDEO_VND_SEALEDMEDIA_SOFTSEAL_MOV = 'video/vnd.sealedmedia.softseal.mov'
    VIDEO_VND_UVVU_MP4 = 'video/vnd.uvvu.mp4'
    VIDEO_VND_YOUTUBE_YT = 'video/vnd.youtube.yt'
    VIDEO_VND_VIVO = 'video/vnd.vivo'
    VIDEO_VP8 = 'video/VP8'
    VIDEO_VP9 = 'video/VP9'


class MimeType(str):
    """MimeType validates MIME types based on [IANA media types](https://www.iana.org/assignments/media-types/media-types.xhtml).

    ```py
    from pydantic import BaseModel

    from pydantic_extra_types.mime_types import MimeType


    class Response(BaseModel):
        content_type: MimeType


    response = Response(content_type='application/json')
    print(response)
    # > content_type='application/json'
    print(response.content_type.category)
    # > application
    ```
    """

    @classmethod
    def _validate(cls, __input_value: str, _: core_schema.ValidationInfo) -> MimeType:
        # Store the original mime_type for retrieval
        mime_info = _index_by_mime_type().get(__input_value.lower())
        if mime_info is None:
            raise PydanticCustomError('mime_type', 'Invalid MIME type')
        # Return the original case from the data
        return cls(mime_info.mime_type)

    @classmethod
    def __get_pydantic_core_schema__(
        cls, source: type[Any], handler: GetCoreSchemaHandler
    ) -> core_schema.AfterValidatorFunctionSchema:
        return core_schema.with_info_after_validator_function(
            cls._validate,
            core_schema.str_schema(),
            serialization=core_schema.to_string_ser_schema(),
        )

    @classmethod
    def __get_pydantic_json_schema__(
        cls, schema: core_schema.CoreSchema, handler: GetJsonSchemaHandler
    ) -> dict[str, Any]:
        json_schema = handler(schema)
        json_schema.update({'pattern': r'^[a-zA-Z0-9][a-zA-Z0-9!#$&\-\^_+.]+/[a-zA-Z0-9][a-zA-Z0-9!#$&\-\^_+.]+$'})
        return json_schema

    @property
    def category(self) -> str:
        """The category of the MIME type (e.g., 'application', 'audio', 'video')."""
        return _index_by_mime_type()[self.lower()].category


# Keep old class names for backward compatibility
Application = _ApplicationEnum
Audio = _AudioEnum
Font = _FontEnum
Haptics = _HapticsEnum
Image = _ImageEnum
Message = _MessageEnum
Model = _ModelEnum
Multipart = _MultipartEnum
Text = _TextEnum
Video = _VideoEnum
