/* This file is part of the KDE project
 * SPDX-FileCopyrightText: 2006 Thomas Zander <zander@kde.org>
 * SPDX-FileCopyrightText: 2004 Cyrille Berger <cberger@cberger.net>
 * SPDX-FileCopyrightText: 2006 Boudewijn Rempt <boud@valdyas.org>
 *
 * SPDX-License-Identifier: LGPL-2.0-or-later
 */

#ifndef _KO_ID_H_
#define _KO_ID_H_

#include <QDebug>
#include <QMetaType>
#include <QString>

#include <KLocalizedString>

/**
 * A KoID is a combination of a user-visible string and a string that uniquely
 * identifies a given resource across languages.
 */
class KoID
{
public:
    KoID()
        : m_id()
        , m_name()
    {
    }

    /**
     * Construct a KoID with the given id, and name, id is the untranslated
     * official name of the id, name should be translatable as it will be used
     * in the UI.
     *
     * @code
     * KoID("id", i18n("name"))
     * @endcode
     */
    explicit KoID(const QString &id, const QString &name = QString())
        : m_id(id)
        , m_name(name)
    {
    }

    /**
     * Use this constructor for static KoID. as KoID("id", ki18n("name"));
     * the name will be translated the first time it is needed. This is
     * important because static objects are constructed before translations
     * are initialized.
     */
    explicit KoID(const QString &id, const KLocalizedString &name)
        : m_id(id)
        , m_localizedString(name)
    {
    }

    KoID(const KoID &rhs)
    {
        operator=(rhs);
    }

    KoID &operator=(const KoID &rhs)
    {
        m_id = rhs.m_id;
        m_name = rhs.name();
        return *this;
    }

    QString id() const
    {
        return m_id;
    }

    QString name() const
    {
        if (m_name.isEmpty() && !m_localizedString.isEmpty()) {
            m_name = m_localizedString.toString();
        }
        return m_name;
    }

    friend inline bool operator==(const KoID &, const KoID &);
    friend inline bool operator!=(const KoID &, const KoID &);
    friend inline bool operator<(const KoID &, const KoID &);
    friend inline bool operator>(const KoID &, const KoID &);

private:
    QString m_id;
    mutable QString m_name;
    KLocalizedString m_localizedString;
};

Q_DECLARE_METATYPE(KoID)

inline bool operator==(const KoID &v1, const KoID &v2)
{
    return v1.m_id == v2.m_id;
}

inline bool operator!=(const KoID &v1, const KoID &v2)
{
    return v1.m_id != v2.m_id;
}

inline bool operator<(const KoID &v1, const KoID &v2)
{
    return v1.m_id < v2.m_id;
}

inline bool operator>(const KoID &v1, const KoID &v2)
{
    return v1.m_id > v2.m_id;
}

inline QDebug operator<<(QDebug dbg, const KoID &id)
{
    dbg.nospace() << id.name() << " (" << id.id() << " )";

    return dbg.space();
}

#endif
