use core:io;

/**
 * An HTTP server that provides routing functionality.
 *
 * The server can be used in two ways: either ignoring the 'host' field entirely, or set up to
 * support multiple hosts. If set up to handle multiple hosts, use the constructor that accepts a
 * default name.
 */
class RoutingServer extends Server {
	// Create.
	init() {
		init {
			defaultHandler = &stdDefaultHandler;
		}
	}

	// Create, supporting multiple hosts.
	init(Str defaultHostname) {
		init {
			defaultHandler = &stdDefaultHandler;
			defaultHostname = defaultHostname;
		}
	}

	// Handle requests.
	Response onRequest(Request request) : override {
		Nat first = 0;
		if (defaultHostname) {
			if (request.path.absolute) {
				root;
			} else {
				root.children.at(defaultHostname);
			}
		} else {
			if (request.path.absolute)
				first = 1;
			root;
		}

		if (found = root.find(first, request.path))
			return found.call(request);
		return defaultHandler.call(request);
	}

	// Handle errors.
	Response onServerError(Exception e) : override {
		print("Server error:\n${e}");
		Response("Internal server error", Status:Internal_Server_Error);
	}

	// Generate error pages.
	Response onError(Status code) : override {
		Response("ERROR: ${code}", code);
	}

	// Prepare responses. We don't really need to do anything extra here at the moment, it is here
	// just to remind that it exists.
	Bool prepareResponse(Request request, Response response) : override {
		super:prepareResponse(request, response);
	}


	/**
	 * Node to keep track of the routing table. It is structured like a tree.
	 */
	private class Node {
		// Function to call.
		Maybe<Fn<Response, Request>> handler;

		// Child nodes.
		Str->Node children;

		// Wildcard node (i.e. if we have a node that matches *)
		Node? wild;

		// Add a node.
		void add(Nat pos, Str[] pieces, Fn<Response, Request> add) {
			if (pos >= pieces.count) {
				if (handler.any)
					throw HttpError("A handler is already defined for ${join(pieces, "/")}.");
				handler = add;
				return;
			}

			Str piece = pieces[pos];
			Node child = if (piece == "*") {
				if (wild) {
					wild;
				} else {
					Node w;
					wild = w;
					w;
				}
			} else {
				children[piece];
			};
			child.add(pos + 1, pieces, add);
		}

		// Find a node.
		Maybe<Fn<Response, Request>> find(Nat pos, Url pieces) {
			if (pos >= pieces.count)
				return handler;

			if (child = children.at(pieces[pos])) {
				child.find(pos + 1, pieces);
			} else if (wild) {
				wild.find(pos + 1, pieces);
			} else {
				return null;
			}
		}
	}

	// Root node.
	private Node root;

	// Default handler.
	private Fn<Response, Request> defaultHandler;

	// Default hostname.
	private Str? defaultHostname;

	// Add a handler for a route. Use * for segments that should be matched as wildcards.
	// If multi-host support is enabled, the first piece should be the hostname.
	void route(Str[] pieces, Fn<Response, Request> handler) {
		root.add(0, pieces, handler);
	}

	// Using an URL.
	void route(Url url, Fn<Response, Request> handler) {
		if (url.absolute) {
			unless (url.protocol as HttpProtocol)
				throw HttpError("It is only possible to pass http urls to the 'route' function!");

			Str[] out;
			for (i, x in url) {
				if (defaultHostname.any | i > 0)
					out << x;
			}
			route(out, handler);
		} else {
			Str[] out;
			if (defaultHostname)
				out << defaultHostname;
			for (x in url)
				out << x;
			route(out, handler);
		}
	}

	// Add a default route, used if no other routes are found.
	void default(Fn<Response, Request> handler) {
		defaultHandler = handler;
	}

	// The standard default handler.
	private Response stdDefaultHandler(Request request) : static {
		// TODO: Should we just call 'onError'?
		Response("The path ${request.path} does not exist on this server.", Status:Not_Found);
	}

}
