/**
 * Converts `TNodeType` into human readable text.
 * Make sure this matches with `TNodeType`
 */
export function toTNodeTypeAsString(tNodeType) {
    let text = '';
    (tNodeType & 1 /* TNodeType.Text */) && (text += '|Text');
    (tNodeType & 2 /* TNodeType.Element */) && (text += '|Element');
    (tNodeType & 4 /* TNodeType.Container */) && (text += '|Container');
    (tNodeType & 8 /* TNodeType.ElementContainer */) && (text += '|ElementContainer');
    (tNodeType & 16 /* TNodeType.Projection */) && (text += '|Projection');
    (tNodeType & 32 /* TNodeType.Icu */) && (text += '|IcuContainer');
    (tNodeType & 64 /* TNodeType.Placeholder */) && (text += '|Placeholder');
    return text.length > 0 ? text.substring(1) : text;
}
// Note: This hack is necessary so we don't erroneously get a circular dependency
// failure based on types.
export const unusedValueExportToPlacateAjd = 1;
/**
 * Returns `true` if the `TNode` has a directive which has `@Input()` for `class` binding.
 *
 * ```
 * <div my-dir [class]="exp"></div>
 * ```
 * and
 * ```
 * @Directive({
 * })
 * class MyDirective {
 *   @Input()
 *   class: string;
 * }
 * ```
 *
 * In the above case it is necessary to write the reconciled styling information into the
 * directive's input.
 *
 * @param tNode
 */
export function hasClassInput(tNode) {
    return (tNode.flags & 16 /* TNodeFlags.hasClassInput */) !== 0;
}
/**
 * Returns `true` if the `TNode` has a directive which has `@Input()` for `style` binding.
 *
 * ```
 * <div my-dir [style]="exp"></div>
 * ```
 * and
 * ```
 * @Directive({
 * })
 * class MyDirective {
 *   @Input()
 *   class: string;
 * }
 * ```
 *
 * In the above case it is necessary to write the reconciled styling information into the
 * directive's input.
 *
 * @param tNode
 */
export function hasStyleInput(tNode) {
    return (tNode.flags & 32 /* TNodeFlags.hasStyleInput */) !== 0;
}
//# sourceMappingURL=data:application/json;base64,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