/**
 * \file pappsomspp/processing/cbor/jsonstreamwriter.cpp
 * \date 19/07/2025
 * \author Olivier Langella
 * \brief PAPPSO JSON stream writer
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

#include "jsonstreamwriter.h"
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonArray>


pappso::cbor::JsonStreamWriter::JsonStreamWriter(QIODevice *device) : QTextStream(device)
{
}

pappso::cbor::JsonStreamWriter::~JsonStreamWriter()
{
}

void
pappso::cbor::JsonStreamWriter::endArray()
{
  *this << "]" << Qt::endl;
  m_isStart     = false;
  m_nextIsValue = false;
}

void
pappso::cbor::JsonStreamWriter::endMap()
{
  *this << "}" << Qt::endl;
  m_isStart     = false;
  m_nextIsValue = false;
}

void
pappso::cbor::JsonStreamWriter::startArray()
{
  if(!m_nextIsValue)
    comma();
  *this << Qt::endl << "[";
  m_isStart = true;
}
void
pappso::cbor::JsonStreamWriter::startMap()
{
  if(!m_nextIsValue)
    comma();
  *this << Qt::endl << "{";
  m_isStart = true;
}

void
pappso::cbor::JsonStreamWriter::writeCborMap(const QCborMap &cbor_map)
{
  QJsonDocument doc;
  doc.setObject(QCborValue(cbor_map).toJsonValue().toObject());

  *this << doc.toJson();
}

void
pappso::cbor::JsonStreamWriter::writeCborArray(const QCborArray &cbor_array)
{
  QJsonDocument doc;
  doc.setArray(QCborValue(cbor_array).toJsonValue().toArray());

  *this << doc.toJson();
}


void
pappso::cbor::JsonStreamWriter::appendKey(const QString &key)
{
  if(!m_isStart)
    comma();
  *this << "\"" << key << "\":";
  m_isStart     = false;
  m_nextIsValue = true;
}

void
pappso::cbor::JsonStreamWriter::appendValue(const QString &value)
{
  *this << "\"" << value << "\"";
  m_nextIsValue = false;
}

void
pappso::cbor::JsonStreamWriter::appendValue(std::size_t integer_size_value)
{
  *this << integer_size_value;
  m_nextIsValue = false;
}


void
pappso::cbor::JsonStreamWriter::writeArray(QStringList &str_list)
{
  *this << "[\"" << str_list.join("\", \"") << "\"]" << Qt::endl;
}

void
pappso::cbor::JsonStreamWriter::comma()
{
  *this << "," << Qt::endl;
  m_isStart = false;
}
