/*
 * Copyright 2022-2023 Soren Stoutner <soren@stoutner.com>.
 *
 * This file is part of Privacy Browser PC <https://www.stoutner.com/privacy-browser-pc>.
 *
 * Privacy Browser PC is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Privacy Browser PC is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Privacy Browser PC.  If not, see <http://www.gnu.org/licenses/>.
 // */

// Application headers.
#include "DomainSettingsDialog.h"
#include "Settings.h"
#include "ui_DomainSettingsDialog.h"
#include "databases/DomainsDatabase.h"

// Qt toolkit headers.
#include <QInputDialog>
#include <QMessageBox>
#include <QPushButton>
#include <QTimer>

// Define the public static int constants.
const int DomainSettingsDialog::SHOW_ALL_DOMAINS = 0;
const int DomainSettingsDialog::ADD_DOMAIN = 1;
const int DomainSettingsDialog::EDIT_DOMAIN = 2;

// Construct the class.
DomainSettingsDialog::DomainSettingsDialog(const int &startType, const QString &domainName) : QDialog(nullptr)
{
    // Set the window title.
    setWindowTitle(i18nc("The domain settings dialog window title", "Domain Settings"));

    // Set the window modality.
    setWindowModality(Qt::WindowModality::ApplicationModal);

    // Instantiate the user agent helper.
    userAgentHelperPointer = new UserAgentHelper();

    // Instantiate the domain settings dialog UI.
    Ui::DomainSettingsDialog domainSettingsDialogUi;

    // Setup the UI.
    domainSettingsDialogUi.setupUi(this);

    // Get handles for the widgets.
    domainsListViewPointer = domainSettingsDialogUi.domainsListView;
    domainSettingsWidgetPointer = domainSettingsDialogUi.domainSettingsWidget;
    domainNameLineEditPointer = domainSettingsDialogUi.domainNameLineEdit;
    javaScriptWidgetPointer = domainSettingsDialogUi.javaScriptWidget;
    javaScriptComboBoxPointer = domainSettingsDialogUi.javaScriptComboBox;
    javaScriptLabelPointer = domainSettingsDialogUi.javaScriptLabel;
    localStorageWidgetPointer = domainSettingsDialogUi.localStorageWidget;
    localStorageComboBoxPointer = domainSettingsDialogUi.localStorageComboBox;
    localStorageLabelPointer = domainSettingsDialogUi.localStorageLabel;
    domStorageWidgetPointer = domainSettingsDialogUi.domStorageWidget;
    domStorageComboBoxPointer = domainSettingsDialogUi.domStorageComboBox;
    domStorageLabelPointer = domainSettingsDialogUi.domStorageLabel;
    userAgentWidgetPointer = domainSettingsDialogUi.userAgentWidget;
    userAgentComboBoxPointer = domainSettingsDialogUi.userAgentComboBox;
    userAgentLabelPointer = domainSettingsDialogUi.userAgentLabel;
    zoomFactorWidgetPointer = domainSettingsDialogUi.zoomFactorWidget;
    zoomFactorComboBoxPointer = domainSettingsDialogUi.zoomFactorComboBox;
    customZoomFactorSpinBoxPointer = domainSettingsDialogUi.customZoomFactorSpinBox;
    QPushButton *addDomainButtonPointer = domainSettingsDialogUi.addDomainButton;
    deleteDomainButtonPointer = domainSettingsDialogUi.deleteDomainButton;
    QDialogButtonBox *dialogButtonBoxPointer = domainSettingsDialogUi.dialogButtonBox;
    applyButtonPointer = dialogButtonBoxPointer->button(QDialogButtonBox::StandardButton::Apply);
    resetButtonPointer = dialogButtonBoxPointer->button(QDialogButtonBox::StandardButton::Reset);

    // Create a table model.
    domainsTableModelPointer = new QSqlTableModel(nullptr, QSqlDatabase::database(DomainsDatabase::CONNECTION_NAME));

    // Set the table for the model.
    domainsTableModelPointer->setTable(DomainsDatabase::DOMAINS_TABLE);

    // Set the edit strategy to be manual.
    domainsTableModelPointer->setEditStrategy(QSqlTableModel::EditStrategy::OnManualSubmit);

    // Sort the output alphabetically.
    domainsTableModelPointer->setSort(1, Qt::SortOrder::AscendingOrder);

    // Set the model for the list view.
    domainsListViewPointer->setModel(domainsTableModelPointer);

    // Set the visible column to be the domain name.
    domainsListViewPointer->setModelColumn(1);

    // Get the domains selection model pointer.
    domainsSelectionModelPointer = domainsListViewPointer->selectionModel();

    // Disable editing of the list view.
    domainsListViewPointer->setEditTriggers(QAbstractItemView::NoEditTriggers);

    // Read the data from the database and apply it to the table model.
    domainsTableModelPointer->select();

    // Get the default palette.
    defaultPalette = javaScriptWidgetPointer->palette();

    // Populate the highlighted palette.
    highlightedPalette = defaultPalette;

    // Get the default highlight color.
    QColor highlightColor = defaultPalette.color(QPalette::Highlight);

    // Set the highlight color to be partially transparent.
    highlightColor.setAlpha(64);

    // Set highlighted background color.
    highlightedPalette.setColor(QPalette::Window, highlightColor);

    // Setup the dialog according to the start type.
    switch (startType)
    {
        case SHOW_ALL_DOMAINS:
        {
            // Select the first entry in the list view.
            domainsListViewPointer->setCurrentIndex(domainsTableModelPointer->index(0, domainsTableModelPointer->fieldIndex(DomainsDatabase::DOMAIN_NAME)));

            // Populate the domain settings.
            domainSelected(domainsSelectionModelPointer->currentIndex());

            break;
        }

        case ADD_DOMAIN:
        {
            // Add the new domain.
            addDomain(domainName);

            // Emit the domain settings updated signal after 100 milliseconds.  This is necessary because the browser window takes time to process the connect command to receive the signal.
            QTimer::singleShot(100, [this] () { emit domainSettingsUpdated();});

            break;
        }

        case EDIT_DOMAIN:
        {
            // Find the index for the new domain.  `1` returns the first match.
            QModelIndexList newDomainIndex = domainsTableModelPointer->match(domainsTableModelPointer->index(0, domainsTableModelPointer->fieldIndex(DomainsDatabase::DOMAIN_NAME)),
                                                                             Qt::DisplayRole, domainName, 1, Qt::MatchWrap);

            // Move to the new domain.
            domainsListViewPointer->setCurrentIndex(newDomainIndex[0]);

            // Populate the domain settings.
            domainSelected(domainsSelectionModelPointer->currentIndex());

            break;
        }
    }

    // Handle clicks on the domains.
    connect(domainsListViewPointer, SIGNAL(activated(QModelIndex)), this, SLOT(domainSelected(QModelIndex)));

    // Process changes to the domain settings.
    connect(domainNameLineEditPointer, SIGNAL(textEdited(QString)), this, SLOT(domainNameChanged(QString)));
    connect(javaScriptComboBoxPointer, SIGNAL(currentIndexChanged(int)), this, SLOT(javaScriptChanged(int)));
    connect(localStorageComboBoxPointer, SIGNAL(currentIndexChanged(int)), this, SLOT(localStorageChanged(int)));
    connect(domStorageComboBoxPointer, SIGNAL(currentIndexChanged(int)), this, SLOT(domStorageChanged(int)));
    connect(userAgentComboBoxPointer, SIGNAL(currentTextChanged(QString)), this, SLOT(userAgentChanged(QString)));
    connect(zoomFactorComboBoxPointer, SIGNAL(currentIndexChanged(int)), this, SLOT(zoomFactorComboBoxChanged(int)));
    connect(customZoomFactorSpinBoxPointer, SIGNAL(valueChanged(double)), this, SLOT(customZoomFactorChanged(double)));

    // Connect the buttons.
    connect(addDomainButtonPointer, SIGNAL(clicked()), this, SLOT(showAddMessageBox()));
    connect(deleteDomainButtonPointer, SIGNAL(clicked()), this, SLOT(showDeleteMessageBox()));
    connect(resetButtonPointer, SIGNAL(clicked()), this, SLOT(reset()));
    connect(dialogButtonBoxPointer, SIGNAL(accepted()), this, SLOT(ok()));
    connect(applyButtonPointer, SIGNAL(clicked()), this, SLOT(apply()));
    connect(dialogButtonBoxPointer, SIGNAL(rejected()), this, SLOT(cancel()));

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::addDomain(const QString &domainName) const
{
    // Create a new domain record.
    QSqlRecord newDomainRecord = QSqlDatabase::database(DomainsDatabase::CONNECTION_NAME).record(DomainsDatabase::DOMAINS_TABLE);

    // Set the values for the new domain.
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::DOMAIN_NAME), domainName);
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::JAVASCRIPT), DomainsDatabase::SYSTEM_DEFAULT);
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::LOCAL_STORAGE), DomainsDatabase::SYSTEM_DEFAULT);
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::DOM_STORAGE), DomainsDatabase::SYSTEM_DEFAULT);
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::USER_AGENT), UserAgentHelper::SYSTEM_DEFAULT_DATABASE);
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::ZOOM_FACTOR), DomainsDatabase::SYSTEM_DEFAULT);
    newDomainRecord.setValue(domainsTableModelPointer->fieldIndex(DomainsDatabase::CUSTOM_ZOOM_FACTOR), 1.0);

    // Insert the new domain.  `-1` appends it to the end.
    domainsTableModelPointer->insertRecord(-1, newDomainRecord);

    // Submit all pending changes.
    domainsTableModelPointer->submitAll();

    // Find the index for the new domain.  `-1` allows for multiple entries to be returned.
    QModelIndexList newDomainIndex = domainsTableModelPointer->match(domainsTableModelPointer->index(0, domainsTableModelPointer->fieldIndex(DomainsDatabase::DOMAIN_NAME)),
                                                                     Qt::DisplayRole, domainName, -1, Qt::MatchWrap);

    // Move to the new domain.  If there are multiple domains with the same name, the new one should be the last in the list.
    domainsListViewPointer->setCurrentIndex(newDomainIndex[newDomainIndex.size() - 1]);

    // Populate the domain settings.
    domainSelected(domainsSelectionModelPointer->currentIndex());

    // Update the UI.
    updateUi();

    // Emit the domain settings updated signal.
    emit domainSettingsUpdated();
}

void DomainSettingsDialog::apply() const
{
    // Get the current index.
    QModelIndex currentIndex = domainsListViewPointer->currentIndex();

    // Get the ID of the current index row.
    QVariant currentId = currentIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::ID)).data();

    // Submit all pending changes.
    domainsTableModelPointer->submitAll();

    // Find the new index for the selected id.  The `1` keeps searching after the first match.
    QModelIndexList newIndexList = domainsTableModelPointer->match(currentIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::ID)), Qt::DisplayRole, currentId,
                                                                   1, Qt::MatchWrap);

    // Select the new index.
    domainsListViewPointer->setCurrentIndex(newIndexList[0].siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::DOMAIN_NAME)));

    // Update the UI.
    updateUi();

    // Emit the domain settings updated signal.
    emit domainSettingsUpdated();
}

void DomainSettingsDialog::cancel()
{
    // Revert all pending changes.
    domainsTableModelPointer->revertAll();

    // Close the dialog.
    reject();
}

void DomainSettingsDialog::customZoomFactorChanged(const double &newValue) const
{
    // Update the domains table model.
    domainsTableModelPointer->setData(domainsSelectionModelPointer->currentIndex().siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::CUSTOM_ZOOM_FACTOR)), newValue);

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::domStorageChanged(const int &newIndex) const
{
    // Update the domains table model.
    domainsTableModelPointer->setData(domainsSelectionModelPointer->currentIndex().siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::DOM_STORAGE)), newIndex);

    // Populate the DOM storage label.
    populateDomStorageLabel();

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::domainNameChanged(const QString &updatedDomainName) const
{
    // Update the domains table model.
    domainsTableModelPointer->setData(domainsSelectionModelPointer->currentIndex(), updatedDomainName);

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::domainSelected(const QModelIndex &modelIndex) const
{
    // Populate the domain name line edit pointer.
    domainNameLineEditPointer->setText(modelIndex.data().toString());

    // Populate the JavaScript combo box.
    javaScriptComboBoxPointer->setCurrentIndex(modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::JAVASCRIPT)).data().toInt());

    // Populate the local storage combo box.
    localStorageComboBoxPointer->setCurrentIndex(modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::LOCAL_STORAGE)).data().toInt());

    // Populate the DOM storage combo box.
    domStorageComboBoxPointer->setCurrentIndex(modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::DOM_STORAGE)).data().toInt());

    // Get the user agent string.
    QString userAgent = modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::USER_AGENT)).data().toString();

    // Get the user agent index.
    int userAgentIndex = UserAgentHelper::getDomainSettingsUserAgentIndex(userAgent);

    // Set the user agent combo box index.
    userAgentComboBoxPointer->setCurrentIndex(userAgentIndex);

    // Set the custom user agent if specified.
    if (userAgentIndex == -1) userAgentComboBoxPointer->setCurrentText(userAgent);

    // Get the zoom factor combo box index.
    int zoomFactorComboBoxIndex = modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::ZOOM_FACTOR)).data().toInt();

    // Populate the zoom factor combo box.
    zoomFactorComboBoxPointer->setCurrentIndex(zoomFactorComboBoxIndex);

    // Populate the custom zoom factor spin box according to the zoom factor combo box.
    if (zoomFactorComboBoxIndex == 0)  // System default zoom factor is selected.
    {
        // Display the default zoom factor.
        customZoomFactorSpinBoxPointer->setValue(Settings::zoomFactor());

        // Use the default palette.
        zoomFactorWidgetPointer->setPalette(defaultPalette);
    }
    else  // Custom zoom factor is selected.
    {
        // Display the custom zoom factor from the domain settings.
        customZoomFactorSpinBoxPointer->setValue(modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::CUSTOM_ZOOM_FACTOR)).data().toDouble());

        // Use the highlighted palette.
        zoomFactorWidgetPointer->setPalette(highlightedPalette);
    }

    // Set the initial status of the custom zoom factor spin box.
    customZoomFactorSpinBoxPointer->setEnabled(zoomFactorComboBoxIndex);

    // Populate the labels.
    populateJavaScriptLabel();
    populateLocalStorageLabel();
    populateDomStorageLabel();
    populateUserAgentLabel(userAgentComboBoxPointer->currentText());

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::javaScriptChanged(const int &newIndex) const
{
    // Update the domains table model.
    domainsTableModelPointer->setData(domainsSelectionModelPointer->currentIndex().siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::JAVASCRIPT)), newIndex);

    // Populate the JavaScript label.
    populateJavaScriptLabel();

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::localStorageChanged(const int &newIndex) const
{
    // Update the domains table model.
    domainsTableModelPointer->setData(domainsSelectionModelPointer->currentIndex().siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::LOCAL_STORAGE)), newIndex);

    // Populate the local storage label.
    populateLocalStorageLabel();

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::ok()
{
    // Submit all pending changes.
    domainsTableModelPointer->submitAll();

    // Emit the domain settings updated signal.
    emit domainSettingsUpdated();

    // Close the dialog.
    accept();
}

void DomainSettingsDialog::populateDomStorageLabel() const
{
    // Populate the label according to the currently selected index.
    switch (domStorageComboBoxPointer->currentIndex())
    {
        case (DomainsDatabase::SYSTEM_DEFAULT):
        {
            // Set the text according to the system default.
            if (Settings::domStorageEnabled())
                domStorageLabelPointer->setText(i18nc("Domain settings DOM storage label.", "DOM storage enabled"));
            else
                domStorageLabelPointer->setText(i18nc("Domain settings DOM storage label.", "DOM storage disabled"));

            // Reset the palette.
            domStorageWidgetPointer->setPalette(defaultPalette);

            break;
        }

        case (DomainsDatabase::ENABLED):
        {
            // Set the enabled text in bold.
            domStorageLabelPointer->setText(i18nc("Domain settings DOM storage label.  The <b> tags should be retained.", "<b>DOM storage enabled</b>"));

            // Set the palette.
            domStorageWidgetPointer->setPalette(highlightedPalette);

            break;
        }

        case (DomainsDatabase::DISABLED):
        {
            // Set the disabled text in bold.
            domStorageLabelPointer->setText(i18nc("Domain settings DOM storage label.  The <b> tags should be retained.", "<b>DOM storage disabled</b>"));

            // Set the palette.
            domStorageWidgetPointer->setPalette(highlightedPalette);

            break;
        }
    }
}

void DomainSettingsDialog::populateJavaScriptLabel() const
{
    // Populate the label according to the currently selected index.
    switch (javaScriptComboBoxPointer->currentIndex())
    {
        case (DomainsDatabase::SYSTEM_DEFAULT):
        {
            // Set the text according to the system default.
            if (Settings::javaScriptEnabled())
                javaScriptLabelPointer->setText(i18nc("Domain settings JavaScript label.", "JavaScript enabled"));
            else
                javaScriptLabelPointer->setText(i18nc("Domain settings JavaScript label.", "JavaScript disabled"));

            // Reset the palette.
            javaScriptWidgetPointer->setPalette(defaultPalette);

            break;
        }

        case (DomainsDatabase::ENABLED):
        {
            // Set the enabled text in bold.
            javaScriptLabelPointer->setText(i18nc("Domain settings JavaScript label.  The <b> tags should be retained.", "<b>JavaScript enabled</b>"));

            // Set the palette.
            javaScriptWidgetPointer->setPalette(highlightedPalette);

            break;
        }

        case (DomainsDatabase::DISABLED):
        {
            // Set the disabled text in bold.
            javaScriptLabelPointer->setText(i18nc("Domain settings JavaScript label.  The <b> tags should be retained.", "<b>JavaScript disabled</b>"));

            // Set the palette.
            javaScriptWidgetPointer->setPalette(highlightedPalette);

            break;
        }
    }
}

void DomainSettingsDialog::populateLocalStorageLabel() const
{
    // Populate the label according to the currently selected index.
    switch (localStorageComboBoxPointer->currentIndex())
    {
        case (DomainsDatabase::SYSTEM_DEFAULT):
        {
            // Set the text according to the system default.
            if (Settings::localStorageEnabled())
                localStorageLabelPointer->setText(i18nc("Domain settings local storage label.", "Local storage enabled"));
            else
                localStorageLabelPointer->setText(i18nc("Domain settings local storage label.", "Local storage disabled"));

            // Reset the palette.
            localStorageWidgetPointer->setPalette(defaultPalette);

            break;
        }

        case (DomainsDatabase::ENABLED):
        {
            // Set the enabled text in bold.
            localStorageLabelPointer->setText(i18nc("Domain settings local storage label.  The <b> tabs should be retained.", "<b>Local storage enabled</b>"));

            // Set the palette.
            localStorageWidgetPointer->setPalette(highlightedPalette);

            break;
        }

        case (DomainsDatabase::DISABLED):
        {
            // Set the disabled text in bold.
            localStorageLabelPointer->setText(i18nc("Domain settings local storage label.  The <b> tags should be retained.", "<b>Local storage disabled</b>"));

            // Set the palette.
            localStorageWidgetPointer->setPalette(highlightedPalette);

            break;
        }
    }
}


void DomainSettingsDialog::populateUserAgentLabel(const QString &userAgentName) const
{
    // Populate the label according to the type.
    if (userAgentName == userAgentHelperPointer->SYSTEM_DEFAULT_TRANSLATED)
    {
        // Display the system default user agent name.
        userAgentLabelPointer->setText(userAgentHelperPointer->getTranslatedUserAgentNameFromDatabaseName(Settings::userAgent()));

        // Reset the palette.
        userAgentWidgetPointer->setPalette(defaultPalette);
    }
    else
    {
        // Display the user agent name in bold.
        userAgentLabelPointer->setText("<strong>" + userAgentName + "</strong>");

        // Set the palette.
        userAgentWidgetPointer->setPalette(highlightedPalette);
    }
}

void DomainSettingsDialog::reset() const
{
    // Cancel all pending changes.
    domainsTableModelPointer->revertAll();

    // Repopulate the domain settings.
    domainSelected(domainsListViewPointer->currentIndex());

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::showAddMessageBox()
{
    // Create an OK flag.
    bool okClicked;

    // Display a dialog to request the new domain name from the user.
    QString newDomainName = QInputDialog::getText(this, i18nc("Add domain dialog title", "Add Domain"),
                                                  i18nc("Add domain message.  The \n\n are newline codes that should be retained",
                                                        "Add a new domain.  Doing so will also save any pending changes that have been made to other domains.\n\n"
                                                        "*. may be prepended to a domain to include all subdomains (eg. *.stoutner.com)."),
                                                  QLineEdit::Normal, QString(), &okClicked);

    // Add the new domain if the user clicked OK.
    if (okClicked) addDomain(newDomainName);
}

void DomainSettingsDialog::showDeleteMessageBox() const
{
    // Instantiate a delete dialog message box.
    QMessageBox deleteDialogMessageBox;

    // Set the icon.
    deleteDialogMessageBox.setIcon(QMessageBox::Warning);

    // Set the window title.
    deleteDialogMessageBox.setWindowTitle(i18nc("Delete domain dialog title", "Delete Domain"));

    // Set the text.
    deleteDialogMessageBox.setText(i18nc("Delete domain dialog main message", "Delete the current domain?"));

    // Set the informative text.
    deleteDialogMessageBox.setInformativeText(i18nc("Delete domain dialog secondary message", "Doing so will also save any pending changes that have been made to other domains."));

    // Set the standard buttons.
    deleteDialogMessageBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);

    // Set the default button.
    deleteDialogMessageBox.setDefaultButton(QMessageBox::No);

    // Display the dialog and capture the return value.
    int returnValue = deleteDialogMessageBox.exec();

    // Delete the domain if instructed.
    if (returnValue == QMessageBox::Yes)
    {
        // Get the current index.
        QModelIndex currentIndex = domainsListViewPointer->currentIndex();

        // Delete the current row.
        domainsTableModelPointer->removeRow(domainsSelectionModelPointer->currentIndex().row());

        // Submit all pending changes.
        domainsTableModelPointer->submitAll();

        // Select the row next to the deleted item if one exists.
        if (domainsTableModelPointer->rowCount() > 0)
        {
            // Check the row of the deleted item.
            if (currentIndex.row() == 0)  // The first row was deleted.
            {
                // Reselect the current index.
                domainsListViewPointer->setCurrentIndex(currentIndex);
            }
            else  // A subsequent row was deleted.
            {
                // Select the crow above the deleted itemm.
                domainsListViewPointer->setCurrentIndex(currentIndex.siblingAtRow(currentIndex.row() - 1));
            }

            // Populate the domain settings.
            domainSelected(domainsListViewPointer->currentIndex());
        }

        // Update the Ui.
        updateUi();

        // Emit the domain settings updated signal.
        emit domainSettingsUpdated();
    }
}

void DomainSettingsDialog::updateUi() const
{
    // Update the delete button status.
    deleteDomainButtonPointer->setEnabled(domainsSelectionModelPointer->hasSelection());

    // Update the reset button status.
    resetButtonPointer->setEnabled(domainsTableModelPointer->isDirty());

    // Update the apply button status.
    applyButtonPointer->setEnabled(domainsTableModelPointer->isDirty());

    // Display the domain settings if there is at least one domain.
    domainSettingsWidgetPointer->setVisible(domainsTableModelPointer->rowCount() > 0);
}

void DomainSettingsDialog::userAgentChanged(const QString &updatedUserAgent) const
{
    // Update the domains table model.
    domainsTableModelPointer->setData(domainsSelectionModelPointer->currentIndex().siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::USER_AGENT)),
                                      userAgentHelperPointer->getDatabaseUserAgentNameFromTranslatedName(updatedUserAgent));

    // Populate the user agent label.
    populateUserAgentLabel(updatedUserAgent);

    // Update the UI.
    updateUi();
}

void DomainSettingsDialog::zoomFactorComboBoxChanged(const int &newIndex) const
{
    // Get the current model index.
    QModelIndex modelIndex = domainsSelectionModelPointer->currentIndex();

    // Update the domains table model.
    domainsTableModelPointer->setData(modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::ZOOM_FACTOR)), newIndex);

    // Populate the custom zoom factor spin box according to the zoom factor combo box.
    if (newIndex == 0)  // System default zoom factor is selected.
    {
        // Display the default zoom factor.
        customZoomFactorSpinBoxPointer->setValue(Settings::zoomFactor());

        // Reset the palette.
        zoomFactorWidgetPointer->setPalette(defaultPalette);
    }
    else  // Custom zoom factor is selected.
    {
        // Display the custom zoom factor from the domain settings.
        customZoomFactorSpinBoxPointer->setValue(modelIndex.siblingAtColumn(domainsTableModelPointer->fieldIndex(DomainsDatabase::CUSTOM_ZOOM_FACTOR)).data().toDouble());

        // Use the highlighted palette.
        zoomFactorWidgetPointer->setPalette(highlightedPalette);
    }

    // Update the status of the custom zoom factor spin box.
    customZoomFactorSpinBoxPointer->setEnabled(newIndex);

    // Update the UI.
    updateUi();
}
