/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.cache;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.time.Duration;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.OpenHAB;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@NonNullByDefault
public class ByteArrayFileCache {
    private final Logger logger = LoggerFactory.getLogger(ByteArrayFileCache.class);
    static final String CACHE_FOLDER_NAME = "cache";
    private static final char EXTENSION_SEPARATOR = '.';
    private final File cacheFolder;
    private final Duration expiry;
    private final Map<String, File> filesInCache = new ConcurrentHashMap<String, File>();

    public ByteArrayFileCache(String servicePID) {
        this(servicePID, Duration.ZERO);
    }

    public ByteArrayFileCache(String servicePID, Duration expiry) {
        this.cacheFolder = new File(new File(OpenHAB.getUserDataFolder(), CACHE_FOLDER_NAME), servicePID);
        if (!this.cacheFolder.exists()) {
            this.logger.debug("Creating cache folder '{}'", (Object)this.cacheFolder.getAbsolutePath());
            this.cacheFolder.mkdirs();
        }
        this.logger.debug("Using cache folder '{}'", (Object)this.cacheFolder.getAbsolutePath());
        if (expiry.isNegative()) {
            throw new IllegalArgumentException("Cache expiration time must be greater than or equal to 0");
        }
        this.expiry = expiry;
    }

    public void put(String key, byte[] content) {
        this.writeFile(this.getUniqueFile(key), content);
    }

    public void putIfAbsent(String key, byte[] content) {
        File fileInCache = this.getUniqueFile(key);
        if (fileInCache.exists()) {
            this.logger.debug("File '{}' present in cache", (Object)fileInCache.getName());
            fileInCache.setLastModified(System.currentTimeMillis());
        } else {
            this.writeFile(fileInCache, content);
        }
    }

    public byte[] putIfAbsentAndGet(String key, byte[] content) throws IOException {
        this.putIfAbsent(key, content);
        return this.get(key);
    }

    private void writeFile(File fileInCache, byte[] content) {
        this.logger.debug("Caching file '{}'", (Object)fileInCache.getName());
        try {
            Files.write(fileInCache.toPath(), content, new OpenOption[0]);
        }
        catch (IOException e) {
            this.logger.warn("Could not write file '{}'\u00a0to cache", (Object)fileInCache.getName(), (Object)e);
        }
    }

    public boolean containsKey(String key) {
        return this.getUniqueFile(key).exists();
    }

    public void remove(String key) {
        this.deleteFile(this.getUniqueFile(key));
    }

    private void deleteFile(File fileInCache) {
        if (fileInCache.exists()) {
            this.logger.debug("Deleting file '{}' from cache", (Object)fileInCache.getName());
            fileInCache.delete();
        } else {
            this.logger.debug("File '{}' not found in cache", (Object)fileInCache.getName());
        }
    }

    public void clear() {
        File[] filesInCache = this.cacheFolder.listFiles();
        if (filesInCache != null && filesInCache.length > 0) {
            this.logger.debug("Deleting all files from cache");
            Arrays.stream(filesInCache).forEach(File::delete);
        }
    }

    public void clearExpired() {
        if (this.expiry.isZero() || this.expiry.isNegative()) {
            return;
        }
        File[] filesInCache = this.cacheFolder.listFiles();
        if (filesInCache != null && filesInCache.length > 0) {
            this.logger.debug("Deleting expired files from cache");
            Arrays.stream(filesInCache).filter(this::isExpired).forEach(File::delete);
        }
    }

    private boolean isExpired(File fileInCache) {
        long expiryInMillis = this.expiry.toMillis();
        return 0L < expiryInMillis && expiryInMillis < System.currentTimeMillis() - fileInCache.lastModified();
    }

    public byte[] get(String key) throws FileNotFoundException, IOException {
        return this.readFile(this.getUniqueFile(key));
    }

    private byte[] readFile(File fileInCache) throws FileNotFoundException, IOException {
        if (fileInCache.exists()) {
            this.logger.debug("Reading file '{}' from cache", (Object)fileInCache.getName());
            fileInCache.setLastModified(System.currentTimeMillis());
            try {
                return Files.readAllBytes(fileInCache.toPath());
            }
            catch (IOException e) {
                this.logger.warn("Could not read file '{}'\u00a0from cache", (Object)fileInCache.getName(), (Object)e);
                throw new IOException(String.format("Could not read file '%s'\u00a0from cache", fileInCache.getName()));
            }
        }
        this.logger.debug("File '{}' not found in cache", (Object)fileInCache.getName());
        throw new FileNotFoundException(String.format("File '%s' not found in cache", fileInCache.getName()));
    }

    File getUniqueFile(String key) {
        String uniqueFileName = this.getUniqueFileName(key);
        File fileInCache = this.filesInCache.get(uniqueFileName);
        if (fileInCache == null) {
            String fileExtension = this.getFileExtension(key);
            fileInCache = new File(this.cacheFolder, uniqueFileName + (String)(fileExtension == null ? "" : "." + fileExtension));
            this.filesInCache.put(uniqueFileName, fileInCache);
        }
        return fileInCache;
    }

    @Nullable String getFileExtension(String fileName) {
        String strippedFileName = fileName.replaceFirst("\\?.*$", "").replace("/", File.separator);
        int extensionPos = strippedFileName.lastIndexOf(46);
        int lastSeparatorPos = strippedFileName.lastIndexOf(File.separator);
        return lastSeparatorPos > extensionPos ? null : strippedFileName.substring(extensionPos + 1);
    }

    String getUniqueFileName(String key) {
        return String.format("%032x", BigInteger.valueOf(key.hashCode()));
    }
}

