/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2005-07-01
 * Description : a widget to draw a control pannel image tool.
 * 
 * Copyright (C) 2005-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqframe.h>
#include <tqvgroupbox.h>
#include <tqlabel.h>
#include <tqpixmap.h>
#include <tqtooltip.h>
#include <tqwhatsthis.h>
#include <tqtimer.h>
#include <tqhbuttongroup.h> 
#include <tqpushbutton.h>
#include <tqlayout.h>
#include <tqpixmap.h>

// KDE includes.

#include <kdialog.h>
#include <tdelocale.h>
#include <kcursor.h>
#include <kprogress.h>
#include <tdeapplication.h>
#include <kiconloader.h>
#include <tdeconfig.h>
#include <tdestandarddirs.h>
#include <kseparator.h>

// Local includes.

#include "ddebug.h"
#include "sidebar.h"
#include "statuszoombar.h"
#include "thumbnailsize.h"
#include "imageregionwidget.h"
#include "imagepaniconwidget.h"
#include "imagepannelwidget.h"
#include "imagepannelwidget.moc"

namespace Digikam
{

class ImagePannelWidgetPriv
{
public:

    ImagePannelWidgetPriv()
    {
        imageRegionWidget  = 0;
        imagePanIconWidget = 0;
        mainLayout         = 0;
        separateView       = 0;
        progressBar        = 0;
        settingsSideBar    = 0;
        splitter           = 0;        
        settingsLayout     = 0;
        settings           = 0;
        previewWidget      = 0;
        zoomBar            = 0;
    }

    TQGridLayout        *mainLayout;
    
    TQHButtonGroup      *separateView;
    
    TQString             settingsSection;
    
    TQWidget            *settings;
    TQWidget            *previewWidget;
    
    TQVBoxLayout        *settingsLayout;
        
    TQSplitter          *splitter;
    
    KProgress          *progressBar;
    
    ImageRegionWidget  *imageRegionWidget;
    ImagePanIconWidget *imagePanIconWidget;
    
    Sidebar            *settingsSideBar;

    StatusZoomBar      *zoomBar;
};
    
ImagePannelWidget::ImagePannelWidget(uint w, uint h, const TQString& settingsSection, 
                                     TQWidget *parent, int separateViewMode)
                 : TQHBox(parent, 0, TQt::WDestructiveClose)
{
    d = new ImagePannelWidgetPriv;
    d->settingsSection = settingsSection;
    d->splitter        = new TQSplitter(this);
    d->previewWidget   = new TQWidget(d->splitter);
    d->mainLayout      = new TQGridLayout( d->previewWidget, 2, 3, 0, KDialog::spacingHint());
    
    d->splitter->setFrameStyle( TQFrame::NoFrame );
    d->splitter->setFrameShadow( TQFrame::Plain );
    d->splitter->setFrameShape( TQFrame::NoFrame );    
    d->splitter->setOpaqueResize(false);

    // -------------------------------------------------------------

    TQFrame *preview      = new TQFrame(d->previewWidget);
    TQVBoxLayout* l1      = new TQVBoxLayout(preview, 5, 0);
    d->imageRegionWidget = new ImageRegionWidget(w, h, preview, false);
    d->imageRegionWidget->setFrameStyle(TQFrame::NoFrame);
    preview->setFrameStyle(TQFrame::Panel|TQFrame::Sunken);
    TQWhatsThis::add( d->imageRegionWidget, i18n("<p>Here you can see the original clip image "
                                                "which will be used for the preview computation."
                                                "<p>Click and drag the mouse cursor in the "
                                                "image to change the clip focus."));
    l1->addWidget(d->imageRegionWidget, 0);

    TQSizePolicy rightSzPolicy(TQSizePolicy::Preferred, TQSizePolicy::Expanding, 2, 1);
    d->previewWidget->setSizePolicy(rightSzPolicy);

    // -------------------------------------------------------------
    
    d->zoomBar = new StatusZoomBar(d->previewWidget);
    TQWhatsThis::add( d->zoomBar, i18n("<p>Here set the zoom factor of the preview area.") );

    // -------------------------------------------------------------
    
    TQString directory;
    d->separateView = new TQHButtonGroup(d->previewWidget);
    d->separateView->setExclusive(true);
    d->separateView->setInsideMargin( 0 );
    d->separateView->setFrameShape(TQFrame::NoFrame);
    
    if (separateViewMode == SeparateViewDuplicate ||
        separateViewMode == SeparateViewAll)
    {
       TQPushButton *duplicateHorButton = new TQPushButton( d->separateView );
       d->separateView->insert(duplicateHorButton, ImageRegionWidget::SeparateViewDuplicateHorz);
       TDEGlobal::dirs()->addResourceType("duplicatebothhorz", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
       directory = TDEGlobal::dirs()->findResourceDir("duplicatebothhorz", "duplicatebothhorz.png");
       duplicateHorButton->setPixmap( TQPixmap( directory + "duplicatebothhorz.png" ) );
       duplicateHorButton->setToggleButton(true);
       TQWhatsThis::add( duplicateHorButton, i18n("<p>If you enable this option, you will separate the preview area "
                                                 "horizontally, displaying the original and target image "
                                                 "at the same time. The target is duplicated from the original "
                                                 "below the red dashed line." ) );
        
       TQPushButton *duplicateVerButton = new TQPushButton( d->separateView );
       d->separateView->insert(duplicateVerButton, ImageRegionWidget::SeparateViewDuplicateVert);
       TDEGlobal::dirs()->addResourceType("duplicatebothvert", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
       directory = TDEGlobal::dirs()->findResourceDir("duplicatebothvert", "duplicatebothvert.png");
       duplicateVerButton->setPixmap( TQPixmap( directory + "duplicatebothvert.png" ) );
       duplicateVerButton->setToggleButton(true);
       TQWhatsThis::add( duplicateVerButton, i18n("<p>If you enable this option, you will separate the preview area "
                                                 "vertically, displaying the original and target image "
                                                 "at the same time. The target is duplicated from the original to "
                                                 "the right of the red dashed line." ) );
    }
        
    if (separateViewMode == SeparateViewNormal ||
        separateViewMode == SeparateViewAll)
    {
       TQPushButton *separateHorButton = new TQPushButton( d->separateView );
       d->separateView->insert(separateHorButton, ImageRegionWidget::SeparateViewHorizontal);
       TDEGlobal::dirs()->addResourceType("bothhorz", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
       directory = TDEGlobal::dirs()->findResourceDir("bothhorz", "bothhorz.png");
       separateHorButton->setPixmap( TQPixmap( directory + "bothhorz.png" ) );
       separateHorButton->setToggleButton(true);
       TQWhatsThis::add( separateHorButton, i18n( "<p>If you enable this option, you will separate the preview area "
                                                 "horizontally, displaying the original and target image "
                                                 "at the same time. The original is above the "
                                                 "red dashed line, the target below it." ) );
        
       TQPushButton *separateVerButton = new TQPushButton( d->separateView );
       d->separateView->insert(separateVerButton, ImageRegionWidget::SeparateViewVertical);
       TDEGlobal::dirs()->addResourceType("bothvert", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
       directory = TDEGlobal::dirs()->findResourceDir("bothvert", "bothvert.png");
       separateVerButton->setPixmap( TQPixmap( directory + "bothvert.png" ) );
       separateVerButton->setToggleButton(true);
       TQWhatsThis::add( separateVerButton, i18n( "<p>If you enable this option, you will separate the preview area "
                                                 "vertically, displaying the original and target image "
                                                 "at the same time. The original is to the left of the "
                                                 "red dashed line, the target to the right of it." ) );
    }
       
    TQPushButton *noSeparateButton = new TQPushButton( d->separateView );
    d->separateView->insert(noSeparateButton, ImageRegionWidget::SeparateViewNone);
    TDEGlobal::dirs()->addResourceType("target", TDEGlobal::dirs()->kde_default("data") + "digikam/data");
    directory = TDEGlobal::dirs()->findResourceDir("target", "target.png");
    noSeparateButton->setPixmap( TQPixmap( directory + "target.png" ) );
    noSeparateButton->setToggleButton(true);
    TQWhatsThis::add( noSeparateButton, i18n( "<p>If you enable this option, the preview area will not "
                                             "be separated." ) );
    
    // -------------------------------------------------------------
    
    d->progressBar = new KProgress(100, d->previewWidget);
    TQWhatsThis::add(d->progressBar ,i18n("<p>This is the percentage of the task which has been completed up to this point."));
    d->progressBar->setProgress(0);
    d->progressBar->setMaximumHeight( fontMetrics().height() );

    // -------------------------------------------------------------
        
    d->mainLayout->addMultiCellWidget(preview, 0, 1, 0, 3);
    d->mainLayout->addMultiCellWidget(d->zoomBar, 2, 2, 0, 0);
    d->mainLayout->addMultiCellWidget(d->progressBar, 2, 2, 2, 2);
    d->mainLayout->addMultiCellWidget(d->separateView, 2, 2, 3, 3);
    
    d->mainLayout->setRowStretch(1, 10);
    d->mainLayout->setColStretch(1, 10);
        
    // -------------------------------------------------------------

    TQString sbName(d->settingsSection + TQString(" Image Plugin Sidebar"));
    d->settingsSideBar = new Sidebar(this, sbName.ascii(), Sidebar::Right);
    d->settingsSideBar->setSplitter(d->splitter);
    
    d->settings       = new TQWidget(d->settingsSideBar);
    d->settingsLayout = new TQVBoxLayout(d->settings);    
    
    TQFrame *frame3 = new TQFrame(d->settings);
    frame3->setFrameStyle(TQFrame::Panel|TQFrame::Sunken);
    TQVBoxLayout* l3 = new TQVBoxLayout(frame3, 5, 0);
    d->imagePanIconWidget = new ImagePanIconWidget(360, 240, frame3);
    TQWhatsThis::add( d->imagePanIconWidget, i18n("<p>Here you can see the original image panel "
                                                 "which can help you to select the clip preview."
                                                 "<p>Click and drag the mouse cursor in the "
                                                 "red rectangle to change the clip focus."));
    l3->addWidget(d->imagePanIconWidget, 0, TQt::AlignCenter);

    d->settingsLayout->addWidget(frame3, 0, TQt::AlignHCenter);
    d->settingsLayout->addSpacing(KDialog::spacingHint());

    d->settingsSideBar->appendTab(d->settings, SmallIcon("configure"), i18n("Settings"));    
    d->settingsSideBar->loadViewState();

    // -------------------------------------------------------------
    
    setProgressVisible(false);
    TQTimer::singleShot(0, this, TQ_SLOT(slotInitGui())); 
    
    // -------------------------------------------------------------
    
    connect(d->imageRegionWidget, TQ_SIGNAL(signalContentsMovedEvent(bool)),
            this, TQ_SLOT(slotOriginalImageRegionChanged(bool)));

    connect(d->imagePanIconWidget, TQ_SIGNAL(signalSelectionMoved(const TQRect&, bool)),
            this, TQ_SLOT(slotSetImageRegionPosition(const TQRect&, bool)));

    connect(d->imagePanIconWidget, TQ_SIGNAL(signalSelectionTakeFocus()),
            this, TQ_SLOT(slotPanIconTakeFocus()));
            
    connect(d->separateView, TQ_SIGNAL(released(int)),
            d->imageRegionWidget, TQ_SLOT(slotSeparateViewToggled(int)));
    
    connect(d->separateView, TQ_SIGNAL(released(int)),
            d->imagePanIconWidget, TQ_SLOT(slotSeparateViewToggled(int)));

    connect(d->zoomBar, TQ_SIGNAL(signalZoomMinusClicked()),
            d->imageRegionWidget, TQ_SLOT(slotDecreaseZoom()));

    connect(d->zoomBar, TQ_SIGNAL(signalZoomPlusClicked()),
            d->imageRegionWidget, TQ_SLOT(slotIncreaseZoom()));

    connect(d->zoomBar, TQ_SIGNAL(signalZoomSliderReleased(int)),
            this, TQ_SLOT(slotZoomSliderChanged(int)));
}

ImagePannelWidget::~ImagePannelWidget()
{
    writeSettings();
    delete d->settingsSideBar;
    delete d;
}

void ImagePannelWidget::readSettings()
{
    TDEConfig *config = tdeApp->config();
    config->setGroup(d->settingsSection);
    int mode = config->readNumEntry("Separate View", ImageRegionWidget::SeparateViewDuplicateVert);
    mode     = TQMAX(ImageRegionWidget::SeparateViewHorizontal, mode);
    mode     = TQMIN(ImageRegionWidget::SeparateViewDuplicateHorz, mode);
    
    d->imageRegionWidget->blockSignals(true);
    d->imagePanIconWidget->blockSignals(true);
    d->separateView->blockSignals(true);
    d->imageRegionWidget->slotSeparateViewToggled( mode );
    d->imagePanIconWidget->slotSeparateViewToggled( mode );
    d->separateView->setButton( mode );
    d->imageRegionWidget->blockSignals(false);
    d->imagePanIconWidget->blockSignals(false);
    d->separateView->blockSignals(false);
}
    
void ImagePannelWidget::writeSettings()
{
    TDEConfig *config = tdeApp->config();
    config->setGroup(d->settingsSection);
    config->writeEntry( "Separate View", d->separateView->selectedId() );
    config->sync();
}

void ImagePannelWidget::slotOriginalImageRegionChanged(bool target)
{
    slotZoomFactorChanged(d->imageRegionWidget->zoomFactor());
    TQRect rect = getOriginalImageRegion();
    d->imagePanIconWidget->setRegionSelection(rect);
    updateSelectionInfo(rect);

    if (target)
    {
        d->imageRegionWidget->backupPixmapRegion();
        emit signalOriginalClipFocusChanged();
    }
}

void ImagePannelWidget::slotZoomFactorChanged(double zoom)
{
    double h    = (double)ThumbnailSize::Huge;
    double s    = (double)ThumbnailSize::Small;
    double zmin = d->imageRegionWidget->zoomMin();
    double zmax = d->imageRegionWidget->zoomMax();
    double b    = (zmin-(zmax*s/h))/(1-s/h);
    double a    = (zmax-b)/h;
    int size    = (int)((zoom - b) /a); 

    d->zoomBar->setZoomSliderValue(size);
    d->zoomBar->setZoomTrackerText(i18n("zoom: %1%").arg((int)(zoom*100.0)));

    d->zoomBar->setEnableZoomPlus(true);
    d->zoomBar->setEnableZoomMinus(true);

    if (d->imageRegionWidget->maxZoom())
        d->zoomBar->setEnableZoomPlus(false);

    if (d->imageRegionWidget->minZoom())
        d->zoomBar->setEnableZoomMinus(false);

    d->imagePanIconWidget->slotZoomFactorChanged(zoom);
}

void ImagePannelWidget::slotZoomSliderChanged(int size)
{
    double h    = (double)ThumbnailSize::Huge;
    double s    = (double)ThumbnailSize::Small;
    double zmin = d->imageRegionWidget->zoomMin();
    double zmax = d->imageRegionWidget->zoomMax();
    double b    = (zmin-(zmax*s/h))/(1-s/h);
    double a    = (zmax-b)/h;
    double z    = a*size+b; 

    d->imageRegionWidget->setZoomFactorSnapped(z);
}

KProgress *ImagePannelWidget::progressBar()
{
    return d->progressBar;
}

void ImagePannelWidget::resizeEvent(TQResizeEvent *)
{
    emit signalResized();
}

void ImagePannelWidget::slotInitGui()
{
    readSettings();
    setCenterImageRegionPosition();
    slotOriginalImageRegionChanged(true);
}

void ImagePannelWidget::setPanIconHighLightPoints(const TQPointArray& pt) 
{
    d->imageRegionWidget->setHighLightPoints(pt);
    d->imagePanIconWidget->setHighLightPoints(pt);
}

void ImagePannelWidget::slotPanIconTakeFocus()
{
    d->imageRegionWidget->restorePixmapRegion();
}

void ImagePannelWidget::setUserAreaWidget(TQWidget *w)
{
    w->reparent( d->settings, TQPoint(0, 0) );
    d->settingsLayout->addSpacing(KDialog::spacingHint());
    d->settingsLayout->addWidget(w);
    d->settingsLayout->addStretch();
}

void ImagePannelWidget::setEnable(bool b)
{
    d->imageRegionWidget->setEnabled(b);
    d->imagePanIconWidget->setEnabled(b);
    d->separateView->setEnabled(b);
    d->zoomBar->setEnabled(b);
}

void ImagePannelWidget::setProgress(int val)
{
    d->progressBar->setValue(val);
}

void ImagePannelWidget::setProgressVisible(bool b)
{
    if (b) d->progressBar->show();
    else d->progressBar->hide();
}

void ImagePannelWidget::setProgressWhatsThis(const TQString& desc)
{
    TQWhatsThis::add( d->progressBar, desc);
}

void ImagePannelWidget::setPreviewImageWaitCursor(bool enable)
{
    if ( enable )
       d->imageRegionWidget->setCursor( KCursor::waitCursor() );
    else 
       d->imageRegionWidget->unsetCursor();
}

TQRect ImagePannelWidget::getOriginalImageRegion()
{
    return ( d->imageRegionWidget->getImageRegion() );
}

TQRect ImagePannelWidget::getOriginalImageRegionToRender()
{
    return ( d->imageRegionWidget->getImageRegionToRender() );
}

DImg ImagePannelWidget::getOriginalRegionImage()
{
    return ( d->imageRegionWidget->getImageRegionImage() );
}

void ImagePannelWidget::setPreviewImage(DImg img)
{
    d->imageRegionWidget->updatePreviewImage(&img);
}    

void ImagePannelWidget::setCenterImageRegionPosition()
{
    d->imageRegionWidget->setCenterContentsPosition();
}

void ImagePannelWidget::slotSetImageRegionPosition(const TQRect& rect, bool targetDone)
{
    d->imageRegionWidget->setContentsPosition(rect.x(), rect.y(), targetDone);
}

void ImagePannelWidget::updateSelectionInfo(const TQRect& rect)
{
    TQToolTip::add( d->imagePanIconWidget,
                   i18n("<nobr>(%1,%2)(%3x%4)</nobr>")
                        .arg(rect.left()).arg(rect.top())
                        .arg(rect.width()).arg(rect.height()));
}

}  // NameSpace Digikam
