/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::common::f_fmla;
use crate::polyeval::f_polyeval5;
use crate::sin_cosf::sincosf_eval::sincosf_eval;

/// Sine function
///
/// Max found ULP 0.5
#[inline]
pub fn f_sinf(x: f32) -> f32 {
    let x_abs = x.to_bits() & 0x7fff_ffffu32;
    let xd = x as f64;

    // |x| <= pi/16
    if x_abs <= 0x3e49_0fdbu32 {
        // |x| < 0.000443633
        if x_abs < 0x39e8_9769u32 {
            if x_abs == 0u32 {
                // For signed zeros.
                return x;
            }
            #[cfg(any(
                all(
                    any(target_arch = "x86", target_arch = "x86_64"),
                    target_feature = "fma"
                ),
                all(target_arch = "aarch64", target_feature = "neon")
            ))]
            {
                use crate::common::f_fmlaf;
                return f_fmlaf(x, f32::from_bits(0xb3000000), x);
            }
            #[cfg(not(any(
                all(
                    any(target_arch = "x86", target_arch = "x86_64"),
                    target_feature = "fma"
                ),
                all(target_arch = "aarch64", target_feature = "neon")
            )))]
            {
                return f_fmla(xd, f64::from_bits(0xbe60000000000000), xd) as f32;
            }
        }

        let xsqr = xd * xd;

        /*
           Generated by Sollya:
           f_sinpi_16 = sin(x)/x;
           Q = fpminimax(f_sinpi_16, [|0, 2, 4, 6, 8|], [|1, D...|], [0, pi/16]);

           See ./notes/sinf.sollya
        */
        let p = f_polyeval5(
            xsqr,
            f64::from_bits(0x3ff0000000000000),
            f64::from_bits(0xbfc55555555554c6),
            f64::from_bits(0x3f81111111085e65),
            f64::from_bits(0xbf2a019f70fb4d4f),
            f64::from_bits(0x3ec718d179815e74),
        );
        return (xd * p) as f32;
    }

    if x_abs >= 0x7f80_0000u32 {
        return x + f32::NAN;
    }

    // Formula:
    //   sin(x) = sin((k + y)*pi/32)
    //          = sin(y*pi/32) * cos(k*pi/32) + cos(y*pi/32) * sin(k*pi/32)
    // The values of sin(k*pi/32) and cos(k*pi/32) for k = 0..31 are precomputed
    // and stored using a vector of 32 doubles. Sin(y*pi/32) and cos(y*pi/32) are
    // computed using degree-7 and degree-6 minimax polynomials generated by
    // Sollya respectively.

    let rs = sincosf_eval(xd, x_abs);
    f_fmla(rs.sin_y, rs.cos_k, f_fmla(rs.cosm1_y, rs.sin_k, rs.sin_k)) as f32
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn f_sinf_test() {
        assert_eq!(f_sinf(0.0), 0.0);
        assert_eq!(f_sinf(1.0), 0.84147096);
        assert_eq!(f_sinf(0.3), 0.29552022);
        assert_eq!(f_sinf(-1.0), -0.84147096);
        assert_eq!(f_sinf(-0.3), -0.29552022);
        assert_eq!(f_sinf(std::f32::consts::PI / 2.), 1.);
        assert!(f_sinf(f32::INFINITY).is_nan());
        assert!(f_sinf(f32::NEG_INFINITY).is_nan());
        assert!((f_sinf(std::f32::consts::PI) - 0f32).abs() < 1e-6);
        assert!((f_sinf(std::f32::consts::FRAC_PI_2) - 1f32).abs() < 1e-6);
    }
}
